/*
 *   This file is part of AkariXB
 *   Copyright 2015-2018  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "activityhandler.h"


ActivityHandler::ActivityHandler(GlobalObject *globalObject,
                                 QObject *parent) : QObject(parent)
{
    m_globalObject = globalObject;

    m_xmppClient = m_globalObject->getXmppClient();
    m_mucManager = m_globalObject->getMucManager();
    m_variableParser = m_globalObject->getVariableParser();


    // Start main timer, ticking once a minute, to check if an activity needs to be started
    m_mainTimer = new QTimer(this);
    m_mainTimer->setInterval(60000); // 60 seconds
    connect(m_mainTimer, &QTimer::timeout,
            this, &ActivityHandler::onMainTimerTick);
    m_mainTimer->start();


    // Activity-specific timer; signals the end of the activity
    m_currentActivityTimer = new QTimer(this);
    m_currentActivityTimer->setSingleShot(true);
    connect(m_currentActivityTimer, &QTimer::timeout,
            this, &ActivityHandler::activityFinished);

    qDebug() << "ActivityHandler created";
}

ActivityHandler::~ActivityHandler()
{
    qDebug() << "ActivityHandler destroyed";
}



void ActivityHandler::setActivityData(QVariantList dataList)
{
    m_activityDataList = dataList;
}


QXmppPresence::AvailableStatusType ActivityHandler::presenceFromIndex(int presenceType)
{
    QXmppPresence::AvailableStatusType statusType;

    switch (presenceType)
    {
    case 1:
        statusType = QXmppPresence::Chat;
        break;

    case 2:
        statusType = QXmppPresence::Away;
        break;

    case 3:
        statusType = QXmppPresence::XA;
        break;

    case 4:
        statusType = QXmppPresence::DND;
        break;

    default:
        statusType = QXmppPresence::Online;
    }

    return statusType;
}


//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


void ActivityHandler::onMainTimerTick()
{
    QTime timeNow = QTime::currentTime();
    QDate dateNow = QDate::currentDate();

    if (timeNow.hour() == 0 && timeNow.minute() == 0) // FIXME: ensure it's not skipped
    {
        qDebug() << "--AH DEBUG-- Midnight! Resetting DoneToday values";
        m_doneToday.clear();
    }

    if (!m_activityInProgress.isEmpty())
    {
        return;
    }


    foreach (QVariant mapVariant, m_activityDataList)
    {
        QVariantMap map = mapVariant.toMap();

        const QString activityName = map.value("name").toString();

        bool shouldDoIt = true; // Until proven guilty

        const int timesDoneToday = m_doneToday.value(activityName).toInt();
        if (timesDoneToday >= map.value("timesPerDay").toInt())
        {
            shouldDoIt = false;
        }

        if (shouldDoIt)
        {
            // FIXME: handle cases such as 22:00 to 02:00
            if (map.value("minTime").toTime() > timeNow
             || map.value("maxTime").toTime() < timeNow)
            {
                shouldDoIt = false;
            }

            // TODO: Check also dateNow
        }

        if (shouldDoIt)
        {
            shouldDoIt = Helpers::probability(map.value("probability").toInt());
        }



        if (shouldDoIt) // It passed all the trials, and survived!
        {
            m_activityInProgress = activityName;
            m_doneToday.insert(activityName, timesDoneToday + 1);

            const int duration = Helpers::randomBetween(map.value("minDuration").toInt(),
                                                        map.value("maxDuration").toInt());

            QTime durationTime;
            durationTime.setHMS(0, 0, 0); // Qt 5.x requires this initialization
            durationTime = durationTime.addSecs(duration * 60);

            QString durationString;
            if (durationTime.hour() > 0)
            {
                durationString.append(tr("%Ln hour(s)", "",
                                         durationTime.hour()));
            }

            if (durationTime.minute() > 0)
            {
                if (!durationString.isEmpty())
                {
                    durationString.append(", ");
                }

                durationString.append(tr("%Ln minute(s)", "",
                                         durationTime.minute()));
            }

                                         // Up to 9 extra seconds, randomly
            const int extraMiliseconds = Helpers::randomBetween(0, 9000);

            m_currentActivityTimer->start(duration * 60000    // Mins to msecs
                                          + extraMiliseconds);


            QTime endTime = timeNow.addSecs(duration * 60);

            m_globalObject->addToLog(tr("Starting activity %1, which "
                                        "will end at %2.")
                                     .arg("'" + m_activityInProgress + "'")
                                     .arg(endTime.toString(Qt::SystemLocaleShortDate))
                                     + " [" + durationString + "]");


            QString timeRange = timeNow.toString(Qt::SystemLocaleShortDate)
                                + QStringLiteral(" ~ ")
                                + endTime.toString(Qt::SystemLocaleShortDate);

            m_globalObject->notifyActivityChange(m_activityInProgress, timeRange);


            // Status type and message
            QXmppPresence presence = m_xmppClient->clientPresence();
            m_oldStatusType = presence.availableStatusType();
            m_oldStatusMessage = presence.statusText();

            presence.setAvailableStatusType(this->presenceFromIndex(map.value("statusType")
                                                                        .toInt()));

            QString statusMsg = Helpers::randomString(map.value("statusMessages")
                                                         .toStringList());
            statusMsg = m_variableParser->getParsed(statusMsg);
            presence.setStatusText(statusMsg);


            m_messagesAfter = map.value("msgAfter").toStringList();


            if (m_globalObject->connectedToServer())
            {
                // Send the new presence
                m_xmppClient->setClientPresence(presence);


                // Send messages to configured recipients about the activity
                m_allRecipientJids.clear();

                const int msgToRooms = map.value("msgToRooms").toInt();
                if (msgToRooms == 0)       // All rooms
                {
                    m_allRecipientJids = m_globalObject->getJoinedRooms();
                }
                else if (msgToRooms == 1)  // Several rooms
                {
                    m_allRecipientJids = Helpers::someStrings(m_globalObject->getJoinedRooms(),
                                                              50);
                }
                else if (msgToRooms == 2)  // A few rooms
                {
                    m_allRecipientJids = Helpers::someStrings(m_globalObject->getJoinedRooms(),
                                                              20);
                }


                // TODO: add JIDs of active private chats


                // Additional JIDs can be added regardless of the other options
                m_allRecipientJids += map.value("specificJids")
                                         .toString()
                                         .split(QStringLiteral(" "));

                QStringList messagesBefore = map.value("msgBefore")
                                                .toStringList();
                QString msgBefore;
                foreach (QString recipient, m_allRecipientJids)
                {
                    msgBefore = Helpers::randomString(messagesBefore);
                    msgBefore = m_variableParser->getParsed(msgBefore);

                    if (msgBefore != QStringLiteral("*")) // TMP FIXME?
                    {
                        m_globalObject->sendMessageToJid(recipient, msgBefore);
                    }
                }
            }

            m_globalObject->notifyStatusChange(presence);


            break;
        }
    }
}


void ActivityHandler::activityFinished()
{
    m_globalObject->addToLog(tr("Activity ended: %1.").arg(m_activityInProgress));

    m_activityInProgress.clear();
    m_globalObject->notifyActivityChange("[ " + tr("None",
                                                   "Regarding an activity")
                                         + " ]",
                                         QString()); // No duration

    QXmppPresence presence = m_xmppClient->clientPresence();
    presence.setAvailableStatusType(m_oldStatusType);
    presence.setStatusText(m_oldStatusMessage);

    if (m_globalObject->connectedToServer())
    {
        m_xmppClient->setClientPresence(presence);

        // Send messages to configured recipients, about the end of the activity
        QString msgAfter;
        foreach (QString recipient, m_allRecipientJids)
        {
            msgAfter = Helpers::randomString(m_messagesAfter);
            msgAfter = m_variableParser->getParsed(msgAfter);

            if (msgAfter != QStringLiteral("*"))  // TMP FIXME?
            {
                m_globalObject->sendMessageToJid(recipient, msgAfter);
            }
        }
    }

    m_globalObject->notifyStatusChange(presence);

    // Cleanup
    m_allRecipientJids.clear();
    m_messagesAfter.clear();
}

