/*
 *   This file is part of AkariXB
 *   Copyright 2015-2016  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "activityhandler.h"


ActivityHandler::ActivityHandler(GlobalObject *globalObject,
                                 QObject *parent) : QObject(parent)
{
    this->globalObj = globalObject;

    this->xmppClient = this->globalObj->getXmppClient();
    this->mucManager = this->globalObj->getMucManager();
    this->varParser = this->globalObj->getVariableParser();


    // Start main timer, ticking once a minute, to check if an activity needs to be started
    this->mainTimer = new QTimer(this);
    mainTimer->setInterval(60000); // 60 seconds
    connect(mainTimer, SIGNAL(timeout()),
            this, SLOT(onMainTimerTick()));
    mainTimer->start();


    // Activity-specific timer; signals the end of the activity
    this->currentActivityTimer = new QTimer(this);
    currentActivityTimer->setSingleShot(true);
    connect(currentActivityTimer, SIGNAL(timeout()),
            this, SLOT(activityFinished()));

    qDebug() << "ActivityHandler created";
}

ActivityHandler::~ActivityHandler()
{
    qDebug() << "ActivityHandler destroyed";
}



void ActivityHandler::setActivityData(QVariantList dataList)
{
    this->activityDataList = dataList;
}


QXmppPresence::AvailableStatusType ActivityHandler::presenceFromIndex(int presenceType)
{
    QXmppPresence::AvailableStatusType statusType;

    switch (presenceType)
    {
    case 1:
        statusType = QXmppPresence::Chat;
        break;

    case 2:
        statusType = QXmppPresence::Away;
        break;

    case 3:
        statusType = QXmppPresence::XA;
        break;

    case 4:
        statusType = QXmppPresence::DND;
        break;

    default:
        statusType = QXmppPresence::Online;
    }

    return statusType;
}


//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


void ActivityHandler::onMainTimerTick()
{
    qDebug() << "--AH DEBUG-- Main timer TICK";

    QTime timeNow = QTime::currentTime();
    QDate dateNow = QDate::currentDate();

    if (timeNow.hour() == 0 && timeNow.minute() == 0) // FIXME: ensure it's not skipped
    {
        qDebug() << "--AH DEBUG-- Midnight! Resetting DoneToday values";
        this->doneToday.clear();
    }

    if (!activityInProgress.isEmpty())
    {
        qDebug() <<"--AH DEBUG-- Activity already in progress: " << activityInProgress;
        return;
    }


    foreach (QVariant mapVariant, this->activityDataList)
    {
        QVariantMap map = mapVariant.toMap();

        QString activityName = map.value("name").toString();
        qDebug() << "Activity:" << activityName;

        bool shouldDoIt = true; // Until proven guilty

        int timesDoneToday = this->doneToday.value(activityName).toInt();
        qDebug() << "Times done today:" << timesDoneToday;
        if (timesDoneToday >= map.value("timesPerDay").toInt())
        {
            qDebug() << "Done enough times today; not considering";
            shouldDoIt = false;
        }

        if (shouldDoIt)
        {
            // FIXME: handle cases such as 22:00 to 02:00
            if (map.value("minTime").toTime() > timeNow
             || map.value("maxTime").toTime() < timeNow)
            {
                qDebug() << "This is not the time for this activity; not considering";
                shouldDoIt = false;
            }

            // TODO: Check also dateNow
        }

        if (shouldDoIt)
        {
            shouldDoIt = Helpers::probability(map.value("probability").toInt());
            qDebug() << "This can still be done today";
            qDebug() << "Probability:" << map.value("probability").toInt() << "%";
            qDebug() << "Should still do it, based on that probability?" << shouldDoIt;
        }



        if (shouldDoIt) // It passed all the trials, and survived!
        {
            this->activityInProgress = activityName;
            this->doneToday.insert(activityName, timesDoneToday + 1);

            int duration = Helpers::randomBetween(map.value("minDuration").toInt(),
                                                  map.value("maxDuration").toInt());

            QTime durationTime;
            durationTime.setHMS(0, 0, 0); // Qt 5.x requires this initialization
            durationTime = durationTime.addSecs(duration * 60);

            QString durationString;
            if (durationTime.hour() > 0)
            {
                durationString.append(tr("%Ln hour(s)", "",
                                         durationTime.hour()));
            }

            if (durationTime.minute() > 0)
            {
                if (!durationString.isEmpty())
                {
                    durationString.append(", ");
                }

                durationString.append(tr("%Ln minute(s)", "",
                                         durationTime.minute()));
            }


            this->currentActivityTimer->start(duration * 60000); // mins to msecs


            QTime endTime = timeNow.addSecs(duration * 60);

            this->globalObj->addToLog(tr("Starting activity %1, which "
                                         "will end at %2.")
                                      .arg("'" + activityInProgress + "'")
                                      .arg(endTime.toString(Qt::SystemLocaleShortDate))
                                      + " [" + durationString + "]");


            QString timeRange = timeNow.toString(Qt::SystemLocaleShortDate)
                                + " ~ "
                                + endTime.toString(Qt::SystemLocaleShortDate);

            this->globalObj->notifyActivityChange(activityInProgress, timeRange);


            // Status type and message
            QXmppPresence presence = this->xmppClient->clientPresence();
            this->oldStatusType = presence.availableStatusType();
            this->oldStatusMessage = presence.statusText();

            presence.setAvailableStatusType(this->presenceFromIndex(map.value("statusType")
                                                                        .toInt()));

            QString statusMsg = Helpers::randomString(map.value("statusMessages")
                                                         .toStringList());
            statusMsg = this->varParser->getParsed(statusMsg);

            this->messageAfter = Helpers::randomString(map.value("msgAfter")
                                                          .toStringList());


            presence.setStatusText(statusMsg);

            if (this->globalObj->connectedToServer())
            {
                // Send the new presence
                this->xmppClient->setClientPresence(presence);


                // Send messages to configured recipients about the activity
                QString recipient;
                this->allRecipientJids = map.value("specificJids")
                                            .toString().split(" ");
                // FIXME: For now, only "specific JIDs" option is used
                if (!allRecipientJids.isEmpty())
                {
                    recipient = allRecipientJids.first(); // TMP, tests, FIXME
                }

                QString msgBefore = Helpers::randomString(map.value("msgBefore")
                                                             .toStringList());
                msgBefore = this->varParser->getParsed(msgBefore);

                this->globalObj->sendMessageToJid(recipient, msgBefore);
            }

            this->globalObj->notifyStatusMessageChange(statusMsg);


            break;
        }
    }
}


void ActivityHandler::activityFinished()
{
    this->globalObj->addToLog(tr("Activity ended: %1.").arg(activityInProgress));

    activityInProgress.clear();
    this->globalObj->notifyActivityChange("[ " + tr("None",
                                                    "Regarding an activity")
                                          + " ]",
                                          ""); // No duration

    QXmppPresence presence = this->xmppClient->clientPresence();
    presence.setAvailableStatusType(this->oldStatusType);
    presence.setStatusText(this->oldStatusMessage);

    if (this->globalObj->connectedToServer())
    {
        this->xmppClient->setClientPresence(presence);

        // Send messages to configured recipients about the end of the activity
        QString recipient;
        // FIXME: For now, only "specific JIDs" option is used
        if (!allRecipientJids.isEmpty())
        {
            recipient = allRecipientJids.first(); // TMP, tests, FIXME
        }

        messageAfter = this->varParser->getParsed(this->messageAfter);
        this->globalObj->sendMessageToJid(recipient, this->messageAfter);
    }

    this->globalObj->notifyStatusMessageChange(this->oldStatusMessage);

    // Cleanup
    this->allRecipientJids.clear();
    this->messageAfter.clear();
}

