/*
 *   This file is part of AkariXB
 *   Copyright 2015-2016  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "roommodule.h"


RoomModule::RoomModule(GlobalObject *globalObject,
                       QWidget *parent) : QWidget(parent)
{
    this->globalObj = globalObject;

    this->xmppClient = this->globalObj->getXmppClient();
    this->mucManager = this->globalObj->getMucManager();


    this->dataFile = new DataFile(globalObj->getDataDirectory()
                                  + "/rooms.axb",
                                  this);


    // List of configured rooms
    this->roomListWidget = new QListWidget(this);
    roomListWidget->setDragDropMode(QAbstractItemView::InternalMove); // Movable items
    connect(roomListWidget, SIGNAL(currentItemChanged(QListWidgetItem*,QListWidgetItem*)), // 2nd arg ignored
            this, SLOT(showRoomDetails(QListWidgetItem*)));



    // Buttons to manage rooms
    this->addRoomButton = new QPushButton(QIcon::fromTheme("list-add",
                                                           QIcon(":/images/list-add.png")),
                                          tr("Add &New Room"),
                                          this);
    addRoomButton->setFlat(true);
    connect(addRoomButton, SIGNAL(clicked()),
            this, SLOT(addRoom()));


    this->removeRoomButton = new QPushButton(QIcon::fromTheme("list-remove",
                                                              QIcon(":/images/list-remove.png")),
                                             tr("&Remove Room"),
                                             this);
    removeRoomButton->setFlat(true);
    removeRoomButton->setDisabled(true);
    connect(removeRoomButton, SIGNAL(clicked()),
            this, SLOT(removeRoom()));


    this->buttonsLayout = new QHBoxLayout();
    buttonsLayout->addWidget(addRoomButton);
    buttonsLayout->addStretch();
    buttonsLayout->addWidget(removeRoomButton);


    // Room details
    this->roomJidLineEdit = new QLineEdit(this);
    roomJidLineEdit->setPlaceholderText(tr("Room address"));

    this->nickLineEdit = new QLineEdit(this);
    nickLineEdit->setPlaceholderText(tr("Nickname"));
    connect(roomJidLineEdit, SIGNAL(returnPressed()),
            nickLineEdit, SLOT(setFocus()));

    this->autojoinCheckbox = new QCheckBox(tr("Autojoin"), this);


    this->joinButton = new QPushButton(QIcon::fromTheme("list-add",
                                                        QIcon(":/images/list-add.png")),
                                       tr("Join"),
                                       this);
    joinButton->hide();
    connect(joinButton, SIGNAL(clicked()),
            this, SLOT(joinSelectedRoom()));

    this->leaveButton = new QPushButton(QIcon::fromTheme("list-remove",
                                                         QIcon(":/images/list-remove.png")),
                                        tr("Leave"),
                                        this);
    leaveButton->hide();
    connect(leaveButton, SIGNAL(clicked()),
            this, SLOT(leaveSelectedRoom()));


    this->editRoomButton = new QPushButton(QIcon::fromTheme("document-edit",
                                                            QIcon(":/images/button-edit.png")),
                                           tr("&Edit"),
                                           this);
    editRoomButton->setDisabled(true);
    connect(editRoomButton, SIGNAL(clicked()),
            this, SLOT(editRoom()));

    this->updateRoomButton = new QPushButton(QIcon::fromTheme("view-refresh",
                                                              QIcon(":/images/button-refresh.png")),
                                             tr("&Update"),
                                             this);
    connect(updateRoomButton, SIGNAL(clicked()),
            this, SLOT(updateRoom()));
    connect(nickLineEdit, SIGNAL(returnPressed()),
            updateRoomButton, SLOT(setFocus()));


    // Stacked widget to hold ChatWidgets
    this->firstPageLabel = new QLabel("<big>"
                                      + tr("Select a room from the list above")
                                      + "</big>",
                                      this);
    firstPageLabel->setAlignment(Qt::AlignCenter);

    this->chatStackedWidget = new QStackedWidget(this);
    chatStackedWidget->addWidget(firstPageLabel);


    this->detailsUpperLayout = new QHBoxLayout();
    detailsUpperLayout->addWidget(roomJidLineEdit, 3);
    detailsUpperLayout->addWidget(nickLineEdit,    2);
    detailsUpperLayout->addWidget(autojoinCheckbox);
    detailsUpperLayout->addStretch();
    detailsUpperLayout->addSpacing(8);
    detailsUpperLayout->addWidget(joinButton);
    detailsUpperLayout->addWidget(leaveButton);
    detailsUpperLayout->addWidget(editRoomButton);
    detailsUpperLayout->addWidget(updateRoomButton);


    this->detailsLayout = new QVBoxLayout();
    detailsLayout->addLayout(detailsUpperLayout);
    detailsLayout->addWidget(chatStackedWidget);

    this->detailsGroupbox = new QGroupBox(tr("Room Details"), this);
    detailsGroupbox->setLayout(detailsLayout);


    this->mainLayout = new QVBoxLayout();
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->addWidget(roomListWidget,  1);
    mainLayout->addLayout(buttonsLayout);
    mainLayout->addWidget(detailsGroupbox, 3);
    this->setLayout(mainLayout);


    this->loadRoomInfo();
    this->toggleRoomDetails(false);

    qDebug() << "RoomModule created";
}


RoomModule::~RoomModule()
{
    this->saveRoomInfo();

    qDebug() << "RoomModule destroyed";
}



void RoomModule::loadRoomInfo()
{
    QVariantList roomDataList = this->dataFile->loadData();

    foreach (QVariant listItem, roomDataList)
    {
        QVariantMap map = listItem.toMap();

        QString roomJid = map.value("jid").toString();
        QString roomNickname = map.value("nickname").toString();
        bool roomAutojoin = map.value("autojoin").toBool();
        QString itemString = this->roomDescriptionLine(roomJid,
                                                       roomNickname,
                                                       roomAutojoin);

        QListWidgetItem *item = new QListWidgetItem(QIcon::fromTheme("user-offline",
                                                                     QIcon(":/images/button-offline.png")),
                                                    itemString);
        item->setData(Qt::UserRole,     roomJid);
        item->setData(Qt::UserRole + 1, roomNickname);
        item->setData(Qt::UserRole + 2, roomAutojoin);
        item->setData(Qt::UserRole + 3, false); // Joined status, initially false

        roomListWidget->addItem(item);
    }

    this->globalObj->addToLog(tr("Loaded %1 rooms.")
                              .arg(roomDataList.length()));
}


void RoomModule::saveRoomInfo()
{
    QVariantList roomDataList;

    for (int counter = 0; counter < roomListWidget->count(); ++counter)
    {
        QListWidgetItem *item = roomListWidget->item(counter);

        QVariantMap roomMap;
        roomMap.insert("jid",        item->data(Qt::UserRole).toString());
        roomMap.insert("nickname",   item->data(Qt::UserRole + 1).toString());
        roomMap.insert("autojoin",   item->data(Qt::UserRole + 2).toBool());

        roomDataList.append(roomMap);
    }

    this->dataFile->saveData(roomDataList);

    // FIXME: This might lead to segfault when called while quitting
    //this->globalObj->addToLog(tr("Saved %1 rooms.")
    //                          .arg(roomDataList.length()));
}



QString RoomModule::roomDescriptionLine(QString roomJid, QString nickname,
                                        bool autojoin)
{
    QString description = tr("%1 as %2").arg(roomJid).arg(nickname);
    if (autojoin)
    {
        description.append(QString::fromUtf8("              \342\234\224  ") // Check mark
                           + "("
                           + tr("Autojoin")
                           + ")");
    }

    return description;
}


/*
 * Find out which item from the list matches a given JID
 *
 */
QListWidgetItem *RoomModule::listItemFromJid(QString roomJid)
{
    for (int count = 0; count < roomListWidget->count(); ++count)
    {
        QListWidgetItem *item = roomListWidget->item(count);

        if (item->data(Qt::UserRole).toString() == roomJid)
        {
            return item;
        }
    }

    return 0;
}


QXmppMucRoom *RoomModule::roomFromJid(QString roomJid)
{
    foreach (QXmppMucRoom *room, mucManager->rooms())
    {
        if (room->jid() == roomJid)
        {
            return room;
        }
    }

    return 0;
}


/*
 * Get the page number in the QStackedWidget that matches
 * the ChatWidget with the given room JID
 *
 */
int RoomModule::stackIndexFromJid(QString roomJid)
{
    int page = 1; // Starts with 1, since 0 is an empty page with no chat
    foreach (ChatWidget *chat, openedRooms)
    {
        if (chat->getRoomJid() == roomJid)
        {
            return page;
        }

        ++page;
    }

    return 0;
}



void RoomModule::joinRoom(QString roomJid, QString nickname)
{
    QXmppMucRoom *room = mucManager->addRoom(roomJid);
    room->setNickName(nickname);
    room->join();

    QString message = tr("Joining room %1...").arg(room->jid());

    this->globalObj->addToLog(message);

    ChatWidget *newChatWidget = new ChatWidget(this->globalObj, this);
    newChatWidget->setRoom(room);
    connect(newChatWidget, SIGNAL(roomJoined(QString)),
            this, SLOT(onRoomJoined(QString)));
    connect(newChatWidget, SIGNAL(roomLeft(QString)),
            this, SLOT(onRoomLeft(QString)));


    this->chatStackedWidget->addWidget(newChatWidget);
    this->openedRooms.append(newChatWidget);

    this->globalObj->addJoinedRoom(roomJid);
}


void RoomModule::leaveRoom(QString roomJid, QString reason)
{
    QXmppMucRoom *room = this->roomFromJid(roomJid);
    if (room)
    {
        QString message = tr("Leaving room %1...").arg(room->jid());
        this->globalObj->addToLog(message);

        room->leave(reason);
        room->deleteLater(); // Should be done in ChatWidget::onLeft() - FIXME?

        for (int count = 0; count < openedRooms.size(); ++count)
        {
            if (openedRooms.at(count)->getRoomJid() == roomJid)
            {
                this->openedRooms.at(count)->deleteLater();
                this->openedRooms.removeAt(count);
                this->chatStackedWidget->widget(count + 1)->deleteLater(); // +1, since first page is not a room
                // break; ?
            }
        }


        QListWidgetItem *item = this->listItemFromJid(roomJid);
        // These item modifications should be handled from onRoomLeft() - FIXME?
        item->setData(Qt::UserRole + 3, false); // Set as not-joined
        item->setIcon(QIcon::fromTheme("user-offline",
                                       QIcon(":/images/button-offline.png")));
        this->showRoomDetails(item);
    }
}



/*
 * Join rooms configured for autojoin
 *
 */
void RoomModule::autojoinRooms()
{
    qDebug() << "autojoinRooms() size:" << mucManager->rooms().size(); // tests

    for (int count = 0; count < roomListWidget->count(); ++count)
    {
        QListWidgetItem *item = roomListWidget->item(count);

        QString roomJid = item->data(Qt::UserRole).toString();
        QString roomNick = item->data(Qt::UserRole + 1).toString();

        if (item->data(Qt::UserRole + 2).toBool()                // autojoin=true
         || this->globalObj->getJoinedRooms().contains(roomJid)) // or previously joined
        {
            this->joinRoom(roomJid, roomNick);
        }
    }
}


bool RoomModule::newRoomIsValid()
{
    QString roomJid = this->roomJidLineEdit->text().trimmed();
    if (!Helpers::isValidJid(roomJid))
    {
        QMessageBox::warning(this, tr("Error"),
                             tr("Room address is not valid."));
        this->roomJidLineEdit->setFocus();

        return false;
    }

    if (this->nickLineEdit->text().trimmed().isEmpty())
    {
        QMessageBox::warning(this, tr("Error"),
                             tr("Nickname is not valid."));
        this->nickLineEdit->setFocus();

        return false;
    }

    return true;
}


void RoomModule::toggleJoinLeaveButtons(QListWidgetItem *item)
{
    // UserRole+3 contains true if joined
    if (item->data(Qt::UserRole + 3).toBool())
    {
        this->joinButton->hide();
        this->leaveButton->show();
    }
    else
    {
        this->joinButton->show();
        this->leaveButton->hide();
    }
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



void RoomModule::addRoom()
{
    this->roomListWidget->addItem(" --- " + tr("New Room") + " ---");

    // Select this new item
    this->roomListWidget->setCurrentRow(roomListWidget->count() - 1);

    this->roomJidLineEdit->clear();
    this->nickLineEdit->clear();
    this->autojoinCheckbox->setChecked(false);

    this->editRoom();
}


/*
 * Enable fields for room editing, and Update button
 *
 */
void RoomModule::editRoom()
{
    this->toggleRoomDetails(true);
}


void RoomModule::updateRoom()
{
    if (this->roomListWidget->selectedItems().size() == 0)
    {
        return;
    }


    QString roomJid = roomJidLineEdit->text().trimmed();
    QString roomNickname = nickLineEdit->text().trimmed();
    bool roomAutojoin = autojoinCheckbox->isChecked();

    if (newRoomIsValid())
    {
        QListWidgetItem *item = this->roomListWidget->currentItem();

        QString itemString = this->roomDescriptionLine(roomJid,
                                                       roomNickname,
                                                       roomAutojoin);
        // Icon depends on current joined status
        if (item->data(Qt::UserRole + 3).toBool())
        {
            item->setIcon(QIcon::fromTheme("user-online",
                                           QIcon(":/images/button-online.png")));
        }
        else
        {
            item->setIcon(QIcon::fromTheme("user-offline",
                                           QIcon(":/images/button-offline.png")));
        }

        item->setText(itemString);
        item->setData(Qt::UserRole,     roomJid);
        item->setData(Qt::UserRole + 1, roomNickname);
        item->setData(Qt::UserRole + 2, roomAutojoin);

        this->toggleRoomDetails(false);
    }
}


void RoomModule::removeRoom()
{
    QListWidgetItem *selectedItem = this->roomListWidget->takeItem(this->roomListWidget->currentRow());
    if (selectedItem)
    {
        // FIXME: Leave room if currently joined!

        delete selectedItem;
        this->toggleRoomDetails(false);
    }
}



void RoomModule::joinSelectedRoom()
{
    this->joinRoom(this->roomJidLineEdit->text().trimmed(),
                   this->nickLineEdit->text().trimmed());


    QListWidgetItem *item = this->roomListWidget->currentItem();
    this->showRoomDetails(item);

    this->joinButton->setDisabled(true);
}


void RoomModule::leaveSelectedRoom()
{
    QString roomJid = this->roomJidLineEdit->text();
    this->leaveRoom(roomJid, "Bye!"); // TMP reason FIXME

    this->globalObj->removeJoinedRoom(roomJid);


    QListWidgetItem *item = this->roomListWidget->currentItem();
    this->showRoomDetails(item);
}


void RoomModule::changeNick()
{
    QXmppMucRoom *room = this->roomFromJid(this->roomJidLineEdit->text());
    if (room)
    {
        room->setNickName(this->nickLineEdit->text().trimmed());
    }
}



void RoomModule::showRoomDetails(QListWidgetItem *item)
{
    //qDebug() << "showRoomDetails()" << item->data(Qt::UserRole).toString();
    this->roomJidLineEdit->setText(item->data(Qt::UserRole).toString());
    this->nickLineEdit->setText(item->data(Qt::UserRole + 1).toString());
    this->autojoinCheckbox->setChecked(item->data(Qt::UserRole + 2).toBool());

    this->toggleRoomDetails(false);

    this->editRoomButton->setEnabled(true);
    this->removeRoomButton->setEnabled(true);


    this->toggleJoinLeaveButtons(item);


    int stackPos = this->stackIndexFromJid(this->roomJidLineEdit->text());
    this->chatStackedWidget->setCurrentIndex(stackPos);
}


void RoomModule::toggleRoomDetails(bool state)
{
    this->roomJidLineEdit->setEnabled(state);
    this->nickLineEdit->setEnabled(state);
    this->autojoinCheckbox->setEnabled(state);

    this->updateRoomButton->setVisible(state);
    this->editRoomButton->setHidden(state); // Reversed state!


    this->joinButton->setEnabled(globalObj->connectedToServer());
    this->leaveButton->setEnabled(globalObj->connectedToServer());


    if (state)
    {
        this->roomJidLineEdit->setFocus();
    }
    else
    {
        this->roomListWidget->setFocus();
    }
}



void RoomModule::onRoomJoined(QString jid)
{
    QListWidgetItem *item = this->listItemFromJid(jid);
    if (item)
    {
        item->setIcon(QIcon::fromTheme("user-online",
                                       QIcon(":/images/button-online.png")));

        item->setData(Qt::UserRole + 3, true); // Mark as joined

        if (item->isSelected())
        {
            this->toggleJoinLeaveButtons(item);
            this->joinButton->setEnabled(true);
        }
    }
}


void RoomModule::onRoomLeft(QString jid)
{
    this->leaveRoom(jid, "---");
}
