/*
 *   This file is part of AkariXB
 *   Copyright 2015-2016  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "mainwindow.h"


MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent)
{
    this->setWindowTitle("AkariXB");
    this->setWindowIcon(QIcon(":/icon/64x64/akarixb.png"));
    this->setMinimumSize(500, 400);

    this->globalObject = new GlobalObject(this);
    connect(globalObject, SIGNAL(showInStatusBar(QString)),
            this, SLOT(setStatusBarMessage(QString)));


#if 0 // Set to 1 to test the fallback icons
    QIcon::setThemeSearchPaths(QStringList() << "./");
#endif

    setupXmppClient();

    // Page 7, runtime log, needed before others for early logging
    this->logModule = new LogModule(this);
    connect(globalObject, SIGNAL(logMessage(QString)),
            logModule, SLOT(addToLog(QString)));

    createDataDir();


    this->configDialog = new ConfigDialog(this->globalObject, this);



    // Page 1, general
    this->generalModule = new GeneralModule(this->globalObject, this);

    // Page 2, MUCs
    this->roomModule = new RoomModule(this->globalObject, this);

    // Page 3, private chats
    this->chatModule = new ChatModule(this->globalObject, this);
    connect(chatModule, SIGNAL(unreadCountChanged(int)),
            this, SLOT(setChatsTitle(int)));


    // Page 4, commands
    this->commandModule = new CommandModule(this->commandHandler,
                                            this->globalObject,
                                            this);

    // Page 5, conversations
    this->conversationModule = new ConversationModule(this);

    // Page 6, activities
    this->activityModule = new ActivityModule(this->globalObject, this);




    // Categories panel
    this->categoriesList = new QListWidget(this);
    categoriesList->setViewMode(QListView::ListMode);
    categoriesList->setFlow(QListView::TopToBottom);
    categoriesList->setWrapping(false);
    categoriesList->setIconSize(QSize(48, 48));
    categoriesList->setMinimumWidth(120); // TMP/FIXME
    categoriesList->setWordWrap(true);
    categoriesList->setMovement(QListView::Static);
    categoriesList->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    categoriesList->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    categoriesList->addItem(tr("General"));
    categoriesList->item(0)->setIcon(QIcon::fromTheme("user-home",
                                                      QIcon(":/images/section-general.png")));
    categoriesList->addItem(tr("Rooms"));
    categoriesList->item(1)->setIcon(QIcon::fromTheme("system-users",
                                                      QIcon(":/images/section-rooms.png")));
    categoriesList->addItem("*chats*");
    categoriesList->item(2)->setIcon(QIcon::fromTheme("system-users",
                                                      QIcon(":/images/section-rooms.png")));
    this->setChatsTitle(0);
    categoriesList->addItem(tr("Commands"));
    categoriesList->item(3)->setIcon(QIcon::fromTheme("system-run",
                                                      QIcon(":/images/section-general.png")));
    categoriesList->addItem(tr("Conversation"));
    categoriesList->item(4)->setIcon(QIcon::fromTheme("view-conversation-balloon",
                                                      QIcon(":/images/section-rooms.png")));
    categoriesList->addItem(tr("Activities"));
    categoriesList->item(5)->setIcon(QIcon::fromTheme("view-categories",
                                                      QIcon(":/images/section-general.png")));
    categoriesList->addItem(tr("Log"));
    categoriesList->item(6)->setIcon(QIcon::fromTheme("text-x-log",
                                                      QIcon(":/images/section-log.png")));

    // Pages on right side
    this->stackedWidget = new QStackedWidget(this);
    stackedWidget->addWidget(generalModule);
    stackedWidget->addWidget(roomModule);
    stackedWidget->addWidget(chatModule);
    stackedWidget->addWidget(commandModule);
    stackedWidget->addWidget(conversationModule);
    stackedWidget->addWidget(activityModule);
    stackedWidget->addWidget(logModule);
    connect(categoriesList, SIGNAL(currentRowChanged(int)),
            stackedWidget, SLOT(setCurrentIndex(int)));
    connect(categoriesList, SIGNAL(currentRowChanged(int)),
            this, SLOT(setTitleAndTrayInfo()));

    this->mainLayout = new QHBoxLayout();
    mainLayout->addWidget(categoriesList, 1);
    mainLayout->addWidget(stackedWidget,  5);

    this->mainWidget = new QWidget(this);
    mainWidget->setLayout(mainLayout);
    this->setCentralWidget(mainWidget);


    loadSettings();


    this->connectionDialog = new ConnectionDialog(this->userJid,
                                                  this->userPassword,
                                                  this->userAutoconnect,
                                                  this->userResource,
                                                  this->userPriority,
                                                  this);
    connect(connectionDialog, SIGNAL(connectionRequested(QString,QString,bool,QString,int)),
            this, SLOT(connectToServer(QString,QString,bool,QString,int)));
    connect(generalModule, SIGNAL(connectionRequested()),
            connectionDialog, SLOT(show()));


    createMenus();
    createTrayIcon();
    createStatusBarWidgets();


    this->categoriesList->setCurrentRow(0);
    this->setTitleAndTrayInfo();

    this->statusBar()->showMessage(tr("Ready."));


    if (this->userJid.isEmpty())
    {
        connectionDialog->show();
    }
    else if (this->userAutoconnect)
    {
        this->connectToServer(this->userJid,
                              this->userPassword,
                              this->userAutoconnect,
                              this->userResource,
                              this->userPriority);
    }

    qDebug() << "MainWindow created";
}


MainWindow::~MainWindow()
{
    this->saveSettings();

    qDebug() << "MainWindow destroyed";
}


void MainWindow::createMenus()
{
    // Session
    this->sessionConnectAction = new QAction(QIcon::fromTheme("network-connect",
                                                              QIcon(":/images/button-online.png")),
                                             tr("Connect"),
                                             this);
    sessionConnectAction->setShortcut(QKeySequence("Ctrl+Shift+C"));
    connect(sessionConnectAction, SIGNAL(triggered()),
            connectionDialog, SLOT(show()));

    this->sessionDisconnectAction = new QAction(QIcon::fromTheme("network-disconnect",
                                                                 QIcon(":/images/button-offline.png")),
                                                tr("Disconnect"),
                                                this);
    sessionDisconnectAction->setDisabled(true);
    sessionDisconnectAction->setShortcut(QKeySequence("Ctrl+Shift+D"));
    connect(sessionDisconnectAction, SIGNAL(triggered()),
            xmppClient, SLOT(disconnectFromServer()));


    this->sessionQuitAction = new QAction(QIcon::fromTheme("application-exit"),
                                   tr("Quit"),
                                   this);
    sessionQuitAction->setShortcut(QKeySequence("Ctrl+Q"));
    connect(sessionQuitAction, SIGNAL(triggered()),
            this, SLOT(quitProgram()));


    this->sessionMenu = new QMenu(tr("&Session"), this);
    sessionMenu->addAction(sessionConnectAction);
    sessionMenu->addAction(sessionDisconnectAction);
    sessionMenu->addSeparator();
    sessionMenu->addAction(sessionQuitAction);
    this->menuBar()->addMenu(sessionMenu);


    // Settings
    this->settingsConfigAction = new QAction(QIcon::fromTheme("configure",
                                                              QIcon(":/images/button-configure.png")),
                                             tr("Configure AkariXB"),
                                             this);
    settingsConfigAction->setShortcut(QKeySequence("Ctrl+Shift+S"));
    connect(settingsConfigAction, SIGNAL(triggered()),
            configDialog, SLOT(show()));

    this->settingsMenu = new QMenu(tr("S&ettings"), this);
    settingsMenu->addAction(settingsConfigAction);
    this->menuBar()->addMenu(settingsMenu);

    this->menuBar()->addSeparator();


    // Help
    this->helpWebsiteAction = new QAction(QIcon::fromTheme("internet-web-browser"),
                                          tr("Visit Website"),
                                          this);
    connect(helpWebsiteAction, SIGNAL(triggered()),
            this, SLOT(visitWebsite()));

    this->helpBugTrackerAction = new QAction(QIcon::fromTheme("tools-report-bug"),
                                             tr("Report a Bug"),
                                             this);
    connect(helpBugTrackerAction, SIGNAL(triggered()),
            this, SLOT(visitBugTracker()));

    this->helpAboutAction = new QAction(QIcon(":/icon/64x64/akarixb.png"),
                                    tr("About AkariXB"),
                                    this);
    connect(helpAboutAction, SIGNAL(triggered()),
            this, SLOT(aboutAkariXB()));

    this->helpMenu = new QMenu(tr("&Help"), this);
    helpMenu->addAction(helpWebsiteAction);
    helpMenu->addAction(helpBugTrackerAction);
    helpMenu->addSeparator();
    helpMenu->addAction(helpAboutAction);
    this->menuBar()->addMenu(helpMenu);


    /// Context menu for the tray icon
    trayTitleSeparatorAction = new QAction("AkariXB", this);
    trayTitleSeparatorAction->setSeparator(true);

    trayShowWindowAction = new QAction(QIcon(":/icon/64x64/akarixb.png"),
                                       "*show-hide-window*",
                                       this);
    connect(trayShowWindowAction, SIGNAL(triggered()),
            this, SLOT(toggleMainWindow()));


    this->systrayMenu = new QMenu("*systray-menu*", this);
    systrayMenu->setSeparatorsCollapsible(false);
    systrayMenu->addAction(trayTitleSeparatorAction);
    systrayMenu->addAction(trayShowWindowAction);
    systrayMenu->addAction(sessionConnectAction);
    systrayMenu->addAction(sessionDisconnectAction);
    systrayMenu->addAction(helpAboutAction);
    systrayMenu->addSeparator();
    systrayMenu->addAction(sessionQuitAction);
}


void MainWindow::createTrayIcon()
{
    this->systrayIcon = new QSystemTrayIcon(QIcon(":/icon/64x64/akarixb.png"),
                                            this);

    if (systrayIcon->isSystemTrayAvailable())
    {
        //trayIconAvailable = true;

        // Catch clicks on icon
        connect(systrayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
                this, SLOT(trayControl(QSystemTrayIcon::ActivationReason)));

        // clicking in a popup notification (balloon-type) will show the window
        connect(systrayIcon, SIGNAL(messageClicked()),
                this, SLOT(show()));  // FIXME: this can mess up the first action in the context menu

        // Set contextual menu for the icon
        systrayIcon->setContextMenu(systrayMenu);
        systrayIcon->setToolTip("AkariXB");
        systrayIcon->show();

        qDebug() << "Tray icon created";
    }
    else
    {
        //trayIconAvailable = false;
        qDebug() << "System tray is NOT available";
    }
}



void MainWindow::createStatusBarWidgets()
{
    this->statusBarLabel = new QLabel(this);
    this->setStatusIconOnline(false);

    this->statusBar()->addPermanentWidget(statusBarLabel);
    this->statusBar()->setSizeGripEnabled(false);
}



void MainWindow::createDataDir()
{
    QString dataDirectory;

#if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
    dataDirectory = QDesktopServices::storageLocation(QDesktopServices::DataLocation);
#else
    dataDirectory = QStandardPaths::standardLocations(QStandardPaths::DataLocation).first();
#endif

    this->globalObject->setDataDirectory(dataDirectory);
    qDebug() << "Data directory:" << dataDirectory;

    QDir dataDir;
    if (!dataDir.exists(dataDirectory))
    {
        qDebug() << "Creating data directory";
        if (dataDir.mkpath(dataDirectory))
        {
            this->globalObject->addToLog(tr("Data directory created: %1")
                                         .arg(dataDirectory));
            qDebug() << "Data directory created:" << dataDirectory;
        }
        else
        {
            this->globalObject->addToLog(tr("Error creating data directory: %1")
                                         .arg(dataDirectory));
            qDebug() << "Error creating data directory!";
        }
    }
    else
    {
        this->globalObject->addToLog(tr("Data directory in use: %1")
                                     .arg(dataDirectory));
    }

    if (!dataDir.exists(dataDirectory + "/logs"))
    {
        qDebug() << "Creating logs directory";
        if (dataDir.mkpath(dataDirectory + "/logs"))
        {
            qDebug() << "Logs directory created";
        }
        else
        {
            qDebug() << "Error creating logs directory!";
        }
    }
}



void MainWindow::setupXmppClient()
{
    this->xmppClient = new QXmppClient(this);
    xmppClient->versionManager().setClientOs("Akari-GNU-OS"); // tmp FIXME xD

    connect(xmppClient, SIGNAL(connected()),
            this, SLOT(onConnected()));
    connect(xmppClient, SIGNAL(disconnected()),
            this, SLOT(onDisconnected()));

    connect(xmppClient, SIGNAL(messageReceived(QXmppMessage)),
            this, SLOT(processMessage(QXmppMessage)));

    connect(xmppClient, SIGNAL(presenceReceived(QXmppPresence)),
            this, SLOT(processPresence(QXmppPresence)));

    connect(xmppClient, SIGNAL(error(QXmppClient::Error)),
            this, SLOT(onNetworkError(QXmppClient::Error)));

    this->mucManager = new QXmppMucManager();
    xmppClient->addExtension(mucManager);


    this->globalObject->setXmppClient(this->xmppClient);
    this->globalObject->setMucManager(this->mucManager);

    this->messageHandler = new MessageHandler(globalObject, this);
    this->commandHandler = new CommandHandler(globalObject, this);
}




void MainWindow::loadSettings()
{
    QSettings settings;
    settings.beginGroup("Connection");
    this->userJid = settings.value("jid").toString();
    this->userPassword = QByteArray::fromBase64(settings.value("password")
                                                        .toByteArray());
    this->userAutoconnect = settings.value("autoconnect").toBool();
    this->userResource = settings.value("resource", "AkariXB").toString();
    this->userPriority = settings.value("priority", 10).toInt();
    settings.endGroup();


    settings.beginGroup("MainWindow");
    this->resize(settings.value("windowSize",
                                QSize(640, 500)).toSize());

    if (settings.contains("windowPosition"))
    {
        this->move(settings.value("windowPosition").toPoint());
    }
    settings.endGroup();
}


void MainWindow::saveSettings()
{
    QSettings settings;
    settings.beginGroup("Connection");
    settings.setValue("jid",         this->userJid);
    settings.setValue("password",
                      this->userPassword.toLocal8Bit().toBase64());
    settings.setValue("autoconnect", this->userAutoconnect);
    settings.setValue("resource",    this->userResource);
    settings.setValue("priority",    this->userPriority);
    settings.endGroup();


    settings.beginGroup("MainWindow");
    settings.setValue("windowSize",     this->size());
    settings.setValue("windowPosition", this->pos());
    settings.endGroup();
    settings.sync();
}


void MainWindow::setStatusIconOnline(bool state)
{
    if (state)
    {
        statusBarLabel->setPixmap(QPixmap(":/images/button-online.png")
                                  .scaled(16, 16));
    }
    else
    {
        statusBarLabel->setPixmap(QPixmap(":/images/button-offline.png")
                                  .scaled(16, 16));
    }
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



void MainWindow::connectToServer(QString jid, QString password,
                                 bool autoconnect, QString resource,
                                 int priority)
{
    this->userJid = jid;
    this->userPassword = password;
    this->userAutoconnect = autoconnect;
    this->userResource = resource;
    this->userPriority = priority;

    QString message = tr("Connecting...");
    this->generalModule->setInfoMessage(message);
    this->globalObject->addToLog(message);

    this->generalModule->setConnecting();
    this->sessionConnectAction->setDisabled(true);

    this->xmppClient->connectToServer(userJid + "/" + userResource,
                                      password);
}



void MainWindow::onConnected()
{
    QString message = tr("%1 connected.").arg(this->userJid);
    this->generalModule->setInfoMessage(message);
    this->globalObject->addToLog(message);


    QXmppPresence presence;
    presence.setPriority(this->userPriority);
    presence.setAvailableStatusType(QXmppPresence::Chat); // TMP presence tests
    presence.setStatusText("Testing status message!");
    this->xmppClient->setClientPresence(presence);

    this->sessionDisconnectAction->setEnabled(true);

    this->roomModule->autojoinRooms();

    this->globalObject->setConnected(true);
    this->setTitleAndTrayInfo();
    this->setStatusIconOnline(true);
}


void MainWindow::onDisconnected()
{
    QString message = tr("Disconnected.");
    this->generalModule->setInfoMessage(message);
    this->globalObject->addToLog(message);

    this->sessionDisconnectAction->setDisabled(true);
    this->sessionConnectAction->setEnabled(true);

    this->globalObject->setConnected(false);
    this->setTitleAndTrayInfo();
    this->setStatusIconOnline(false);
}


void MainWindow::processMessage(QXmppMessage message)
{
    if (message.body().startsWith(this->globalObject->getCommandPrefix())
     && message.type() != QXmppMessage::Error)
    {
        this->commandHandler->processCommand(message);
    }
    else
    {
        this->messageHandler->processMessage(message);
    }

    // Send private messages to their tabs
    if (message.type() == QXmppMessage::Chat
     || message.type() == QXmppMessage::Normal)
    {
        this->chatModule->addChatTab(message);
    }
}


void MainWindow::processPresence(QXmppPresence presence)
{
    this->generalModule->addPresence(presence); // TMP...
}



void MainWindow::onNetworkError(QXmppClient::Error error)
{
    QString errorString;
    switch (error)
    {
    case QXmppClient::SocketError:
        errorString = tr("Socket Error");
        break;

    case QXmppClient::KeepAliveError:
        errorString = tr("KeepAlive Error");
        break;

    case QXmppClient::XmppStreamError:
        errorString = tr("XMPP Stream Error");
        break;

    default:
        errorString = "NoError";
    }


    QString message = tr("Network error: %1").arg(errorString);
    this->globalObject->addToLog(message);
}


void MainWindow::setChatsTitle(int count)
{
    QString title = tr("Chats");
    if (count > 0)
    {
        title.append(QString("  (%1)").arg(count));
    }
    this->categoriesList->item(2)->setText(title);

    // FIXME: title bar needs updating when this changes!
}



void MainWindow::visitWebsite()
{
    qDebug() << "Opening website in browser";
    QDesktopServices::openUrl(QUrl("https://jancoding.wordpress.com/akarixb"));
}


void MainWindow::visitBugTracker()
{
    qDebug() << "Opening bugtracker in browser";
    QDesktopServices::openUrl(QUrl("https://gitlab.com/akarixb/akarixb-dev/issues"));
}


/*
 * About... window
 *
 */
void MainWindow::aboutAkariXB()
{
    QMessageBox::about(this,
                       tr("About AkariXB"),
                       QString("<big><b>AkariXB v%1</b></big>")
                       .arg(qApp->applicationVersion())
                       + "<br>"
                         "Copyright 2015-2016  JanKusanagi<br><br>"
                         "<a href=\"https://jancoding.wordpress.com/akarixb\">"
                         "https://jancoding.wordpress.com/akarixb</a><br>"
                         "<br>"
                         "<br>"
                       + tr("AkariXB is a Jabber/XMPP bot.")
                       + "<br>"

                       + tr("You can talk to it, make it join chatrooms, etc.")
                       + "<br>"

                         "<hr>" // ---
                         "<br>"

                       + tr("English translation by JanKusanagi.",
                            "TRANSLATORS: Change this with your language "
                            "and name. If there was another translator before "
                            "you, add your name after theirs ;)")
                       + "<br>"
                         "<br>"

                       + tr("Thanks to all the testers, translators and "
                            "packagers, who help make AkariXB better!")
                       + "<br>"

                         "<hr>" // ---
                         "<br>"

                       + tr("AkariXB is Free Software, licensed under the "
                            "GNU GPL license, and uses some Oxygen icons "
                            "under LGPL license.")
                       + "<br><br>"

                         "<a href=\"http://www.gnu.org/licenses/gpl-2.0.html\">"
                         "GNU GPL v2</a> - "
                         "<a href=\"http://www.gnu.org/licenses/lgpl-2.1.html\">"
                         "GNU LGPL v2.1</a>"
                         "<br><br>"
                         "<a href=\"https://techbase.kde.org/Projects/Oxygen\">"
                         "techbase.kde.org/Projects/Oxygen</a>"
                         "<br>");

    // FIXME: ugly workaround to avoid closing the program after closing
    if (this->isHidden()) // the About dialog, if the main window was hidden
    {
        this->show();
        this->hide();
        qDebug() << "MainWindow was hidden, showing and hiding again";
    }
}


void MainWindow::setTitleAndTrayInfo()
{
    QString currentTabTitle = "#";
    QString offlineString = " (" + tr("Offline") + ")";

    if (this->categoriesList->currentRow() != -1)
    {
        currentTabTitle = this->categoriesList->currentItem()->text();
    }

    QString title = currentTabTitle;
    if (!this->userJid.isEmpty())
    {
        title.append(" - " + this->userJid);
    }

    if (this->globalObject->connectedToServer())
    {
        this->systrayIcon->setIcon(QIcon(":/icon/64x64/akarixb.png"));
    }
    else
    {
        this->systrayIcon->setIcon(QIcon(":/icon/64x64/akarixb-offline.png"));
        title.append(offlineString);
    }

    title.append(" - AkariXB");

    this->setWindowTitle(title);
    if (/*trayIconAvailable*/true) // FIXME
    {
        QString idToShow = this->userJid.isEmpty()
                           ? tr("Account is not configured")
                             : userJid + "/" + userResource;
#ifdef Q_OS_UNIX
        title = "<b>AkariXB</b>"
                "<hr>"
                "<br>"
                + idToShow
                + (this->globalObject->connectedToServer() ?
                   "" : offlineString)
                + "<br>";
#else
        // Some OSes don't render HTML in the tray tooltip
        title = "AkariXB: "
                "\n\n"
                + idToShow
                + (this->globalObject->connectedToServer() ?
                   "" : offlineString)
                + "\n";
#endif


        this->systrayIcon->setToolTip(title);

        this->trayTitleSeparatorAction->setText("AkariXB - " + this->userJid);
    }
}


void MainWindow::setStatusBarMessage(QString message)
{
    this->statusBar()->showMessage("[" + QTime::currentTime().toString() + "] "
                                   + message, 0);
}


/*
 * Control interaction with the system tray icon
 *
 */
void MainWindow::trayControl(QSystemTrayIcon::ActivationReason reason)
{
    qDebug() << "Tray icon activation reason:" << reason;

    if (reason != QSystemTrayIcon::Context) // Simple "main button" click in icon
    {
        // Hide or show the main window
        if (this->isMinimized())
        {
            // hide and show, because raise() wouldn't work
            this->hide();
            this->showNormal();
            qDebug() << "RAISING from minimized state";
        }
        else
        {
            this->toggleMainWindow();
        }
    }
}


void MainWindow::toggleMainWindow()
{
    if (this->isHidden() /*|| !this->trayIconAvailable*/)
    {
        this->trayShowWindowAction->setText(tr("&Hide Window"));
        this->show();
        qDebug() << "SHOWING main window";
    }
    else
    {
        this->trayShowWindowAction->setText(tr("&Show Window"));
        this->hide();
        qDebug() << "HIDING main window";
    }
}



void MainWindow::quitProgram()
{
    if (this->isHidden())
    {
        this->show();
    }

    std::cout << "\nShutting down AkariXB (" + this->userJid.toStdString()
                 + ")...\n";
    std::cout.flush();

    this->xmppClient->disconnectFromServer();
    qApp->closeAllWindows();

    std::cout << "All windows closed, bye! o/\n";
    std::cout.flush();

    qApp->quit();
}


//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


void MainWindow::resizeEvent(QResizeEvent *event)
{
    event->accept();
}


void MainWindow::closeEvent(QCloseEvent *event)
{
    qDebug() << "Trying to close window; hiding instead";
    this->hide(); // FIXME: only if tray icon was available; also, manage environment shutdown
    event->ignore();
}
