/*
 *   This file is part of AkariXB
 *   Copyright 2015-2016  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "datafile.h"


DataFile::DataFile(QString filename, QObject *parent) : QObject(parent)
{
    this->dFile = new QFile(filename);

    qDebug() << "DataFile created for " << filename;
}


DataFile::~DataFile()
{
    qDebug() << "DataFile destroyed for " << this->dFile->fileName();
}


/*
 * Read command data from disk in JSON format.
 *
 * Uses QJSON when building with Qt 4.x,
 * and built-in Qt 5 methods otherwise.
 *
 */
QVariantList DataFile::loadData()
{
    QVariantList commandDataList;
    bool parsedOk = false;

    dFile->open(QIODevice::ReadOnly);

    QByteArray rawData = dFile->readAll();
    dFile->close();

#if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
    // Qt 4 version, requires QJSON
    QJson::Parser jsonParser;
    commandDataList = jsonParser.parse(rawData, &parsedOk).toList();
#else
    // Qt 5 version, using built-in classes
    QJsonDocument jsonDocument;
    jsonDocument = QJsonDocument::fromJson(rawData);
    commandDataList = jsonDocument.toVariant().toList();
    parsedOk = !jsonDocument.isNull();
#endif

    if (!parsedOk)
    {
        // TMP FIXME
        commandDataList.clear();
    }

    return commandDataList;
}


/*
 * Save command data to disk in JSON format.
 *
 * Uses QJSON when building with Qt 4.x, and
 * built-in Qt 5 methods when building with Qt 5.x.
 *
 */
bool DataFile::saveData(QVariantList list)
{
    QByteArray commandData;

#if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
    // Qt 4 version, using QJSON
    QJson::Serializer serializer;
    serializer.setIndentMode(QJson::IndentFull); // More readable output
    commandData = serializer.serialize(list);
#else
    // Qt 5 version, with built-in methods
    QJsonDocument jsonDocument;
    jsonDocument = QJsonDocument::fromVariant(list);
    commandData = jsonDocument.toJson(QJsonDocument::Indented);
#endif


    dFile->open(QIODevice::WriteOnly);
    int written = dFile->write(commandData);
    dFile->close();

    if (written < 1) // 0 bytes, or -1 for Error
    {
        return false;
    }

    return true;
}

