/*
 *   This file is part of AkariXB
 *   Copyright 2015-2016  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "commandhandler.h"


CommandHandler::CommandHandler(GlobalObject *globalObject,
                               QObject *parent) : QObject(parent)
{
    this->globalObj = globalObject;

    this->xmppClient = this->globalObj->getXmppClient();
    this->mucManager = this->globalObj->getMucManager();


    // Give seed based on current time to random number generator
    uint randomSeed;
    randomSeed = (QTime::currentTime().hour())
               + (QTime::currentTime().minute() * 4)
               + (QTime::currentTime().second() * 5)
               + (QTime::currentTime().msec() * 6);
    randomSeed *= 8;
    randomSeed += qrand() % (randomSeed / (QTime::currentTime().second() + 1));

    qsrand(randomSeed);


    qDebug() << "CommandHandler created";
}


CommandHandler::~CommandHandler()
{
    qDebug() << "CommandHandler destroyed";
}


void CommandHandler::processCommand(QXmppMessage message)
{
    QStringList from = message.from().split("/");
    QString bareFrom = from.takeFirst();

    int userAccessLevel = 0;
    if (this->globalObj->getAdminJids().contains(bareFrom))
    {
        userAccessLevel = 100; // Admins get level 100
    }

    if (message.type() == QXmppMessage::GroupChat)
    {
        // Find the matching room object
        foreach (QXmppMucRoom *room, mucManager->rooms()) // FIXME: use roomFromJid()?
        {
            if (bareFrom == room->jid())
            {
                QString fromNick = from.join("/");
                if (fromNick == room->nickName()) // Command from ourselves!
                {
                    return;
                }

                // FIXME: log only if it's a recognized command?
                this->globalObj->addToLog(tr("Command from %1 in room %2: %3")
                                          .arg(fromNick)
                                          .arg(bareFrom)
                                          .arg(message.body()));


                QString reply = this->parseCommand(message.body(),
                                                   userAccessLevel);
                if (!reply.isEmpty())
                {
                    room->sendMessage(QString("%1: %2")
                                      .arg(fromNick)
                                      .arg(reply));
                }
            }
        }
    }
    else // Any other type; Error type is handled before entering this method
    {
        this->globalObj->addToLog(tr("Command from %1: %2")
                                  .arg(message.from())
                                  .arg(message.body()));

        // TODO: handle per-user flood

        QString reply = this->parseCommand(message.body(),
                                           userAccessLevel);
        if (!reply.isEmpty())
        {
            xmppClient->sendMessage(message.from(), reply);
        }
    }
}


/*
 * Look for the requested command and return the matching reply string
 *
 */
QString CommandHandler::parseCommand(QString commandString, int userAccess)
{
    // Remove prefix
    commandString.remove(0, this->globalObj->getCommandPrefix().length());

    QStringList commandStringList = commandString.split(" ");

    QString command = commandStringList.takeFirst(); // FIXME: to lowercase?
    QString parameters = commandStringList.join(" ").trimmed();


    QString commandFailedString = tr("The command failed.");
    QString commandReply;
    QString outputPattern;


    if (command == "help")
    {
        if (parameters.isEmpty())
        {
            commandReply = QString("Use %1commands to list "
                                   "the available commands.")
                           .arg(globalObj->getCommandPrefix())
                           + "\n\n"
                           + QString("Use %1help [command] to get help "
                                     "about a specific command")
                           .arg(globalObj->getCommandPrefix());
        }
        else
        {
            QString query = parameters.trimmed();
            QString helpMessage;
            foreach (QVariant mapVariant, this->commandDataList)
            {
                QVariantMap map = mapVariant.toMap();
                if (map.value("trigger").toString() == query)
                {
                    helpMessage = map.value("helpText").toString().trimmed();
                    break;
                }
            }

            if (helpMessage.isEmpty())
            {
                commandReply = QString("I don't have help about: %1")
                               .arg(query);
            }
            else
            {
                commandReply = QString("Help for [%1]:\n-- %2")
                               .arg(query)
                               .arg(helpMessage);
            }
        }
    }
    else if (command == "commands")
    {
        QStringList availableCommands = this->getCommandList(userAccess);

        commandReply = tr("Available commands")
                       + ":\n > ";
        foreach (QString trigger, availableCommands)
        {
            commandReply.append(trigger + ", ");
        }

        commandReply.remove(-2, 2); // Remove last comma and space
    }
    else if (command == "version")
    {
        commandReply = QString("AkariXB v%1").arg(qApp->applicationVersion());
    }
    else if (command == "date")
    {
        commandReply = QDateTime::currentDateTime()
                                 .toLocalTime()
                                 .toString(Qt::SystemLocaleLongDate);
    }
    else
    {
        foreach (QVariant mapVariant, this->commandDataList)
        {
            QVariantMap map = mapVariant.toMap();
            if (map.value("trigger").toString() == command)
            {
                if (userAccess < map.value("accessLevel").toInt())
                {
                    // FIXME: user JID should be shown here.
                    this->globalObj->addToLog(tr("Command %1 requested without "
                                                 "the right access level."));

                    qDebug() << "Command" << command << "requested without privileges!";
                    qDebug() << "-- Ignoring.";
                    break;
                }

                outputPattern = map.value("outputPattern").toString();

                int commandType = map.value("type").toInt();
                if (commandType == 0)
                {
                    commandReply = map.value("staticReply").toString();
                }
                else if (commandType == 1)
                {
                    QStringList replies = map.value("randomSentence")
                                             .toStringList();
                    int stringNum = qrand() % replies.count();
                    commandReply = replies.at(stringNum);
                }
                else if (commandType == 2)
                {
                    /*
                     * TODO: Optimize. Load at startup and notify clearly
                     *       if there's a problem loading the file.
                     */
                    QFile textFile(map.value("filename").toString());
                    if (textFile.open(QIODevice::ReadOnly))
                    {
                        QString fileLines = QString::fromUtf8(textFile.readAll())
                                                    .trimmed();
                        QStringList linesList = fileLines.split("\n");

                        int lineNum = qrand() % linesList.length();
                        if (lineNum < linesList.length())
                        {
                            // FIXME: parse %LINE% or something as \n
                            commandReply = linesList.at(lineNum).trimmed();
                            commandReply.replace("~LINE~", "\n"); // TMP
                        }
                    }
                    else
                    {
                        this->globalObj->addToLog(tr("Error loading file %1 "
                                                     "for command: %2")
                                                  .arg(map.value("filename")
                                                          .toString())
                                                  .arg(command));
                        commandReply = commandFailedString;
                    }
                }
                else if (commandType == 3)
                {
                    // Keyword
                }
                else if (commandType == 4)
                {
                    QProcess programProcess;
                    programProcess.start(map.value("programToRun").toString());
                    if (!programProcess.waitForFinished())
                    {
                        commandReply = commandFailedString;
                    }
                    else
                    {
                        commandReply = QString::fromUtf8(programProcess
                                                         .readAllStandardOutput()
                                                         .trimmed());

                        // Nothing on stdout, read stderr
                        if (commandReply.isEmpty())
                        {
                            commandReply = QString::fromUtf8(programProcess
                                                             .readAllStandardError()
                                                             .trimmed());
                        }

                        // Still empty...
                        if (commandReply.isEmpty())
                        {
                            commandReply = tr("Command gave no output text.");
                        }
                    }
                }
                else if (commandType == 5)
                {
                    // Alias -- TODO
                }

                break;
            }
        }
    }

    if (!outputPattern.isEmpty())
    {
        outputPattern.replace("%output%", commandReply);
        commandReply = outputPattern;
    }

    return commandReply;
}


void CommandHandler::setCommandData(QVariantList dataList)
{
    //qDebug() << "Sync'ing command data:\n\n" << dataList;
    this->commandDataList = dataList;
}


QStringList CommandHandler::getCommandList(int accessLevel)
{
    QStringList list;
    foreach (QVariant mapVariant, this->commandDataList)
    {
        if (mapVariant.toMap().value("accessLevel").toInt() <= accessLevel)
        {
            list.append(mapVariant.toMap().value("trigger").toString());
        }
    }

    return list;
}

