/*
 *   This file is part of AkariXB
 *   Copyright 2015-2016  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "chatwidget.h"


ChatWidget::ChatWidget(GlobalObject *globalObject,
                       QWidget *parent) : QWidget(parent)
{
    this->globalObj = globalObject;

    this->xmppClient = this->globalObj->getXmppClient();
    this->isMuc = false;

    this->unreadCount = 0;


    this->chatBrowser = new QTextBrowser(this);

    this->messageLineEdit = new QLineEdit(this);
    connect(messageLineEdit, SIGNAL(returnPressed()),
            this, SLOT(sendMessage()));

    // Give focus to line edit every time the text browser is focused
    //this->chatBrowser->setFocusProxy(this->messageLineEdit);
    // Disabled for now; it removes context menu in text browser -- FIXME


    this->mainLayout = new QVBoxLayout();
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->addWidget(chatBrowser);
    mainLayout->addWidget(messageLineEdit);
    this->setLayout(mainLayout);

    qDebug() << "ChatWidget created";
}


ChatWidget::~ChatWidget()
{
    qDebug() << "ChatWidget destroyed";
}



void ChatWidget::setRoom(QXmppMucRoom *room)
{
    this->muc = room;
    this->isMuc = true;


    connect(muc, SIGNAL(messageReceived(QXmppMessage)),
            this, SLOT(appendMessage(QXmppMessage)));

    connect(muc, SIGNAL(joined()),
            this, SLOT(onJoined()));

    connect(muc, SIGNAL(left()),
            this, SLOT(onLeft()));
}


QString ChatWidget::getRoomJid()
{
    if (isMuc)
    {
        return this->muc->jid();
    }

    return QString();
}


void ChatWidget::setContact(QString jid)
{
    this->contactJid = jid;
    this->contactNick = jid.split("@").first();

    this->isMuc = false;
}

QString ChatWidget::getContactJid()
{
    return this->contactJid;
}

QString ChatWidget::getContactBareJid()
{
    return this->contactJid.split("/").first();
}

void ChatWidget::setContactNick(QString nickname)
{
    this->contactNick = nickname;
}


void ChatWidget::setChatTitle(QString title)
{
    this->chatTitle = title;
}

QString ChatWidget::getChatTitle()
{
    return this->chatTitle;
}


int ChatWidget::getUnreadCount()
{
    return this->unreadCount;
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



void ChatWidget::appendMessage(QXmppMessage message)
{
    QString subject = Helpers::safeForHtml(message.subject().trimmed());
    if (!subject.isEmpty())
    {
        subject.append("<br>");
    }

    QString body = Helpers::safeForHtml(message.body());

    QString divStyle;
    QString nick;
    if (isMuc)
    {
        if (message.type() != QXmppMessage::GroupChat)
        {
            // A private message from a room, ignore it; handled by ChatModule
            return;
        }

        nick = Helpers::nickFromRoomJid(message.from());

        if (nick == this->muc->nickName())
        {
            divStyle = " style=\"color:#00AAEE; font-weight:bold\"";
        }
        else if (body.contains(this->muc->nickName(), Qt::CaseInsensitive))
        {
            divStyle = " style=\"color:#9910EE; font-weight:bold\"";
        }
    }
    else
    {
        nick = this->contactNick;
    }

    nick = Helpers::safeForHtml(nick);
    QString logMessage = "<div" + divStyle + ">"
                         "<b>[" + QTime::currentTime().toString() + "]</b> ";

    if (message.stamp().isValid())
    {
        logMessage.append("{" + message.stamp()
                                       .toLocalTime()
                                       .toString(Qt::SystemLocaleShortDate)
                          + "} ");
    }


    if (body.startsWith("/me "))
    {
        body.remove(0, 4);
        logMessage.append("<i>*" + nick + " "
                          + subject + body + "</i>");
    }
    else
    {
        logMessage.append("<i>&lt;" + nick + "&gt;</i> "
                          + subject + body);
    }

    /****
     **** FIXME: Handle chatstates
     ****
    QString state;
    switch (message.state())
    {
    case QXmppMessage::None:
        break;
    case QXmppMessage::Active:
        state="Active";
        break;
    case QXmppMessage::Inactive:
        state="Inactive";
        break;
    case QXmppMessage::Gone:
        state="Gone";
        break;
    case QXmppMessage::Composing:
        state="Composing";
        break;
    case QXmppMessage::Paused:
        state="Paused";
    }
    if (!state.isEmpty())
    {
        logMessage.append("  [state: " + state + "]");
    }

    ****
    ****/

    logMessage.append("</div>");


    this->chatBrowser->append(logMessage);
    if (!this->isVisible())
    {
        ++unreadCount;
    }
}



void ChatWidget::sendMessage()
{
    QString msg = this->messageLineEdit->text().trimmed();

    if (!msg.isEmpty())
    {
        if (isMuc)
        {
            muc->sendMessage(msg);
        }
        else
        {
            xmppClient->sendMessage(this->contactJid, msg);
        }

        this->messageLineEdit->clear();
    }
}



void ChatWidget::onParticipantJoined(QString nickname)
{
    QString nick = Helpers::nickFromRoomJid(nickname);
    nick = Helpers::safeForHtml(nick);

    this->chatBrowser->append("<div style=\"color:#DD0000\">"
                              "<b>[" + QTime::currentTime().toString()
                              + "]</b> <i>+++++ "
                              + nick
                              + " joined.</i></div>");

    //muc->sendMessage("Hi, " + nick);
}



void ChatWidget::onParticipantLeft(QString nickname)
{
    QString nick = Helpers::nickFromRoomJid(nickname);
    nick = Helpers::safeForHtml(nick);

    this->chatBrowser->append("<div style=\"color:#DD0000\">"
                              "<b>[" + QTime::currentTime().toString()
                              + "]</b> <i>----- "
                              + nick
                              + " left.</i></div>");
}


void ChatWidget::onKicked(QString jid, QString reason)
{
    if (!reason.isEmpty())
    {
        reason = " " + tr("Reason:") + " "
                 + Helpers::safeForHtml(reason);
    }

    if (!jid.isEmpty())
    {
        reason.append(" (" + jid + ")");
    }

    this->chatBrowser->append("<hr>"
                              "<div style=\"color:#10BB10\">"
                              "<b>[" + QTime::currentTime().toString()
                              + "] "
                              + tr("Kicked from the room!")
                              + reason
                              + "</b></div>");

    // TMP/FIXME: should be handled in onLeft()
    disconnect(muc, SIGNAL(participantRemoved(QString)),
               this, SLOT(onParticipantLeft(QString)));
}


void ChatWidget::onJoined()
{
    connect(muc, SIGNAL(participantAdded(QString)),
            this, SLOT(onParticipantJoined(QString)));

    connect(muc, SIGNAL(participantRemoved(QString)),
            this, SLOT(onParticipantLeft(QString)));

    connect(muc, SIGNAL(kicked(QString,QString)),
            this, SLOT(onKicked(QString,QString)));


    QString mucName = muc->jid();
    if (!muc->name().isEmpty())
    {
        mucName.append(" (" + muc->name() + ")");
    }
    QString joinedMessage = tr("Joined room %1, as %2.").arg(mucName)
                                                        .arg(muc->nickName());

    this->chatBrowser->append("<b><u>*** " + joinedMessage + " ***</u></b><br>");
    this->globalObj->addToLog(joinedMessage);

    // Inform RoomModule
    emit roomJoined(muc->jid());
}


void ChatWidget::onLeft()
{
    QString mucName = muc->jid();
    if (!muc->name().isEmpty())
    {
        mucName.append(" (" + muc->name() + ")");
    }

    this->globalObj->addToLog(tr("Left room %1.").arg(mucName));

    // Confirm leave status to RoomModule
    emit roomLeft(muc->jid());
}


//////////////////////////////////////////////////////////////////////////////
/////////////////////////////////// PROTECTED ////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


void ChatWidget::showEvent(QShowEvent *event)
{
    int unreadBefore = this->unreadCount;
    this->unreadCount = 0;

    emit messagesRead(unreadBefore);

    event->accept();
}

