/*
 *   This file is part of AkariXB
 *   Copyright 2015-2016  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "chatmodule.h"

ChatModule::ChatModule(GlobalObject *globalObject,
                       QWidget *parent) : QWidget(parent)
{
    this->globalObj = globalObject;

    this->xmppClient = this->globalObj->getXmppClient();


    this->unreadCount = 0;


    this->tabWidget = new QTabWidget(this);
    this->tabWidget->setTabsClosable(true);

    this->mainLayout = new QVBoxLayout();
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->addWidget(tabWidget);
    this->setLayout(mainLayout);

    qDebug() << "ChatModule created";
}


ChatModule::~ChatModule()
{
    qDebug() << "ChatModule destroyed";
}


void ChatModule::setTabTitle(int tabNumber, QString name, int unread)
{
    if (unread > 0)
    {
        name.append(QString("  (%1)").arg(unread));
    }

    this->tabWidget->setTabText(tabNumber, name);
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



void ChatModule::addChatTab(QXmppMessage message)
{
    QString bareFrom = message.from().split("/").first();

    // If this is a private message via MUC room
    bool fromMuc = false;
    if (this->globalObj->getJoinedRooms().contains(bareFrom))
    {
        // Will use full JID as title, and resource as nickname
        fromMuc = true;
    }

    int tabCount = 0;
    bool tabFound = false;
    for (; tabCount < openedChats.count(); ++tabCount)
    {
        if (fromMuc)
        {
            if (openedChats.at(tabCount)->getContactJid() == message.from())
            {
                tabFound = true;
                break;
            }
        }
        else
        {
            if (openedChats.at(tabCount)->getContactBareJid() == bareFrom)
            {
                tabFound = true;
                break;
            }
        }
    }


    ChatWidget *currentChatWidget;
    if (tabFound)
    {
        currentChatWidget = openedChats.at(tabCount);
        currentChatWidget->appendMessage(message);
    }
    else
    {
        currentChatWidget = new ChatWidget(this->globalObj, this);
        openedChats.append(currentChatWidget);

        tabWidget->addTab(currentChatWidget,
                          QIcon::fromTheme("user-identity",
                                           QIcon(":/images/no-avatar.png")),
                          bareFrom);

        currentChatWidget->setContact(message.from());

        QString tabTitle;
        if (fromMuc)
        {
            tabTitle = message.from(); // Full JID, room@server/nickname
            QStringList contactNick = message.from().split("/");
            contactNick.removeFirst(); // Can't just use .last(), since nickname could contain "/"
            currentChatWidget->setContactNick(contactNick.join("/"));
        }
        else
        {
            tabTitle = message.from().split("/").first(); // Username part of JID
        }

        currentChatWidget->setChatTitle(tabTitle);
        currentChatWidget->appendMessage(message);

        // It's important to connect() *after* the tab has been added
        connect(currentChatWidget, SIGNAL(messagesRead(int)),
                this, SLOT(updateTabTitle(int)));

        globalObj->addToLog(tr("Chat started by %1.",
                               "%1 = A nickname").arg(message.from()));

        // TODO: popup/email notification for this event
    }


    // Update title of that tab, not necessarily current tab
    this->setTabTitle(tabCount,
                      currentChatWidget->getChatTitle(),
                      currentChatWidget->getUnreadCount());


    if (!currentChatWidget->isVisible())
    {
        ++unreadCount;
        emit unreadCountChanged(unreadCount);
    }
}


void ChatModule::updateTabTitle(int count)
{
    this->unreadCount -= count;
    emit unreadCountChanged(unreadCount);

    int currentTab = tabWidget->currentIndex();

    this->setTabTitle(currentTab,
                      openedChats.at(currentTab)->getChatTitle(),
                      openedChats.at(currentTab)->getUnreadCount());
}

