/*
 *   This file is part of AkariXB
 *   Copyright 2015-2016  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "activitymodule.h"


ActivityModule::ActivityModule(GlobalObject *globalObject,
                               QWidget *parent) : QWidget(parent)
{
    this->globalObj = globalObject;

    this->dataFile = new DataFile(globalObj->getDataDirectory()
                                  + "/activities.axb",
                                  this);


    // Left side
    this->syncActivitiesButton = new QPushButton(QIcon::fromTheme("document-save",
                                                                  QIcon(":/images/button-save.png")),
                                                 tr("&Sync Activities"),
                                                 this);
    connect(syncActivitiesButton, SIGNAL(clicked()),
            this, SLOT(syncActivities()));



    this->activityListWidget = new QListWidget(this);
    activityListWidget->setDragDropMode(QAbstractItemView::InternalMove); // Movable items
    connect(activityListWidget, SIGNAL(currentRowChanged(int)),
            this, SLOT(onActivitySelected(int)));

    this->addActivityButton = new QPushButton(QIcon::fromTheme("list-add",
                                                               QIcon(":/images/list-add.png")),
                                              tr("&Add Activity"),
                                              this);
    connect(addActivityButton, SIGNAL(clicked()),
            this, SLOT(addActivity()));


    this->removeActivityButton = new QPushButton(QIcon::fromTheme("list-remove",
                                                                  QIcon(":/images/list-remove.png")),
                                                 tr("&Remove Activity"),
                                                 this);
    connect(removeActivityButton, SIGNAL(clicked()),
            this, SLOT(removeActivity()));




    // Right side
    this->nameLineEdit = new QLineEdit(this);
    connect(nameLineEdit, SIGNAL(textChanged(QString)),
            this, SLOT(onNameChanged(QString)));


    this->timesPerDaySpinbox = new QSpinBox(this);
    timesPerDaySpinbox->setRange(1, 100);

    this->probabilitySpinbox = new QSpinBox(this);
    probabilitySpinbox->setRange(1, 100);
    probabilitySpinbox->setValue(50);
    probabilitySpinbox->setSuffix("%");

    this->minTimeEdit = new QTimeEdit(QTime::currentTime(), this);
    this->maxTimeEdit = new QTimeEdit(QTime::currentTime().addSecs(60), this);

    this->minDateEdit = new QDateEdit(QDate::currentDate(), this);
    minDateEdit->setCalendarPopup(true);
    minDateEdit->calendarWidget()->setGridVisible(true);
    minDateEdit->setDisplayFormat("dd MMMM");
    this->maxDateEdit = new QDateEdit(QDate::currentDate().addDays(1), this);
    maxDateEdit->setCalendarPopup(true);
    maxDateEdit->calendarWidget()->setGridVisible(true);
    maxDateEdit->setDisplayFormat("dd MMMM");

    this->minDurationSpinbox = new QSpinBox(this);
    minDurationSpinbox->setRange(1, 300);
    this->maxDurationSpinbox = new QSpinBox(this);
    maxDurationSpinbox->setRange(1, 300);
    maxDurationSpinbox->setSuffix(" " + tr("minutes"));

    this->statusTypeCombobox = new QComboBox(this);
    statusTypeCombobox->addItem(tr("Online"));
    statusTypeCombobox->addItem(tr("Free for Chat"));
    statusTypeCombobox->addItem(tr("Away"));
    statusTypeCombobox->addItem(tr("Extended Away"));
    statusTypeCombobox->addItem(tr("Do not Disturb"));

    this->availabilitySpinbox = new QSpinBox(this);
    availabilitySpinbox->setRange(0, 100);
    availabilitySpinbox->setValue(50);
    availabilitySpinbox->setSuffix("%");


    this->minReactionSpinbox = new QSpinBox(this);
    minReactionSpinbox->setRange(0, 300);
    this->maxReactionSpinbox = new QSpinBox(this);
    maxReactionSpinbox->setRange(1, 300);
    maxReactionSpinbox->setSuffix(" " + tr("seconds"));


    this->descriptionListWidget = new StringListWidget(this);

    this->msgBeforeListWidget = new StringListWidget(this);

    this->msgAfterListWidget = new StringListWidget(this);


    // Send messsages to: all rooms/some rooms/few rooms + active private conversations
    this->msgToRoomsCombobox = new QComboBox(this);
    msgToRoomsCombobox->addItem(tr("All rooms"));
    msgToRoomsCombobox->addItem(tr("Several rooms"));
    msgToRoomsCombobox->addItem(tr("A few rooms"));

    this->msgToChatsCheckbox = new QCheckBox(tr("Also to active private chats"),
                                             this);


    this->saveActivityButton = new QPushButton(QIcon::fromTheme("document-save",
                                                                QIcon(":/images/button-save.png")),
                                               tr("Sa&ve Activity"),
                                               this);
    connect(saveActivityButton, SIGNAL(clicked()),
            this, SLOT(saveActivity()));


    // Layout
    this->leftLayout = new QVBoxLayout();
    leftLayout->addWidget(syncActivitiesButton);
    leftLayout->addWidget(activityListWidget);
    leftLayout->addWidget(addActivityButton);
    leftLayout->addWidget(removeActivityButton);


    this->timeLayout = new QHBoxLayout();
    timeLayout->addWidget(minTimeEdit);
    timeLayout->addSpacing(8);
    timeLayout->addWidget(maxTimeEdit);

    this->dateLayout = new QHBoxLayout();
    dateLayout->addWidget(minDateEdit);
    dateLayout->addSpacing(8);
    dateLayout->addWidget(maxDateEdit);

    this->durationLayout = new QHBoxLayout();
    durationLayout->addWidget(minDurationSpinbox);
    durationLayout->addSpacing(8);
    durationLayout->addWidget(maxDurationSpinbox);

    this->msgRecipientsLayout = new QHBoxLayout();
    msgRecipientsLayout->addWidget(msgToRoomsCombobox);
    msgRecipientsLayout->addWidget(msgToChatsCheckbox);

    this->reactionLayout = new QHBoxLayout();
    reactionLayout->addWidget(minReactionSpinbox);
    reactionLayout->addSpacing(8);
    reactionLayout->addWidget(maxReactionSpinbox);


    this->formLayout = new QFormLayout();
    formLayout->addRow(tr("Name"),                  nameLineEdit);
    formLayout->addRow(tr("Maximum Times Per Day"), timesPerDaySpinbox);
    formLayout->addRow(tr("Probability"),           probabilitySpinbox);
    formLayout->addRow(tr("Time Range"),            timeLayout);
    formLayout->addRow(tr("Date Range"),            dateLayout);
    formLayout->addRow(tr("Duration"),              durationLayout);
    formLayout->addRow(tr("Set Status Type"),       statusTypeCombobox);
    formLayout->addRow(tr("Availability"),          availabilitySpinbox);
    formLayout->addRow(tr("Reaction Time"),         reactionLayout);
    formLayout->addRow(tr("Descriptions"),          descriptionListWidget);
    formLayout->addRow(tr("Messages Before"),       msgBeforeListWidget);
    formLayout->addRow(tr("Messages Afterwards"),   msgAfterListWidget);
    formLayout->addRow(tr("Send Messages To"),      msgRecipientsLayout);


    this->rightLayout = new QVBoxLayout();
    rightLayout->addLayout(formLayout);
    rightLayout->addStretch();
    rightLayout->addWidget(saveActivityButton, 0, Qt::AlignRight);

    this->rightWidget = new QWidget(this);
    this->rightWidget->setLayout(rightLayout);


    this->mainLayout = new QHBoxLayout();
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->addLayout(leftLayout,  1);
    mainLayout->addWidget(rightWidget, 4);
    this->setLayout(mainLayout);


    this->rightWidget->setDisabled(true);

    this->loadActivities();


    qDebug() << "ActivityModule created";
}


ActivityModule::~ActivityModule()
{
    qDebug() << "ActivityModule destroyed";
}



void ActivityModule::loadActivities()
{
    QVariantList activityDataList = this->dataFile->loadData();

    foreach (QVariant listItem, activityDataList)
    {
        QVariantMap map = listItem.toMap();
        QListWidgetItem *item = new QListWidgetItem();

        item->setText(map.value("name").toString());
        item->setData(Qt::UserRole,      map.value("name"));
        item->setData(Qt::UserRole + 1,  map.value("timesPerDay"));
        item->setData(Qt::UserRole + 2,  map.value("probability"));
        item->setData(Qt::UserRole + 3,  map.value("minTime"));
        item->setData(Qt::UserRole + 4,  map.value("maxTime"));
        item->setData(Qt::UserRole + 5,  map.value("minDate"));
        item->setData(Qt::UserRole + 6,  map.value("maxDate"));
        item->setData(Qt::UserRole + 7,  map.value("minDuration"));
        item->setData(Qt::UserRole + 8,  map.value("maxDuration"));
        item->setData(Qt::UserRole + 9,  map.value("statusType"));
        item->setData(Qt::UserRole + 10, map.value("availability"));
        item->setData(Qt::UserRole + 11, map.value("minReaction"));
        item->setData(Qt::UserRole + 12, map.value("maxReaction"));
        item->setData(Qt::UserRole + 13, map.value("descriptions"));
        item->setData(Qt::UserRole + 14, map.value("msgBefore"));
        item->setData(Qt::UserRole + 15, map.value("msgAfter"));
        item->setData(Qt::UserRole + 16, map.value("msgToRooms"));
        item->setData(Qt::UserRole + 17, map.value("msgToChats"));


        this->activityListWidget->addItem(item);
    }

    this->globalObj->addToLog(tr("Loaded %1 activities.")
                              .arg(activityDataList.length()));
}


bool ActivityModule::validateActivity()
{
    return true; /////// tmp
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



void ActivityModule::onNameChanged(QString newName)
{
    this->saveActivityButton->setDisabled(newName.trimmed().isEmpty());
}


void ActivityModule::addActivity()
{
    QListWidgetItem *item = new QListWidgetItem("-- "
                                                + tr("New Activity")
                                                + " --");
    ////// Defaults
    item->setData(Qt::UserRole,     "");

    // Times per day
    item->setData(Qt::UserRole + 1, 1);

    // Probability %
    item->setData(Qt::UserRole + 2, 50);

    // Time and date ranges
    item->setData(Qt::UserRole + 3, QTime(0,  0));
    item->setData(Qt::UserRole + 4, QTime(23, 59));
    item->setData(Qt::UserRole + 5, QDate(2000,  1,  1));
    item->setData(Qt::UserRole + 6, QDate(2000, 12, 31));

    // Duration range
    item->setData(Qt::UserRole + 7, 1);
    item->setData(Qt::UserRole + 8, 10);

    // Status type
    item->setData(Qt::UserRole + 9, 0);

    // Availability %
    item->setData(Qt::UserRole + 10, 50);

    // Reaction time range
    item->setData(Qt::UserRole + 11, 1);
    item->setData(Qt::UserRole + 12, 10);

    // Nothing to set here for the lists (UR + 13, 14, 15)

    item->setData(Qt::UserRole + 16, 0);
    item->setData(Qt::UserRole + 17, true);


    this->activityListWidget->addItem(item);

    int lastRow = activityListWidget->count() - 1;
    activityListWidget->setCurrentRow(lastRow);
    activityListWidget->setCurrentItem(item);
    item->setSelected(true);

    this->nameLineEdit->setFocus();
}


void ActivityModule::removeActivity()
{
    int row = this->activityListWidget->currentRow();
    if (row != -1)
    {
        QListWidgetItem *item = this->activityListWidget->takeItem(row);
        delete item;
    }

    if (activityListWidget->count() == 0) // No activities left
    {
        this->rightWidget->setDisabled(true);
    }
}


void ActivityModule::onActivitySelected(int row)
{
    if (row < 0) // Protection
    {
        return;
    }

    QListWidgetItem *item = activityListWidget->item(row);

    this->nameLineEdit->setText(item->data(Qt::UserRole).toString());
    this->timesPerDaySpinbox->setValue(item->data(Qt::UserRole + 1).toInt());
    this->probabilitySpinbox->setValue(item->data(Qt::UserRole + 2).toInt());

    this->minTimeEdit->setTime(item->data(Qt::UserRole + 3).toTime());
    this->maxTimeEdit->setTime(item->data(Qt::UserRole + 4).toTime());
    this->minDateEdit->setDate(item->data(Qt::UserRole + 5).toDate());
    this->maxDateEdit->setDate(item->data(Qt::UserRole + 6).toDate());

    this->minDurationSpinbox->setValue(item->data(Qt::UserRole + 7).toInt());
    this->maxDurationSpinbox->setValue(item->data(Qt::UserRole + 8).toInt());

    this->statusTypeCombobox->setCurrentIndex(item->data(Qt::UserRole + 9).toInt());

    this->availabilitySpinbox->setValue(item->data(Qt::UserRole + 10).toInt());

    this->minReactionSpinbox->setValue(item->data(Qt::UserRole + 11).toInt());
    this->maxReactionSpinbox->setValue(item->data(Qt::UserRole + 12).toInt());

    this->descriptionListWidget->clearList();
    this->descriptionListWidget->addStringsToList(item->data(Qt::UserRole + 13).toStringList());

    this->msgBeforeListWidget->clearList();
    this->msgBeforeListWidget->addStringsToList(item->data(Qt::UserRole + 14).toStringList());

    this->msgAfterListWidget->clearList();
    this->msgAfterListWidget->addStringsToList(item->data(Qt::UserRole + 15).toStringList());

    this->msgToRoomsCombobox->setCurrentIndex(item->data(Qt::UserRole + 16).toInt());
    this->msgToChatsCheckbox->setChecked(item->data(Qt::UserRole + 17).toBool());


    this->rightWidget->setEnabled(true);
}


void ActivityModule::saveActivity()
{
    QString newName = nameLineEdit->text().trimmed();

    // Check for duplicates
    for (int counter = 0; counter < activityListWidget->count(); ++counter)
    {
        QListWidgetItem *activityItem = activityListWidget->item(counter);
        if (activityItem->data(Qt::UserRole).toString() == newName)
        {
            if (activityItem != this->activityListWidget->currentItem())
            {
                QMessageBox::warning(this,
                                     tr("Activity exists"),
                                     tr("Error: There is already an activity "
                                        "in the list with this name."));
                this->nameLineEdit->setFocus();

                return;
            }
        }
    }

    // Check minimum required fields are filled
    if (!validateActivity())
    {
        return;
    }

    QListWidgetItem *item = this->activityListWidget->currentItem();

    nameLineEdit->setText(newName); // Show it as it will be stored
    item->setText(newName);
    item->setData(Qt::UserRole,     newName);
    item->setData(Qt::UserRole + 1, timesPerDaySpinbox->value());
    item->setData(Qt::UserRole + 2, probabilitySpinbox->value());

    item->setData(Qt::UserRole + 3, minTimeEdit->time());
    item->setData(Qt::UserRole + 4, maxTimeEdit->time());
    item->setData(Qt::UserRole + 5, minDateEdit->date());
    item->setData(Qt::UserRole + 6, maxDateEdit->date());

    item->setData(Qt::UserRole + 7, minDurationSpinbox->value());
    item->setData(Qt::UserRole + 8, maxDurationSpinbox->value());

    item->setData(Qt::UserRole + 9, statusTypeCombobox->currentIndex());

    item->setData(Qt::UserRole + 10, availabilitySpinbox->value());

    item->setData(Qt::UserRole + 11, minReactionSpinbox->value());
    item->setData(Qt::UserRole + 12, maxReactionSpinbox->value());

    item->setData(Qt::UserRole + 13, descriptionListWidget->getStringsFromList());
    item->setData(Qt::UserRole + 14, msgBeforeListWidget->getStringsFromList());
    item->setData(Qt::UserRole + 15, msgAfterListWidget->getStringsFromList());

    item->setData(Qt::UserRole + 16, msgToRoomsCombobox->currentIndex());
    item->setData(Qt::UserRole + 17, msgToChatsCheckbox->isChecked());



    this->activityListWidget->setFocus();
}


void ActivityModule::syncActivities()
{
    QVariantList activityDataList;

    for (int counter = 0; counter < activityListWidget->count(); ++counter)
    {
        QListWidgetItem *item = activityListWidget->item(counter);

        QVariantMap activityMap;
        activityMap.insert("name",          item->data(Qt::UserRole));
        activityMap.insert("timesPerDay",   item->data(Qt::UserRole + 1));
        activityMap.insert("probability",   item->data(Qt::UserRole + 2));
        activityMap.insert("minTime",       item->data(Qt::UserRole + 3));
        activityMap.insert("maxTime",       item->data(Qt::UserRole + 4));
        activityMap.insert("minDate",       item->data(Qt::UserRole + 5));
        activityMap.insert("maxDate",       item->data(Qt::UserRole + 6));
        activityMap.insert("minDuration",   item->data(Qt::UserRole + 7));
        activityMap.insert("maxDuration",   item->data(Qt::UserRole + 8));
        activityMap.insert("statusType",    item->data(Qt::UserRole + 9));
        activityMap.insert("availability",  item->data(Qt::UserRole + 10));
        activityMap.insert("minReaction",   item->data(Qt::UserRole + 11));
        activityMap.insert("maxReaction",   item->data(Qt::UserRole + 12));
        activityMap.insert("descriptions",  item->data(Qt::UserRole + 13));
        activityMap.insert("msgBefore",     item->data(Qt::UserRole + 14));
        activityMap.insert("msgAfter",      item->data(Qt::UserRole + 15));
        activityMap.insert("msgToRooms",    item->data(Qt::UserRole + 16));
        activityMap.insert("msgToChats",    item->data(Qt::UserRole + 17));


        activityDataList.append(activityMap);
    }

    this->dataFile->saveData(activityDataList);

    this->globalObj->addToLog(tr("Saved %1 activities.")
                              .arg(activityDataList.length()));
}

