/*  ADCD - A Diminutive CD player for GNU/Linux
    Copyright (C) 2004, 2005 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
*/
/*
    Return values: 0 for a normal exit, 1 for environmental problems
    (file not found, invalid flags, I/O errors, etc), 2 to indicate a
    corrupt or invalid input file, 3 for an internal consistency error
    (eg, bug) which caused adcd to panic.
*/

#include <cstdio>
#include <cstring>
#include <vector>
#include <getopt.h>

#include "msf_time.h"
#include "cd.h"
#include "player.h"


namespace {

// Date of this version: 2005-10-19

const char * invocation_name = 0;
const char * const program_name    = "ADCD";
const char * const program_version = "0.6";
const char * const program_year    = "2005";


void show_error( const char * msg, const int errcode = 0, const bool help = false ) throw()
  {
  if( msg && msg[0] != 0 )
    {
    std::fprintf( stderr, "%s: %s", program_name, msg );
    if( errcode > 0 ) std::fprintf( stderr, ": %s", strerror( errcode ) );
    std::fprintf( stderr, "\n" );
    }
  if( help && invocation_name && invocation_name[0] != 0 )
    std::fprintf( stderr, "Try `%s --help' for more information.\n", invocation_name );
  }


void show_help() throw()
  {
  std::printf( "%s - A Diminutive CD player for GNU/Linux.\n", program_name );
  std::printf( "If invoked without options enters interactive mode.\n" );
  std::printf( "\nUsage: %s [options] [device]\n", invocation_name );
  std::printf( "Options:\n" );
  std::printf( "  -h, --help             display this help and exit\n" );
  std::printf( "  -V, --version          output version information and exit\n" );
  std::printf( "  -i, --info             display info about current status and exit\n" );
  std::printf( "  -p, --play=<n>         start playing at track <n> and exit\n" );
  std::printf( "  -s, --stop             stop playing and exit\n" );
  std::printf( "  -u, --pause            pause/resume playing and exit\n" );
  std::printf( "  -v, --volume=[+-]<n>   adjust or set volume and exit\n" );
  std::printf( "\nKeys not shown on screen:\n" );
  std::printf( "  1 to 9             Start playing at track 1 to 9\n" );
  std::printf( "  0                  Start playing at track 10\n" );
  std::printf( "  F1 to F12          Start playing at track 11 to 22\n" );
  std::printf( "  up arrow           Seek backward 10 seconds\n" );
  std::printf( "  down arrow         Seek forward 10 seconds\n" );
  std::printf( "  Page up            Seek backward 30 seconds\n" );
  std::printf( "  Page down          Seek forward 30 seconds\n" );
  std::printf( "  Home               Seek backward 1 minute\n" );
  std::printf( "  End                Seek forward 1 minute\n" );
  std::printf( "  Ins                Seek backward 5 minutes\n" );
  std::printf( "  Del                Seek forward 5 minutes\n" );
  std::printf( "  + -                increase/decrease volume by  1\n" );
  std::printf( "  * /                increase/decrease volume by 10\n" );
  std::printf( "  L                  Loop mode (no_loop, Disc, Track)\n" );
  std::printf( "  M                  Change playing mode (Linear/Playlist)\n" );
  std::printf( "  R                  Generate random playlist\n" );
  std::printf( "  T                  Change displayed time\n" );
  std::printf( "\nKeys used for edit playlist:\n" );
  std::printf( "  arrow keys         Move the cursor\n" );
  std::printf( "  0..9, F1..F12      Insert corresponding track\n" );
  std::printf( "  + -                Increase/decrease track by  1\n" );
  std::printf( "  * /                Increase/decrease track by 10\n" );
  std::printf( "  Del                Delete track at cursor\n" );
  std::printf( "  Backspace          Delete track preceding cursor\n" );
  std::printf( "  Enter              Store playlist and exit edit mode\n" );
  std::printf( "  Q                  Discard changes and exit edit mode\n" );
  std::printf( "\nReport bugs to adcd-bug@nongnu.org\n" );
  }


void show_version() throw()
  {
  std::printf( "%s version %s\n", program_name, program_version );
  std::printf( "Copyright (C) %s Antonio Diaz Diaz.\n", program_year );
  std::printf( "This program is free software; you may redistribute it under the terms of\n" );
  std::printf( "the GNU General Public License.  This program has absolutely no warranty.\n" );
  }

} // end namespace


int main( int argc, char * argv[] ) throw()
  {
  enum Volcommand { vnone, vadjust, vset };
  Volcommand vc = vnone;
  int track = 1, volume = 0;
  bool info = false, pause = false, play = false, stop = false;
  const char *devname = "/dev/cdrom";
  invocation_name = argv[0];

  // scan arguments

  while( true )			// process options
    {
    static const struct option long_options[] =
      {
      {"help",         no_argument, 0, 'h'},
      {"info",         no_argument, 0, 'i'},
      {"pause",        no_argument, 0, 'u'},
      {"play",   required_argument, 0, 'p'},
      {"stop",         no_argument, 0, 's'},
      {"version",      no_argument, 0, 'V'},
      {"volume", required_argument, 0, 'v'},
      {0, 0, 0, 0}
      };

    int c = getopt_long( argc, argv, "Vhip:suv:", long_options, 0 );
    if( c == -1 ) break;		// all options processed

    switch( c )
      {
      case 'V':	show_version(); return 0;
      case 'h': show_help(); return 0;
      case 'i': info = true; break;
      case 'p': play = true; track = std::strtol( optarg, 0, 0 ); break;
      case 's': stop = true; break;
      case 'u': pause = true; break;
      case 'v': if( isdigit( optarg[0] ) ) vc = vset;
                else if( ( optarg[0] == '+' || optarg[0] == '-' ) && isdigit( optarg[1] ) )
                  vc = vadjust;
                else { show_error( "bad volume value" ); return 1; }
                volume = std::strtol( optarg, 0, 0 );
                break;
      case '?': show_error( 0, 0, true ); return 1;	// bad option
      default : show_error( argv[optind], 0, true ); return 1;
      }
    } // end process options

  if( optind < argc ) devname = argv[optind++];
  if( optind < argc )
    { show_error( "too many devices", 0, true ); return 1; }

  // end scan arguments

  CD cd( devname );
  bool interactive = true;

  if( vc != vnone )
    {
    interactive = false;
    if( vc == vadjust ) cd.volume( cd.volume() + volume );
    else if( vc == vset ) cd.volume( volume );
    std::printf( "New volume = %d\n", cd.volume() );
    }

  if( stop ) { cd.stop(); interactive = false; }
  if( play )
    {
    if( !cd.track( track, true ) ) { show_error( "bad track number" ); return 1; }
    interactive = false;
    }
  if( info ) { cd.show_info(); interactive = false; }
  if( pause ) { cd.pause(); interactive = false; if( info ) cd.show_info(); }

  if( interactive ) Player::main_loop( cd );
  return 0;
  }
