/*  ADCD - A Diminutive CD player for Linux
    Copyright (C) 2004 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <algorithm>
#include <cctype>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <ctime>
#include <vector>
#include <signal.h>
#include <unistd.h>
#include <ncurses.h>
#include "cd.h"

namespace {

const size_t maxsize = 100;

// Date of this version: 2004-08-21
const char *const program_version = "0.3";
const char *const program_year    = "2004";


void show_version() throw()
  {
  std::printf( "ADCD version %s\n", program_version );
  std::printf( "Copyright (C) %s Antonio Diaz Diaz.\n", program_year );
  std::printf( "This program is free software; you may redistribute it under the terms of\n" );
  std::printf( "the GNU General Public License.  This program has absolutely no warranty.\n" );
  }


void show_help( const char * program_name ) throw()
  {
  std::printf( "ADCD, A Diminutive CD player for Linux\n" );
  std::printf( "Usage: %s [options] [device]\n", program_name );
  std::printf( "Options:\n");
  std::printf( "  -h, --help         display this help and exit\n");
  std::printf( "  -V, --version      output version information and exit\n");
  std::printf( "\nKeys not shown on screen:\n");
  std::printf( "  1 to 9             Start playing at track 1 to 9\n");
  std::printf( "  0                  Start playing at track 10\n");
  std::printf( "  F1 to F12          Start playing at track 11 to 22\n");
  std::printf( "  up arrow           Seek backward 10 seconds\n");
  std::printf( "  down arrow         Seek forward 10 seconds\n");
  std::printf( "  Page up            Seek backward 30 seconds\n");
  std::printf( "  Page down          Seek forward 30 seconds\n");
  std::printf( "  Home               Seek backward 1 minute\n");
  std::printf( "  End                Seek forward 1 minute\n");
  std::printf( "  Ins                Seek backward 5 minutes\n");
  std::printf( "  Del                Seek forward 5 minutes\n");
  std::printf( "  + -                increase/decrease volume by  1\n");
  std::printf( "  * /                increase/decrease volume by 10\n");
  std::printf( "  L                  Loop mode on/off\n");
  std::printf( "  M                  Change playing mode (Linear/Playlist)\n");
  std::printf( "  R                  Generate random playlist\n");
  std::printf( "  T                  Change displayed time\n");
  std::printf( "\nKeys used for edit playlist:\n");
  std::printf( "  arrow keys         Move the cursor\n");
  std::printf( "  0..9, F1..F12      Insert corresponding track\n");
  std::printf( "  + -                Increase/decrease track by  1\n");
  std::printf( "  * /                Increase/decrease track by 10\n");
  std::printf( "  Del                Delete track at cursor\n");
  std::printf( "  Backspace          Delete track preceding cursor\n");
  std::printf( "  Enter              Store playlist and exit edit mode\n");
  std::printf( "  Q                  Discard changes and exit edit mode\n");
  std::printf( "\nReport bugs to ant_diaz@teleline.es\n");
  }


void dummy_endwin() throw() { endwin(); }

void initialize_ncurses() throw()
  {
  std::atexit( dummy_endwin );
  initscr();			// initializes curses data structures
  cbreak();			// read from keyboard a char at a time
  keypad( stdscr, true );	// enables single value for function keys
  nodelay( stdscr, false );	// forces getch() to wait for key
  noecho();			// disables echoing of getch()
  nonl();			// disables CR LF translation
  scrollok( stdscr, false );	// disables automatic scrolling
  }


int kbhit() throw()
  { 
  nodelay( stdscr, true );
  int c = getch();
  nodelay( stdscr, false );
  if( c == ERR ) return 0;
  else { ungetch( c ); return c; }
  }


void show_panel() throw()
  {
  mvaddstr( 0, 0, " Play [P]  [O] Open" );
  mvaddstr( 1, 0, "Pause [U]  [C] Close" );
  mvaddstr( 2, 0, " Stop [S]  [Q] Quit       Volume :" );
  mvaddstr( 3, 0, "Prev [<-]  [->] Next      Track  :" );
  mvaddstr( 4, 0, "                          Time   :" );
  mvaddstr( 5, 0, "Mode:" );
  }


void show_playlist( const std::vector< int > * pl, int index = -1 ) throw()
  {
  int size = ( pl ? pl->size() : 0 );
  for( int i = 0; i < size; ++i )
    {
    if( i == index ) attrset( A_REVERSE );
    mvprintw( 6 + (i / 10), 3 * (i % 10), "%2d", (*pl)[i] );
    if( i == index ) attrset( A_NORMAL );
    addch(' ');
    }
  for( size_t i = size; i < maxsize; ++i )
    mvaddstr( 6 + (i / 10), 3 * (i % 10), "   " );
  }


CD::Time_mode time_mode( bool next = false ) throw()
  {
  static CD::Time_mode mode = CD::relative;
  if( next ) switch( mode )
    {
    case CD::relative: mode = CD::rem_rel;  break;
    case CD::rem_rel : mode = CD::absolute; break;
    case CD::absolute: mode = CD::rem_abs;  break;
    case CD::rem_abs : mode = CD::relative; break;
    }
  return mode;
  }


void show_data( const CD & cd ) throw()
  {
  mvaddstr( 1, 26, cd.loop() ? "Loop" : "    " );
  mvprintw( 2, 35, "%3d ", cd.volume() );
  mvprintw( 3, 35, "%2d/%d  ", cd.track(), cd.last_track() );
  CD::Time_mode tt = time_mode();
  Msf_time msf = cd.time( tt );
  const char *s = ( tt == CD::relative || tt == CD::rem_rel ) ? "   " : "(d)";
  char c = ( tt == CD::rem_rel || tt == CD::rem_abs ) ? '-' : ' ';
  mvprintw( 4, 30, "%3s:%c%2d:%02d ", s, c, msf.minute(), msf.second() );
  mvaddstr( 5,  6, cd.linear() ? "Linear  " : "Playlist" );
  if( cd.linear() ) show_playlist( 0 );
  else show_playlist( &cd.playlist(), cd.index() );
  mvaddstr( 0, 26, cd.status_name() );
  wrefresh( stdscr );
  }


void edit_playlist( CD & cd ) throw()
  {
  static size_t i = 0;
  std::vector< int > temp = cd.playlist();
  if( i > temp.size() ) i = temp.size();
  while( true )
    {
    show_playlist( &temp );
    move( 6 + (i / 10), (3 * (i % 10)) + 1 );
    int key = std::toupper( getch() );
    if( key == '\r' ) { cd.playlist( temp ); break; }
    if( key == 'Q' ) break;
    int track = cd.first_track() - 1;
    if( std::isdigit( key ) ) track = ( key != '0' ) ? key - '0' : 10;
    else if( key >= KEY_F(1) && key <= KEY_F(12) ) track = 10 + key - KEY_F(0);
    if( track >= cd.first_track() && track <= cd.last_track() )
      {
      if( temp.size() >= maxsize ) temp[i] = track;
      else { temp.insert( temp.begin()+i, track ); if( i < maxsize-1 ) ++i; }
      }
    else switch( key )
      {
      case '-': if( i < temp.size() && temp[i] > cd.first_track() ) --temp[i]; break;
      case '+': if( i < temp.size() && temp[i] < cd.last_track() ) ++temp[i]; break;
      case '/': if( i < temp.size() )
                  temp[i] = std::max( temp[i] - 10, cd.first_track() ); break;
      case '*': if( i < temp.size() )
                  temp[i] = std::min( temp[i] + 10, cd.last_track() ); break;
      case KEY_HOME: i = 0; break;
      case KEY_END: i = std::min( temp.size(), maxsize - 1 ); break;
      case KEY_UP: if( i >= 10 ) i -= 10; break;
      case KEY_DOWN: if( i+9 < temp.size() && i+9 < maxsize - 1 ) i+=10; break;
      case KEY_LEFT: if( i > 0 ) --i; break;
      case KEY_RIGHT: if( i < temp.size() && i < maxsize - 1 ) ++i; break;
      case KEY_DC: if( i < temp.size() ) temp.erase( temp.begin() + i ); break;
      case KEY_BACKSPACE: if( i > 0 ) temp.erase( temp.begin() + --i ); break;
      }
    }
  }


void change_mode( CD & cd ) throw()
  {
  if( !cd.linear() ) { cd.linear( true ); return; }
  mvaddstr( 5, 15, "(editing list)" );
  edit_playlist( cd ); if( cd.playlist().size() > 0 ) cd.linear( false );
  mvaddstr( 5, 15, "              " );
  }


void random_playlist( CD & cd ) throw()
  {
  if( !cd.linear() )
    {
    mvaddstr( 5, 15, "(overwrite list ?)" );
    int ch = std::toupper( getch() );
    mvaddstr( 5, 15, "                  " );
    if( ch != 'Y' ) return;
    }
  std::vector< int > temp1, temp2;
  for( int i = cd.first_track(); i <= cd.last_track() && temp1.size() < maxsize; ++i )
    temp1.push_back( i );
  while( temp1.size() )
    {
    int i = std::rand() % temp1.size();
    temp2.push_back( temp1[i] ); temp1.erase( temp1.begin() + i );
    }
  cd.playlist( temp2 );
  if( cd.playlist().size() > 0 ) cd.linear( false );
  }


void main_loop( CD & cd ) throw()
  {
  int key = 0;
  while( key != 'Q' )
    {
    alarm( 1 );					// start polling cd drive
    while( kbhit() ) getch();			// clear input buffer
    do key = getch(); while( key < 0 );
    if( std::islower( key ) ) key = std::toupper( key );
    alarm( 0 );					// stop polling cd drive
    switch( key )
      {
      case '-': cd.volume( cd.volume() - 1 ); break;
      case '+': cd.volume( cd.volume() + 1 ); break;
      case '/': cd.volume( cd.volume() - 10 ); break;
      case '*': cd.volume( cd.volume() + 10 ); break;
      case 'C': cd.close(); break;
      case 'L': cd.loop( !cd.loop() ); break;
      case 'M': if( cd.tracks() ) change_mode( cd ); break;
      case 'O': cd.open(); break;
      case 'P': cd.play(); break;
      case 'R': if( cd.tracks() ) random_playlist( cd ); break;
      case 'S': cd.stop(); break;
      case 'T': time_mode( true ); break;
      case ' ':
      case 'U': cd.pause(); break;
      case KEY_LEFT  : cd.prev_track(); break;
      case KEY_RIGHT : cd.next_track(); break;
      case KEY_UP    : cd.seek_backward( 10 ); break;
      case KEY_DOWN  : cd.seek_forward( 10 ); break;
      case KEY_PPAGE : cd.seek_backward( 30 ); break;
      case KEY_NPAGE : cd.seek_forward( 30 ); break;
      case KEY_HOME  : cd.seek_backward( 60 ); break;
      case KEY_END   : cd.seek_forward( 60 ); break;
      case KEY_IC    : cd.seek_backward( 300 ); break;
      case KEY_DC    : cd.seek_forward( 300 ); break;
      case '0': cd.track( 10, true ); break;
      default : if( std::isdigit( key ) ) cd.track( key - '0', true );
                else if( key >= KEY_F(1) && key <= KEY_F(12) )
                       cd.track( 10 + key - KEY_F(0), true );
                else continue;
      }
    show_data( cd );
    }
  }


CD *cd_ptr;

void poll_cd_status( int )			// polls the cd drive
  {
  if( cd_ptr->read_status( false ) ) show_data( *cd_ptr );
  alarm( 1 );
  }


void restore_signals() throw()
  {
  signal( SIGALRM, SIG_DFL );
  }


void set_signals() throw()
  {
  std::atexit( restore_signals );
  signal( SIGALRM, poll_cd_status );
  }

} // end namespace


int main( int argc, char *argv[] ) throw()
  {
  char *str = "/dev/cdrom";
  for( int i = 1; i < argc; ++i )
    {
    if( !strcmp( argv[i], "-h" ) || !strcmp( argv[i], "--help" ) )
      { show_help( argv[0] ); return 0; }
    if( !strcmp( argv[i], "-V" ) || !strcmp( argv[i], "--version" ) )
      { show_version(); return 0; }
    str = argv[i]; break;
    }

  CD cd( str );
  cd_ptr = &cd;
  srand( time(0) );
  initialize_ncurses();
  show_panel(); show_data( cd );
  set_signals();
  main_loop( cd );
  return 0;
  }
