/* 
 *  Unix SMB/Netbios implementation.
 *  Version 1.9.
 *  RPC Pipe client / server routines
 *  Copyright (C) Andrew Tridgell              1992-1997,
 *  Copyright (C) Luke Kenneth Casson Leighton 1996-1997,
 *  Copyright (C) Paul Ashton                       1997.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../includes.h"
#include "../nterr.h"

extern int DEBUGLEVEL;


/*******************************************************************
makes an LSA_OBJ_ATTR structure.
********************************************************************/
void make_lsa_obj_attr(LSA_OBJ_ATTR *attr, uint32 attributes, uint32 sec_qos)
{
	if (attr == NULL) return;

	DEBUG(5,("make_lsa_obj_attr\n"));

	attr->len = 0x18; /* length of object attribute block, in bytes */
	attr->ptr_root_dir = 0;
	attr->ptr_obj_name = 0;
	attr->attributes = attributes;
	attr->ptr_sec_desc = 0;
	attr->sec_qos = sec_qos;
}

/*******************************************************************
reads or writes an LSA_OBJ_ATTR structure.
********************************************************************/
void lsa_io_obj_attr(char *desc,  LSA_OBJ_ATTR *attr, prs_struct *ps)
{
	int start;

	if (attr == NULL) return;

	prs_debug(ps, desc, "lsa_io_obj_attr");
	

	prs_align(ps);
	
	start = ps->offset;

	/* these pointers had _better_ be zero, because we don't know
	   what they point to!
	 */
	prs_uint32("len"         , ps, &(attr->len         )); /* 0x18 - length (in bytes) inc. the length field. */
	prs_uint32("ptr_root_dir", ps, &(attr->ptr_root_dir)); /* 0 - root directory (pointer) */
	prs_uint32("ptr_obj_name", ps, &(attr->ptr_obj_name)); /* 0 - object name (pointer) */
	prs_uint32("attributes"  , ps, &(attr->attributes  )); /* 0 - attributes (undocumented) */
	prs_uint32("ptr_sec_desc", ps, &(attr->ptr_sec_desc)); /* 0 - security descriptior (pointer) */
	prs_uint32("sec_qos"     , ps, &(attr->sec_qos     )); /* 0 - security quality of service */

	if (attr->len != ps->offset - start)
	{
		DEBUG(3,("lsa_io_obj_attr: length %lx does not match size %lx\n",
		         attr->len, ps->offset - start));
	}
}
/*******************************************************************
makes an LSA_Q_OPEN_POL structure.
********************************************************************/
void make_q_open_pol(LSA_Q_OPEN_POL *r_q, char *server_name,
			uint32 attributes, uint32 sec_qos,
			uint32 desired_access)
{
	if (r_q == NULL) return;

	DEBUG(5,("make_open_pol\n"));

	r_q->ptr = 1; /* undocumented pointer */

	make_unistr2     (&(r_q->uni_server_name), server_name, strlen(server_name));
	make_lsa_obj_attr(&(r_q->attr           ), attributes, sec_qos);

	r_q->des_access = desired_access;
}

/*******************************************************************
reads or writes an LSA_Q_OPEN_POL structure.
********************************************************************/
void lsa_io_q_open_pol(char *desc,  LSA_Q_OPEN_POL *r_q, prs_struct *ps)
{
	if (r_q == NULL) return;

	prs_debug(ps, desc, "lsa_io_q_open_pol");
	

	prs_uint32("ptr       ", ps, &(r_q->ptr       ));

	smb_io_unistr2 ("", &(r_q->uni_server_name), r_q->ptr, ps);
	lsa_io_obj_attr("", &(r_q->attr           ), ps);

	prs_uint32("des_access", ps, &(r_q->des_access));
}

/*******************************************************************
reads or writes an LSA_R_OPEN_POL structure.
********************************************************************/
void lsa_io_r_open_pol(char *desc,  LSA_R_OPEN_POL *r_p, prs_struct *ps)
{
	if (r_p == NULL) return;

	prs_debug(ps, desc, "lsa_io_r_open_pol");
	


	smb_io_pol_hnd("", &(r_p->pol), ps);

	prs_uint32("status", ps, &(r_p->status));
}

/*******************************************************************
makes an LSA_Q_QUERY_INFO structure.
********************************************************************/
void make_q_query(LSA_Q_QUERY_INFO *q_q, POLICY_HND *hnd, uint16 info_class)
{
	if (q_q == NULL || hnd == NULL) return;

	DEBUG(5,("make_q_query\n"));

	memcpy(&(q_q->pol), hnd, sizeof(q_q->pol));

	q_q->info_class = info_class;
}

/*******************************************************************
reads or writes an LSA_Q_QUERY_INFO structure.
********************************************************************/
void lsa_io_q_query(char *desc,  LSA_Q_QUERY_INFO *q_q, prs_struct *ps)
{
	if (q_q == NULL) return;

	prs_debug(ps, desc, "lsa_io_q_query");
	

	smb_io_pol_hnd("", &(q_q->pol), ps);

	prs_uint16("info_class", ps, &(q_q->info_class));
}

/*******************************************************************
reads or writes an LSA_Q_ENUM_TRUST_DOM structure.
********************************************************************/
void lsa_io_q_enum_trust_dom(char *desc,  LSA_Q_ENUM_TRUST_DOM *q_e, prs_struct *ps)
{
	if (q_e == NULL) return;

	prs_debug(ps, desc, "lsa_io_q_enum_trust_dom");
	


	smb_io_pol_hnd("", &(q_e->pol), ps);

	prs_uint32("enum_context ", ps, &(q_e->enum_context ));
	prs_uint32("preferred_len", ps, &(q_e->preferred_len));
}

/*******************************************************************
makes an LSA_R_ENUM_TRUST_DOM structure.
********************************************************************/
void make_r_enum_trust_dom(LSA_R_ENUM_TRUST_DOM *r_e,
				uint32 enum_context, char *domain_name, char *domain_sid,
				uint32 status)
{
	if (r_e == NULL) return;

	DEBUG(5,("make_r_enum_trust_dom\n"));

	r_e->enum_context = enum_context;

	if (status == 0)
	{
		int len_domain_name = strlen(domain_name);

		r_e->num_domains  = 1;
		r_e->ptr_enum_domains = 1;
		r_e->num_domains2 = 1;

		make_uni_hdr2(&(r_e->hdr_domain_name ), len_domain_name, len_domain_name, 4);
		make_unistr2 (&(r_e->uni_domain_name ), domain_name, len_domain_name);
		make_dom_sid2(&(r_e->other_domain_sid), domain_sid);
	}
	else
	{
		r_e->num_domains = 0;
		r_e->ptr_enum_domains = 0;
	}

	r_e->status = status;
}

/*******************************************************************
reads or writes an LSA_R_ENUM_TRUST_DOM structure.
********************************************************************/
void lsa_io_r_enum_trust_dom(char *desc,  LSA_R_ENUM_TRUST_DOM *r_e, prs_struct *ps)
{
	if (r_e == NULL) return;

	prs_debug(ps, desc, "lsa_io_r_enum_trust_dom");
	

	prs_uint32("enum_context    ", ps, &(r_e->enum_context    ));
	prs_uint32("num_domains     ", ps, &(r_e->num_domains     ));
	prs_uint32("ptr_enum_domains", ps, &(r_e->ptr_enum_domains));

	if (r_e->ptr_enum_domains != 0)
	{
		prs_uint32("num_domains2", ps, &(r_e->num_domains2));
		smb_io_unihdr2 ("", &(r_e->hdr_domain_name ), ps);
		smb_io_unistr2 ("", &(r_e->uni_domain_name ), r_e->hdr_domain_name.buffer, ps);
		smb_io_dom_sid2("", &(r_e->other_domain_sid), ps);
	}

	prs_uint32("status", ps, &(r_e->status));
}

/*******************************************************************
makes an LSA_Q_CLOSE structure.
********************************************************************/
void make_lsa_q_close(LSA_Q_CLOSE *q_c, POLICY_HND *hnd)
{
	if (q_c == NULL || hnd == NULL) return;

	DEBUG(5,("make_lsa_q_close\n"));

	memcpy(&(q_c->pol), hnd, sizeof(q_c->pol));
}


/*******************************************************************
reads or writes an LSA_Q_CLOSE structure.
********************************************************************/
void lsa_io_q_close(char *desc,  LSA_Q_CLOSE *q_c, prs_struct *ps)
{
	if (q_c == NULL) return;

	prs_debug(ps, desc, "lsa_io_q_close");
	

	smb_io_pol_hnd("", &(q_c->pol), ps);
}

/*******************************************************************
makes an LSA_R_CLOSE structure.
********************************************************************/
void make_lsa_r_close(LSA_R_CLOSE *q_r, POLICY_HND *hnd)
{
	if (q_r == NULL || hnd == NULL) return;

	DEBUG(5,("make_lsa_r_close\n"));

	memcpy(&(q_r->pol), hnd, sizeof(q_r->pol));
}


/*******************************************************************
reads or writes an LSA_R_CLOSE structure.
********************************************************************/
void lsa_io_r_close(char *desc,  LSA_R_CLOSE *r_c, prs_struct *ps)
{
	if (r_c == NULL) return;

	prs_debug(ps, desc, "lsa_io_r_close");
	

	smb_io_pol_hnd("", &(r_c->pol), ps);

	prs_uint32("status", ps, &(r_c->status));
}

/*******************************************************************
reads or writes an LSA_Q_QUERY_INFO structure.
********************************************************************/
void lsa_io_r_query(char *desc,  LSA_R_QUERY_INFO *r_q, prs_struct *ps)
{
	if (r_q == NULL) return;

	prs_debug(ps, desc, "lsa_io_r_query");
	

	prs_uint32("undoc_buffer", ps, &(r_q->undoc_buffer));

	if (r_q->undoc_buffer != 0)
	{
		prs_uint16("info_class", ps, &(r_q->info_class));

		switch (r_q->info_class)
		{
			case 3:
			{
				smb_io_dom_query_3("", &(r_q->dom.id3), ps);
				break;
			}
			case 5:
			{
				smb_io_dom_query_5("", &(r_q->dom.id3), ps);
				break;
			}
			default:
			{
				/* PANIC! */
				break;
			}
		}
	}

	prs_uint32("status", ps, &(r_q->status));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void lsa_io_q_lookup_sids(char *desc,  LSA_Q_LOOKUP_SIDS *q_s, prs_struct *ps)
{
	int i;

	if (q_s == NULL) return;

	prs_debug(ps, desc, "lsa_io_q_lookup_sids");
	

	prs_align(ps);
	
    smb_io_pol_hnd("", &(q_s->pol_hnd), ps); /* policy handle */

	prs_uint32("num_entries          ", ps, &(q_s->num_entries));
	prs_uint32("buffer_dom_sid       ", ps, &(q_s->buffer_dom_sid)); /* undocumented domain SID buffer pointer */
	prs_uint32("buffer_dom_name      ", ps, &(q_s->buffer_dom_name)); /* undocumented domain name buffer pointer */

	for (i = 0; i < q_s->num_entries; i++)
	{	
		fstring temp;
		sprintf(temp, "buffer_lookup_sids[%d] ", i);
		prs_uint32(temp, ps, &(q_s->buffer_lookup_sids[i])); /* undocumented domain SID pointers to be looked up. */
	}

	for (i = 0; i < q_s->num_entries; i++)
	{
		smb_io_dom_sid2("", &(q_s->dom_sids[i]), ps); /* domain SIDs to be looked up. */
	}

	prs_uint8s (False, "undoc                ", ps, q_s->undoc, 16);
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void lsa_io_r_lookup_sids(char *desc,  LSA_R_LOOKUP_SIDS *r_s, prs_struct *ps)
{
	int i;

	if (r_s == NULL) return;

	prs_debug(ps, desc, "lsa_io_r_lookup_sids");
	

	prs_align(ps);
	
	smb_io_dom_r_ref("", &(r_s->dom_ref), ps); /* domain reference info */

	prs_uint32("num_entries ", ps, &(r_s->num_entries));
	prs_uint32("undoc_buffer", ps, &(r_s->undoc_buffer));
	prs_uint32("num_entries2", ps, &(r_s->num_entries2));

	for (i = 0; i < r_s->num_entries2; i++)
	{
		smb_io_dom_str_sid("", &(r_s->str_sid[i]), ps); /* domain SIDs being looked up */
	}

	prs_uint32("num_entries3", ps, &(r_s->num_entries3));

	prs_uint32("status      ", ps, &(r_s->status));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void lsa_io_q_lookup_rids(char *desc,  LSA_Q_LOOKUP_RIDS *q_r, prs_struct *ps)
{
	int i;

	if (q_r == NULL) return;

	prs_debug(ps, desc, "lsa_io_q_lookup_rids");
	

	prs_align(ps);
	
    smb_io_pol_hnd("", &(q_r->pol_hnd), ps); /* policy handle */

	prs_uint32("num_entries    ", ps, &(q_r->num_entries));
	prs_uint32("num_entries2   ", ps, &(q_r->num_entries2));
	prs_uint32("buffer_dom_sid ", ps, &(q_r->buffer_dom_sid)); /* undocumented domain SID buffer pointer */
	prs_uint32("buffer_dom_name", ps, &(q_r->buffer_dom_name)); /* undocumented domain name buffer pointer */

	for (i = 0; i < q_r->num_entries; i++)
	{
		smb_io_dom_name("", &(q_r->lookup_name[i]), ps); /* names to be looked up */
	}

	prs_uint8s (False, "undoc          ", ps, q_r->undoc, UNKNOWN_LEN);
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void lsa_io_r_lookup_rids(char *desc,  LSA_R_LOOKUP_RIDS *r_r, prs_struct *ps)
{
	int i;

	if (r_r == NULL) return;

	prs_debug(ps, desc, "lsa_io_r_lookup_rids");
	

	prs_align(ps);
	
	smb_io_dom_r_ref("", &(r_r->dom_ref), ps); /* domain reference info */

	prs_uint32("num_entries ", ps, &(r_r->num_entries));
	prs_uint32("undoc_buffer", ps, &(r_r->undoc_buffer));
	prs_uint32("num_entries2", ps, &(r_r->num_entries2));

	for (i = 0; i < r_r->num_entries2; i++)
	{
		smb_io_dom_rid2("", &(r_r->dom_rid[i]), ps); /* domain RIDs being looked up */
	}

	prs_uint32("num_entries3", ps, &(r_r->num_entries3));

	prs_uint32("status      ", ps, &(r_r->status));
}

/*************************************************************************
 make_lsa_user_info3
 *************************************************************************/
void make_lsa_user_info3(LSA_USER_INFO_3 *usr,

	NTTIME *logon_time,
	NTTIME *logoff_time,
	NTTIME *kickoff_time,
	NTTIME *pass_last_set_time,
	NTTIME *pass_can_change_time,
	NTTIME *pass_must_change_time,

	char *user_name,
	char *full_name,
	char *logon_script,
	char *profile_path,
	char *home_dir,
	char *dir_drive,

	uint16 logon_count,
	uint16 bad_pw_count,

	uint32 user_id,
	uint32 group_id,
	uint32 num_groups,
	DOM_GID *gids,
	uint32 user_flgs,

	char sess_key[16],

	char *logon_srv,
	char *logon_dom,

	char *dom_sid,
	char *other_sids)
{
	/* only cope with one "other" sid, right now. */
	/* need to count the number of space-delimited sids */
	int i;
	int num_other_sids = 0;

	int len_user_name    = strlen(user_name   );
	int len_full_name    = strlen(full_name   );
	int len_logon_script = strlen(logon_script);
	int len_profile_path = strlen(profile_path);
	int len_home_dir     = strlen(home_dir    );
	int len_dir_drive    = strlen(dir_drive   );

	int len_logon_srv    = strlen(logon_srv);
	int len_logon_dom    = strlen(logon_dom);

	usr->ptr_user_info = 1; /* yes, we're bothering to put USER_INFO data here */

	usr->logon_time            = *logon_time;
	usr->logoff_time           = *logoff_time;
	usr->kickoff_time          = *kickoff_time;
	usr->pass_last_set_time    = *pass_last_set_time;
	usr->pass_can_change_time  = *pass_can_change_time;
	usr->pass_must_change_time = *pass_must_change_time;

	make_uni_hdr(&(usr->hdr_user_name   ), len_user_name   , len_user_name   , 4);
	make_uni_hdr(&(usr->hdr_full_name   ), len_full_name   , len_full_name   , 4);
	make_uni_hdr(&(usr->hdr_logon_script), len_logon_script, len_logon_script, 4);
	make_uni_hdr(&(usr->hdr_profile_path), len_profile_path, len_profile_path, 4);
	make_uni_hdr(&(usr->hdr_home_dir    ), len_home_dir    , len_home_dir    , 4);
	make_uni_hdr(&(usr->hdr_dir_drive   ), len_dir_drive   , len_dir_drive   , 4);

	usr->logon_count = logon_count;
	usr->bad_pw_count = bad_pw_count;

	usr->user_id = user_id;
	usr->group_id = group_id;
	usr->num_groups = num_groups;
	usr->buffer_groups = 1; /* indicates fill in groups, below, even if there are none */
	usr->user_flgs = user_flgs;

	if (sess_key != NULL)
	{
		memcpy(usr->user_sess_key, sess_key, sizeof(usr->user_sess_key));
	}
	else
	{
		bzero(usr->user_sess_key, sizeof(usr->user_sess_key));
	}

	make_uni_hdr(&(usr->hdr_logon_srv), len_logon_srv, len_logon_srv, 4);
	make_uni_hdr(&(usr->hdr_logon_dom), len_logon_dom, len_logon_dom, 4);

	usr->buffer_dom_id = dom_sid ? 1 : 0; /* yes, we're bothering to put a domain SID in */

	bzero(usr->padding, sizeof(usr->padding));

	num_other_sids = make_dom_sid2s(other_sids, usr->other_sids, LSA_MAX_SIDS);

	usr->num_other_sids = num_other_sids;
	usr->buffer_other_sids = num_other_sids != 0 ? 1 : 0; 
	
	make_unistr2(&(usr->uni_user_name   ), user_name   , len_user_name   );
	make_unistr2(&(usr->uni_full_name   ), full_name   , len_full_name   );
	make_unistr2(&(usr->uni_logon_script), logon_script, len_logon_script);
	make_unistr2(&(usr->uni_profile_path), profile_path, len_profile_path);
	make_unistr2(&(usr->uni_home_dir    ), home_dir    , len_home_dir    );
	make_unistr2(&(usr->uni_dir_drive   ), dir_drive   , len_dir_drive   );

	usr->num_groups2 = num_groups;
	for (i = 0; i < num_groups; i++)
	{
		usr->gids[i] = gids[i];
	}

	make_unistr2(&(usr->uni_logon_srv), logon_srv, len_logon_srv);
	make_unistr2(&(usr->uni_logon_dom), logon_dom, len_logon_dom);

	make_dom_sid2(&(usr->dom_sid), dom_sid);
	/* "other" sids are set up above */
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void lsa_io_user_info3(char *desc,  LSA_USER_INFO_3 *usr, prs_struct *ps)
{
	int i;

	if (usr == NULL) return;

	prs_debug(ps, desc, "lsa_io_lsa_user_info");
	

	prs_align(ps);
	
	prs_uint32("ptr_user_info ", ps, &(usr->ptr_user_info));

	if (usr->ptr_user_info != 0)
	{
		smb_io_time("time", &(usr->logon_time)           , ps); /* logon time */
		smb_io_time("time", &(usr->logoff_time)          , ps); /* logoff time */
		smb_io_time("time", &(usr->kickoff_time)         , ps); /* kickoff time */
		smb_io_time("time", &(usr->pass_last_set_time)   , ps); /* password last set time */
		smb_io_time("time", &(usr->pass_can_change_time) , ps); /* password can change time */
		smb_io_time("time", &(usr->pass_must_change_time), ps); /* password must change time */

		smb_io_unihdr("unihdr", &(usr->hdr_user_name)   , ps); /* username unicode string header */
		smb_io_unihdr("unihdr", &(usr->hdr_full_name)   , ps); /* user's full name unicode string header */
		smb_io_unihdr("unihdr", &(usr->hdr_logon_script), ps); /* logon script unicode string header */
		smb_io_unihdr("unihdr", &(usr->hdr_profile_path), ps); /* profile path unicode string header */
		smb_io_unihdr("unihdr", &(usr->hdr_home_dir)    , ps); /* home directory unicode string header */
		smb_io_unihdr("unihdr", &(usr->hdr_dir_drive)   , ps); /* home directory drive unicode string header */

		prs_uint16("logon_count   ", ps, &(usr->logon_count ));  /* logon count */
		prs_uint16("bad_pw_count  ", ps, &(usr->bad_pw_count)); /* bad password count */

		prs_uint32("user_id       ", ps, &(usr->user_id      ));       /* User ID */
		prs_uint32("group_id      ", ps, &(usr->group_id     ));      /* Group ID */
		prs_uint32("num_groups    ", ps, &(usr->num_groups   ));    /* num groups */
		prs_uint32("buffer_groups ", ps, &(usr->buffer_groups)); /* undocumented buffer pointer to groups. */
		prs_uint32("user_flgs     ", ps, &(usr->user_flgs    ));     /* user flags */

		prs_uint8s (False, "user_sess_key", ps, usr->user_sess_key, 16); /* unused user session key */

		smb_io_unihdr("unihdr", &(usr->hdr_logon_srv), ps); /* logon server unicode string header */
		smb_io_unihdr("unihdr", &(usr->hdr_logon_dom), ps); /* logon domain unicode string header */

		prs_uint32("buffer_dom_id ", ps, &(usr->buffer_dom_id)); /* undocumented logon domain id pointer */
		prs_uint8s (False, "padding       ", ps, usr->padding, 40); /* unused padding bytes? */

		prs_uint32("num_other_sids", ps, &(usr->num_other_sids)); /* 0 - num_sids */
		prs_uint32("buffer_other_sids", ps, &(usr->buffer_other_sids)); /* NULL - undocumented pointer to SIDs. */
		
		smb_io_unistr2("unistr2", &(usr->uni_user_name)   , usr->hdr_user_name   .buffer, ps); /* username unicode string */
		smb_io_unistr2("unistr2", &(usr->uni_full_name)   , usr->hdr_full_name   .buffer, ps); /* user's full name unicode string */
		smb_io_unistr2("unistr2", &(usr->uni_logon_script), usr->hdr_logon_script.buffer, ps); /* logon script unicode string */
		smb_io_unistr2("unistr2", &(usr->uni_profile_path), usr->hdr_profile_path.buffer, ps); /* profile path unicode string */
		smb_io_unistr2("unistr2", &(usr->uni_home_dir)    , usr->hdr_home_dir    .buffer, ps); /* home directory unicode string */
		smb_io_unistr2("unistr2", &(usr->uni_dir_drive)   , usr->hdr_dir_drive   .buffer, ps); /* home directory drive unicode string */

		prs_align(ps);
		prs_uint32("num_groups2   ", ps, &(usr->num_groups2));        /* num groups */
		for (i = 0; i < usr->num_groups2; i++)
		{
			smb_io_gid("", &(usr->gids[i]), ps); /* group info */
		}

		smb_io_unistr2("unistr2", &( usr->uni_logon_srv), usr->hdr_logon_srv.buffer, ps); /* logon server unicode string */
		smb_io_unistr2("unistr2", &( usr->uni_logon_dom), usr->hdr_logon_srv.buffer, ps); /* logon domain unicode string */

		smb_io_dom_sid2("", &(usr->dom_sid), ps);           /* domain SID */

		for (i = 0; i < usr->num_other_sids; i++)
		{
			smb_io_dom_sid2("", &(usr->other_sids[i]), ps); /* other domain SIDs */
		}
	}
}

