/* test_smb.c                                                               */
/* Copyright (C) Richard Sharpe, 1996                                       */
/*                                                                          */
/* We test some things out in SMB servers ...                               */

/* A test for smblib that does nasty things ...... */

#include <sys/types.h>
#include <unistd.h>

#include "smblib.h"

#define DEFAULT "msfax"

struct error_info {
  int SMBlib_Error;
  int error_details;
  int errno_info;
} error_info;

#define strerror_r(err_num, str, str_len) \
          strncpy(strerror(err_num), str, str_len - 1)

void *SMB_Create_Tree_Handle(void *con);

#define NOT_EQ(e1, e2) ((e1.SMBlib_Error != e2.SMBlib_Error) | \
			(e1.error_details != e2.error_details) | \
			(e1.errno_info    != e2.errno_info))

char err_string[1024];

int logon = FALSE;
int verbose = FALSE;

char *prots_to_use[1000];
int prot_count = 0;

char data[65536] = "Now is the time for all birds to roost.";
char *SMB_Prots[] = {"PC NETWORK PROGRAM 1.0", 
			    "MICROSOFT NETWORKS 1.03",
			    "MICROSOFT NETWORKS 3.0",
		            "DOS LANMAN1.0",
			    "LANMAN1.0",
			    "DOS LM1.2X002",
			    "LM1.2X002",
		            "DOS LANMAN2.1",
		            "LANMAN2.1",
			    "NT LM 0.12",
			    "NT LANMAN 1.0",
			    NULL};

void usage()

{

  fprintf(stderr, "test_discon-nt [-b <start-TID>] [-e <end-TID>] [-n <prot>]\\\n");
  fprintf(stderr, "               [-l] [-u <user>] [-p <password>] server\n");
  fprintf(stderr, "Where: -b <start-TID> sets the TID to start testing with\n");
  fprintf(stderr, "                      defaults to 0\n");
  fprintf(stderr, "Where: -e <end-TID>   sets the last TID to test\n");
  fprintf(stderr, "                      defaults to 0xFFFF\n");
  fprintf(stderr, "Where: -n <prot>      adds another protocol to negotiate\n");
  fprintf(stderr, "                      you can have multiple -n entries\n");
  fprintf(stderr, "                      defaults to a builtin list\n");
  fprintf(stderr, "Where: -l             says to do a logon to the server\n");
  fprintf(stderr, "                      defaults to not logging on\n");

}


void print_errors(struct error_info errors)

{ char err_string[1024];

  if (errors.SMBlib_Error == SMBlibE_Remote) {

    SMB_Get_SMB_Error_Msg(SMBlib_Error_Class(errors.error_details),
			  SMBlib_Error_Code(errors.error_details),
			  err_string,
			  sizeof(err_string) - 1);

    fprintf(stderr, "  %s\n", err_string);

  }
  else {

    if (errors.SMBlib_Error < 0) { /* Pick up the extra stuff */

      SMB_Get_Error_Msg(-errors.SMBlib_Error,
			err_string, sizeof(err_string) -1);

      fprintf(stderr, " %s\n", err_string);

      RFCNB_Get_Error_Msg(errors.error_details, 
			err_string, sizeof(err_string) -1);

      fprintf(stderr, "  %s\n", err_string);

      strerror_r(errors.errno_info, err_string, sizeof(err_string) -1);
      fprintf(stderr, "    %s\n", err_string);

    }
    else {

      SMB_Get_Error_Msg(errors.SMBlib_Error, 
			err_string, sizeof(err_string) - 1);

      fprintf(stderr, "  %s\n", err_string);

      strerror_r(errors.errno_info, err_string, sizeof(err_string) -1);
      fprintf(stderr, "    %s\n", err_string);

    }
  }


}

void *connect_server(char *server, char *user, char *password)

{ void *con;
  int prot, SMB_Error;
  char **use_prots;

  fprintf(stderr, "Connecting to server \\\\%s ...\n", server);

  con = SMB_Connect_Server(NULL, server);

  if (con == NULL) { /* Error ... */

    fprintf(stderr, "Unable to connect to server \"%s\" ... \n", server);
    SMB_Get_Error_Msg(SMB_Get_Last_Error(), 
		      err_string, sizeof(err_string) - 1);
    printf("Error in calling: %s ...\n", err_string);
    exit(1);

  }

  /* Now do a negotiate ... */

  fprintf(stderr, "Now we negotiate a protocol ... \n");

  if (prot_count > 0) /* Then use what asked for */
    use_prots = prots_to_use;
  else
    use_prots = SMB_Prots;

  if (SMB_Negotiate(con, use_prots) < 0) { /* An error */

    fprintf(stderr, "Unable to negotiate a protocol of the %i offered.\n",
	    prot_count);
    if (SMB_Get_Last_Error() == SMBlibE_Remote) {

      SMB_Error = SMB_Get_Last_SMB_Err();
      SMB_Get_SMB_Error_Msg(SMBlib_Error_Class(SMB_Error),
			    SMBlib_Error_Code(SMB_Error),
			    err_string,
			    sizeof(err_string) - 1);

    }
    else {
      SMB_Get_Error_Msg(SMB_Get_Last_Error(), err_string, sizeof(err_string) - 1);
    }


    printf("  %s ...\n", err_string);
    exit(1);

  }

  prot = SMB_Get_Protocol_IDX(con);

  fprintf(stderr, "  The protocol accepted was %i, %s, of %i offered.\n", 
	  prot, use_prots[prot], prot_count);

  /* Now we do a logon ... */

  if (logon == TRUE) {

    fprintf(stderr, "Now try to logon as %s ...\n", user);

    if (SMB_Logon_Server(con, user, password) < 0) {

      fprintf(stderr, "Unable to logon to server ...\n");

      if (SMB_Get_Last_Error() == SMBlibE_Remote) {

	SMB_Error = SMB_Get_Last_SMB_Err();
	SMB_Get_SMB_Error_Msg(SMBlib_Error_Class(SMB_Error),
			      SMBlib_Error_Code(SMB_Error),
			      err_string,
			      sizeof(err_string) - 1);

      }
      else {
	SMB_Get_Error_Msg(SMB_Get_Last_Error(), err_string, 
			  sizeof(err_string) - 1);
      }

      fprintf(stderr, "  %s\n", err_string);
      fprintf(stderr, "This error ignored. We will keep going and hope ...\n");
      
    }
  }
  else {

    fprintf(stderr, "We will not try to logon ...\n");

  }

  return(con);

}

/* Do various tests on tree disconnects ... we step through TID space    */

void test_tree_disconnect(void *con, char *service, char *server, char* user,
			  char *password, int start_tid, int end_tid)

{ int i, SMB_Error, error;
  char err_string[1024];
  void *tree;

  struct error_info *err_list, last_error;

  /* First, allocate space for errors */

  err_list = (struct error_info *)malloc(((end_tid - start_tid) + 1) * 
					 sizeof(struct error_info));

  if (err_list == NULL) { /* Could not get space ... Abort */

    fprintf(stderr, "Could not get space to store the errors for the TID range ");
    fprintf(stderr, "you requested: \n\t%i to %i\n", start_tid, end_tid);
    exit(1);

  }

  fprintf(stderr, "We are now going to go through the TID range you requested");
  fprintf(stderr, "\n(%i to %i) requesting a disconnect without there being ",
	  start_tid, end_tid);
  fprintf(stderr, "anything connected on\nthe TIDs. This may take a while.\n\n");

  tree = SMB_Create_Tree_Handle(con);

  for (i = start_tid; i <= end_tid; i++) {

    SMB_Set_Tree_Con(tree, con);     /* Make sure this is right     */
    SMB_Set_Tree_TID(tree, i);       /* Set the tid to what we want */

    if (verbose == TRUE) {

      fprintf(stderr, "We are trying TID %i ...\n", i);

    }

    if (SMB_TreeDisconnect(tree, FALSE) < 0) { /* Well, we got an error ... */

      if (SMB_Get_Last_Error() == SMBlibE_Remote) {


	err_list[i - start_tid].SMBlib_Error = SMBlibE_Remote;
	err_list[i - start_tid].error_details = SMB_Get_Last_SMB_Err();
	err_list[i - start_tid].errno_info = errno;

      }
      else {

	err_list[i - start_tid].SMBlib_Error = SMB_Get_Last_Error();
	err_list[i - start_tid].error_details = RFCNB_Get_Last_Error();
	err_list[i - start_tid].errno_info = errno;

      }

      error = abs(SMB_Get_Last_Error());

      if (error == SMBlibE_SendFailed || error == SMBlibE_RecvFailed) {

	/* Some sort of problem at the protocol layer, drop the connection */
	/* and reconnect                                                   */

	SMB_Discon(con, FALSE);

	con = connect_server(server, user, password);
      }

    }
    else { /* We should have got an error as there was nothing connected */

	err_list[i - start_tid].SMBlib_Error = 0;   /* No errors ... */
	err_list[i - start_tid].error_details = 0; 
	err_list[i - start_tid].errno_info = errno;

    }

  }

  /* Well, now process the error messages. Print out a list of the errors */

  last_error = err_list[0];

  fprintf(stderr, "The results of the test were ...\n");
  fprintf(stderr, "  From TID %i to ", start_tid);

  for (i = start_tid + 1; i <= end_tid; i++) {

    if (NOT_EQ(last_error,err_list[i - start_tid])) { /* Finish the message */

      /* Print out the end of the range and the error message ... */

      fprintf(stderr, "%i we got the following message:\n", i - 1);

      print_errors(last_error);

      last_error = err_list[i - start_tid];

      fprintf(stderr, "\n  From TID %i to ", i);

    }

  }

  /* Print the last entry ... */

  fprintf(stderr, "%i we got the following message:\n", end_tid);

  print_errors(last_error);

}

main(int argc, char *argv[])

{ void *con, *file;

  int written, actual_size, i, prot, start_tid = 0, end_tid = 0x0FFFF;
  int opt;
  extern char *optarg;
  extern int optind;
  char server[80], service[80], called[80];
  char service_unc[80], password[80], user[80];
  actual_size = 8192;

  strcpy(password, "nurk");
  strcpy(user, "fred");

  while ((opt = getopt(argc, argv, "n:b:e:p:u:lv")) != EOF) {

    switch (opt) {
    case 'b':       /* beginning tid to use */

      start_tid = atoi(optarg);
      break;

    case 'e':       /* Ending TID to use */

      end_tid = atoi(optarg);
      break;

    case 'l':     /* do a logon */
      logon = TRUE;
      break;

    case 'p':     /* Pick up the password */
      strncpy(password, optarg, sizeof(password) - 1);
      break;

    case 'u':     /* pick up the username */
      strncpy(user, optarg, sizeof(user) - 1);
      break;

    case 'n':     /* Pick up another protocol string to negotiate */
      prots_to_use[prot_count] = optarg;
      prot_count = prot_count + 1;
      break;

    case 'v':     /* Set verbose flag */
      verbose = TRUE;
      break;

    default:

      usage();
      exit(1);
      break;
    }

  }

  if (optind < argc) { /* Some more parameters, assume is the server */
    strncpy(server, argv[optind], sizeof(server) - 1);
    optind++;

    if (optind < argc) { /* Pick up password ... */
      strncpy(password, argv[optind], sizeof(password) -1);
    }
    else {
      strcpy(password, "nurk");
    }
  }
  else {
    strcpy(server, "nemesis");
  }

  /* Now go to work ... */

  SMB_Init();

  con = connect_server(server, user, password);

  sprintf(service_unc, "\\\\%s\\%s", server, service);

  test_tree_disconnect(con, service, server, user, password, 
		       start_tid, end_tid);

  SMB_Discon(con, FALSE);

}
