/*
    Windows NT Security functions library.
    Copyright (C) 1995  Jeremy R. Allison

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Log:	sidobj.cpp,v $
// Revision 1.4  95/10/17  18:14:29  18:14:29  jra (Jeremy Allison)
// Added lsa code.
// 
// Revision 1.3  1995/09/27  17:34:50  jra
// Fixed incorrect comparison when getting user name/domain.
//
// Revision 1.2  1995/09/07  00:39:09  jra
// Fixed error codes.
//
// Revision 1.1  1995/06/30  18:32:55  jra
// Initial revision
//^M
*/

#include "autoheap.h"
#include "sidobj.h"

// Class functions for the embedded class SIDinfo

// Create a SIDinfo from a SID *
SIDobj::SIDinfo::SIDinfo(const SID *psid, const TCHAR *mach) : name_(0), domain_(0), 
		type_(SidTypeUser), refcnt_(1) {

	DWORD namesize = 0, domsize = 0;
	LookupAccountSid(mach, (SID *)psid, name_, &namesize, domain_, &domsize,
					   &type_);
	if( GetLastError() != ERROR_INSUFFICIENT_BUFFER)
		throw (DWORD)GetLastError();

	autoHeapChar namep = new char [ namesize ];
	if(namep.getPtr() == 0) {
		SetLastError(ERROR_OUTOFMEMORY);
		throw (DWORD)(ERROR_OUTOFMEMORY);
	}
	autoHeapChar domp = new char [ domsize ];
	if(domp.getPtr() == 0) {
		SetLastError(ERROR_OUTOFMEMORY);
		throw (DWORD)(ERROR_OUTOFMEMORY);
	}
	if(!LookupAccountSid(mach, (SID *)psid, (TCHAR *)namep.getPtr(), &namesize, 
										 (TCHAR *)domp.getPtr(), &domsize, &type_))
		throw (DWORD)GetLastError();

	name_ = namep.relinquishOwnership();
	domain_ = domp.relinquishOwnership();
}

// Destructor.
SIDobj::SIDinfo::~SIDinfo() {
	delete [] name_;
	delete [] domain_;
}

//
// SIDobj functions.
//

// Ensure we have a valid SIDinfo pointer.
BOOL SIDobj::validSIDinfo() {
	if(psidinfo_ == 0) {
		DWORD err = 0;
		try {
			psidinfo_ = new SIDinfo(psid_, mach_lookup_);
		} catch (DWORD caerr) {
			err = caerr;
		}
		if(err) {
			SetLastError(err);
			return FALSE;
		}
	}
	return TRUE;
}

// Utility function to copy a SIDobj
BOOL SIDobj::deepcopy( const SIDobj& sid) {

	if(this == &sid)
		return TRUE;

	// Copy the SID pointer
	if(!copySid(sid.psid_))
		return FALSE; 

	// Throw away our own psidinfo_
	if(psidinfo_)
		psidinfo_->deleteRef();
	psidinfo_ = 0;

	// Throw away our own mach_lookup_
	if(mach_lookup_)
		delete [] mach_lookup_;
	mach_lookup_ = 0;

	// Copy the mahine to lookup the SIDinfo class if there is one.
	autoHeapChar mlp;
	if(sid.mach_lookup_) {
		mlp = new char [ _tcslen(sid.mach_lookup_) + sizeof(TCHAR) ];
		if(mlp.getPtr() == 0) {
			SetLastError(ERROR_OUTOFMEMORY);
		 	return FALSE;
		}
		_tcscpy((TCHAR *)mlp.getPtr(), sid.mach_lookup_);
	}

	mach_lookup_ = (TCHAR *)mlp.relinquishOwnership();
	psidinfo_ = sid.psidinfo_->addRef();
	return TRUE;
}

// Utility function to copy a SID
BOOL SIDobj::copySid( const SID *sid) {

	if(!IsValidSid((SID *)sid))
		return FALSE;

	// Copy the SID itself.
	DWORD sz = GetLengthSid((SID *)sid);
	if(psid_ != 0 && (sz <= GetLengthSid(psid_))) {
		// Use SID space as is.
		if(!CopySid(sz, (SID *)psid_, (SID *)sid))
			return FALSE;
	} else {
		// We need to allocate a new SID.
		delete [] (char *)psid_;
		psid_ = 0;

		autoHeapChar sidptr = new char [ sz ];
		if(sidptr.getPtr() == 0) {
			SetLastError(ERROR_OUTOFMEMORY);
		 	return FALSE;
		}
		if(!CopySid(sz, (SID *)sidptr.getPtr(), (SID *)sid))
			return FALSE;
		psid_ = (SID *)sidptr.relinquishOwnership();
	}

	return TRUE;
}

// Construct a new SIDobj from a SID_IDENTIFIER_AUTHORITY

SIDobj::SIDobj( PSID_IDENTIFIER_AUTHORITY pia, BYTE cnt, DWORD s1, 
				DWORD s2, DWORD s3, DWORD s4, DWORD s5, DWORD s6, DWORD s7, DWORD s8)
	:psid_(0), psidinfo_(0), mach_lookup_(0) {

	PSID psid;
	if(!AllocateAndInitializeSid( pia, cnt, s1, s2, s3, s4, s5, s6, s7, s8, &psid))
		throw GetLastError();
	
	DWORD sz = GetSidLengthRequired(cnt);	 
	autoHeapChar sidptr = new char [ sz ];
	if(sidptr.getPtr() == 0) {
		FreeSid(psid);
		SetLastError(ERROR_OUTOFMEMORY);
	 	throw (DWORD)(ERROR_OUTOFMEMORY);
	}
	if(!CopySid( sz, (SID *)sidptr.getPtr(), psid)) {
		FreeSid(psid);
	 	throw GetLastError();
	}
	FreeSid(psid);
	psid_ = (SID *)sidptr.relinquishOwnership();
}

// Construct a new SIDobj from a pointer to a SID.
SIDobj::SIDobj(const SID *sid) : psid_(0), psidinfo_(0), mach_lookup_(0) {
	if(!copySid(sid))
		throw GetLastError();
}

// Construct a new SIDobj from a name.
SIDobj::SIDobj(const TCHAR *name, const TCHAR *mach) : psid_(0), psidinfo_(0), mach_lookup_(0)  {
	DWORD sidsize = 0;
	DWORD domsize = 0;
	SID_NAME_USE type;
	autoHeapChar pmach;

	if(mach) {
		pmach = new char [ _tcslen(mach) + sizeof(TCHAR) ];
		if(pmach.getPtr() == 0) {
			SetLastError(ERROR_OUTOFMEMORY);
		 	throw (DWORD)(ERROR_OUTOFMEMORY);
		}
		_tcscpy( (TCHAR *)pmach.getPtr(), mach);
	}

	LookupAccountName(mach, name, 0, &sidsize, 0, &domsize, &type);
	if( GetLastError() != ERROR_INSUFFICIENT_BUFFER)
		throw (DWORD)GetLastError();
	autoHeapChar psid = new char [ sidsize ];
	if(psid.getPtr() == 0) {
		SetLastError(ERROR_OUTOFMEMORY);
	 	throw (DWORD)(ERROR_OUTOFMEMORY);
	}
	autoHeapChar pdom = new char [ domsize ];
	if(pdom.getPtr() == 0) {
		SetLastError(ERROR_OUTOFMEMORY);
	 	throw (DWORD)(ERROR_OUTOFMEMORY);
	}
	if(!LookupAccountName(mach, name, (SID *)psid.getPtr(), &sidsize, 
			(TCHAR *)pdom.getPtr(), &domsize, &type))
		throw (DWORD)GetLastError();

	mach_lookup_ = (TCHAR *)pmach.relinquishOwnership();
	psid_ = (SID *)psid.relinquishOwnership();
}

// Copy Constructor
SIDobj::SIDobj(const SIDobj& so) {
	if(!deepcopy(so))
		throw GetLastError();
}

// Private Copy operator =
SIDobj& SIDobj::operator=(const SIDobj& so) {
	if(!deepcopy(so))
		throw GetLastError();
	return *this;
}

// Utility function to copy SID	- deals with exceptions.
BOOL SIDobj::CopyIntoSIDobj(SIDobj *dest) {
	DWORD err = 0;
	try {
		*dest = *this;
	} catch (DWORD caerr) {
		err = caerr;
	}
	if(err)	{
		SetLastError(err);
		return FALSE;
	}
	return TRUE;
}

// Equivalence operator
int SIDobj::operator==(const SIDobj& so) {
	return EqualSid( so.psid_, psid_);
}

// Setup the machine that will be used to do SID to Name lookups.
BOOL SIDobj::setLookupMachine(const TCHAR *mach) {
	// Delete any saved SIDinfo.
	if(psidinfo_)
		psidinfo_->deleteRef();
	psidinfo_ = 0;

	if(mach_lookup_) {
		if(!_tcscmp(mach_lookup_, mach))
			return TRUE;
		delete [] mach_lookup_;
		mach_lookup_ = 0;
	}
	if(mach) {
		mach_lookup_ = new char [ _tcslen(mach) + sizeof(TCHAR) ];
		if(mach_lookup_ == 0) {
			SetLastError(ERROR_OUTOFMEMORY);
			return FALSE;
		}
		_tcscpy( mach_lookup_, mach);
	} else {
		mach_lookup_ = 0;
	}
	return TRUE;
}

// Get the user name corresponding to this SID.
BOOL SIDobj::getUserName(TCHAR **ppname, DWORD *pnamesize) {
	if(!validSIDinfo())
		return FALSE;
	DWORD len = _tcslen(psidinfo_->getName());
	if(*pnamesize > len) {
		_tcscpy(*ppname, psidinfo_->getName());
		return TRUE;
	} else {
		*pnamesize = len + sizeof(TCHAR);
		return FALSE;
	}
}

// Get the domain name corresponding to this SID.
BOOL SIDobj::getUserDomain(TCHAR **ppdom, DWORD *pdomsize) {
	if(!validSIDinfo())
		return FALSE;
	DWORD len = _tcslen(psidinfo_->getDomain());
	if(*pdomsize > len) {
		_tcscpy(*ppdom, psidinfo_->getDomain());
		return TRUE;
	} else {
		*pdomsize = len + sizeof(TCHAR);
		return FALSE;
	}
}

// Get the SID type.
BOOL SIDobj::getUserType(SID_NAME_USE *psnu) {
	if(!validSIDinfo())
		return FALSE;
	*psnu = psidinfo_->getType();
	return TRUE;
}

