/*
    Windows NT Security functions library.
    Copyright (C) 1995  Jeremy R. Allison

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Log: seclib.cpp,v $
// Revision 1.1  1995/06/30  18:32:55  jra
// Initial revision
//^M
*/

#include "seclib.h"
#include "acl.h"
#include "autoheap.h"
#include "objhandle.h"

/*
 * The C Linkage for these routines
 */

extern "C" {

BOOL AddAllowedSIDKernel( HANDLE hkern, PSID psid, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	SECURITY_DESCRIPTOR_HANDLE sdh;
	if(!CreateSecurityDescriptorHandleFromKernelObject(&sdh, hkern, DACL_SECURITY_INFORMATION))
		return FALSE;
	if(!AddAllowedSID(sdh, psid, mask, flags)) {
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	SECURITY_DESCRIPTOR *psd;
	if(!AllocateSecurityDescriptorFromHandle(sdh, &psd)) { 
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	kernel_obj_handle obj( hkern );
	ret = obj.SetObjectSecurity( DACL_SECURITY_INFORMATION, psd);
	CloseSecurityDescriptorHandle(sdh);
	return ret;
}

BOOL AddAllowedNameKernel( HANDLE hkern, const TCHAR *name, const TCHAR *machine, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid(name, machine);
		ret = AddAllowedSIDKernel( hkern, sid, mask, flags);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	return ret;
}

BOOL AddDeniedSIDKernel( HANDLE hkern, PSID psid, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	SECURITY_DESCRIPTOR_HANDLE sdh;
	if(!CreateSecurityDescriptorHandleFromKernelObject(&sdh, hkern, DACL_SECURITY_INFORMATION))
		return FALSE;
	if(!AddDeniedSID(sdh, psid, mask, flags)) {
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	SECURITY_DESCRIPTOR *psd;
	if(!AllocateSecurityDescriptorFromHandle(sdh, &psd)) { 
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	kernel_obj_handle obj( hkern );
	ret = obj.SetObjectSecurity( DACL_SECURITY_INFORMATION, psd);
	CloseSecurityDescriptorHandle(sdh);
	return ret;
}

BOOL AddDeniedNameKernel( HANDLE hkern, const TCHAR *name, const TCHAR *machine, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid(name, machine);
		ret = AddDeniedSIDKernel( hkern, sid, mask, flags);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	return ret;
}

BOOL AddAllowedSIDFile( const TCHAR *file, PSID psid, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	SECURITY_DESCRIPTOR_HANDLE sdh;
	if(!CreateSecurityDescriptorHandleFromFile(&sdh, file, DACL_SECURITY_INFORMATION))
		return FALSE;
	if(!AddAllowedSID(sdh, psid, mask, flags)) {
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	SECURITY_DESCRIPTOR *psd;
	if(!AllocateSecurityDescriptorFromHandle(sdh, &psd)) { 
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	file_obj_handle obj( file );
	ret = obj.SetObjectSecurity( DACL_SECURITY_INFORMATION, psd);
	CloseSecurityDescriptorHandle(sdh);
	return ret;
}

BOOL AddAllowedNameFile( const TCHAR *file, const TCHAR *name, const TCHAR *machine, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid(name, machine);
		ret = AddAllowedSIDFile( file, sid, mask, flags);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	return ret;
}

BOOL AddDeniedSIDFile( const TCHAR *file, PSID psid, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	SECURITY_DESCRIPTOR_HANDLE sdh;
	if(!CreateSecurityDescriptorHandleFromFile(&sdh, file, DACL_SECURITY_INFORMATION))
		return FALSE;
	if(!AddDeniedSID(sdh, psid, mask, flags)) {
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	SECURITY_DESCRIPTOR *psd;
	if(!AllocateSecurityDescriptorFromHandle(sdh, &psd)) { 
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	file_obj_handle obj( file );
	ret = obj.SetObjectSecurity( DACL_SECURITY_INFORMATION, psd);
	CloseSecurityDescriptorHandle(sdh);
	return ret;
}

BOOL AddDeniedNameFile( const TCHAR *file, const TCHAR *name, const TCHAR *machine, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid(name, machine);
		ret = AddDeniedSIDFile( file, sid, mask, flags);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	return ret;
}

BOOL AddAllowedSIDUser( HANDLE huser, PSID psid, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	SECURITY_DESCRIPTOR_HANDLE sdh;
	if(!CreateSecurityDescriptorHandleFromUserObject(&sdh, huser, DACL_SECURITY_INFORMATION))
		return FALSE;
	if(!AddAllowedSID(sdh, psid, mask, flags)) {
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	SECURITY_DESCRIPTOR *psd;
	if(!AllocateSecurityDescriptorFromHandle(sdh, &psd)) { 
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	user_obj_handle obj( huser );
	ret = obj.SetObjectSecurity( DACL_SECURITY_INFORMATION, psd);
	CloseSecurityDescriptorHandle(sdh);
	return ret;
}

BOOL AddAllowedNameUser( HANDLE huser, const TCHAR *name, const TCHAR *machine, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid(name, machine);
		ret = AddAllowedSIDUser( huser, sid, mask, flags);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	return ret;
}

BOOL AddDeniedSIDUser( HANDLE huser, PSID psid, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	SECURITY_DESCRIPTOR_HANDLE sdh;
	if(!CreateSecurityDescriptorHandleFromUserObject(&sdh, huser, DACL_SECURITY_INFORMATION))
		return FALSE;
	if(!AddDeniedSID(sdh, psid, mask, flags)) {
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	SECURITY_DESCRIPTOR *psd;
	if(!AllocateSecurityDescriptorFromHandle(sdh, &psd)) { 
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	user_obj_handle obj( huser );
	ret = obj.SetObjectSecurity( DACL_SECURITY_INFORMATION, psd);
	CloseSecurityDescriptorHandle(sdh);
	return ret;
}

BOOL AddDeniedNameUser( HANDLE huser, const TCHAR *name, const TCHAR *machine, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid(name, machine);
		ret = AddDeniedSIDUser( huser, sid, mask, flags);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	return ret;
}

/* Add access allowed ACE to a Registry key. */
BOOL AddAllowedSIDRegistry( HKEY hkey, PSID psid, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	SECURITY_DESCRIPTOR_HANDLE sdh;
	if(!CreateSecurityDescriptorHandleFromRegistryKey(&sdh, hkey, DACL_SECURITY_INFORMATION))
		return FALSE;
	if(!AddAllowedSID(sdh, psid, mask, flags)) {
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	SECURITY_DESCRIPTOR *psd;
	if(!AllocateSecurityDescriptorFromHandle(sdh, &psd)) { 
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	reg_obj_handle obj( hkey );
	ret = obj.SetObjectSecurity( DACL_SECURITY_INFORMATION, psd);
	CloseSecurityDescriptorHandle(sdh);
	return ret;
}

BOOL AddAllowedNameRegistry( HKEY hkey, const TCHAR *name, const TCHAR *mach, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid(name, mach);
		ret = AddAllowedSIDRegistry( hkey, sid, mask, flags);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	return ret;
}

/* Add access denied ACE to a Registry key */
BOOL AddDeniedSIDRegistry( HKEY hkey, PSID psid, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	SECURITY_DESCRIPTOR_HANDLE sdh;
	if(!CreateSecurityDescriptorHandleFromRegistryKey(&sdh, hkey, DACL_SECURITY_INFORMATION))
		return FALSE;
	if(!AddDeniedSID(sdh, psid, mask, flags)) {
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	SECURITY_DESCRIPTOR *psd;
	if(!AllocateSecurityDescriptorFromHandle(sdh, &psd)) { 
		CloseSecurityDescriptorHandle(sdh);
		return FALSE;
	}
	reg_obj_handle obj( hkey );
	ret = obj.SetObjectSecurity( DACL_SECURITY_INFORMATION, psd);
	CloseSecurityDescriptorHandle(sdh);
	return ret;
}
BOOL AddDeniedNameRegistry( HKEY hkey, const TCHAR *name, const TCHAR *mach, ACCESS_MASK mask, BYTE flags)
{
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid(name, mach);
		ret = AddDeniedSIDRegistry( hkey, sid, mask, flags);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	return ret;
}

/* File and directory permission manipulation routines */
static BOOL make_file_perms(const TCHAR *cmask, ACCESS_MASK *mask, BOOL *allow)
{
	*mask = 0;
	*allow = TRUE;
	const char *p;
	for( p = cmask; *p; p++) {
		switch(*p) {
			case 'F':
				*mask = FILE_ALL_ACCESS;
				return TRUE;
			case 'N':
				*mask = FILE_ALL_ACCESS;
				*allow = FALSE;
				return TRUE;
			case 'R':
				*mask |= GENERIC_READ;
				break;
			case 'W':
				*mask |= GENERIC_WRITE;
				break;
			case 'X':
				*mask |= GENERIC_EXECUTE;
				break;
			case 'D':
				*mask |= DELETE;
				break;
			case 'P':
				*mask |= WRITE_DAC;
				break;
			case 'O':
				*mask |= WRITE_OWNER;
				break;
			default:
				SetLastErrorEx(ERROR_INVALID_PARAMETER, SLE_ERROR);
				return FALSE;
			}
		}
	return TRUE;
}

BOOL AddDirectoryAccessPermissionsByName( const TCHAR *path, const TCHAR *name, const TCHAR *mach, 
			BOOL inherit, const TCHAR *dir_cmask, const TCHAR *file_cmask)
{
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid( name, mach);
		ret = AddDirectoryAccessPermissionsBySID( path, sid, inherit, 
				dir_cmask, file_cmask);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	return ret;
}


BOOL AddDirectoryAccessPermissionsBySID( const TCHAR *path, PSID sid, BOOL inherit,
				const TCHAR *dir_cmask, const TCHAR *file_cmask)
{
	ACCESS_MASK mask;
	BOOL allow = TRUE;

	if(dir_cmask && *dir_cmask) {
		if(!make_file_perms( dir_cmask, &mask, &allow))
			return FALSE;
		if(allow) { 
			if(!AddAllowedSIDFile(path, sid, mask, inherit ? CONTAINER_INHERIT_ACE : 0))
				return FALSE;
		} else {
			if(!AddDeniedSIDFile(path, sid, mask, inherit ? CONTAINER_INHERIT_ACE : 0))
				return FALSE;
		}
	} 
	if(file_cmask && *file_cmask) {
		if(!make_file_perms( file_cmask, &mask, &allow))
			return FALSE;
		if(allow) { 
			if(!AddAllowedSIDFile(path, sid, mask, OBJECT_INHERIT_ACE|INHERIT_ONLY_ACE))
				return FALSE;
		} else {
			if(!AddDeniedSIDFile(path, sid, mask, OBJECT_INHERIT_ACE|INHERIT_ONLY_ACE))
				return FALSE;
		}
	} 
	return TRUE;
}

BOOL AddFileAccessPermissionsByName( const TCHAR *path, const TCHAR *name, const TCHAR *mach,  
			const TCHAR *file_cmask)
{
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid( name, mach);
		ret = AddFileAccessPermissionsBySID( path, sid, file_cmask); 
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	return ret;
}

BOOL AddFileAccessPermissionsBySID( const TCHAR *path, PSID sid, const TCHAR *file_cmask)
{
	ACCESS_MASK mask;
	BOOL allow = TRUE;

	if(file_cmask && *file_cmask) {
		if(!make_file_perms( file_cmask, &mask, &allow))
			return FALSE;
		if(allow) { 
			if(!AddAllowedSIDFile( path, sid, mask, 0))
				return FALSE;
		} else {
			if(!AddDeniedSIDFile( path, sid, mask, 0))
				return FALSE;
		}
	} 
	return TRUE;
}

}
