#ifndef _LIST_H_
#define _LIST_H_

/*
    Windows NT Security functions library.
    Copyright (C) 1995  Jeremy R. Allison

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Log:	list.h,v $
 * Revision 1.2  95/10/17  18:15:40  18:15:40  jra (Jeremy Allison)
 * Added lsa code.
 * 
 * Revision 1.1  1995/06/30  18:32:55  jra
 * Initial revision
 *^M
*/

#include <assert.h>

template <class Item>
class ListItem {
public:
	Item i_;
	ListItem *next_;
	ListItem *prev_;
	ListItem(const Item& it) : i_(it), next_(0), prev_(0) {};
};

#define bool int

template <class Item>
class List {
	ListItem<Item> *head_;
	ListItem<Item> *tail_;
	unsigned long count_;
	void deepcopy( const List &);
public:
	List() : head_(0), count_(0), tail_(0) {}
	List(const List&l) { deepcopy(l); }
	virtual ~List() { RemoveAll(); }
	virtual List& operator=(const List& l) { deepcopy(l); return *this; }

	unsigned long Count() const { return count_;}
	virtual Item& Get(unsigned long index) const;
	virtual Item& First() const { return head_->i_;}
	virtual Item& Last() const { return tail_->i_;}
	virtual bool Includes(const Item&, unsigned long *) const;

	virtual unsigned long Insert(const Item&);

	virtual bool Remove(const Item&);
	virtual void RemoveAt(unsigned long index);
	virtual void RemoveLast();
	virtual void RemoveFirst();
	virtual void RemoveAll();
	
	friend class ListIterator<Item>;																			
};

template <class Item>
class ListIterator {
	const List<Item> *list_;
	ListItem<Item> *current_;
public:
	
	ListIterator(const List<Item> *l) :list_(l), current_(l->head_) {}

	bool Done() const { return current_ == 0; }
	void Next() { current_ = current_->next_;  }
	Item& Get() const { return current_->i_; }
};


//
// Deepcopy - copies all elements of a list
//
template <class Item>
void List<Item>::deepcopy(const List<Item>& l) {
	if( &l != this) {
		RemoveAll();
		// TODO : Replace this with an iterator....
		ListIterator<Item> iter(&l);
		for(; !iter.Done(); iter.Next() ) {
			Insert(iter.Get());
		}
	}
}

//
// Get an Item at a specified index
//
template <class Item>
Item& List<Item>::Get(unsigned long index) const {
	unsigned long i;

	assert(index < count_);

	ListItem<Item> *p = head_;
	for( i = 0; i < index; i++, p = p->next_)
		;
	return p->i_;
}

//
// Check if a List includes a particular Item
// Returns index of item if found and returns 1
// Returns 0 if not found.
//
template <class Item>
bool List<Item>::Includes(const Item& it, unsigned long *pos) const {
	unsigned long i;

	ListItem<Item> *p = head_;
	for( i = 0; i < count_; i++, p = p->next_)
		if( p->i_ == it) {
			*pos = i;
			return 1;
		}
    return 0;
}

//
// Insert an Item into the doubly linked list
//
template <class Item>
unsigned long List<Item>::Insert( const Item& it) {
	ListItem<Item> *li = new ListItem<Item>(it);

	assert( li != 0);

	li->next_ = head_;
	li->prev_ = 0;
	if(head_ != 0)
		head_->prev_ = li;
	if(0 == tail_)
		tail_ = li;

	head_ = li;
	count_++;
	return 0;
}

//
// Remove a particular item from the list
//
template <class Item>
bool List<Item>::Remove(const Item& it) {
	unsigned long pos;

	if(!Includes(it, &pos))
		return 1;
	RemoveAt(pos);
	return 0;
}

//
// Remove an item at a particular position in the list.
//
template <class Item>
void List<Item>::RemoveAt(unsigned long index) {
	assert(index < count_);
	ListItem<Item> *p = head_;
	ListItem<Item> *last = 0;

	unsigned long i;
	for( i = 0; i < index; i++) {
		last = p;
		p = p->next_;
	}

	if(p == head_)
		head_ = p->next_;
	if(p->next_) {
		// Correct backlink
		p->next_->prev_ = last;
		if(last)
			last->next_ = p->next_;
	} else {
		// End of chain, link to tail_
		tail_ = last;
	}
	delete p;
	count_--;
}

//
// Remove the last item in the list
//
template <class Item>
void List<Item>::RemoveLast() {
	assert(count_ > 0);
	ListItem<Item> *last = tail_;
	tail_ = last->prev_;
	delete last;
	if(tail_ == 0)
		head_ = 0;
	count_--;
}

//
// Remove the first item in the list
//
template <class Item>
void List<Item>::RemoveFirst() {
	assert(count_ > 0);
	ListItem<Item> *first = head_;
	head_ = first->next_;
	delete first;
	if(head_ == 0)
		tail_ = 0;
	count_--;
}

//
// Remove everything in the list
//
template <class Item>
void List<Item>::RemoveAll() {
	ListItem<Item> *p, *last;

	for( p = head_; p; count_--) {
		last = p;
		p = p->next_;
		delete last;
	}
	head_ = tail_ = 0;
}
#endif
