/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   Samba utility functions
   Copyright (C) Andrew Tridgell 1992-1998
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"


extern int DEBUGLEVEL;
extern pstring scope;
extern pstring global_myname;

/*
 * This is set on startup - it defines the SID for this
 * machine, and therefore the SAM database for which it is
 * responsible.
 */

DOM_SID global_sam_sid;

/*
 * This is the name associated with the SAM database for
 * which this machine is responsible.  In the case of a PDC
 * or PDC, this name is the same as the workgroup.  In the
 * case of "security = domain" mode, this is the same as
 * the name of the server (global_myname).
 */

fstring global_sam_name; 

/*
 * This is obtained on startup - it defines the SID for which
 * this machine is a member.  It is therefore only set, and
 * used, in "security = domain" mode.
 */

DOM_SID global_member_sid;

/*
 * note the lack of a "global_member_name" - this is because
 * this is the same as "global_myworkgroup".
 */

extern fstring global_myworkgroup;
/* fstring global_member_dom_name; */

/*
 * some useful sids
 */

DOM_SID global_sid_S_1_5_20; /* local well-known domain */
DOM_SID global_sid_S_1_1;    /* everyone */
DOM_SID global_sid_S_1_3;    /* Creator Owner */
DOM_SID global_sid_S_1_5;    /* NT Authority */

struct sid_map
{
	DOM_SID *sid;
	char *name;

};

struct sid_map static_sid_name_map[] =
{
	{ &global_sid_S_1_5_20, "BUILTIN" },
	{ &global_sid_S_1_1   , "Everyone" },
	{ &global_sid_S_1_3   , "Creator Owner" },
	{ &global_sid_S_1_5   , "NT Authority" },
	{ &global_sam_sid     , global_sam_name },
	{ &global_member_sid  , global_myworkgroup },
	{ NULL                , NULL      }
};

struct sid_map **sid_name_map = NULL;
uint32 num_maps = 0;

static struct sid_map *sid_map_dup(const struct sid_map *from)
{
	if (from != NULL)
	{
		struct sid_map *copy = (struct sid_map *)
		                        malloc(sizeof(struct sid_map));
		if (copy != NULL)
		{
			ZERO_STRUCTP(copy);
			if (from->name != NULL)
			{
				copy->name  = strdup(from->name );
			}
			if (from->sid != NULL)
			{
				copy->sid = sid_dup(from->sid);
			}
		}
		return copy;
	}
	return NULL;
}

static void sid_map_free(struct sid_map *map)
{
	if (map->name != NULL)
	{
		free(map->name);
	}
	if (map->sid != NULL)
	{
		free(map->sid);
	}
	free(map);
}

/****************************************************************************
free a sid map array
****************************************************************************/
static void free_sidmap_array(uint32 num_entries, struct sid_map **entries)
{
	void(*fn)(void*) = (void(*)(void*))&sid_map_free;
	free_void_array(num_entries, (void**)entries, *fn);
}

/****************************************************************************
add a sid map state to the array
****************************************************************************/
struct sid_map* add_sidmap_to_array(uint32 *len, struct sid_map ***array,
				const struct sid_map *name)
{
	void*(*fn)(const void*) = (void*(*)(const void*))&sid_map_dup;
	return (struct sid_map*)add_copy_to_array(len,
	                     (void***)array, (const void*)name, *fn, False);
				
}
/****************************************************************************
 sets up the name associated with the SAM database for which we are responsible
****************************************************************************/
void get_sam_domain_name(void)
{
	switch (lp_server_role())
	{
		case ROLE_DOMAIN_PDC:
		case ROLE_DOMAIN_BDC:
		{
			/* we are PDC (or BDC) for a Domain */
			fstrcpy(global_sam_name, lp_workgroup());
			break;
		}
		case ROLE_DOMAIN_MEMBER:
		{
			/* we are a "PDC", but FOR LOCAL SAM DATABASE ONLY */
			fstrcpy(global_sam_name, global_myname);
			break;
		}
		default:
		{
			/* no domain role, probably due to "security = share" */
			memset(global_sam_name, 0, sizeof(global_sam_name));
			break;
		}
	}
}

/****************************************************************************
 obtain the sid from the PDC.
****************************************************************************/
BOOL get_member_domain_sid(void)
{
	switch (lp_server_role())
	{
		case ROLE_DOMAIN_NONE:
		{
			ZERO_STRUCT(global_member_sid);
			return True;
		}
		case ROLE_DOMAIN_PDC:
		{
			sid_copy(&global_member_sid, &global_sam_sid);
			return True;
		}
		default:
		{
			/* member or BDC, we're going for connection to PDC */
			break;
		}
	}

	return get_domain_sids(lp_workgroup(), NULL, &global_member_sid);
}


/****************************************************************************
 creates some useful well known sids
****************************************************************************/
void generate_wellknown_sids(void)
{
	string_to_sid(&global_sid_S_1_5_20, "S-1-5-32");
	string_to_sid(&global_sid_S_1_1   , "S-1-1"   );
	string_to_sid(&global_sid_S_1_3   , "S-1-3"   );
	string_to_sid(&global_sid_S_1_5   , "S-1-5"   );
}

/****************************************************************************
 create a sid map table
****************************************************************************/
BOOL create_sidmap_table(void)
{
	int i;
	char **doms = NULL;
	uint32 num_doms = 0;

	for (i = 0; static_sid_name_map[i].name != NULL; i++)
	{
		add_sidmap_to_array(&num_maps, &sid_name_map,
		                    &static_sid_name_map[i]);
	}

	enumtrustdoms(&doms, &num_doms);

	for (i = 0; i < num_doms; i++)
	{
		struct sid_map map;
		DOM_SID sid;

		map.name = doms[i];
		map.sid  = &sid;

		if (!read_sid(map.name, map.sid))
		{
			DEBUG(0,("Could not read Domain SID %s\n", map.name));
			return False;
		}
		add_sidmap_to_array(&num_maps, &sid_name_map, &map);
	}


	for (i = 0; i < num_maps; i++)
	{
		fstring sidstr;
		sid_to_string(sidstr, sid_name_map[i]->sid);
		DEBUG(10,("Map:\tDomain:\t%s\tSID:\t%s\n",
		         sid_name_map[i]->name, sidstr));
	}


	free_char_array(num_doms, doms);

	return True;
}

/****************************************************************************
 Generate the global machine sid. Look for the DOMAINNAME.SID file first, if
 not found then look in smb.conf and use it to create the DOMAINNAME.SID file.
****************************************************************************/
BOOL generate_sam_sid(char *domain_name, DOM_SID *sid)
{
	char *p;
	pstring sid_file;
	pstring machine_sid_file;
	fstring file_name;

	pstrcpy(sid_file, lp_smb_passwd_file());

	if (sid_file[0] == 0)
	{
		DEBUG(0,("cannot find smb passwd file\n"));
		return False;
	}

	p = strrchr(sid_file, '/');
	if (p != NULL)
	{
		*++p = '\0';
	}

	if (!directory_exist(sid_file, NULL)) {
		if (mkdir(sid_file, 0700) != 0) {
			DEBUG(0,("can't create private directory %s : %s\n",
				 sid_file, strerror(errno)));
			return False;
		}
	}

	pstrcpy(machine_sid_file, sid_file);
	pstrcat(machine_sid_file, "MACHINE.SID");

	slprintf(file_name, sizeof(file_name)-1, "%s.SID", domain_name);
	strupper(file_name);
	pstrcat(sid_file, file_name);
    
	if (file_exist(machine_sid_file, NULL))
	{
		if (file_exist(sid_file, NULL))
		{
			DEBUG(0,("both %s and %s exist when only one should, unable to continue\n",
			          machine_sid_file, sid_file));
			return False;
		}
		if (file_rename(machine_sid_file, sid_file))
		{
			DEBUG(0,("could not rename %s to %s.  Error was %s\n",
			          machine_sid_file, sid_file, strerror(errno)));
			return False;
		}
	}
	
	/* attempt to read the SID from the file */
	if (read_sid(domain_name, sid))
	{
		return True;
	}
  
	if (!create_new_sid(sid))
	{
		return False;
	}
	/* attempt to read the SID from the file */
	if (!write_sid(domain_name, sid))
	{
		return True;
	}
  
	/* during the attempt to write, someone else wrote? */

	/* attempt to read the SID from the file */
	if (read_sid(domain_name, sid))
	{
		return True;
	}
  
	return True;
}   

/*************************************************************
 initialise password databases, domain names, domain sid.
**************************************************************/
BOOL pwdb_initialise(BOOL is_server)
{
	get_sam_domain_name();

	if (!init_myworkgroup())
	{
		return False;
	}

	generate_wellknown_sids();

	if (is_server)
	{
		if (!generate_sam_sid(global_sam_name, &global_sam_sid))
		{
			DEBUG(0,("ERROR: Samba cannot create a SAM SID for its domain (%s).\n",
				  global_sam_name));
			return False;
		}
	}
	else
	{
		if (!get_domain_sids(lp_workgroup(), &global_member_sid,
		                      &global_sam_sid))
		{
			return False;
		}
	}

	create_sidmap_table();

	return True;
}

/**************************************************************************
 turns a domain name into a SID.

 *** side-effect: if the domain name is NULL, it is set to our domain ***

***************************************************************************/
BOOL map_domain_name_to_sid(DOM_SID *sid, char **nt_domain)
{
	int i = 0;

	if (nt_domain == NULL)
	{
		sid_copy(sid, &global_sam_sid);
		return True;
	}

	if ((*nt_domain) == NULL)
	{
		DEBUG(5,("map_domain_name_to_sid: overriding NULL name to %s\n",
		          global_sam_name));
		(*nt_domain) = strdup(global_sam_name);
		sid_copy(sid, &global_sam_sid);
		return True;
	}

	if ((*nt_domain)[0] == 0)
	{
		free(*nt_domain);
		(*nt_domain) = strdup(global_sam_name);
		DEBUG(5,("map_domain_name_to_sid: overriding blank name to %s\n",
		          (*nt_domain)));
		sid_copy(sid, &global_sam_sid);
		return True;
	}

	DEBUG(5,("map_domain_name_to_sid: %s\n", (*nt_domain)));

	for (i = 0; i < num_maps; i++)
	{
		DEBUG(5,("compare: %s\n", sid_name_map[i]->name));
		if (strequal(sid_name_map[i]->name, (*nt_domain)))
		{
			fstring sid_str;
			sid_copy(sid, sid_name_map[i]->sid);
			sid_to_string(sid_str, sid_name_map[i]->sid);
			DEBUG(5,("found %s\n", sid_str));
			return True;
		}
	}

	DEBUG(5,("map_domain_name_to_sid: mapping to %s not known\n",
		  (*nt_domain)));
	return False;
}

/**************************************************************************
 turns a domain SID into a name.

***************************************************************************/
BOOL map_domain_sid_to_name(DOM_SID *sid, char *nt_domain)
{
	fstring sid_str;
	int i = 0;
	sid_to_string(sid_str, sid);

	DEBUG(5,("map_domain_sid_to_name: %s\n", sid_str));

	if (nt_domain == NULL)
	{
		return False;
	}

	for (i = 0; i < num_maps; i++)
	{
		sid_to_string(sid_str, sid_name_map[i]->sid);
		DEBUG(5,("compare: %s\n", sid_str));
		if (sid_equal(sid_name_map[i]->sid, sid))
		{
			fstrcpy(nt_domain, sid_name_map[i]->name);
			DEBUG(5,("found %s\n", nt_domain));
			return True;
		}
	}

	DEBUG(0,("map_domain_sid_to_name: mapping NOT IMPLEMENTED\n"));

	return False;
}
/**************************************************************************
 turns a domain SID into a domain controller name.
***************************************************************************/
BOOL map_domain_sid_to_any_dc(DOM_SID *sid, char *dc_name)
{
	fstring domain;

	if (!map_domain_sid_to_name(sid, domain))
	{
		return False;
	}

	return get_any_dc_name(domain, dc_name);
}

/**************************************************************************
 splits a name of format \DOMAIN\name or name into its two components.
 sets the DOMAIN name to global_sam_name if it has not been specified.
***************************************************************************/
BOOL split_domain_name(const char *fullname, char *domain, char *name)
{
	fstring full_name;
	char *p;

	if (fullname == NULL || domain == NULL || name == NULL)
	{
		return False;
	}

	if (fullname[0] == '\\')
	{
		fullname++;
	}
	fstrcpy(full_name, fullname);
	p = strchr(full_name+1, '\\');

	if (p != NULL)
	{
		*p = 0;
		fstrcpy(domain, full_name);
		fstrcpy(name, p+1);
	}
	else
	{
		fstrcpy(domain, global_sam_name);
		fstrcpy(name, full_name);
	}

	DEBUG(10,("name '%s' split into domain:%s and nt name:%s'\n", fullname, domain, name));
	return True;
}

/**************************************************************************
 enumerates all trusted domains
***************************************************************************/
BOOL enumtrustdoms(char ***doms, uint32 *num_entries)
{
	fstring tmp;
	char *tok;

	/* add trusted domains */

	tok = lp_trusted_domains();
	if (next_token(&tok, tmp, NULL, sizeof(tmp)))
	{
		do
		{
			fstring domain;
			split_at_first_component(tmp, domain, '=', NULL);
			add_chars_to_array(num_entries, doms, domain);

		} while (next_token(NULL, tmp, NULL, sizeof(tmp)));
	}

	return True;
}

/**************************************************************************
 enumerates all domains for which the SAM server is responsible
***************************************************************************/
BOOL enumdomains(char ***doms, uint32 *num_entries)
{
	add_chars_to_array(num_entries, doms, global_sam_name);
	add_chars_to_array(num_entries, doms, "Builtin");

	return True;
}

