/* 
   Unix SMB/Netbios implementation.
   Version 0.1
   WINS server routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Module name: nameservreply.c

   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   04 jul 96: lkcl@pires.co.uk
   created module nameservreply containing NetBIOS reply functions

*/

#include "includes.h"

extern int ServerNMB[];
extern int DEBUGLEVEL;

extern struct name_record *namelist;
extern time_t names_last_modified;


/* a name query challenge structure */
struct nmb_query_chllng
{
	/* see rfc1002.txt 5.1.4.1 p62 */
	struct in_addr reply_ip;
	uint16 response_id;
	int nb_flags;
	int port;
};

static void response_name_release(time_t timestamp, struct packet_struct *p,
                                  struct response_record *n);

/****************************************************************************
reply to a name release
****************************************************************************/
void reply_name_release(struct packet_struct *p)
{
  struct nmb_packet *nmb = &p->packet.nmb;
  struct nmb_name* reply_name = &nmb->question.question_name;
  struct nmb_ip release_ip;
  BOOL bcast = nmb->header.nm_flags.bcast;
  struct name_record *n;
  BOOL success = False;
  uint16 id = nmb->header.name_trn_id;
  
  if (!nmb->header.nm_flags.recursion_desired)
  {
    /* packet is not for name server. */
    proxy_forward_packet(p->timestamp, p);
    return;
  }

  release_ip.nb_flags = nmb->additional->rdata[0];
  putip((char *)&release_ip.ip,&nmb->additional->rdata[2]);  
  
  DEBUG(3,("Name release on %s name %s\n",
			inet_ntoa(release_ip.ip), namestr(reply_name)));
  
  if (bcast) return;
  
  n = find_name(namelist, &nmb->question.question_name,0);
  
  /* XXXX under what conditions should we reject the removal?? */
  if (n && find_name_idx(n, release_ip.ip, 0) != -1)
  {
      success = True;
      
      remove_name(p->timestamp,&namelist,&names_last_modified,
                        n,release_ip.ip,True);
      n = NULL;
  }
  
  /* is there a top-level wins server that we need to release this
     name with?
   */
  if (*lp_wins_register())
  {
    struct in_addr wins_ip, send_ip;
    struct nmb_query_chllng *nmb_data;
	nmb_data = (struct nmb_query_chllng*)(malloc(sizeof(*nmb_data)));

    send_ip = *interpret_addr2(lp_wins_register());
    if (zero_ip(send_ip))
    {
      DEBUG(1,("could not find ip address of recursive WINS server\n"));
    }
    if (ismyip(send_ip))
    {
      DEBUG(0,("ERROR: recursive name registration with ourself!\n"));
      return;
    }

    if (nmb_data)
    {
      int fd = ServerNMB[iface_idx(send_ip)];

	  send_wait_ack(p, id, 15*1000, release_ip.nb_flags);

      /* rfc1002.txt 5.1.4.1 p62 - save the id, ip and port number */
      putip(&nmb_data->reply_ip, &p->ip);
      nmb_data->port = p->port;
      nmb_data->response_id = id;
      nmb_data->nb_flags = release_ip.nb_flags;

      /* RECURSIVE NAME RELEASE */
	  netbios_name_register(p->timestamp, fd,NMB_REL,response_name_release,
                         (void*)nmb_data,reply_name,
                         &release_ip,1,
                         False,False,wins_ip);
    }
  }
  else
  {
    /* Send a NAME RELEASE RESPONSE (pos/neg) see rfc1002.txt 4.2.10-11 */
    send_name_response(p, id, NMB_REL,
						success ? RCODE_REL_OK : RCODE_REL_ACT_ERR,
                        False,False,
						reply_name, release_ip.nb_flags, 0, release_ip.ip);
  }
}


/****************************************************************************
  response or no response for a reg release received. 
  **************************************************************************/
static void response_name_release(time_t timestamp, struct packet_struct *p,
                                  struct response_record *n)
{
	BOOL success = False;
	struct nmb_name *ans_name = &n->name;
	struct nmb_ip release_ip;
	struct nmb_query_chllng *nb_data = (struct nmb_query_chllng*)n->nmb_data;

	memcpy(&release_ip, &n->nb, sizeof(release_ip));

	if (!p)
	{
		success = True;
		ans_name = &n->name;

		if (!n->bcast)
		{
			DEBUG(1,("WINS server did not respond to name release!\n"));
			/* XXXX whoops. we have problems. must deal with this */
		}
	}
	else
	{
		struct nmb_packet *nmb = &p->packet.nmb;
		int rcode = nmb->header.rcode;
		char *rdata = nmb->answers->rdata;

		ans_name = &nmb->answers->rr_name;

		if (rcode == 0 && rdata)
		{
			/* copy the netbios flags and the ip address out of reply data */
			release_ip.nb_flags = rdata[0];
			putip((char*)&release_ip.ip,&rdata[2]);
			
			success = True;
		}

		if (n->num_msgs > 1)
		{
			DEBUG(1,("more than one name register response received!\n"));
			return;
		}

		DEBUG(4,("response name release received\n"));
	}
	if (success)
	{
		remove_netbios_name(timestamp, &namelist, &names_last_modified,
				            ans_name, release_ip.ip, True);
	}
	else
	{
		DEBUG(2,("name release for %s rejected by %s\n", namestr(ans_name),
		           inet_ntoa(n->send_ip)));
	}

	/* report results of registration to original name registerer */
 	p->port = nb_data->port;
	p->timestamp = timestamp;
	p->ip = nb_data->reply_ip;
	p->fd = ServerNMB[iface_idx(p->ip)];

	send_name_response(p, nb_data->response_id, NMB_REL,
	                      success ? RCODE_REL_OK : RCODE_REL_ACT_ERR,
	                      False, False, ans_name,
	                      release_ip.nb_flags, 0,
	                      nb_data->reply_ip);
}


