/* 
   Unix SMB/Netbios implementation.
   Version 0.1
   WINS server routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Module name: winsquery.c

   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   04 jul 96: lkcl@pires.co.uk
   created module nameservreply containing NetBIOS reply functions

*/

#include "includes.h"

extern int ServerNMB[];
extern int DEBUGLEVEL;

extern struct name_record *namelist;
extern time_t names_last_modified;

extern struct in_addr ipgrp;


/***************************************************************************
  assume a WINS name is a dns name, and do a gethostbyname() on it.
  ****************************************************************************/
struct name_record *dns_name_search(time_t time_now, struct nmb_name *question)
{
	int name_type = question->name_type;
	char *qname = question->name;
	BOOL dns_type = (name_type == 0x20 || name_type == 0);
	struct in_addr dns_ip;
	struct nmb_ip nb;

	DEBUG(3,("Search for %s - ", namestr(question)));

	/* only do DNS lookups if the query is for type 0x20 or type 0x0 */
	if (!dns_type)
	{
		DEBUG(3,("types 0x20 and 0x0 for DNS lookup only.\n"));
		return NULL;
	}

	/* look it up with DNS */      
	dns_ip.s_addr = interpret_addr(qname);

	nb.nb_flags = NB_ACTIVE;
	nb.death_time = 60*60;
	nb.ip = dns_ip;

	if (!dns_ip.s_addr)
	{
		/* no luck with DNS. */
		DEBUG(3,("not found. no recursion.\n"));
		/* add the fail to WINS cache of names. give it 1 hour in the cache */
		nb.source = DNSFAIL;
		add_netbios_entry(time_now, &namelist, &names_last_modified,
		                  question, &nb,
		                  True, False);
		return NULL;
	}

	DEBUG(3,("found with DNS: %s\n", inet_ntoa(dns_ip)));

	/* add it to our WINS cache of names. give it 2 hours in the cache */
	nb.source = DNS;
	nb.death_time = 2*60*60;
	return add_netbios_entry(time_now, &namelist, &names_last_modified,
	                         question,&nb,
	                         True,False);
}


/***************************************************************************
  reply to a name query
  ****************************************************************************/
static struct name_record *search_for_name(struct nmb_name *question,
                    struct in_addr ip, int search,
                    int *idx)
{
	struct name_record *n;

	DEBUG(3,("Search for %s from %s - ", namestr(question), inet_ntoa(ip)));

	/* first look up name in cache */
	if ((n = find_name(namelist,question,search)) == NULL) return NULL;

    *idx = find_name_idx(n, ip, search);

    if (*idx == -1) *idx = 0;

	if (ip_equal(n->ip_flgs[*idx].ip, ip) ||
	    (ip_equal(n->ip_flgs[*idx].ip, ipgrp) &&
		 NAME_GROUP(n->ip_flgs[*idx].nb_flags)))
	{
		/* it may have been an earlier failure */
		if (n->ip_flgs[*idx].source == DNSFAIL)
		{
			DEBUG(3,("DNSFAIL\n"));
			return NULL;
		}

		DEBUG(3,("OK %s\n",inet_ntoa(n->ip_flgs[*idx].ip)));

		return n;
	}
	return NULL;
}


static void reply_to_name_query(int fd, int port,
				struct in_addr reply_ip, time_t timestamp, int response_id,
				struct nmb_name *question, 
				struct name_record *n, int idx,
                BOOL success)
{
	struct packet_struct p2;
	pstring rdata;
	time_t ttl = 0;
	int rcode = 0;

	if (n)
	{
		/* name is directed query, or it's a Domain Master type
		   name, or we're replying on behalf of a caller because they are on a
		   different subnet and cannot hear the broadcast. XXXX lp_wins_proxy
		   should be switched off in environments where broadcasts are forwarded
		 */

		/* XXXX note: for proxy servers, we should forward the query on to
		   another WINS server if the name is not in our database, or we are
		   not a WINS server ourselves
		 */
		if (n->ip_flgs[idx].source != DNSFAIL)
		{
			ttl = n->ip_flgs[idx].death_time ? 
			      n->ip_flgs[idx].death_time - timestamp : GET_TTL(0);
		}
		else
		{
			success = False;
		}
	}
	else
	{
		success = False;
	}

	if (n && zero_ip(n->ip_flgs[idx].ip))
	{
		DEBUG(0,("ERROR in nmbd name query: zero IP address\n"));
		success = False;
	}

	if (success)
	{
		rcode = RCODE_QUERY_OK;
		DEBUG(3,("OK\n"));      
	}
	else
	{
		rcode = RCODE_QUERY_NAM_ERR;
		DEBUG(3,("UNKNOWN\n"));      
	}

	if (success && n)
	{
		int i;

		if (question->name_type == 0x1e)
		{
			rdata[0] = NB_GROUP;
			rdata[1] = 0;
			putip(&rdata[2],(char *)&ipgrp);
		}
		else
		{
			for (i = 0; i < n->num_ips; i++)
			{
				rdata[6*i+0] = n->ip_flgs[i].nb_flags;
				rdata[6*i+1] = 0;
				putip(&rdata[6*i+2],(char *)&n->ip_flgs[i].ip);
			}
		}
	}

	p2.ip = reply_ip;
	p2.port = port;
	p2.fd = fd,
	p2.timestamp = timestamp;
	p2.packet_type = NMB_PACKET;

	reply_netbios_packet(&p2,response_id,
	                     rcode,NMB_QUERY,0,True,True,
	                     question,
	                     success ? RR_TYPE_NBIP : RR_TYPE_NULL, RR_CLASS_IP,
	                     ttl,
	                     rdata, success ? n->num_ips * 6 : 0);
}

/* a name query challenge structure */
struct nmb_query_expire
{
	/* see rfc1002.txt 5.1.4.1 p62 */
	struct in_addr reply_ip;
	uint16 response_id;
	BOOL expired;
	int port;
};

static void response_name_check(time_t timestamp, struct packet_struct *p,
                                    struct response_record *n);

/***************************************************************************
reply to a name query.

with broadcast name queries:

	- ignore them. this is a WINS server.

	- the exception to this is if the query is for a Primary Domain Controller
	  type name (0x1b), in which case, a reply is sent.

	- NEVER send a negative response to a broadcast query. no-one else will,
	  and it causes problems - the owner is expected to respond, not the WINS
	  server.

with directed name queries:

	- if you are the WINS server, you are expected to respond with either
      a negative response, a positive response, or a wait-for-acknowledgement
      packet, and then later on a pos/neg response.

****************************************************************************/
void reply_name_query(struct packet_struct *p)
{
	struct nmb_packet *nmb = &p->packet.nmb;
	struct nmb_name *question = &nmb->question.question_name;
	int name_type = question->name_type;
	BOOL bcast = nmb->header.nm_flags.bcast;
	BOOL success = True;
	struct name_record *n = NULL;
	int idx;
    uint16 id = nmb->header.name_trn_id;
	struct in_addr check_ip, send_ip;
	int nb_flags = 0;
	BOOL name_check = False;
	BOOL expired = False;

	if (!nmb->header.nm_flags.recursion_desired)
	{
		/* packet is not for name server. */
		proxy_forward_packet(p->timestamp, p);
		return;
	}

	/* directed queries are for WINS server: broadcasts are local queries.
	the exception is Domain Master names.  */

	debug_nmb_packet(p);

	DEBUG(3,("Name query "));

	if (!bcast && name_type == 0x1d)
	{
		/* see WINS manager HELP - 'How WINS Handles Special Names' */
		/* a WINS query (unicasted) for a 0x1d name must always return False */
		success = False;
	}

	if (success)
	{
		n = search_for_name(question,p->ip,FIND_LOCAL,&idx);
	}

	if (n)
	{
		DEBUG(4,("found %s %s %d\n", namestr(&n->name),
		          inet_ntoa(n->ip_flgs[idx].ip),n->ip_flgs[idx].death_time));
	}

	/* is our entry already dead? */
	if (n && n->ip_flgs[idx].death_time)
	{
		if (n->ip_flgs[idx].death_time < p->timestamp)
		{
			expired = True;
			name_check = True;

			putip(&check_ip, &n->ip_flgs[idx].ip);
			putip(&send_ip,&check_ip);
			nb_flags = n->ip_flgs[idx].nb_flags;
		}
	}

	/* don't have this name in our records. can we recursively look it
       up with the top-level wins server?

       the lp_wins_query() parameter is the wins server above this one
       that we can do name queries on if the name is unknown to us.

       the lp_wins_register() parameter is the wins server above this
       one that we can do name registration and release with when we
       receive a wins client's name registration or release.

       this structure gives us a hierarchical NetBIOS name database.

       it allows for the load to be taken off a top-level wins server
       (to which all the other wins servers below it point) for name
       queries.

       it allows an isolated network (e.g by a dial-on-demand
       isdn link) to remain isolated. lp_wins_poll() should probably
       be set to False for a dial-on-demand local wins server.

       note that without polling, the status of the NetBIOS name
       database will become out-of-sync. even with polling enabled, there
       is no mechanism to pass name releases instantaneously back down the
       hierarchical layers.

     */

	if (!n && *lp_wins_query())
	{
		send_ip = *interpret_addr2(lp_wins_query());
		if (zero_ip(send_ip))
		{
		  DEBUG(1,("could not find ip address of recursive WINS server\n"));
		}
		if (ismyip(send_ip))
		{
		  DEBUG(0,("ERROR: recursive name registration with ourself!\n"));
		  return;
		}

		name_check = True;
		expired = False;
		putip(&check_ip, &p->ip);
		nb_flags = 0;
	}

	if (name_check)
	{
		/* check that the name is valid before replying */

		struct nmb_query_expire *nmb_data;
		nmb_data = (struct nmb_query_expire*)(malloc(sizeof(*nmb_data)));

		if (nmb_data)
		{
			int fd = ServerNMB[iface_idx(p->ip)];

			send_wait_ack(p, id, 15*1000, nb_flags);

			/* rfc1002.txt 5.1.4.1 p62 - save the id, ip and port number */
			putip(&nmb_data->reply_ip, &check_ip);
			nmb_data->response_id = id;
			nmb_data->port = p->port;
			nmb_data->expired = expired;

			/* initiate some enquiries to see if the current owner exists */
			netbios_name_query(p->timestamp,fd, response_name_check,
							   (void*)nmb_data, question,
							   False, False, send_ip);
		}
		return;
	}

	if (!n && lp_dns_resolution())
	{
		n = dns_name_search(p->timestamp, question);
	}

	if (n)
	{
		/* XXXX update namelist records */

		/* don't respond to broadcast queries unless the query is for
		   a name we own or it is for a Primary Domain Controller name */

		if (bcast && name_type != 0x1b)
		{
			return;
		}
	}
	else
	{
		/* XXXX remove name from records (expire it?) */
		if (bcast)
		{
			return; /* never reply negative response to bcasts */
		}
	}

	reply_to_name_query(p->fd, p->port, p->ip, p->timestamp,
					nmb->header.name_trn_id,
                    question, n, idx, success);
}

/****************************************************************************
  response from a name query - has a name expired or not?
  ****************************************************************************/
static void response_name_check(time_t timestamp, struct packet_struct *p,
                                    struct response_record *n)
{
	struct nmb_query_expire *nb_data = (struct nmb_query_expire*)n->nmb_data;
	struct in_addr reply_ip;
	struct nmb_ip found;
	BOOL success = False;
	struct name_record *nm = NULL;
	int idx;
	int fd;

	putip(&reply_ip, &nb_data->reply_ip);

	if (!p)
	{
        /* name challenge: no reply. we can reply to the person that
           wanted the unique name and tell them that they can have it
         */

		success = False;

		DEBUG(4,("owner %s %s did not respond to name query.\n",
					namestr(&n->name), inet_ntoa(n->send_ip)));
	}
	else
	{
		struct nmb_packet *nmb = &p->packet.nmb;
		struct nmb_name *ans_name = NULL;
		char *rdata = nmb->answers->rdata;

		int rcode = nmb->header.rcode;

		if (rcode == 0 && rdata)
		{
			/* copy the netbios flags and the ip address out of reply data */
			found.nb_flags = rdata[0];
			putip((char*)&found.ip,&rdata[2]);
			
			/* first check the details are correct */
			if (!ip_equal(n->send_ip, found.ip))
			{
				/* someone gave us the wrong ip as a reply. oops. */
				/* XXXX should say to them 'oi! release that name!' */

				DEBUG(4,("expected ip: %s\n", inet_ntoa(n->send_ip)));
				DEBUG(4,("unexpected ip: %s\n", inet_ntoa(found.ip)));
			}

			success = True;
		}

		DEBUG(4, ("Name query at %s ip %s - ",
			  namestr(&n->name), inet_ntoa(n->send_ip)));

		if (!nb_data) return;

		if (!name_equal(&n->name, ans_name))
		{
			/* someone gave us the wrong name as a reply. oops. */
			/* XXXX should say to them 'oi! release that name!' */

			DEBUG(4,("unexpected name received: %s\n", namestr(ans_name)));
			success = False;
		}

		if (n->num_msgs > 1)
		{
			if ((!NAME_GROUP(found.nb_flags)) && !ip_equal(n->from_ip, p->ip))
			{
				struct packet_struct p2;

				DEBUG(3,("Unique Name conflict detected!\n"));

				p2.ip = p->ip;
				p2.port = p->port;
				p2.fd = p->fd;
				p2.timestamp = p->timestamp;
				p2.packet_type = NMB_PACKET;

				/* XXXX note that the nb flags and ip should be sent here -
				   see rfc1002.txt 4.2.8
				 */
				reply_netbios_packet(&p2,nmb->header.name_trn_id,
									 RCODE_QUERY_CFT_ERR,NMB_QUERY,0,True,True,
									 ans_name,
									 RR_TYPE_NULL, RR_CLASS_IP,
									 0, NULL, 0);
				success = False;
			}
		}
		else
		{
			/* record who this packet is from, so that if another packet
			   is received, and it's a unique name, we can detect a conflict
			 */
			putip(&n->from_ip, &p->ip);
		}
	}

	if (success)
	{
		nm = search_for_name(&n->name,n->send_ip,FIND_LOCAL,&idx);

		/* we had sent out a name query to the current owner
		   of a name because someone else wanted it. now they
		   have responded saying that they still want the name,
		   so the other host can't have it.
		 */

		DEBUG(4, (" OK: %s\n", inet_ntoa(found.ip)));
	}
	else
	{
		DEBUG(4, (" NEGATIVE RESPONSE!\n"));
	}

	fd = ServerNMB[iface_idx(nb_data->reply_ip)];

	reply_to_name_query(fd, nb_data->port, nb_data->reply_ip, timestamp,
					nb_data->response_id,
                    &n->name, nm, idx, success);
}
