/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   NBT netbios library routines
   Copyright (C) Andrew Tridgell 1994-1995
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/

#include "includes.h"

extern int DEBUGLEVEL;


/****************************************************************************
send a registration / release response: pos/neg
**************************************************************************/
void send_name_response(struct packet_struct *p, 
				int name_trn_id, int opcode, int rcode,
				BOOL recurse_avail,
				BOOL recurse_desired,
				struct nmb_name *reply_name, int nb_flags, int ttl,
				struct in_addr ip)
{
  char rdata[6];

  rdata[0] = nb_flags;
  rdata[1] = 0;
  putip(&rdata[2],(char *)&ip);
  
  reply_netbios_packet(p,name_trn_id,
		       rcode,opcode,opcode,
			   recurse_avail, recurse_desired,
		       reply_name, RR_TYPE_NBIP, RR_CLASS_IP,
		       ttl, 
		       rdata, 6);
}


/****************************************************************************
  reply to a netbios name packet 
  ****************************************************************************/
void reply_netbios_packet(struct packet_struct *p1,int trn_id,
                int rcode, int rcv_code, int opcode,
				BOOL from_wins_server,
				BOOL to_wins_server,
                struct nmb_name *rr_name,int rr_type,int rr_class,int ttl,
                char *data,int len)
{
  struct packet_struct p;
  struct nmb_packet *nmb = &p.packet.nmb;
  struct res_rec answers;
  char *packet_type = "unknown";
  
  BOOL authoritative = from_wins_server; /* see rfc1002.txt 4.2.1.1 p10 */

  p = *p1;

  switch (rcv_code)
  {
    case NMB_STATUS:
    {
      packet_type = "nmb_status";
      break;
    }
    case NMB_QUERY:
    {
      packet_type = "nmb_query";
      authoritative = True; /* see rfc1002.txt 4.2.1.1 p10 - it's _my_ name! */
      break;
    }
    case NMB_REG_MULTI:
    {
      packet_type = "nmb_reg_multi";
      break;
    }
    case NMB_REG:
    {
      packet_type = "nmb_reg";
      break;
    }
    case NMB_REL:
    {
      packet_type = "nmb_rel";
      break;
    }
    case NMB_WAIT_ACK:
    {
      packet_type = "nmb_wack";
      break;
    }
    default:
    {
      DEBUG(1,("replying netbios packet: %s %s\n",
                packet_type, namestr(rr_name), inet_ntoa(p.ip)));

      return;
    }
  }

  DEBUG(4,("replying netbios packet: %s %s\n",
       packet_type, namestr(rr_name), inet_ntoa(p.ip)));

  nmb->header.name_trn_id = trn_id;
  nmb->header.opcode = opcode;
  nmb->header.response = True;
  nmb->header.nm_flags.bcast = False;
  nmb->header.nm_flags.recursion_available = from_wins_server;
  nmb->header.nm_flags.recursion_desired   = to_wins_server;
  nmb->header.nm_flags.trunc = False;
  nmb->header.nm_flags.authoritative = authoritative;
  
  nmb->header.qdcount = 0;
  nmb->header.ancount = 1;
  nmb->header.nscount = 0;
  nmb->header.arcount = 0;
  nmb->header.rcode = rcode;
  
  bzero((char*)&nmb->question,sizeof(nmb->question));
  
  nmb->answers = &answers;
  bzero((char*)nmb->answers,sizeof(*nmb->answers));
  
  nmb->answers->rr_name  = *rr_name;
  nmb->answers->rr_type  = rr_type;
  nmb->answers->rr_class = rr_class;
  nmb->answers->ttl      = ttl;
  
  if (data && len)
  {
    nmb->answers->rdlength = len;
    memcpy(nmb->answers->rdata, data, len);
  }
  
  p.packet_type = NMB_PACKET;
  
  debug_nmb_packet(&p);
  
  send_packet(&p);
}


/****************************************************************************
  initiate a netbios packet
  ****************************************************************************/
BOOL initiate_netbios_packet(time_t time_now, struct response_record *n)
{
  struct packet_struct p;
  struct nmb_packet *nmb = &p.packet.nmb;
  struct res_rec additional_rec;
  char *pkt_type = "unknown";
  int opcode = -1;

  switch (n->quest_type)
  {
    case NMB_STATUS:
    {
      pkt_type = "nmb_status";
      opcode = 0;
      break;
    }
    case NMB_QUERY:
    {
      pkt_type = "nmb_query";
      opcode = 0;
      break;
    }
    case NMB_REG:
    {
      pkt_type = "nmb_reg";
      opcode = n->quest_type;
      break;
    }
    case NMB_REG_MULTI:
    {
      pkt_type = "nmb_reg_multi";
      opcode = n->quest_type;
      break;
    }
    case NMB_REG_REFRESH:
    {
      pkt_type = "nmb_reg_refresh";
      opcode = n->quest_type;
      break;
    }
    case NMB_REL:
    {
      pkt_type = "nmb_rel";
      opcode = n->quest_type;
      break;
    }
  }

  DEBUG(4,("init netbios: %d %d %s %s (bcast=%s) %s\n",
       n->repeat_time, n->repeat_count,
       pkt_type, namestr(&n->name), BOOLSTR(n->bcast), inet_ntoa(n->send_ip)));

  if (opcode == -1) return False;

  bzero((char *)&p,sizeof(p));

  nmb->header.name_trn_id = n->response_id;
  nmb->header.opcode = opcode;
  nmb->header.response = False;

  nmb->header.nm_flags.bcast = n->bcast;
  nmb->header.nm_flags.recursion_available = False;
  nmb->header.nm_flags.recursion_desired = n->recurse;
  nmb->header.nm_flags.trunc = False;
  nmb->header.nm_flags.authoritative = False;

  nmb->header.rcode = 0;
  nmb->header.qdcount = 1;
  nmb->header.ancount = 0;
  nmb->header.nscount = 0;
  nmb->header.arcount = (n->quest_type==NMB_REG || 
			 n->quest_type==NMB_REL ||
			 n->quest_type==NMB_REG_MULTI ||
			 n->quest_type==NMB_REG_REFRESH) ? 1 : 0;
  
  memcpy(&nmb->question.question_name,&n->name,sizeof(n->name));
  
  nmb->question.question_class = RR_CLASS_IP;
  nmb->question.question_type  = RR_TYPE_NBIP;
  
  if (nmb->header.arcount)
  {
      nmb->additional = &additional_rec;
      bzero((char *)nmb->additional,sizeof(*nmb->additional));
      
      nmb->additional->rr_name  = nmb->question.question_name;
      nmb->additional->rr_type  = RR_TYPE_NBIP;
      nmb->additional->rr_class = RR_CLASS_IP;
      
      if (n->quest_type != NMB_REL)
      {
        nmb->additional->ttl = lp_max_ttl();
      }
      else
      {
        nmb->additional->ttl = 0;
      }

      nmb->additional->rdlength = n->num_ips * 6;;
      nmb->additional->rdata[0] = n->nb.nb_flags;
      putip(&nmb->additional->rdata[2],(char *)iface_ip(n->nb.ip));
#if 0
      int i;

      for (i = 0; i < n->num_ips; i++)
      {
        nmb->additional->rdata[i*6] = n->nb[i].nb_flags;
        putip(&nmb->additional->rdata[i*6+2],(char *)iface_ip(n->nb[i].ip));
      }
#endif
  }
  
  p.ip = n->send_ip;
  p.port = NMB_PORT;
  p.fd = n->fd;
  p.timestamp = time_now;
  p.packet_type = NMB_PACKET;
  
  debug_nmb_packet(&p);
  if (!send_packet(&p))
  {
    DEBUG(3,("send_packet to %s %d failed\n",inet_ntoa(p.ip),p.port));
    return False;
  }

  return True;
}


/****************************************************************************
  report the response record type
  ****************************************************************************/
static BOOL debug_rr_type(int rr_type)
{
  switch (rr_type)
  {
      case NMB_STATUS: DEBUG(3,("Name status ")); break;
      case NMB_QUERY : DEBUG(3,("Name query ")); break;
      case NMB_REG   : DEBUG(3,("Name registration ")); break;
      case NMB_REG_MULTI: DEBUG(3,("Name reg (Multi-Home) ")); break;
      case NMB_REL   : DEBUG(3,("Name release ")); break;
      default        : DEBUG(1,("wrong response packet type received")); break;
  }
  return False;
}


/****************************************************************************
  response from a netbios packet.

  return False if we were _not_ expecting a response for this
  packet. then, winsd (for example) can proxy it off to a port/ip
  that _was_ expecting it. 

  ****************************************************************************/
static BOOL response_netbios_packet(struct packet_struct *p)
{
  struct nmb_packet *nmb = &p->packet.nmb;
  struct nmb_name *ans_name = NULL;
  BOOL bcast = nmb->header.nm_flags.bcast;
  struct response_record *n;

  if (!(n = find_response_record(nmb->header.name_trn_id)))
  {
    DEBUG(3,("unknown netbios response\n"));
    return False;
  }

  DEBUG(4,("netbios response packet: %s %d\n",inet_ntoa(p->ip),n->response_id));

  if (nmb->answers == NULL)
  {
      /* hm. the packet received was a response, but with no answer. wierd! */
      DEBUG(2,("NMB packet response from %s (bcast=%s) - NO ANSWER!\n",
           inet_ntoa(p->ip), BOOLSTR(bcast)));
      return True;
  }

  ans_name = &nmb->answers->rr_name;
  DEBUG(3,("response for %s from %s (bcast=%s)\n",
       namestr(ans_name), inet_ntoa(p->ip), BOOLSTR(bcast)));
  
  if (nmb->header.opcode == NMB_WAIT_ACK)
  {
    time_t new_ttl = nmb->answers->ttl;
    new_ttl = new_ttl ? new_ttl / 1000 : 30; /* add on waiting time */
    DEBUG(3,("WAIT for ACKNOWLEDGEMENT\n"));
    n->repeat_time += new_ttl;

    update_response_record(p->timestamp+new_ttl+1,n->response_id,False);

    return True;
  }

  if (debug_rr_type(nmb->answers->rr_type)) return True;

  /* count number of responses received */
  n->num_msgs++; 

  /* don't resend - let packet be deleted by expire_netbios_packets() */
  n->repeat_count = 0; 

  /* now carry on processing where we left off... */
  if (n->fn) n->fn(p->timestamp, p, n);

  return True;
}


/****************************************************************************
  process a nmb packet
  ****************************************************************************/
BOOL process_nmb(struct packet_struct *p)
{
	struct nmb_packet *nmb = &p->packet.nmb;
	static struct in_addr local_ip;

	local_ip.s_addr = inet_addr("127.0.0.1");

	debug_nmb_packet(p);

	if (ismyip(p->ip) && p->port == NMB_PORT && !nmb->header.response)
	{
		DEBUG(4,("*** disregarding request packet from ourself ***\n"));
		return True;
	}

	if (ip_equal(p->ip, local_ip))
	{
		if (!nmb->header.response)
		{
			/* nmbd's way of identifying that the packet has in fact been
			   forwarded to us from winsd. see proxy_forward_packet()
			 */
			uint16 id;
			if (!confirm_response_record(p->timestamp,nmb->header.name_trn_id,
			    &p->port,&p->ip,&id,True))
			{
				DEBUG(4,("discard own nmb packet from %s\n", inet_ntoa(p->ip)));
				return True;
			}
			else
			{
				nmb->header.name_trn_id = id;
				DEBUG(4,("forwarded packet received\n"));
			}
		}
	}

	switch (nmb->header.opcode) 
	{
		case NMB_WAIT_ACK:
		{
			if (nmb->header.response)
			{
				if (nmb->header.ancount > 0) 
				{
					return response_netbios_packet(p);
				}
				break;
			}
			else
			{
				DEBUG(1,("Wait Acknowledgement non-response packet!\n"));
				break;
			}
		}

		case NMB_REG:
		case NMB_REG_MULTI:
		case NMB_REG_REFRESH:
		case 8: /* same as an NMB_REG_REFRESH but rfc1001/2.txt is messed up */
		{
			if (nmb->header.response)
			{
				if (nmb->header.ancount > 0) 
				{
					return response_netbios_packet(p);
				}
				break;
			}
			else
			{
				if (nmb->header.qdcount==0 || nmb->header.arcount==0) break;
				reply_name_reg(p);
			}
			break;
		}

		case 0:
		{
			if (nmb->header.response)
			{
				if (nmb->header.ancount > 0) 
				{
					return response_netbios_packet(p);
				}
				break;
			}
			else if (nmb->header.qdcount>0) 
			{
				switch (nmb->question.question_type)
				{
					case NMB_QUERY:
					{
						reply_name_query(p);
						break;
					}
					case NMB_STATUS:
					{
						reply_name_status(p);
						break;
					}
				}
			}
			break;
		}

		case NMB_REL:
		{
			if (nmb->header.response)
			{
				if (nmb->header.ancount>0) 
				{
					return response_netbios_packet(p); 
				}
				break;
			}
			else
			{
				if (nmb->header.qdcount==0 || nmb->header.arcount==0)
				{
					DEBUG(2,("netbios release packet rejected\n"));
					break;
				}

				reply_name_release(p);
			}
			break;
		}
	}
	return True;
}


