/*
   Unix SMB/Netbios implementation.
   Version 1.9.
   NBT netbios routines and daemon - version 2
   Copyright (C) Andrew Tridgell 1994-1995
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support


   30 July 96: David.Chappell@mail.trincoll.edu
   Expanded multiple workgroup domain master browser support.

*/

#include "includes.h"

extern int DEBUGLEVEL;

extern pstring debugf;
pstring servicesf = CONFIGFILE;

extern struct subnet_record *subnetlist;

int ClientNMB   = -1;
int ServerNMB   = -1;
int ClientDGRAM = -1;

extern pstring myhostname;

/* are we running as a daemon ? */
static BOOL is_daemon = False;

/* what server type are we currently */

time_t StartupTime =0;

extern struct in_addr ipgrp;
extern struct in_addr ipzero;

int ClientPort;

 /****************************************************************************
  catch a sigterm
  ****************************************************************************/
static int sig_term()
{
  BlockSignals(True,SIGTERM);
  
  DEBUG(0,("Got SIGTERM: going down...\n"));
  
  /* remove all samba names, with wins server if necessary. */
  remove_my_names(time(NULL));
  
  /* announce all server entries as 0 time-to-live, 0 type */
  /* XXXX don't care if we never receive a response back... yet */
  remove_my_servers(time(NULL));

  /* XXXX other things: if we are a master browser, force an election? */
  
  exit(0);
}


/****************************************************************************
catch a sighup
****************************************************************************/
static int sig_hup(void)
{
  BlockSignals(True,SIGHUP);

  DEBUG(0,("Got SIGHUP (reload not implemented)\n"));
  reload_services(True);

  BlockSignals(False,SIGHUP);
#ifndef DONT_REINSTALL_SIG
  signal(SIGHUP,SIGNAL_CAST sig_hup);
#endif
  return(0);
}

/****************************************************************************
catch a sigpipe
****************************************************************************/
static int sig_pipe(void)
{
  BlockSignals(True,SIGPIPE);

  DEBUG(0,("Got SIGPIPE\n"));
  if (!is_daemon)
    exit(1);
  BlockSignals(False,SIGPIPE);
  return(0);
}

#if DUMP_CORE
/*******************************************************************
prepare to dump a core file - carefully!
********************************************************************/
static BOOL dump_core(void)
{
  char *p;
  pstring dname;
  strcpy(dname,debugf);
  if ((p=strrchr(dname,'/'))) *p=0;
  strcat(dname,"/corefiles");
  mkdir(dname,0700);
  sys_chown(dname,getuid(),getgid());
  chmod(dname,0700);
  if (chdir(dname)) return(False);
  umask(~(0700));

#ifndef NO_GETRLIMIT
#ifdef RLIMIT_CORE
  {
    struct rlimit rlp;
    getrlimit(RLIMIT_CORE, &rlp);
    rlp.rlim_cur = MAX(4*1024*1024,rlp.rlim_cur);
    setrlimit(RLIMIT_CORE, &rlp);
    getrlimit(RLIMIT_CORE, &rlp);
    DEBUG(3,("Core limits now %d %d\n",rlp.rlim_cur,rlp.rlim_max));
  }
#endif
#endif


  DEBUG(0,("Dumping core in %s\n",dname));
  return(True);
}
#endif


/****************************************************************************
possibly continue after a fault
****************************************************************************/
static void fault_continue(void)
{
#if DUMP_CORE
  dump_core();
#endif
}


/*******************************************************************
  expire old names from the namelist and server list
  ******************************************************************/
static void expire_names_and_servers(time_t time_now)
{
  static time_t lastrun = 0;
  
  if (!lastrun) lastrun = time_now;
  if (time_now < lastrun + 15) return; /* give samba time to check its names */
  lastrun = time_now;
  
  check_expire_names(time_now); /* this checks samba's NetBIOS names */
  expire_servers(time_now);
}

/*****************************************************************************
  reload the services file
  **************************************************************************/
BOOL reload_services(BOOL test)
{
  BOOL ret;
  extern struct nmb_name remote_machine;

  make_nmb_name(&remote_machine, "nmbd", 0x0, NULL);

  if (lp_loaded())
    {
      pstring fname;
      strcpy(fname,lp_configfile());
      if (file_exist(fname,NULL) && !strcsequal(fname,servicesf))
    {
      strcpy(servicesf,fname);
      test = False;
    }
    }

  if (test && !lp_file_list_changed())
    return(True);

  ret = lp_load(servicesf,True);

  /* perhaps the config filename is now set */
  if (!test) {
    DEBUG(3,("services not loaded\n"));
    reload_services(True);
  }


  return(ret);
}



/****************************************************************************
  The main select loop.
  ***************************************************************************/
static void process(void)
{
  while (True)
  {
      time_t time_now = time(NULL);
      listen_for_packets(check_elections());

      run_elections(time_now);

      /* announce as member of workgroup locally and to PDC; or 
         announce as local master browser, locally
       */
      announce_host(time_now); 

      /* announce as local master browser to PDC */
      announce_master(time_now);

      /* announce as 'sort-of' member to a remote server */
      announce_remote(time_now);

      /* removing expired NetBIOS names, and removing old browse entries */
      expire_names_and_servers(time_now);

      /* remove and deal with NetBIOS responses not having been received */
      expire_netbios_response_entries(time_now);

      write_browse_list(time_now);
      do_browser_lists(time_now);

      /* see if there is a local master browser - if not, attempt to be one */
      check_master_browser(time_now);

      /* write out names - if anything's changed */
      dump_names(time_now);
  }
}


/****************************************************************************
  open the socket communication
****************************************************************************/
static BOOL open_sockets(BOOL isdaemon, int nmb_port)
{
  struct hostent *hp;
  int dgram_port = DGRAM_PORT;
  struct in_addr wins_ip = *interpret_addr2(lp_wins_server());

  ClientPort = 0;

  /* get host info */
  if ((hp = Get_Hostbyname(myhostname)) == 0) {
    DEBUG(0,( "Get_Hostbyname: Unknown host. %s\n",myhostname));
    return False;
  }   

  if (!ip_equal(ipzero,wins_ip))
  {
    ServerNMB   = open_socket_in(SOCK_DGRAM,&ClientPort,3,
                               interpret_addr(lp_socket_address()));
  }

  if (isdaemon)
    ClientNMB = open_socket_in(SOCK_DGRAM,&nmb_port,3,
                               interpret_addr(lp_socket_address()));
  else
    ClientNMB = 0;
  
  ClientDGRAM = open_socket_in(SOCK_DGRAM,&dgram_port,3,
                               interpret_addr(lp_socket_address()));

  if (ClientNMB == -1) return(False);

  signal(SIGPIPE, SIGNAL_CAST sig_pipe);

  set_socket_options(ClientNMB  ,"SO_BROADCAST");
  set_socket_options(ServerNMB  ,"SO_BROADCAST");
  set_socket_options(ClientDGRAM,"SO_BROADCAST");

  DEBUG(3,("Sockets opened.\n"));
  return True;
}


/****************************************************************************
  initialise connect, service and file structs
****************************************************************************/
static BOOL init_structs()
{
  extern struct nmb_name local_machine;

  /* XXXX must get correct scope */
  make_nmb_name(&local_machine,lp_server_alias(),0x0,NULL);

  return True;
}

/****************************************************************************
usage on the program
****************************************************************************/
static void usage(char *pname)
{
  DEBUG(0,("Incorrect program usage - is the command line correct?\n"));

  printf("Usage: %s [-n name] [-D] [-p port] [-d debuglevel] [-l log basename]\n",pname);
  printf("Version %s\n",VERSION);
  printf("\t-D                    become a daemon\n");
  printf("\t-p port               listen on the specified port\n");
  printf("\t-d debuglevel         set the debuglevel\n");
  printf("\t-l log basename.      Basename for log/debug files\n");
  printf("\t-n netbiosname.       the netbios name to advertise for this host\n");
  printf("\n");
}


/****************************************************************************
  main program
  **************************************************************************/
int main(int argc,char *argv[])
{
  int port = NMB_PORT;
  int opt;
  extern FILE *dbf;
  extern char *optarg;
  time_t time_now;

  time_now = StartupTime = time(NULL);

  TimeInit();

  strcpy(debugf,NMBLOGFILE);

  setup_logging(argv[0],False);

  charset_initialise();

  /* this is for people who can't start the program correctly */
  while (argc > 1 && (*argv[1] != '-')) {
    argv++;
    argc--;
  }

  fault_setup(fault_continue);

  signal(SIGHUP ,SIGNAL_CAST sig_hup);
  signal(SIGTERM,SIGNAL_CAST sig_term);

  while ((opt = getopt(argc, argv, "s:T:I:C:bAi:B:N:H:l:d:Dp:hSG:")) != EOF)
    {
      switch (opt)
    {
    case 's':
      strcpy(servicesf,optarg);
      break;      
    case 'N':
    case 'B':
    case 'I':
    case 'C':
    case 'G':
    case 'H':
    case 'i':
      DEBUG(0,("Obsolete option '%c' used\n",opt));
      break;
    case 'l':
      sprintf(debugf,"%s.nmb",optarg);
      break;
    case 'D':
      is_daemon = True;
      break;
    case 'd':
      DEBUGLEVEL = atoi(optarg);
      break;
    case 'p':
      port = atoi(optarg);
      break;
    case 'h':
      usage(argv[0]);
      exit(0);
      break;
    default:
      if (!is_a_socket(0)) {
        usage(argv[0]);
      }
      break;
    }
    }

  DEBUG(1,("%s nmbd browser version %s started\n",timestring(),VERSION));
  DEBUG(1,("Copyright Andrew Tridgell 1994-1996 Luke Leighton 1996\n"));

  get_myname(myhostname,NULL);

  init_structs();

  if (!reload_services(False))
    return(-1); 
  
  reload_services(True);

  if (!is_daemon && !is_a_socket(0)) {
    DEBUG(0,("standard input is not a socket, assuming -D option\n"));
    is_daemon = True;
  }
  
  if (is_daemon) {
    DEBUG(2,("%s becoming a daemon\n",timestring()));
    become_daemon();
  }

  sleep(10);

  time_now = time(NULL);

  /* XXXX BIG NOTE: reloading services results in changes to
     interfaces, which may mean that sockets need to be closed

     and re-opened...
   */

  DEBUG(3,("Opening sockets %d\n", port));

  if (!open_sockets(is_daemon,port)) return 1;

  load_interfaces();
  add_subnet_interfaces(ServerNMB != -1);

  add_my_names(time_now);

  if (strequal(lp_workgroup(),"*")) {
    DEBUG(0,("ERROR: a workgroup name of * is no longer supported\n"));
  }

  add_workgroups_to_subnets(time_now);

  DEBUG(3,("Loaded names\n"));

  write_browse_list(time_now);
  dump_names(time_now);

  DEBUG(3,("Dumped names\n"));

  process();

  close(ClientNMB);
  close(ServerNMB);
  close(ClientDGRAM);

  if (dbf)
    fclose(dbf);
  return(0);
}
