/* 
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Module name: namedbname.c

   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   04 jul 96: lkcl@pires.co.uk
   created module namedbname containing name database functions

   30 July 96: David.Chappell@mail.trincoll.edu
   Expanded multiple workgroup domain master browser support.

*/

#include "includes.h"

extern int DEBUGLEVEL;


char *source_description[] =
{
	"LMHOSTS",
	"REG",
	"SELF",
	"DNS",
	"DNSFAIL",
	"EXPIRED"
};


/****************************************************************************
  true if two netbios names are equal
****************************************************************************/
BOOL name_equal(struct nmb_name *n1,struct nmb_name *n2)
{
  return n1->name_type == n2->name_type &&
         strequal(n1->name ,n2->name ) &&
         strequal(n1->scope,n2->scope);
}


/****************************************************************************
  true if the netbios name is ^1^2__MSBROWSE__^2^1

  note: this name is registered if as a master browser or backup browser
  you are responsible for a workgroup (when you announce a domain by
  broadcasting on your local subnet, you announce it as coming from this
  name: see announce_host()).

  **************************************************************************/
BOOL ms_browser_name(char *name, int type)
{
  return strequal(name,MSBROWSE) && type == 0x01;
}


/****************************************************************************
  add a netbios name into the namelist
  **************************************************************************/
static void add_name(struct name_record **namelist, struct name_record *n)
{
  struct name_record *n2;

  if (!namelist)
  {
    DEBUG(2,("NULL namelist in add_name()!\n"));
    return;
  }

  if (!(*namelist))
  {
    *namelist = n;
    n->prev = NULL;
    n->next = NULL;
    return;
  }

  for (n2 = *namelist; n2->next; n2 = n2->next) ;

  n2->next = n;
  n->next = NULL;
  n->prev = n2;
}


/****************************************************************************
  remove a name from the namelist. The pointer must be an element just 
  retrieved
  **************************************************************************/
struct name_record *remove_name(time_t time_now,
				struct name_record **namelist, time_t *last_mod,
				struct name_record *n, struct in_addr ip, BOOL expire)
{
  struct name_record *nlist;
  int i;
  int j;

  if (!namelist) return NULL;

  if ((*last_mod) == 0) *last_mod = time_now;

  nlist = *namelist;

  /* double-check the item being removed is in this list */
  while (nlist && nlist != n) nlist = nlist->next;

  for (i = 0; nlist && i < nlist->num_ips; i++)
  {
    if (!ip_equal(n->ip_flgs[i].ip, ip)) continue;

    if (expire)
    {
      DEBUG(3,("Expiring ip %s ", inet_ntoa(n->ip_flgs[i].ip)));
      n->ip_flgs[i].source = EXPIRED;
      n->ip_flgs[i].refresh_time = 0;
      n->ip_flgs[i].death_time = time_now;

      return nlist;
    }

    DEBUG(3,("Removing ip %s ", inet_ntoa(n->ip_flgs[i].ip)));
			  
    for (j = nlist->num_ips-2; j >= i; j--)
    {
	    nlist->ip_flgs[j] = nlist->ip_flgs[j+1];
    }

    nlist->num_ips--;

	if (nlist->num_ips == 0)
	{
		free(nlist->ip_flgs);

	    DEBUG(3,("and dead name %s", namestr(&nlist->name)));
			  
	    if (nlist->prev) nlist->prev->next = nlist->next;
	    if (nlist->next) nlist->next->prev = nlist->prev;
			  
	    if (*namelist == nlist) *namelist = nlist->next; 
			  
	    free(nlist);
        nlist = NULL;
	}
	else
	{
    	nlist->ip_flgs = realloc(nlist->ip_flgs, nlist->num_ips);
    }
  }
  DEBUG(3,("\n"));
  return nlist;
}


int find_name_idx(struct name_record *n, struct in_addr ip, int search)
{
	int i;

	if (!n) return -1;

	for (i = 0; i < n->num_ips; i++)
	{
		/* self search: self names only */
		if ((search & FIND_SELF) == FIND_SELF && 
			n->ip_flgs[i].source != EXPIRED &&
			n->ip_flgs[i].source != SELF &&
			n->name.name_type != 0x1c)
		{
			continue;
		}

		if (!ip_equal(ip, n->ip_flgs[i].ip))
		{
			continue;
		}

		return i;
	}
	return -1;
}


/****************************************************************************
  find a name in a namelist.
  **************************************************************************/
struct name_record *find_name(struct name_record *n, struct nmb_name *name,
            int search)
{
    struct name_record *ret;
  
    for (ret = n; ret; ret = ret->next)
    {
        if (name_equal(&ret->name,name))
        {
			return ret;
        }
    }
    return NULL;
}


/****************************************************************************
  dump a copy of the name table
  **************************************************************************/
void write_netbios_names(time_t time_now,
		char *file_save, struct name_record *namelist,
		time_t *last_mod, BOOL flush)
{
    struct name_record *n;
    fstring fname, fnamenew;
    
    FILE *f;
    
    if (!flush && ((*last_mod) == 0 || time_now - (*last_mod) < 10)) return;

    strcpy(fname,NMBDIR);
    trim_string(fname,NULL,"/");
    strcat(fname,"/");
    strcat(fname,file_save);
    strcpy(fnamenew,fname);
    strcat(fnamenew,".");
    
    f = fopen(fnamenew,"w");
  
    if (!f)
    {
      DEBUG(4,("Can't open %s - %s\n",fnamenew,strerror(errno)));
    }
  
    DEBUG(3,("Writing name table:\n"));
  
    for (n = namelist; n; n = n->next)
    {
      int i;
      BOOL reg = False;


      for (i = 0; i < n->num_ips; i++)
      {
        if (i == 0)
        {
          DEBUG(3,("%-19s ", namestr(&n->name)));
        }
        else
        {
          DEBUG(3,("%-19s ", ""));
        }

        reg |= n->ip_flgs[i].source == REGISTER;
        reg |= n->ip_flgs[i].source == EXPIRED;
        reg |= n->ip_flgs[i].source == SELF;

        DEBUG(3,("%7s ", source_description[n->ip_flgs[i].source]));
        DEBUG(3,("TTL=%7ld %-15s NB=%2x\n",
                        n->ip_flgs[i].death_time?
                           n->ip_flgs[i].death_time-time_now:0,
                        inet_ntoa(n->ip_flgs[i].ip),
                        (unsigned char)n->ip_flgs[i].nb_flags));
      }

      if (f && reg)
      {
        /* XXXX ignore scope for now */
        fprintf(f, "%s#%02x ", n->name.name,n->name.name_type); 

        for (i = 0; i < n->num_ips; i++)
        {
           if (n->ip_flgs[i].source == REGISTER ||
               n->ip_flgs[i].source == EXPIRED ||
               n->ip_flgs[i].source == SELF)
           {
             fprintf(f, "%ld %s %2x ",
                        n->ip_flgs[i].death_time,
                        inet_ntoa(n->ip_flgs[i].ip),
                        (unsigned char)n->ip_flgs[i].nb_flags);
           }
        }
        fprintf(f, "\n");
      }

    }

    fclose(f);
    unlink(fname);
    chmod(fnamenew,0644);
    rename(fnamenew,fname);   

	(*last_mod) = 0;

	DEBUG(3,("Wrote wins database %s\n",fname));
}


/****************************************************************************
  load a netbios name database file

  XXXX we cannot cope with loading Internet Group names, yet
  ****************************************************************************/
void load_netbios_names(time_t time_now, 
				struct name_record **namelist, time_t *last_mod)
{
  fstring fname;

  FILE *f;
  pstring line;

  if (!namelist) return;

  strcpy(fname,NMBDIR);
  trim_string(fname,NULL,"/");
  strcat(fname,"/");
  strcat(fname,WINS_LIST);

  f = fopen(fname,"r");

  if (!f) {
    DEBUG(2,("Can't open wins database file %s\n",fname));
    return;
  }

  DEBUG(4,("Time now: %d\n", time_now));

  while (!feof(f))
  {
    BOOL same_line; /* True if we are still reading in tokens for a line */

    pstring name_str, ip_str, ttd_str, nb_flags_str;
    pstring name;
    struct nmb_name nb_name;
    int type = 0;
    time_t ttd;
    struct nmb_ip nb;

    char *ptr;
    int count = 0;

    char *p;

    if (!fgets_slash(line,sizeof(pstring),f)) continue;

    if (*line == '#') continue;

    ptr = line;

    if (next_token(&ptr,name_str,NULL)) ++count;

    if (count <= 0) continue;

    /* netbios name. # divides the name from the type (hex): netbios#xx */
    strcpy(name,name_str);

    p = strchr(name,'#');

    if (p)
    {
      *p = 0;
      sscanf(p+1,"%x",&type);
    }

    same_line = True;

    while (same_line)
    {
      int nb_flags;
      if (same_line &= next_token(&ptr,ttd_str     ,NULL)) ++count;
      if (same_line &= next_token(&ptr,ip_str      ,NULL)) ++count;
      if (same_line &= next_token(&ptr,nb_flags_str,NULL)) ++count;

      if (same_line && (count-1) % 3 != 0)
      {
        DEBUG(0,("Ill formed wins line"));
        DEBUG(0,("[%s]: name#type abs_time ip nb_flags\n",line));
        same_line = False;
      }

      if (!same_line) break;

      /* decode the netbios flags (hex) and the time-to-die (seconds) */
      sscanf(nb_flags_str,"%x",&nb_flags);
      sscanf(ttd_str,"%ld",&ttd);

      nb.death_time = (ttd?ttd-time_now:0) / 3;
      nb.nb_flags = nb_flags;
      nb.ip = *interpret_addr2(ip_str);
      nb.source = REGISTER;

      DEBUG(4, ("add WINS line: %s#%02x %ld %s %2x\n",
             name,type, ttd, inet_ntoa(nb.ip), nb.nb_flags));

      /* XXXX must get scope loaded from file */

      make_nmb_name(&nb_name,name,type,NULL);
      /* add netbios entry read from the wins.dat file. IF it's ok */
      add_netbios_entry(time_now, namelist,last_mod,&nb_name,&nb,True,count>4);
    }
  }
  fclose(f);
}


/****************************************************************************
  remove an entry from the name list
  ****************************************************************************/
void remove_netbios_name(time_t time_now,
			struct name_record **namelist, time_t *last_mod,
            struct nmb_name *name, struct in_addr ip, BOOL expire)
{
	struct name_record *n = find_name(*namelist, name, FIND_LOCAL);
	if (n)
	{
  		remove_name(time_now, namelist,last_mod,n,ip,expire);
	}
}


/****************************************************************************
  update and entry in the name list
  ****************************************************************************/
static void update_name(time_t time_now,struct name_record *n, time_t *last_mod,
                        int idx, struct nmb_ip *nb, BOOL add)
{
  if ((*last_mod) == 0) *last_mod = time_now;

  memcpy(&n->ip_flgs[idx],nb,sizeof(*nb));

  if (ismyip(nb->ip))
  {
    if (n->ip_flgs[idx].source != DNSFAIL) n->ip_flgs[idx].source = SELF;
  }

  if (nb->death_time >= 0)
  {
    n->ip_flgs[idx].death_time = nb->death_time?time_now+nb->death_time*3:0;
    n->ip_flgs[idx].refresh_time = time_now+NAME_POLL_REFRESH_TIME;
  }
  else
  {
    n->ip_flgs[idx].source = EXPIRED;
    n->ip_flgs[idx].death_time = time_now + nb->death_time;
    n->ip_flgs[idx].refresh_time = 0;
  }

  if (add)
  {
    DEBUG(3,("Adding "));
  }
  else
  {
    DEBUG(3,("Updating "));
  }

  DEBUG(3,("name %s ", namestr(&n->name)));
  DEBUG(3,("(%s) ", source_description[nb->source]));
  DEBUG(3,("%s ", inet_ntoa(nb->ip)));
  DEBUG(3,("ttl=%d ", nb->death_time));
  DEBUG(3,("nb_flags=%2x\n", (unsigned char)nb->nb_flags));
}


/****************************************************************************
  add an internet group entry to the name list.

  ****************************************************************************/
BOOL add_netbios_entry_ip(time_t time_now,
		struct name_record *n, time_t *last_mod,
        struct nmb_ip *nb)
{
  int idx = n->num_ips; /* select the last (or most recently added) item */
  BOOL add = True;
  int i;

  DEBUG(4,("add_netbios_entry_ip: %x %s\n",
                    (unsigned char)nb->nb_flags, inet_ntoa(nb->ip)));

  for (i = 0; i < n->num_ips; i++)
  {
    /* don't add duplicate ip addresses */
    if (ip_equal(n->ip_flgs[i].ip, nb->ip))
    {
      /* update the current entry instead of overwriting one */
      add = False;
      idx = i;
      break;
    }
  }

  if (add && n->num_ips < 25)
  {
    n->num_ips++;
    n->ip_flgs = (struct nmb_ip*)realloc(n->ip_flgs,
                                       sizeof(*n->ip_flgs) * n->num_ips);
  }

  if (!n->ip_flgs)
  {
     DEBUG(4,("memory reallocation error in add_netbios_entry_ip()\n"));
     /* oops. delete the name */
     n->num_ips = 0;
     return False;
  }

  /* 25 items. hm. select the oldest non-static one */
  if (n->num_ips == 25 && add)
  {
    time_t oldest_time = n->ip_flgs[0].death_time;
    idx = -1;

    for (i = 0; i < n->num_ips-1; i++)
    {
      time_t this_time = n->ip_flgs[i].death_time;
      if (this_time && this_time < oldest_time)
      {
        idx = i;
        oldest_time = this_time;
      }
    }
    /* all entries are static. cannot add another dynamic one. sorry */
    if (idx == -1) return False;
  }

  update_name(time_now, n, last_mod, idx, nb, add);

  return True;
}

/****************************************************************************
  add an entry to the name list.

  this is a multi-purpose function.

  it adds samba's own names in to its records on each interface, keeping a
  record of whether it is a master browser, domain master, or WINS server.

  it also keeps a record of WINS entries.

  ****************************************************************************/
struct name_record *add_netbios_entry(time_t time_now,
		struct name_record **namelist, time_t *last_mod,
        struct nmb_name *name, struct nmb_ip *nb,
        BOOL wins,BOOL multi)
{
  struct name_record *n;
  struct name_record *n2=NULL;
  int search = 0;
  int idx;
  BOOL self = ismyip(nb->ip) || nb->source == SELF;
  BOOL add = False;

  /* add the name to the WINS list if the name comes from a directed query */
  search |= wins ? FIND_WINS : FIND_LOCAL;
  /* search for SELF names only */
  search |= self ? FIND_SELF : 0;

  DEBUG(4,("add_netbios_entry: %x %s\n",
                    (unsigned char)nb->nb_flags, inet_ntoa(nb->ip)));

  if (!self)
  {
    if (!wins && name->name_type != 0x1b)
    {
       /* the only broadcast (non-WINS) names we are adding are ours
          (SELF) and Domain Master type names */
       return NULL;
    }
  }

  n = (struct name_record *)malloc(sizeof(*n));
  if (!n) return(NULL);

  bzero((char *)n,sizeof(*n));

  idx = 0;
  n->num_ips = 1; /* start off adding one entry */
  n->ip_flgs = (struct nmb_ip*)malloc(sizeof(*n->ip_flgs) * n->num_ips);
  if (!n->ip_flgs)
  {
     free(n);
     return NULL;
  }

  memcpy(&n->name,name,sizeof(*name));

  if ((n2 = find_name(*namelist, name, search)))
  {
    free(n->ip_flgs);
    free(n);

    multi |= NAME_GROUP(nb->nb_flags) &&
           n2->name.name_type == 0x1c &&
           lp_dynamic_ipgroups();

    if (wins)
    {
      if (multi)
      {
        add_netbios_entry_ip(time_now, n2, last_mod, nb);
        return n2;
      }
    }

    n = n2;

    idx = find_name_idx(n, nb->ip, search);

    if (idx == -1) idx = 0;

    add = False;
  }
  else
  {
    add_name(namelist,n);
    add = True;
  }

  update_name(time_now, n, last_mod, idx, nb, add);

  return(n);
}
