/*
 *   fs/cifs/cifssmb.c
 *
 *   Copyright (c) International Business Machines  Corp., 2002
 *   Author(s): Steve French (sfrench@us.ibm.com)
 *
 *   Contains the routines for constructing the SMB PDUs themselves
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation; either version 2.1 of the License, or
 *   (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

 /* All SMB/CIFS PDU handling routines go here - except for a few leftovers in connect.c */

#include <linux/fs.h>
#include <linux/kernel.h>
#include "cifspdu.h"
#include "cifsglob.h"
#include "cifsproto.h"
#include "cifs_unicode.h"
#include "cifs_debug.h"

static struct {
	int index;
	char *name;
} protocols[] = {
	{
	CIFS_PROT, "\2NT LM 0.12"}, {
	BAD_PROT, "\2"}
};

int smb_init(int smb_command, int wct, const struct smbTconInfo *tcon,
	     void **request_buf /* returned */ ,
	     void **response_buf /* returned */ )
{
	int rc = 0;

	*request_buf = buf_get();
	if (request_buf == 0) {
		return ENOMEM;
	}
	*response_buf = buf_get();
	if (response_buf == 0) {
		buf_release(request_buf);
		return ENOMEM;
	}

	header_assemble((struct smb_hdr *) *request_buf, smb_command, tcon,
			wct /*wct */ );
	return rc;
}

int CIFSSMBNegotiate(unsigned int xid, struct smbSesInfo *ses,
		     char cryptokey[CIFS_CRYPTO_KEY_SIZE])
{
	NEGOTIATE_REQ *pSMB;
	NEGOTIATE_RSP *pSMBr;
	int rc = 0;
	int bytes_returned;

	/* smb_buffer = buf_get();
	   if (smb_buffer == 0) {
	   return ENOMEM;
	   }
	   smb_buffer_response = buf_get();
	   if (smb_buffer_response == 0) {
	   buf_release(smb_buffer);
	   return ENOMEM;
	   }

	   header_assemble(smb_buffer, SMB_COM_NEGOTIATE, 0, 0 ); */

	rc = smb_init(SMB_COM_NEGOTIATE, 0, 0 /* no tcon yet */ ,
		      (void **) &pSMB, (void **) &pSMBr);
	if (rc)
		return rc;


	pSMB->hdr.Flags2 |= SMBFLG2_UNICODE;
	pSMB->ByteCount = strlen(protocols[0].name) + 1;
	strncpy(pSMB->DialectsArray, protocols[0].name, 30);	/* null guaranteed to be at end of source and target buffers anyway */

	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc == 0) {
		ses->dialectIndex = le16_to_cpu(pSMBr->DialectIndex);
		ses->secMode = pSMBr->SecurityMode;	/* one byte - no need to convert this or EncryptionKeyLen field from le,  */
		ses->maxReq = le16_to_cpu(pSMBr->MaxMpxCount);
		/* probably no need to store and check maxvcs */
		ses->maxBuf =
		    min(le32_to_cpu(pSMBr->MaxBufferSize),
			(__u32) CIFS_MAX_MSGSIZE + MAX_CIFS_HDR_SIZE);
		ses->maxRw = le32_to_cpu(pSMBr->MaxRawSize);	/* BB le_to_host needed around here and ff */
		GETU32(ses->sessid) = le32_to_cpu(pSMBr->SessionKey);
		ses->capabilities = le32_to_cpu(pSMBr->Capabilities);
		ses->timeZone = le16_to_cpu(pSMBr->ServerTimeZone);
		if (pSMBr->EncryptionKeyLength == CIFS_CRYPTO_KEY_SIZE) {
			memcpy(cryptokey, pSMBr->EncryptionKey,
			       CIFS_CRYPTO_KEY_SIZE);
		} else
			rc = EIO;
		/* BB save domain of server here */
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSSMBTDis(const int xid, struct smbTconInfo *tcon)
{
	struct smb_hdr *smb_buffer;
	struct smb_hdr *smb_buffer_response;
	int rc = 0;
	int length;

	cFYI(1, ("\nIn tree disconnect"));
	/*
	 *  If last user of the connection and
	 *  connection alive - disconnect it
	 *  If this is the last connection on the server session disconnect it
	 *  (and inside session disconnect we should check if tcp socket needs to 
	 *  be freed and kernel thread woken up).
	 */
	if (tcon)
		down(&tcon->tconSem);
	else
		return -EIO;

	atomic_dec(&tcon->useCount);
	if (atomic_read(&tcon->useCount) > 0) {
		up(&tcon->tconSem);
		return -EBUSY;
	}

/* BB remove (from server) list of shares - but with smp safety  BB */
/* BB is ses active - do we need to check here - but how? BB */

	rc = smb_init(SMB_COM_TREE_DISCONNECT, 0, tcon,
		      (void **) &smb_buffer,
		      (void **) &smb_buffer_response);
	if (rc) {
		up(&tcon->tconSem);
		return rc;
	}
	rc = SendReceive(xid, tcon->ses, smb_buffer, smb_buffer_response,
			 &length, 0);
	if (rc)
		cFYI(1, (" Tree disconnect failed %d", rc));

	if (smb_buffer)
		buf_release(smb_buffer);
	if (smb_buffer_response)
		buf_release(smb_buffer_response);
	up(&tcon->tconSem);
	return rc;
}

int CIFSSMBLogoff(const int xid, struct smbSesInfo *ses)
{
	struct smb_hdr *smb_buffer_response;
	LOGOFF_ANDX_REQ *pSMB;
	int rc = 0;
	int length;

	cFYI(1, ("\nIn SMBLogoff for session disconnect"));

	if (ses)
		down(&ses->sesSem);
	else
		return -EIO;

	atomic_dec(&ses->inUse);
	if (atomic_read(&ses->inUse) > 0) {
		up(&ses->sesSem);
		return -EBUSY;
	}

	rc = smb_init(SMB_COM_LOGOFF_ANDX, 2, 0 /* no tcon anymore */ ,
		      (void **) &pSMB, (void **) &smb_buffer_response);
	if (rc) {
		up(&ses->sesSem);
		return rc;
	}

	pSMB->hdr.Uid = ses->Suid;

	pSMB->AndXCommand = 0xFF;

	rc = SendReceive(xid, ses, (struct smb_hdr *) pSMB,
			 smb_buffer_response, &length, 0);

	if (pSMB)
		buf_release(pSMB);
	if (smb_buffer_response)
		buf_release(smb_buffer_response);
	up(&ses->sesSem);
	return rc;
}

int CIFSSMBDelFile(const int xid, const struct smbTconInfo *tcon,
		   const char *fileName,
		   const struct nls_table *nls_codepage)
{
	DELETE_FILE_REQ *pSMB = NULL;
	DELETE_FILE_RSP *pSMBr = NULL;
	int rc = 0;
	int bytes_returned;
	int name_len;

	rc = smb_init(SMB_COM_DELETE, 1, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->fileName, fileName, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(fileName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->fileName, fileName, name_len);
	}
	pSMB->SearchAttributes =
	    cpu_to_le16(ATTR_READONLY | ATTR_HIDDEN | ATTR_SYSTEM);
	pSMB->ByteCount = name_len + 1;
	pSMB->BufferFormat = 0x04;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);
	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cERROR(1, ("\nSend error in RMFile = %d\n", rc));
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSSMBRmDir(const int xid, const struct smbTconInfo *tcon,
		 const char *dirName, const struct nls_table *nls_codepage)
{
	DELETE_DIRECTORY_REQ *pSMB = NULL;
	DELETE_DIRECTORY_RSP *pSMBr = NULL;
	int rc = 0;
	int bytes_returned;
	int name_len;

	cFYI(1, ("\nIn CIFSSMBRmDir"));

	rc = smb_init(SMB_COM_DELETE_DIRECTORY, 0, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->DirName, dirName, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(dirName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->DirName, dirName, name_len);
	}

	pSMB->ByteCount = name_len + 1;
	pSMB->BufferFormat = 0x04;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);
	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cERROR(1, ("\nSend error in RMDir = %d\n", rc));
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSSMBMkDir(const int xid, const struct smbTconInfo *tcon,
		 const char *name, const struct nls_table *nls_codepage)
{
	int rc = 0;
	CREATE_DIRECTORY_REQ *pSMB = NULL;
	CREATE_DIRECTORY_RSP *pSMBr = NULL;
	int bytes_returned;
	int name_len;

	cFYI(1, ("\nIn CIFSSMBMkDir"));

	rc = smb_init(SMB_COM_CREATE_DIRECTORY, 0, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->DirName, name, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(name, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->DirName, name, name_len);
	}

	pSMB->ByteCount = name_len + 1 /* for buf format */ ;
	pSMB->BufferFormat = 0x04;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);
	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cERROR(1, ("\nSend error in RMDir = %d\n", rc));
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);

	return rc;
}

/* BB Need to continue endian neutral code fixing from here down and in SessSetup and tConx */

int CIFSSMBOpen(const int xid, const struct smbTconInfo *tcon,
		const char *fileName, const int openDisposition,
		const int access_flags, const int omode, __u16 * netfid,
		const struct nls_table *nls_codepage)
{
	int rc = -EACCES;

	OPEN_REQ *pSMB = NULL;
	OPEN_RSP *pSMBr = NULL;
	int bytes_returned;
	int name_len;

	rc = smb_init(SMB_COM_NT_CREATE_ANDX, 24, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	pSMB->AndXCommand = 0xFF;	/* none */

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		pSMB->ByteCount = 1;	/* account for one byte pad to word boundary */
		name_len =
		    cifs_strtoUCS((UniChar *) (pSMB->fileName + 1),
				  fileName, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
		pSMB->NameLength = name_len;
	} else {		/* BB improve the check for buffer overruns BB */

		pSMB->ByteCount = 0;	/* no pad */
		name_len = strnlen(fileName, 530);
		name_len++;	/* trailing null */
		pSMB->NameLength = name_len;
		strncpy(pSMB->fileName, fileName, name_len);
	}

	pSMB->DesiredAccess = access_flags;
	pSMB->AllocationSize = 0;
	pSMB->FileAttributes = ATTR_NORMAL;	/* XP does not handle ATTR_POSIX_SEMANTICS */
	if ((omode & S_IWUGO) == 0)
		pSMB->FileAttributes |= ATTR_READONLY;	/* BB might as well do this until CIFS ACL support is put in */
	pSMB->ShareAccess = FILE_SHARE_ALL;
	pSMB->CreateDisposition = openDisposition;
	pSMB->CreateOptions = CREATE_NOT_DIR;	/* BB what are these? BB */
	pSMB->ImpersonationLevel = SECURITY_IMPERSONATION;	/* BB ?? BB */
	pSMB->SecurityFlags =
	    SECURITY_CONTEXT_TRACKING | SECURITY_EFFECTIVE_ONLY;

	pSMB->ByteCount += name_len;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;

	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);
	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cERROR(1, ("\nSend error in Open = %d\n", rc));
	} else
		*netfid = pSMBr->Fid;

	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);

	return rc;
}

int CIFSSMBRead(const int xid, const struct smbTconInfo *tcon,
		const int netfid, const unsigned int count,
		const __u64 lseek, unsigned int *nbytes, char *buf)
{
	int rc = -EACCES;
	READ_REQ *pSMB = NULL;
	READ_RSP *pSMBr = NULL;
	char *pReadData;
	int bytes_returned;

	rc = smb_init(SMB_COM_READ_ANDX, 12, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	pSMB->AndXCommand = 0xFF;	/* none */
	pSMB->Fid = netfid;
	pSMB->OffsetLow = lseek & 0xFFFFFFFF;
	pSMB->OffsetHigh = lseek >> 32;
	pSMB->Remaining = 0;
	pSMB->MaxCount =
	    min_t(int, count,
		  (tcon->ses->maxBuf - MAX_CIFS_HDR_SIZE) & 0xFFFFFF00);
	pSMB->MaxCountHigh = 0;
	pSMB->ByteCount = 0;	/* no need to do le conversion since it is 0 */

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cERROR(1, ("\nSend error in read = %d\n", rc));
		*nbytes = 0;
	} else {
		*nbytes = pSMBr->DataLength;
		/* BB check that DataLength would not go beyond end of SMB BB */
/* if(pSMBr->DataOffset < pSMBr->ByteCount + sizeof(READ_RSP) - 1 *//* BB fix this length check */
		if (pSMBr->DataLength >
		    CIFS_MAX_MSGSIZE + MAX_CIFS_HDR_SIZE) {
			rc = -EIO;
			*nbytes = 0;
		} else {
			pReadData =
			    (char *) (&pSMBr->hdr.Protocol) +
			    pSMBr->DataOffset;
			memcpy(buf, pReadData, pSMBr->DataLength);
		}
	}

	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);

	return rc;
}

int CIFSSMBWrite(const int xid, const struct smbTconInfo *tcon,
		 const int netfid, const unsigned int count,
		 const __u64 offset, unsigned int *nbytes, const char *buf,
		 const int long_op)
{
	int rc = -EACCES;
	WRITE_REQ *pSMB = NULL;
	WRITE_RSP *pSMBr = NULL;
	int bytes_returned;


	rc = smb_init(SMB_COM_WRITE_ANDX, 14, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	pSMB->AndXCommand = 0xFF;	/* none */
	pSMB->Fid = netfid;
	pSMB->OffsetLow = offset & 0xFFFFFFFF;
	pSMB->OffsetHigh = (offset >> 32);
	pSMB->Remaining = 0;
	if (count > ((tcon->ses->maxBuf - MAX_CIFS_HDR_SIZE) & 0xFFFFFF00))
		pSMB->DataLengthLow =
		    (tcon->ses->maxBuf - MAX_CIFS_HDR_SIZE) & 0xFFFFFF00;
	else
		pSMB->DataLengthLow = count;
	pSMB->DataLengthHigh = 0;
	pSMB->DataOffset = (int) &(pSMB->Data) - (int) pSMB->hdr.Protocol;
	memcpy(pSMB->Data, buf, pSMB->DataLengthLow);

	pSMB->ByteCount += pSMB->DataLengthLow + 1 /* pad */ ;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned,
			 long_op);
	if (rc) {
		cERROR(1, ("\nSend error in write = %d\n", rc));
		*nbytes = 0;
	} else
		*nbytes = pSMBr->Count;

	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);

	return rc;
}

int CIFSSMBLock(const int xid, const struct smbTconInfo *tcon,
		const __u16 smb_file_id, const __u64 len,
		const __u64 offset, const __u32 numUnlock,
		const __u32 numLock, const __u32 lockType,
		const int waitFlag)
{
	int rc = 0;
	LOCK_REQ *pSMB = NULL;
	LOCK_RSP *pSMBr = NULL;
	int bytes_returned;

	cFYI(1, ("\nIn CIFSSMBLock"));

	rc = smb_init(SMB_COM_LOCKING_ANDX, 8, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	pSMB->NumberOfLocks = cpu_to_le32(numLock);
	pSMB->NumberOfUnlocks = cpu_to_le32(numUnlock);
	pSMB->LockType = cpu_to_le32(lockType);
	pSMB->AndXCommand = 0xFF;	/* none */
	pSMB->Fid = cpu_to_le16(smb_file_id);

	pSMB->Locks[0].Pid = cpu_to_le16(current->pid);
	pSMB->Locks[0].Length = cpu_to_le64(len);
	pSMB->Locks[0].Offset = cpu_to_le64(offset);
	pSMB->ByteCount = sizeof(LOCKING_ANDX_RANGE);
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);

	if (rc) {
		cERROR(1, ("\nSend error in Lock = %d\n", rc));
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);

	return rc;
}


int CIFSSMBClose(const int xid, const struct smbTconInfo *tcon,
		 int smb_file_id)
{
	int rc = 0;
	CLOSE_REQ *pSMB = NULL;
	CLOSE_RSP *pSMBr = NULL;
	int bytes_returned;
	cFYI(1, ("\nIn CIFSSMBClose"));

	rc = smb_init(SMB_COM_CLOSE, 3, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	pSMB->FileID = (__u16) smb_file_id;
	pSMB->LastWriteTime = 0;
	pSMB->ByteCount = 0;
	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cERROR(1, ("\nSend error in Close = %d\n", rc));
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);

	return rc;
}

int CIFSSMBRename(const int xid, const struct smbTconInfo *tcon,
		  const char *fromName, const char *toName,
		  const struct nls_table *nls_codepage)
{
	int rc = 0;
	RENAME_REQ *pSMB = NULL;
	RENAME_RSP *pSMBr = NULL;
	int bytes_returned;
	int name_len, name_len2;

	cFYI(1, ("\nIn CIFSSMBRename"));

	rc = smb_init(SMB_COM_RENAME, 1, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	pSMB->BufferFormat = 0x04;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->OldFileName, fromName,
				  530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
		pSMB->OldFileName[name_len] = 0;	/* pad */
		pSMB->OldFileName[name_len + 1] = 0x04;	/* strange that protocol requires an ASCII signature byte on Unicode string */
		name_len2 =
		    cifs_strtoUCS((UniChar *) & pSMB->
				  OldFileName[name_len + 2], toName, 530,
				  nls_codepage);
		name_len2 +=
		    1 /* trailing null */  + 1 /* Signature word */ ;
		name_len2 *= 2;	/* convert to bytes */
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(fromName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->OldFileName, fromName, name_len);
		name_len2 = strnlen(toName, 530);
		name_len2++;	/* trailing null */
		pSMB->OldFileName[name_len] = 0x04;	/* 2nd buffer format */
		strncpy(&pSMB->OldFileName[name_len + 1], toName,
			name_len2);
		name_len2++;	/* trailing null */
		name_len2++;	/* signature byte */
	}

	pSMB->ByteCount = 1 /* 1st signature byte */  + name_len + name_len2;	/* we could also set search attributes but not needed */
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cFYI(1, ("\nSend error in RMDir = %d\n", rc));
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);

	return rc;
}


int CIFSUnixCreateSymLink(const int xid, const struct smbTconInfo *tcon,
			  const char *fromName, const char *toName,
			  const struct nls_table *nls_codepage)
{
	TRANSACTION2_SPI_REQ *pSMB = NULL;
	TRANSACTION2_SPI_RSP *pSMBr = NULL;
	char *data_offset;
	int name_len;
	int name_len_target;
	int rc = 0;
	int bytes_returned = 0;

	cFYI(1, ("\nIn Symlink Unix style"));


	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->FileName, fromName, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;

	} else {		/* BB improve the check for buffer overruns BB */
		name_len = strnlen(fromName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->FileName, fromName, name_len);
	}
	pSMB->TotalParameterCount = 6 + name_len;
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataOffset =
	    pSMB->ParameterOffset + pSMB->TotalParameterCount;

	data_offset = (char *) (&pSMB->hdr.Protocol) + pSMB->DataOffset;
	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len_target =
		    cifs_strtoUCS((UniChar *) data_offset, toName, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len_target++;	/* trailing null */
		name_len_target *= 2;
	} else {		/* BB improve the check for buffer overruns BB */
		name_len_target = strnlen(toName, 530);
		name_len_target++;	/* trailing null */
		strncpy(data_offset, toName, name_len_target);
	}

	pSMB->TotalDataCount = name_len_target;
	pSMB->DataCount = pSMB->TotalDataCount;
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 1000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_SET_PATH_INFORMATION;
	pSMB->ByteCount =
	    3 /* pad */  + pSMB->ParameterCount + pSMB->DataCount;
	pSMB->InformationLevel = SMB_SET_FILE_UNIX_LINK;
	pSMB->Reserved4 = 0;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);
	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cFYI(1,
		     ("\nSend error in SetPathInfo (create symlink) = %d\n",
		      rc));
	}

	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSUnixCreateHardLink(const int xid, const struct smbTconInfo *tcon,
			   const char *fromName, const char *toName,
			   const struct nls_table *nls_codepage)
{
	TRANSACTION2_SPI_REQ *pSMB = NULL;
	TRANSACTION2_SPI_RSP *pSMBr = NULL;
	char *data_offset;
	int name_len;
	int name_len_target;
	int rc = 0;
	int bytes_returned = 0;

	cFYI(1, ("\nIn Create Hard link Unix style"));


	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->FileName, toName, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;

	} else {		/* BB improve the check for buffer overruns BB */
		name_len = strnlen(toName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->FileName, toName, name_len);
	}
	pSMB->TotalParameterCount = 6 + name_len;
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataOffset =
	    pSMB->ParameterOffset + pSMB->TotalParameterCount;

	data_offset = (char *) (&pSMB->hdr.Protocol) + pSMB->DataOffset;
	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len_target =
		    cifs_strtoUCS((UniChar *) data_offset, fromName, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len_target++;	/* trailing null */
		name_len_target *= 2;
	} else {		/* BB improve the check for buffer overruns BB */
		name_len_target = strnlen(fromName, 530);
		name_len_target++;	/* trailing null */
		strncpy(data_offset, fromName, name_len_target);
	}

	pSMB->TotalDataCount = name_len_target;
	pSMB->DataCount = pSMB->TotalDataCount;
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 1000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_SET_PATH_INFORMATION;
	pSMB->ByteCount =
	    3 /* pad */  + pSMB->ParameterCount + pSMB->DataCount;
	pSMB->InformationLevel = SMB_SET_FILE_UNIX_HLINK;
	pSMB->Reserved4 = 0;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);
	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cFYI(1,
		     ("\nSend error in SetPathInfo (hard link) = %d\n",
		      rc));
	}

	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSCreateHardLink(const int xid, const struct smbTconInfo *tcon,
		       const char *fromName, const char *toName,
		       const struct nls_table *nls_codepage)
{
	int rc = 0;
	NT_RENAME_REQ *pSMB = NULL;
	RENAME_RSP *pSMBr = NULL;
	int bytes_returned;
	int name_len, name_len2;

	cFYI(1, ("\nIn CIFSCreateHardLink"));

	rc = smb_init(SMB_COM_NT_RENAME, 4, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	pSMB->SearchAttributes = 0x16;
	pSMB->Flags = CREATE_HARD_LINK;
	pSMB->ClusterCount = 0;

	pSMB->BufferFormat = 0x04;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->OldFileName, fromName,
				  530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
		pSMB->OldFileName[name_len] = 0;	/* pad */
		pSMB->OldFileName[name_len + 1] = 0x04;	/* strange that protocol requires an ASCII signature byte on Unicode string */
		name_len2 =
		    cifs_strtoUCS((UniChar *) & pSMB->
				  OldFileName[name_len + 2], toName, 530,
				  nls_codepage);
		name_len2 +=
		    1 /* trailing null */  + 1 /* Signature word */ ;
		name_len2 *= 2;	/* convert to bytes */
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(fromName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->OldFileName, fromName, name_len);
		name_len2 = strnlen(toName, 530);
		name_len2++;	/* trailing null */
		pSMB->OldFileName[name_len] = 0x04;	/* 2nd buffer format */
		strncpy(&pSMB->OldFileName[name_len + 1], toName,
			name_len2);
		name_len2++;	/* trailing null */
		name_len2++;	/* signature byte */
	}

	pSMB->ByteCount = 1 /* string type byte */  + name_len + name_len2;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cFYI(1,
		     ("\nSend error in hard link (NT rename) = %d\n", rc));
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);

	return rc;
}


int CIFSSMBUnixQuerySymLink(const int xid, const struct smbTconInfo *tcon,
			    const unsigned char *searchName,
			    char *symlinkinfo, const int buflen,
			    const struct nls_table *nls_codepage)
{
/* SMB_QUERY_FILE_UNIX_LINK */
	TRANSACTION2_QPI_REQ *pSMB = NULL;
	TRANSACTION2_QPI_RSP *pSMBr = NULL;
	int rc = 0;
	int bytes_returned;
	int name_len;

	cFYI(1, ("\nIn QPathSymLinkInfo (Unix) the path %s", searchName));
	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->FileName, searchName,
				  530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(searchName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->FileName, searchName, name_len);
	}

	pSMB->TotalParameterCount = 2 /* level */  + 4 /* reserved */  +
	    name_len /* includes null */ ;
	pSMB->TotalDataCount = 0;
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 4000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataCount = 0;
	pSMB->DataOffset = 0;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_QUERY_PATH_INFORMATION;
	pSMB->ByteCount = pSMB->TotalParameterCount + 1 /* pad */ ;
	pSMB->InformationLevel = SMB_QUERY_FILE_UNIX_LINK;
	pSMB->Reserved4 = 0;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cFYI(1, ("\nSend error in QuerySymLinkInfo = %d\n", rc));
	} else {		/* decode response */

		if ((pSMBr->ByteCount < 2) || (pSMBr->DataOffset > 512))	/* BB also check enough total bytes returned */
			rc = EIO;	/* bad smb */
		else {
			cFYI(1,
			     ("\nData: %s ",
			      (char *) &pSMBr->hdr.Protocol +
			      pSMBr->DataOffset));
			strncpy(symlinkinfo, (char *) &pSMBr->hdr.Protocol + pSMBr->DataOffset, buflen);	/* BB remember to test with small buflen and large link data */
			symlinkinfo[buflen] = 0;
		}
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}



int CIFSSMBQPathInfo(const int xid, const struct smbTconInfo *tcon,
		     const unsigned char *searchName,
		     FILE_ALL_INFO * pFindData,
		     const struct nls_table *nls_codepage)
{
/* level 263 SMB_QUERY_FILE_ALL_INFO */
	TRANSACTION2_QPI_REQ *pSMB = NULL;
	TRANSACTION2_QPI_RSP *pSMBr = NULL;
	int rc = 0;
	int bytes_returned;
	int name_len;

	cFYI(1, ("\nIn QPathInfo path %s", searchName));
	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->FileName, searchName,
				  530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(searchName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->FileName, searchName, name_len);
	}

	pSMB->TotalParameterCount = 2 /* level */  + 4 /* reserved */  +
	    name_len /* includes null */ ;
	pSMB->TotalDataCount = 0;
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 4000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataCount = 0;
	pSMB->DataOffset = 0;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_QUERY_PATH_INFORMATION;
	pSMB->ByteCount = pSMB->TotalParameterCount + 1 /* pad */ ;
	pSMB->InformationLevel = SMB_QUERY_FILE_ALL_INFO;
	pSMB->Reserved4 = 0;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cFYI(1, ("\nSend error in QPathInfo = %d\n", rc));
	} else {		/* decode response */

		if ((pSMBr->ByteCount < 40) || (pSMBr->DataOffset > 512))	/* BB also check enough total bytes returned */
			rc = EIO;	/* bad smb */
		else {
			memcpy((char *) pFindData,
			       (char *) &pSMBr->hdr.Protocol +
			       pSMBr->DataOffset, sizeof(FILE_ALL_INFO));
		}
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSSMBUnixQPathInfo(const int xid, const struct smbTconInfo *tcon,
			 const unsigned char *searchName,
			 FILE_UNIX_BASIC_INFO * pFindData,
			 const struct nls_table *nls_codepage)
{
/* SMB_QUERY_FILE_UNIX_BASIC */
	TRANSACTION2_QPI_REQ *pSMB = NULL;
	TRANSACTION2_QPI_RSP *pSMBr = NULL;
	int rc = 0;
	int bytes_returned;
	int name_len;

	cFYI(1, ("\nIn QPathInfo (Unix) the path %s", searchName));
	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->FileName, searchName,
				  530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(searchName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->FileName, searchName, name_len);
	}

	pSMB->TotalParameterCount = 2 /* level */  + 4 /* reserved */  +
	    name_len /* includes null */ ;
	pSMB->TotalDataCount = 0;
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 4000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataCount = 0;
	pSMB->DataOffset = 0;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_QUERY_PATH_INFORMATION;
	pSMB->ByteCount = pSMB->TotalParameterCount + 1 /* pad */ ;
	pSMB->InformationLevel = SMB_QUERY_FILE_UNIX_BASIC;
	pSMB->Reserved4 = 0;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cFYI(1, ("\nSend error in QPathInfo = %d\n", rc));
	} else {		/* decode response */

		if ((pSMBr->ByteCount < 40) || (pSMBr->DataOffset > 512))	/* BB also check enough total bytes returned */
			rc = EIO;	/* bad smb */
		else {
			memcpy((char *) pFindData,
			       (char *) &pSMBr->hdr.Protocol +
			       pSMBr->DataOffset,
			       sizeof(FILE_UNIX_BASIC_INFO));
		}
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSFindSingle(const int xid, const struct smbTconInfo *tcon,
		   const char *searchName, FILE_ALL_INFO * findData,
		   const struct nls_table *nls_codepage)
{
/* level 257 SMB_ */
	TRANSACTION2_FFIRST_REQ *pSMB = NULL;
	TRANSACTION2_FFIRST_RSP *pSMBr = NULL;
	int rc = 0;
	int bytes_returned;
	int name_len;

	cFYI(1, ("\nIn FindUnique"));
	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->FileName, searchName,
				  530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(searchName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->FileName, searchName, name_len);
	}

	pSMB->TotalParameterCount = 12 + name_len /* includes null */ ;
	pSMB->TotalDataCount = 0;	/* no EAs */
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 4000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &pSMB->InformationLevel - (int) &pSMB->hdr.Protocol;
	pSMB->DataCount = 0;
	pSMB->DataOffset = 0;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_FIND_FIRST;
	pSMB->ByteCount = pSMB->TotalParameterCount + 1 /* pad */ ;
	pSMB->SearchAttributes = ATTR_READONLY | ATTR_HIDDEN | ATTR_SYSTEM;
	pSMB->SearchCount = 10;	/* BB increase */
	pSMB->SearchFlags = 1;	/* close search after this request - BB remove BB */
	pSMB->InformationLevel = SMB_FIND_FILE_DIRECTORY_INFO;
	pSMB->SearchStorageType = 0;	/* BB what should we set this to? BB */
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);

	if (rc) {
		cFYI(1, ("\nSend error in FindFileDirInfo = %d\n", rc));
	} else {		/* decode response */

		/* BB fill in */
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSFindFirst(const int xid, const struct smbTconInfo *tcon,
		  const char *searchName, FILE_DIRECTORY_INFO * findData,
		  T2_FFIRST_RSP_PARMS * findParms,
		  const struct nls_table *nls_codepage, int *pUnicodeFlag,
		  int *pUnixFlag)
{
/* level 257 SMB_ */
	TRANSACTION2_FFIRST_REQ *pSMB = NULL;
	TRANSACTION2_FFIRST_RSP *pSMBr = NULL;
	char *response_data;
	int rc = 0;
	int bytes_returned;
	int name_len;

	cFYI(1, ("\nIn FindFirst"));
	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->FileName, searchName,
				  530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(searchName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->FileName, searchName, name_len);
	}

	pSMB->TotalParameterCount =
	    cpu_to_le16(12 + name_len /* includes null */ );
	pSMB->TotalDataCount = 0;	/* no EAs */
	pSMB->MaxParameterCount = cpu_to_le16(10);
	pSMB->MaxDataCount = cpu_to_le16((tcon->ses->maxBuf -
					  MAX_CIFS_HDR_SIZE) & 0xFFFFFF00);
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset = cpu_to_le16((int) &pSMB->SearchAttributes -
					    (int) &pSMB->hdr.Protocol);
	pSMB->DataCount = 0;
	pSMB->DataOffset = 0;
	pSMB->SetupCount = 1;	/* one byte no need to make endian neutral */
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = cpu_to_le16(TRANS2_FIND_FIRST);
	pSMB->ByteCount = pSMB->TotalParameterCount + 1 /* pad */ ;
	pSMB->SearchAttributes =
	    cpu_to_le16(ATTR_READONLY | ATTR_HIDDEN | ATTR_SYSTEM |
			ATTR_DIRECTORY);
	pSMB->SearchCount = cpu_to_le16(CIFS_MAX_MSGSIZE / sizeof(FILE_DIRECTORY_INFO));	/* should this be shrunk even more ? */
	pSMB->SearchFlags = cpu_to_le16(CIFS_SEARCH_CLOSE_AT_END);

	/* test for Unix extensions */
	if (tcon->ses->capabilities & CAP_UNIX) {
		pSMB->InformationLevel = cpu_to_le16(SMB_FIND_FILE_UNIX);
		*pUnixFlag = TRUE;
	} else {
		pSMB->InformationLevel =
		    cpu_to_le16(SMB_FIND_FILE_DIRECTORY_INFO);
		*pUnixFlag = FALSE;
	}
	pSMB->SearchStorageType = 0;	/* BB what should we set this to? It is not clear if it matters BB */
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);

	if (rc) {		/* BB add logic to retry regular search if Unix search rejected unexpectedly by server */
		cFYI(1, ("\nError in FindFirst = %d\n", rc));
	} else {		/* decode response */
		/* BB add safety checks for these memcpys */
		if (pSMBr->hdr.Flags2 & le16_to_cpu(SMBFLG2_UNICODE))
			*pUnicodeFlag = TRUE;
		else
			*pUnicodeFlag = FALSE;
		memcpy(findParms,
		       (char *) &pSMBr->hdr.Protocol +
		       le16_to_cpu(pSMBr->ParameterOffset),
		       sizeof(T2_FFIRST_RSP_PARMS));
		response_data =
		    (char *) &pSMBr->hdr.Protocol + pSMBr->DataOffset;
		memcpy(findData, response_data,
		       le16_to_cpu(pSMBr->DataCount));
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSFindNext(const int xid, const struct smbTconInfo *tcon,
		 FILE_DIRECTORY_INFO * findData,
		 T2_FNEXT_RSP_PARMS * findParms, const __u16 searchHandle,
		 __u32 resumeKey, int *pUnicodeFlag, int *pUnixFlag)
{
/* level 257 SMB_ */
	TRANSACTION2_FNEXT_REQ *pSMB = NULL;
	TRANSACTION2_FNEXT_RSP *pSMBr = NULL;
	char *response_data;
	int rc = 0;
	int bytes_returned;

	cFYI(1, ("\nIn FindNext"));
	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	pSMB->TotalParameterCount = 14;	/* includes 2 bytes of null string */
	pSMB->TotalDataCount = 0;	/* no EAs */
	pSMB->MaxParameterCount = 8;
	pSMB->MaxDataCount =
	    (tcon->ses->maxBuf - MAX_CIFS_HDR_SIZE) & 0xFFFFFF00;
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &pSMB->SearchHandle - (int) &pSMB->hdr.Protocol;
	pSMB->DataCount = 0;
	pSMB->DataOffset = 0;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_FIND_NEXT;
	pSMB->ByteCount = pSMB->TotalParameterCount + 1 /* pad */ ;
	pSMB->SearchHandle = searchHandle;
	pSMB->SearchCount =
	    min_t(int, CIFS_MAX_MSGSIZE / sizeof(FILE_DIRECTORY_INFO),
		  findParms->SearchCount);
	findParms->SearchCount = 0;	/* reset to zero in case of error */
	/* test for Unix extensions */
	if (tcon->ses->capabilities & CAP_UNIX) {
		pSMB->InformationLevel = cpu_to_le16(SMB_FIND_FILE_UNIX);
		*pUnixFlag = TRUE;
	} else {
		pSMB->InformationLevel =
		    cpu_to_le16(SMB_FIND_FILE_DIRECTORY_INFO);
		*pUnixFlag = FALSE;
	}
	pSMB->ResumeKey = resumeKey;
	pSMB->SearchFlags =
	    CIFS_SEARCH_CLOSE_AT_END | CIFS_SEARCH_CONTINUE_FROM_LAST;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);

	if (rc) {
		if (rc == EBADF)
			rc = 0;	/* search probably was closed at end of search above */
		else
			cFYI(1, ("\nFindNext returned = %d\n", rc));
	} else {		/* decode response */
		/* BB add safety checks for these memcpys */
		if (pSMBr->hdr.Flags2 & SMBFLG2_UNICODE)
			*pUnicodeFlag = TRUE;
		else
			*pUnicodeFlag = FALSE;
		memcpy(findParms,
		       (char *) &pSMBr->hdr.Protocol +
		       pSMBr->ParameterOffset, sizeof(T2_FNEXT_RSP_PARMS));
		response_data =
		    (char *) &pSMBr->hdr.Protocol + pSMBr->DataOffset;
		memcpy(findData, response_data, pSMBr->DataCount);
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSGetDFSRefer(const int xid, const struct smbTconInfo *tcon,
		    const unsigned char *searchName,
		    unsigned char **targetUNCs,
		    int *number_of_UNC_in_array,
		    const struct nls_table *nls_codepage)
{
/* TRANS2_GET_DFS_REFERRAL */
	TRANSACTION2_GET_DFS_REFER_REQ *pSMB = NULL;
	TRANSACTION2_GET_DFS_REFER_RSP *pSMBr = NULL;
	int rc = 0;
	int bytes_returned;
	int name_len;

	cFYI(1, ("\nIn GetDFSRefer the path %s", searchName));
	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	/* BB prepend UNC path to real local mount/share relative path - 
	   and also we really should connect to IPC$ here and use that tid instead of our file share tid BB */

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->RequestFileName,
				  searchName, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(searchName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->RequestFileName, searchName, name_len);
	}

	pSMB->TotalParameterCount =
	    2 /* level */  + name_len /*includes null */ ;
	pSMB->TotalDataCount = 0;
	pSMB->MaxParameterCount = 0;
	pSMB->MaxDataCount = 4000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->MaxReferralLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataCount = 0;
	pSMB->DataOffset = 0;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_GET_DFS_REFERRAL;
	pSMB->ByteCount = pSMB->TotalParameterCount + 1 /* pad */ ;
	pSMB->MaxReferralLevel = 3;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cFYI(1, ("\nSend error in GetDFSRefer = %d\n", rc));
	} else {		/* decode response */
/* BB Add logic to parse referrals here */
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
        return rc;
}

int CIFSSMBQFSInfo(const int xid, const struct smbTconInfo *tcon,
		   struct statfs *FSData,
		   const struct nls_table *nls_codepage)
{
/* level 0x103 SMB_QUERY_FILE_SYSTEM_INFO */
	TRANSACTION2_QFSI_REQ *pSMB = NULL;
	TRANSACTION2_QFSI_RSP *pSMBr = NULL;
	FILE_SYSTEM_INFO *response_data;
	int rc = 0;
	int bytes_returned = 0;

	cFYI(1, ("\nIn QFSInfo"));

	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	pSMB->TotalParameterCount = 2;	/* level */
	pSMB->TotalDataCount = 0;
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 1000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataCount = 0;
	pSMB->DataOffset = 0;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_QUERY_FS_INFORMATION;
	pSMB->ByteCount = pSMB->TotalParameterCount + 1 /* pad */ ;
	pSMB->InformationLevel = SMB_QUERY_FS_SIZE_INFO;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cERROR(1, ("\nSend error in QFSInfo = %d\n", rc));
	} else {		/* decode response */

		cFYI(1,
		     ("\nDecoding qfsinfo response.  BCC: %d  Offset %d\n",
		      pSMBr->ByteCount, pSMBr->DataOffset));
		if ((pSMBr->ByteCount < 24) || (pSMBr->DataOffset > 512))	/* BB also check enough total bytes returned */
			rc = EIO;	/* bad smb */
		else {
			response_data =
			    (FILE_SYSTEM_INFO
			     *) (((char *) &pSMBr->hdr.Protocol) +
				 pSMBr->DataOffset);
			FSData->f_bsize =
			    response_data->BytesPerSector *
			    response_data->SectorsPerAllocationUnit;
			FSData->f_blocks =
			    response_data->TotalAllocationUnits;
			FSData->f_bfree = FSData->f_bavail =
			    response_data->FreeAllocationUnits;
			cFYI(1,
			     ("\nBlocks: %ld  Free: %ld BytesPerSector %d\n",
			      FSData->f_blocks, FSData->f_bfree,
			      response_data->BytesPerSector));
		}
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSSMBQFSAttributeInfo(int xid, struct smbTconInfo *tcon,
			    const struct nls_table *nls_codepage)
{
/* level 0x105  SMB_QUERY_FILE_SYSTEM_INFO */
	TRANSACTION2_QFSI_REQ *pSMB = NULL;
	TRANSACTION2_QFSI_RSP *pSMBr = NULL;
	FILE_SYSTEM_ATTRIBUTE_INFO *response_data;
	int rc = 0;
	int bytes_returned = 0;

	cFYI(1, ("\nIn QFSAttributeInfo"));
	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	pSMB->TotalParameterCount = 2;	/* level */
	pSMB->TotalDataCount = 0;
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 1000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataCount = 0;
	pSMB->DataOffset = 0;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_QUERY_FS_INFORMATION;
	pSMB->ByteCount = pSMB->TotalParameterCount + 1 /* pad */ ;
	pSMB->InformationLevel = SMB_QUERY_FS_ATTRIBUTE_INFO;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cERROR(1, ("\nSend error in QFSAttributeInfo = %d\n", rc));
	} else {		/* decode response */

		if ((pSMBr->ByteCount < 13) || (pSMBr->DataOffset > 512)) {	/* BB also check enough bytes returned */
			rc = EIO;	/* bad smb */
		} else {
			response_data =
			    (FILE_SYSTEM_ATTRIBUTE_INFO
			     *) (((char *) &pSMBr->hdr.Protocol) +
				 pSMBr->DataOffset);
			memcpy(&tcon->fsAttrInfo, response_data,
			       sizeof(FILE_SYSTEM_ATTRIBUTE_INFO));
		}
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSSMBQFSDeviceInfo(int xid, struct smbTconInfo *tcon,
			 const struct nls_table *nls_codepage)
{
/* level 0x104 SMB_QUERY_FILE_SYSTEM_INFO */
	TRANSACTION2_QFSI_REQ *pSMB = NULL;
	TRANSACTION2_QFSI_RSP *pSMBr = NULL;
	FILE_SYSTEM_DEVICE_INFO *response_data;
	int rc = 0;
	int bytes_returned = 0;

	cFYI(1, ("\nIn QFSDeviceInfo"));

	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	pSMB->TotalParameterCount = 2;	/* level */
	pSMB->TotalDataCount = 0;
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 1000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataCount = 0;
	pSMB->DataOffset = 0;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_QUERY_FS_INFORMATION;
	pSMB->ByteCount = pSMB->TotalParameterCount + 1 /* pad */ ;
	pSMB->InformationLevel = SMB_QUERY_FS_DEVICE_INFO;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cERROR(1, ("\nSend error in QFSDeviceInfo = %d\n", rc));
	} else {		/* decode response */

		if ((pSMBr->ByteCount < sizeof(FILE_SYSTEM_DEVICE_INFO)) || (pSMBr->DataOffset > 512))	/* BB also check enough bytes returned */
			rc = EIO;	/* bad smb */
		else {
			response_data =
			    (FILE_SYSTEM_DEVICE_INFO
			     *) (((char *) &pSMBr->hdr.Protocol) +
				 pSMBr->DataOffset);
			memcpy(&tcon->fsDevInfo, response_data,
			       sizeof(FILE_SYSTEM_DEVICE_INFO));
		}
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSSMBQFSUnixInfo(int xid, struct smbTconInfo *tcon,
		       const struct nls_table *nls_codepage)
{
/* level 0x200  SMB_QUERY_CIFS_UNIX_INFO */
	TRANSACTION2_QFSI_REQ *pSMB = NULL;
	TRANSACTION2_QFSI_RSP *pSMBr = NULL;
	FILE_SYSTEM_UNIX_INFO *response_data;
	int rc = 0;
	int bytes_returned = 0;

	cFYI(1, ("\nIn QFSUnixInfo"));
	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	pSMB->TotalParameterCount = 2;	/* level */
	pSMB->TotalDataCount = 0;
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 100;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataCount = 0;
	pSMB->DataOffset = 0;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_QUERY_FS_INFORMATION;
	pSMB->ByteCount = pSMB->TotalParameterCount + 1 /* pad */ ;
	pSMB->InformationLevel = SMB_QUERY_CIFS_UNIX_INFO;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);

	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cERROR(1, ("\nSend error in QFSUnixInfo = %d\n", rc));
	} else {		/* decode response */

		if ((pSMBr->ByteCount < 13) || (pSMBr->DataOffset > 512)) {	/* BB also check enough bytes returned */
			rc = EIO;	/* bad smb */
		} else {
			response_data =
			    (FILE_SYSTEM_UNIX_INFO
			     *) (((char *) &pSMBr->hdr.Protocol) +
				 pSMBr->DataOffset);
			memcpy(&tcon->fsUnixInfo, response_data,
			       sizeof(FILE_SYSTEM_UNIX_INFO));
		}
	}
	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}


int CIFSSMBSetEOF(int xid, struct smbTconInfo *tcon, char *fileName,
		  __u64 size, const struct nls_table *nls_codepage)
{
/* level 0x104 T2SetPathInfo - we can not use write of zero bytes trick to set file size due to need for large file support*/
	TRANSACTION2_SPI_REQ *pSMB = NULL;
	TRANSACTION2_SPI_RSP *pSMBr = NULL;
	FILE_END_OF_FILE_INFO *parm_data;
	int name_len;
	int rc = 0;
	int bytes_returned = 0;

	cFYI(1, ("\nIn SetEOF"));


	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->FileName, fileName, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(fileName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->FileName, fileName, name_len);
	}
	pSMB->TotalParameterCount = 6 + name_len;
	pSMB->TotalDataCount = sizeof(FILE_END_OF_FILE_INFO);
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 1000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataCount = pSMB->TotalDataCount;
	pSMB->DataOffset =
	    pSMB->ParameterOffset + pSMB->TotalParameterCount;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_SET_PATH_INFORMATION;
	pSMB->ByteCount =
	    3 /* pad */  + pSMB->ParameterCount + pSMB->DataCount;
	if (tcon->ses->capabilities & CAP_INFOLEVEL_PASSTHRU)
		pSMB->InformationLevel = SMB_SET_FILE_END_OF_FILE_INFO2;
	else
		pSMB->InformationLevel = SMB_SET_FILE_END_OF_FILE_INFO;
	pSMB->Reserved4 = 0;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	parm_data =
	    (FILE_END_OF_FILE_INFO *) (((char *) &pSMB->hdr.Protocol) +
				       pSMB->DataOffset);
	parm_data->FileSize = size;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);
	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cFYI(1, ("\nSetPathInfo (file size) returned %d\n", rc));
	}

	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSSMBSetAllocationSize(int xid, struct smbTconInfo *tcon,
			     char *fileName, __u64 size,
			     const struct nls_table *nls_codepage)
{
/* level 0x104 T2SetPathInfo - we can not use write of zero bytes trick to set file size due to need for large file support*/
	TRANSACTION2_SPI_REQ *pSMB = NULL;
	TRANSACTION2_SPI_RSP *pSMBr = NULL;
	FILE_END_OF_FILE_INFO *parm_data;
	int name_len;
	int rc = 0;
	int bytes_returned = 0;

	cFYI(1, ("\nIn SetEOF"));


	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->FileName, fileName, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(fileName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->FileName, fileName, name_len);
	}
	pSMB->TotalParameterCount = 6 + name_len;
	pSMB->TotalDataCount = sizeof(FILE_END_OF_FILE_INFO);
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 1000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataCount = pSMB->TotalDataCount;
	pSMB->DataOffset =
	    pSMB->ParameterOffset + pSMB->TotalParameterCount;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_SET_PATH_INFORMATION;
	pSMB->ByteCount =
	    3 /* pad */  + pSMB->ParameterCount + pSMB->DataCount;
	if (tcon->ses->capabilities & CAP_INFOLEVEL_PASSTHRU)
		pSMB->InformationLevel = SMB_SET_FILE_ALLOCATION_INFO2;
	else
		pSMB->InformationLevel = SMB_SET_FILE_ALLOCATION_INFO;

	pSMB->Reserved4 = 0;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	parm_data =
	    (FILE_END_OF_FILE_INFO *) (((char *) &pSMB->hdr.Protocol) +
				       pSMB->DataOffset);
	parm_data->FileSize = size;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);
	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 2);
	if (rc) {
		cFYI(1,
		     ("\nSetPathInfo (allocation size) returned %d\n",
		      rc));
	}

	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}


int CIFSSMBSetTimes(int xid, struct smbTconInfo *tcon, char *fileName,
		    FILE_BASIC_INFO * data,
		    const struct nls_table *nls_codepage)
{
	TRANSACTION2_SPI_REQ *pSMB = NULL;
	TRANSACTION2_SPI_RSP *pSMBr = NULL;
	int name_len;
	int rc = 0;
	int bytes_returned = 0;
	char *data_offset;

	cFYI(1, ("\nIn SetTimes"));


	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->FileName, fileName, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(fileName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->FileName, fileName, name_len);
	}

	pSMB->TotalParameterCount = 6 + name_len;
	pSMB->TotalDataCount = sizeof(FILE_BASIC_INFO);
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 1000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataCount = pSMB->TotalDataCount;
	pSMB->DataOffset = pSMB->ParameterOffset + pSMB->ParameterCount;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_SET_PATH_INFORMATION;
	pSMB->ByteCount =
	    3 /* pad */  + pSMB->ParameterCount + pSMB->DataCount;
	if (tcon->ses->capabilities & CAP_INFOLEVEL_PASSTHRU)
		pSMB->InformationLevel = SMB_SET_FILE_BASIC_INFO2;
	else
		pSMB->InformationLevel = SMB_SET_FILE_BASIC_INFO;
	pSMB->Reserved4 = 0;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	data_offset = (char *) (&pSMB->hdr.Protocol) + pSMB->DataOffset;
	memcpy(data_offset, data, sizeof(FILE_BASIC_INFO));
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);
	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cFYI(1, ("\nSetPathInfo (times) returned %d\n", rc));
	}

	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}

int CIFSSMBUnixSetPerms(const int xid, struct smbTconInfo *tcon,
			char *fileName, __u64 mode, __u64 uid, __u64 gid,
			const struct nls_table *nls_codepage)
{
	TRANSACTION2_SPI_REQ *pSMB = NULL;
	TRANSACTION2_SPI_RSP *pSMBr = NULL;
	int name_len;
	int rc = 0;
	int bytes_returned = 0;
	FILE_UNIX_BASIC_INFO *data_offset;

	cFYI(1, ("\nIn SetUID/GID/Mode"));


	rc = smb_init(SMB_COM_TRANSACTION2, 15, tcon, (void **) &pSMB,
		      (void **) &pSMBr);
	if (rc)
		return rc;

	if (pSMB->hdr.Flags2 & SMBFLG2_UNICODE) {
		name_len =
		    cifs_strtoUCS((UniChar *) pSMB->FileName, fileName, 530
				  /* find define for this maxpathcomponent */
				  , nls_codepage);
		name_len++;	/* trailing null */
		name_len *= 2;
	} else {		/* BB improve the check for buffer overruns BB */

		name_len = strnlen(fileName, 530);
		name_len++;	/* trailing null */
		strncpy(pSMB->FileName, fileName, name_len);
	}

	pSMB->TotalParameterCount = 6 + name_len;
	pSMB->TotalDataCount = sizeof(FILE_UNIX_BASIC_INFO);
	pSMB->MaxParameterCount = 2;
	pSMB->MaxDataCount = 1000;	/* BB find exact max SMB PDU from sess structure BB */
	pSMB->MaxSetupCount = 0;
	pSMB->Reserved = 0;
	pSMB->Flags = 0;
	pSMB->Timeout = 0;
	pSMB->Reserved2 = 0;
	pSMB->ParameterCount = pSMB->TotalParameterCount;
	pSMB->ParameterOffset =
	    (int) &(pSMB->InformationLevel) - (int) pSMB->hdr.Protocol;
	pSMB->DataCount = pSMB->TotalDataCount;
	pSMB->DataOffset = pSMB->ParameterOffset + pSMB->ParameterCount;
	pSMB->SetupCount = 1;
	pSMB->Reserved3 = 0;
	pSMB->SubCommand = TRANS2_SET_PATH_INFORMATION;
	pSMB->ByteCount =
	    3 /* pad */  + pSMB->ParameterCount + pSMB->DataCount;
	pSMB->InformationLevel = SMB_SET_FILE_UNIX_BASIC;
	pSMB->Reserved4 = 0;
	pSMB->hdr.smb_buf_length += pSMB->ByteCount;
	data_offset =
	    (FILE_UNIX_BASIC_INFO *) ((char *) &pSMB->hdr.Protocol +
				      pSMB->DataOffset);
	data_offset->Uid = uid;
	cFYI(1, ("\nUid = %lld from %lld ", data_offset->Uid, uid));
	data_offset->Gid = gid;
	cFYI(1, ("\nGid = %lld from %lld ", data_offset->Gid, gid));
	data_offset->Permissions = mode;
	pSMB->ByteCount = cpu_to_le16(pSMB->ByteCount);
	rc = SendReceive(xid, tcon->ses, (struct smb_hdr *) pSMB,
			 (struct smb_hdr *) pSMBr, &bytes_returned, 0);
	if (rc) {
		cFYI(1, ("\nSetPathInfo (perms) returned %d\n", rc));
	}

	if (pSMB)
		buf_release(pSMB);
	if (pSMBr)
		buf_release(pSMBr);
	return rc;
}
