#!/bin/sh
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.
cmdname=`basename "$0"`
MOZ_DIST_BIN=`dirname "$0"`
MOZ_DEFAULT_NAME="./${cmdname}-bin"
MOZ_APPRUNNER_NAME="./mozilla-bin"
MOZ_PROGRAM=""

exitcode=1
#
##
## Functions
##
##########################################################################
moz_usage()
{
echo "Usage:  ${cmdname} [options] [program]"
echo ""
echo "  options:"
echo ""
echo "    -g                   Run in debugger."
echo "    --debug"
echo ""
echo "    -d debugger          Debugger to use."
echo "    --debugger debugger"
echo ""
echo "    -a debugger_args     Arguments passed to [debugger]."
echo "    --debugger-args debugger_args"
echo ""
echo "  Examples:"
echo ""
echo "  Run the mozilla-bin binary"
echo ""
echo "    ${cmdname} mozilla-bin"
echo ""
echo "  Debug the mozilla-bin binary in gdb"
echo ""
echo "    ${cmdname} -g mozilla-bin -d gdb"
echo ""
echo "  Run mozilla-bin under valgrind with arguments"
echo ""
echo "    ${cmdname} -g -d valgrind -a '--tool=memcheck --leak-check=full' mozilla-bin"
echo ""
	return 0
}
##########################################################################
moz_bail()
{
	message=$1
	echo
	echo "$cmdname: $message"
	echo
	exit 1
}
##########################################################################
moz_test_binary()
{
	binary=$1
	if [ -f "$binary" ]
	then
		if [ -x "$binary" ]
		then
			return 1
		fi
	fi
	return 0
}
##########################################################################
moz_get_debugger()
{
	debuggers="ddd gdb dbx bdb native-gdb"
	debugger="notfound"
	done="no"
	for d in $debuggers
	do
		moz_test_binary /bin/which
		if [ $? -eq 1 ]
		then
			dpath=`which ${d}`	
		else 	
			dpath=`LC_MESSAGES=C type ${d} | awk '{print $3;}' | sed -e 's/\.$//'`	
		fi
		if [ -x "$dpath" ]
		then
			debugger=$dpath
			break
		fi
	done
	echo $debugger
	return 0
}
##########################################################################
moz_run_program()
{
	prog=$MOZ_PROGRAM
	##
	## Make sure the program is executable
	##
	if [ ! -x "$prog" ]
	then
		moz_bail "Cannot execute $prog."
	fi
	##
	## Run the program
	##
	exec "$prog" ${1+"$@"}
	exitcode=$?
}
##########################################################################
moz_debug_program()
{
	prog=$MOZ_PROGRAM
	##
	## Make sure the program is executable
	##
	if [ ! -x "$prog" ]
	then
		moz_bail "Cannot execute $prog."
	fi
	if [ -n "$moz_debugger" ]
	then
		moz_test_binary /bin/which
		if [ $? -eq 1 ]
		then	
			debugger=`which $moz_debugger` 
		else
			debugger=`LC_MESSAGES=C type $moz_debugger | awk '{print $3;}' | sed -e 's/\.$//'` 
		fi	
	else
		debugger=`moz_get_debugger`
	fi
    if [ -x "$debugger" ] 
    then
# If you are not using ddd, gdb and know of a way to convey the arguments 
# over to the prog then add that here- Gagan Saksena 03/15/00
        case `basename $debugger` in
            native-gdb) echo "$debugger $moz_debugger_args --args $prog" ${1+"$@"}
                exec "$debugger" $moz_debugger_args --args "$prog" ${1+"$@"}
                exitcode=$?
                ;;
            gdb) echo "$debugger $moz_debugger_args --args $prog" ${1+"$@"}
                exec "$debugger" $moz_debugger_args --args "$prog" ${1+"$@"}
		exitcode=$?
                ;;
            ddd) echo "$debugger $moz_debugger_args --gdb -- --args $prog" ${1+"$@"}
		exec "$debugger" $moz_debugger_args --gdb -- --args "$prog" ${1+"$@"}
		exitcode=$?
                ;;
            *) echo "$debugger $moz_debugger_args $prog ${1+"$@"}"
                exec $debugger $moz_debugger_args "$prog" ${1+"$@"}
		exitcode=$?
                ;;
        esac
    else
        moz_bail "Could not find a debugger on your system."
    fi
}
##########################################################################
##
## Command line arg defaults
##
moz_debug=0
moz_debugger=""
moz_debugger_args=""
#
##
## Parse the command line
##
while [ $# -gt 0 ]
do
  case $1 in
    -g | --debug)
      moz_debug=1
      shift
      ;;
    -d | --debugger)
      moz_debugger=$2;
      if [ "${moz_debugger}" != "" ]; then
	shift 2
      else
        echo "-d requires an argument"
        exit 1
      fi
      ;;
    -a | --debugger-args)
      moz_debugger_args=$2;
      if [ "${moz_debugger_args}" != "" ]; then
	shift 2
      else
        echo "-a requires an argument"
        exit 1
      fi
      ;;
    *)
      break;
      ;;
  esac
done
#
##
## Program name given in $1
##
if [ $# -gt 0 ]
then
	MOZ_PROGRAM=$1
	shift
fi
##
## Program not given, try to guess a default
##
if [ -z "$MOZ_PROGRAM" ]
then
	##
	## Try this script's name with '-bin' appended
	##
	if [ -x "$MOZ_DEFAULT_NAME" ]
	then
		MOZ_PROGRAM=$MOZ_DEFAULT_NAME
	##
	## Try mozilla-bin
	## 
	elif [ -x "$MOZ_APPRUNNER_NAME" ]
	then
		MOZ_PROGRAM=$MOZ_APPRUNNER_NAME
	fi
fi
#
#
##
## Make sure the program is executable
##
if [ ! -x "$MOZ_PROGRAM" ]
then
	moz_bail "Cannot execute $MOZ_PROGRAM."
fi
#
##
## Set MOZILLA_FIVE_HOME
##
MOZILLA_FIVE_HOME=$MOZ_DIST_BIN

if [ -z "$MRE_HOME" ]; then
    MRE_HOME=$MOZILLA_FIVE_HOME
fi
##
## Set LD_LIBRARY_PATH
##
## On Solaris we use $ORIGIN (set in RUNPATH) instead of LD_LIBRARY_PATH 
## to locate shared libraries. 
##
## When a shared library is a symbolic link, $ORIGIN will be replaced with
## the real path (i.e., what the symbolic link points to) by the runtime
## linker.  For example, if dist/bin/libxul.so is a symbolic link to
## toolkit/library/libxul.so, $ORIGIN will be "toolkit/library" instead of "dist/bin".
## So the runtime linker will use "toolkit/library" NOT "dist/bin" to locate the
## other shared libraries that libxul.so depends on.  This only happens
## when a user (developer) tries to start firefox, thunderbird, or seamonkey
## under dist/bin. To solve the problem, we should rely on LD_LIBRARY_PATH
## to locate shared libraries.
##
## Note: 
##  We test $MOZ_DIST_BIN/*.so. If any of them is a symbolic link,
##  we need to set LD_LIBRARY_PATH.
##########################################################################
moz_should_set_ld_library_path()
{
	[ `uname -s` != "SunOS" ] && return 0
	for sharedlib in $MOZ_DIST_BIN/*.so
	do
		[ -h $sharedlib ] && return 0
	done
	return 1
}
if moz_should_set_ld_library_path
then
	LD_LIBRARY_PATH=${MOZ_DIST_BIN}:${MOZ_DIST_BIN}/plugins:${MRE_HOME}${LD_LIBRARY_PATH:+":$LD_LIBRARY_PATH"}
fi 

if [ -n "$LD_LIBRARYN32_PATH" ]
then
	LD_LIBRARYN32_PATH=${MOZ_DIST_BIN}:${MOZ_DIST_BIN}/plugins:${MRE_HOME}${LD_LIBRARYN32_PATH:+":$LD_LIBRARYN32_PATH"}
fi
if [ -n "$LD_LIBRARYN64_PATH" ]
then
	LD_LIBRARYN64_PATH=${MOZ_DIST_BIN}:${MOZ_DIST_BIN}/plugins:${MRE_HOME}${LD_LIBRARYN64_PATH:+":$LD_LIBRARYN64_PATH"}
fi
if [ -n "$LD_LIBRARY_PATH_64" ]; then
	LD_LIBRARY_PATH_64=${MOZ_DIST_BIN}:${MOZ_DIST_BIN}/plugins:${MRE_HOME}${LD_LIBRARY_PATH_64:+":$LD_LIBRARY_PATH_64"}
fi
#
#
## Set SHLIB_PATH for HPUX
SHLIB_PATH=${MOZ_DIST_BIN}:${MRE_HOME}${SHLIB_PATH:+":$SHLIB_PATH"}
#
## Set LIBPATH for AIX
LIBPATH=${MOZ_DIST_BIN}:${MRE_HOME}${LIBPATH:+":$LIBPATH"}
#
## Set DYLD_LIBRARY_PATH for Mac OS X (Darwin)
DYLD_LIBRARY_PATH=${MOZ_DIST_BIN}:${MRE_HOME}${DYLD_LIBRARY_PATH:+":$DYLD_LIBRARY_PATH"}
#
## Solaris Xserver(Xsun) tuning - use shared memory transport if available
if [ "$XSUNTRANSPORT" = "" ]
then 
        XSUNTRANSPORT="shmem" 
        XSUNSMESIZE="512"
        export XSUNTRANSPORT XSUNSMESIZE
fi

# Disable Gnome crash dialog
GNOME_DISABLE_CRASH_DIALOG=1
export GNOME_DISABLE_CRASH_DIALOG

if [ "$moz_debug" -eq 1 ]
then
  echo "MOZILLA_FIVE_HOME=$MOZILLA_FIVE_HOME"
  echo "  LD_LIBRARY_PATH=$LD_LIBRARY_PATH"
  if [ -n "$LD_LIBRARYN32_PATH" ]
  then
  	echo "LD_LIBRARYN32_PATH=$LD_LIBRARYN32_PATH"
  fi
  if [ -n "$LD_LIBRARYN64_PATH" ]
  then
  	echo "LD_LIBRARYN64_PATH=$LD_LIBRARYN64_PATH"
  fi
  if [ -n "$LD_LIBRARY_PATH_64" ]; then
  	echo "LD_LIBRARY_PATH_64=$LD_LIBRARY_PATH_64"
  fi
  if [ -n "$DISPLAY" ]; then
       echo "DISPLAY=$DISPLAY"
  fi
  if [ -n "$FONTCONFIG_PATH" ]; then
	echo "FONTCONFIG_PATH=$FONTCONFIG_PATH"
  fi
  if [ -n "$MOZILLA_POSTSCRIPT_PRINTER_LIST" ]; then
       echo "MOZILLA_POSTSCRIPT_PRINTER_LIST=$MOZILLA_POSTSCRIPT_PRINTER_LIST"
  fi
  echo "DYLD_LIBRARY_PATH=$DYLD_LIBRARY_PATH"
  echo "     LIBRARY_PATH=$LIBRARY_PATH"
  echo "       SHLIB_PATH=$SHLIB_PATH"
  echo "          LIBPATH=$LIBPATH"
  echo "       ADDON_PATH=$ADDON_PATH"
  echo "      MOZ_PROGRAM=$MOZ_PROGRAM"
  echo "      MOZ_TOOLKIT=$MOZ_TOOLKIT"
  echo "        moz_debug=$moz_debug"
  echo "     moz_debugger=$moz_debugger"
  echo "moz_debugger_args=$moz_debugger_args"
fi
#
export MOZILLA_FIVE_HOME LD_LIBRARY_PATH
export SHLIB_PATH LIBPATH LIBRARY_PATH ADDON_PATH DYLD_LIBRARY_PATH

mkdir -p $HOME/.mozilla/firefox/
touch $HOME/.mozilla/firefox/.refresh

OLD_PATH=`cat $HOME/.mozilla/firefox/.refresh`
NEW_PATH=`cd ${MOZ_DIST_BIN}; pwd`
unset LD_LIBRARY_PATH_32
PANGO_RC_FILE=$HOME/.mozilla/firefox/etc/pango/pangorc
LD_LIBRARY_PATH=${NEW_PATH}/depend/lib
GTK_IM_MODULE_FILE=$HOME/.mozilla/firefox/etc/gtk-2.0/gtk.immodules
GDK_PIXBUF_MODULE_FILE=$HOME/.mozilla/firefox/etc/gtk-2.0/gdk-pixbuf.loaders
export PANGO_RC_FILE LD_LIBRARY_PATH GTK_IM_MODULE_FILE GDK_PIXBUF_MODULE_FILE

# SPW 08Jul2008
# set GTK_PATH so that the GTK libprintbackend-* libraries are
# picked up. WARNING: the libprintbackend-papi.so library currently
# (Jul 2008) causes Firefox to crash, so you must remove it from the
# distribution directory before running firefox with GTK_PATH set
GTK_PATH=${NEW_PATH}/depend/lib/gtk-2.0${GTK_PATH:+:${GTK_PATH}}
export GTK_PATH
# SPW 08Jul2008

if [ "${OLD_PATH}" != "${NEW_PATH}" ]
then
  mkdir -p $HOME/.mozilla/firefox/etc/pango
  mkdir -p $HOME/.mozilla/firefox/etc/gtk-2.0
  echo "[Pango]" > $HOME/.mozilla/firefox/etc/pango/pangorc
  echo "ModuleFiles=$HOME/.mozilla/firefox/etc/pango/pango.modules" >> $HOME/.mozilla/firefox/etc/pango/pangorc
  ${NEW_PATH}/depend/bin/pango-querymodules ${NEW_PATH}/depend/lib/pango/1.6.0/modules/* > $HOME/.mozilla/firefox/etc/pango/pango.modules
  ${NEW_PATH}/depend/bin/gtk-query-immodules-2.0 ${NEW_PATH}/depend/lib/gtk-2.0/2.10.0/immodules/* > $HOME/.mozilla/firefox/etc/gtk-2.0/gtk.immodules
  ${NEW_PATH}/depend/bin/gdk-pixbuf-query-loaders ${NEW_PATH}/depend/lib/gtk-2.0/2.10.0/loaders/* > $HOME/.mozilla/firefox/etc/gtk-2.0/gdk-pixbuf.loaders
  echo "${NEW_PATH}" > $HOME/.mozilla/firefox/.refresh
fi

if [ $moz_debug -eq 1 ]
then
	moz_debug_program ${1+"$@"}
else
	moz_run_program ${1+"$@"}
fi

exit $exitcode
