/* Any copyright is dedicated to the Public Domain.
   http://creativecommons.org/publicdomain/zero/1.0/ */

// Tests functionality of the mozIAsyncLivemarks interface.

const FEED_URI = NetUtil.newURI("http://feed.rss/");
const SITE_URI = NetUtil.newURI("http://site.org/");


add_task(function test_addLivemark_noArguments_throws()
{
  try {
    yield PlacesUtils.livemarks.addLivemark();
    do_throw("Invoking addLivemark with no arguments should throw");
  } catch (ex) {
    // The error is actually generated by XPConnect.
    do_check_eq(ex.result, Cr.NS_ERROR_XPC_NOT_ENOUGH_ARGS);
  }
});

add_task(function test_addLivemark_emptyObject_throws()
{
  try {
    yield PlacesUtils.livemarks.addLivemark({});
    do_throw("Invoking addLivemark with empty object should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_addLivemark_badParentId_throws()
{
  try {
    yield PlacesUtils.livemarks.addLivemark({ parentId: "test" });
    do_throw("Invoking addLivemark with a bad parent id should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_addLivemark_invalidParentId_throws()
{
  try {
    yield PlacesUtils.livemarks.addLivemark({ parentId: -2 });
    do_throw("Invoking addLivemark with an invalid parent id should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_addLivemark_noIndex_throws()
{
  try {
    yield PlacesUtils.livemarks.addLivemark({
      parentId: PlacesUtils.unfiledBookmarksFolderId });
    do_throw("Invoking addLivemark with no index should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_addLivemark_badIndex_throws()
{
  try {
    yield PlacesUtils.livemarks.addLivemark(
      { parentId: PlacesUtils.unfiledBookmarksFolderId
      , index: "test" });
    do_throw("Invoking addLivemark with a bad index should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_addLivemark_invalidIndex_throws()
{
  try {
    yield PlacesUtils.livemarks.addLivemark(
      { parentId: PlacesUtils.unfiledBookmarksFolderId
      , index: -2
      });
    do_throw("Invoking addLivemark with an invalid index should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_addLivemark_noFeedURI_throws()
{
  try {
    yield PlacesUtils.livemarks.addLivemark(
      { parentId: PlacesUtils.unfiledBookmarksFolderId
      , index: PlacesUtils.bookmarks.DEFAULT_INDEX });
    do_throw("Invoking addLivemark with no feedURI should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_addLivemark_badFeedURI_throws()
{
  try {
    yield PlacesUtils.livemarks.addLivemark(
      { parentId: PlacesUtils.unfiledBookmarksFolderId
      , index: PlacesUtils.bookmarks.DEFAULT_INDEX
      , feedURI: "test" });
    do_throw("Invoking addLivemark with a bad feedURI should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_addLivemark_badSiteURI_throws()
{
  try {
    yield PlacesUtils.livemarks.addLivemark(
      { parentId: PlacesUtils.unfiledBookmarksFolderId
      , index: PlacesUtils.bookmarks.DEFAULT_INDEX
      , feedURI: FEED_URI
      , siteURI: "test" });
    do_throw("Invoking addLivemark with a bad siteURI should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_addLivemark_badGuid_throws()
{
  try {
    yield PlacesUtils.livemarks.addLivemark(
      { parentId: PlacesUtils.unfiledBookmarksFolderId
      , index: PlacesUtils.bookmarks.DEFAULT_INDEX
      , feedURI: FEED_URI
      , guid: "123456" });
    do_throw("Invoking addLivemark with a bad guid should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_addLivemark_badCallback_throws()
{
  try {
    yield PlacesUtils.livemarks.addLivemark(
      { parentId: PlacesUtils.unfiledBookmarksFolderId
      , index: PlacesUtils.bookmarks.DEFAULT_INDEX
      , feedURI: FEED_URI
      }, "test");
    do_throw("Invoking addLivemark with a bad callback should throw");
  } catch (ex) {
    // The error is actually generated by XPConnect.
    do_check_eq(ex.result, Cr.NS_ERROR_XPC_BAD_CONVERT_JS);
  }
});

add_task(function test_addLivemark_noCallback_succeeds()
{
  let onItemAddedCalled = false;
  PlacesUtils.bookmarks.addObserver({
    __proto__: NavBookmarkObserver.prototype,
    onItemAdded: function onItemAdded(aItemId, aParentId, aIndex, aItemType,
                                      aURI, aTitle)
    {
      onItemAddedCalled = true;
      PlacesUtils.bookmarks.removeObserver(this);
      do_check_eq(aParentId, PlacesUtils.unfiledBookmarksFolderId);
      do_check_eq(aIndex, 0);
      do_check_eq(aItemType, Ci.nsINavBookmarksService.TYPE_FOLDER);
      do_check_eq(aTitle, "test");
    }
  }, false);

  yield PlacesUtils.livemarks.addLivemark(
    { title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI });
  do_check_true(onItemAddedCalled);
});


add_task(function test_addLivemark_noSiteURI_callback_succeeds()
{
  let checkLivemark = aLivemark => {
    do_check_true(aLivemark.id > 0);
    do_check_valid_places_guid(aLivemark.guid);
    do_check_eq(aLivemark.title, "test");
    do_check_eq(aLivemark.parentId, PlacesUtils.unfiledBookmarksFolderId);
    do_check_eq(aLivemark.index, PlacesUtils.bookmarks.getItemIndex(aLivemark.id));
    do_check_eq(aLivemark.lastModified, PlacesUtils.bookmarks.getItemLastModified(aLivemark.id));
    do_check_true(aLivemark.feedURI.equals(FEED_URI));
    do_check_eq(aLivemark.siteURI, null);
  };

  // The deprecated callback is called before resolving the promise.
  let callbackCalled = false;
  let livemark = yield PlacesUtils.livemarks.addLivemark(
    { title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI
    },
    (aStatus, aLivemark) => {
      callbackCalled = true;
      do_check_true(Components.isSuccessCode(aStatus));
      checkLivemark(aLivemark);
    } );
  do_check_true(callbackCalled);
  checkLivemark(livemark);
});

add_task(function test_addLivemark_callback_succeeds()
{
  let checkLivemark = aLivemark => {
    do_check_true(aLivemark.id > 0);
    do_check_valid_places_guid(aLivemark.guid);
    do_check_eq(aLivemark.title, "test");
    do_check_eq(aLivemark.parentId, PlacesUtils.unfiledBookmarksFolderId);
    do_check_eq(aLivemark.index, PlacesUtils.bookmarks.getItemIndex(aLivemark.id));
    do_check_eq(aLivemark.lastModified, PlacesUtils.bookmarks.getItemLastModified(aLivemark.id));
    do_check_true(aLivemark.feedURI.equals(FEED_URI));
    do_check_true(aLivemark.siteURI.equals(SITE_URI));
    do_check_true(PlacesUtils.annotations
                             .itemHasAnnotation(aLivemark.id,
                                                PlacesUtils.LMANNO_FEEDURI));
    do_check_true(PlacesUtils.annotations
                             .itemHasAnnotation(aLivemark.id,
                                                PlacesUtils.LMANNO_SITEURI));
  };

  // The deprecated callback is called before resolving the promise.
  let callbackCalled = false;
  let livemark = yield PlacesUtils.livemarks.addLivemark(
    { title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI
    , siteURI: SITE_URI
    },
    (aStatus, aLivemark) => {
      callbackCalled = true;
      do_check_true(Components.isSuccessCode(aStatus));
      checkLivemark(aLivemark);
    } );
  do_check_true(callbackCalled);
  checkLivemark(livemark);
});

add_task(function test_addLivemark_bogusid_callback_succeeds()
{
  let checkLivemark = aLivemark => {
    do_check_true(aLivemark.id > 0);
    do_check_neq(aLivemark.id, 100);
  };

  // The deprecated callback is called before resolving the promise.
  let callbackCalled = false;
  let livemark = yield PlacesUtils.livemarks.addLivemark(
    { id: 100 // Should be ignored.
    , title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI
    , siteURI: SITE_URI
    },
    (aStatus, aLivemark) => {
      callbackCalled = true;
      do_check_true(Components.isSuccessCode(aStatus));
      checkLivemark(aLivemark);
    } );
  do_check_true(callbackCalled);
  checkLivemark(livemark);
});

add_task(function test_addLivemark_bogusParent_callback_fails()
{
  // The deprecated callback is called before resolving the promise.
  let callbackCalled = false;
  try {
    yield PlacesUtils.livemarks.addLivemark(
      { title: "test"
      , parentId: 187
      , index: PlacesUtils.bookmarks.DEFAULT_INDEX
      , feedURI: FEED_URI
      },
      (aStatus, aLivemark) => {
        callbackCalled = true;
        do_check_false(Components.isSuccessCode(aStatus));
        do_check_eq(aLivemark, null);
      } );
    do_throw("Adding a livemark with a bogus parent should fail");
  }
  catch(ex) {
    do_check_true(callbackCalled);
  }
});

add_task(function test_addLivemark_intoLivemark_callback_fails()
{
  // The deprecated callback is called before resolving the promise.
  let callbackCalled = false;
  let livemark = yield PlacesUtils.livemarks.addLivemark(
    { title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI
    },
    (aStatus, aLivemark) => {
      callbackCalled = true;
      do_check_true(Components.isSuccessCode(aStatus));
    } );
  do_check_true(callbackCalled);
  do_check_true(Boolean(livemark));

  callbackCalled = false;
  try {
    yield PlacesUtils.livemarks.addLivemark(
      { title: "test"
      , parentId: livemark.id
      , index: PlacesUtils.bookmarks.DEFAULT_INDEX
      , feedURI: FEED_URI
      },
      (aStatus, aLivemark) => {
        callbackCalled = true;
        do_check_false(Components.isSuccessCode(aStatus));
        do_check_eq(aLivemark, null);
      } );
    do_throw("Adding a livemark into a livemark should fail");
  }
  catch(ex) {
    do_check_true(callbackCalled);
  }
});

add_task(function test_addLivemark_forceGuid_callback_succeeds()
{
  let checkLivemark = aLivemark => {
    do_check_eq(aLivemark.guid, "1234567890AB");
    do_check_guid_for_bookmark(aLivemark.id, "1234567890AB");
  };

  // The deprecated callback is called before resolving the promise.
  let callbackCalled = false;
  let livemark = yield PlacesUtils.livemarks.addLivemark(
    { title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI
    , guid: "1234567890AB"
    },
    (aStatus, aLivemark) => {
      callbackCalled = true;
      do_check_true(Components.isSuccessCode(aStatus));
      checkLivemark(aLivemark);
    } );
  do_check_true(callbackCalled);
  checkLivemark(livemark);
});

add_task(function test_addLivemark_lastModified_callback_succeeds()
{
  let now = Date.now() * 1000;
  let callbackCalled = false;
  let livemark = yield PlacesUtils.livemarks.addLivemark(
    { title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI
    , lastModified: now
    },
    (aStatus, aLivemark) => {
      callbackCalled = true;
      do_check_true(Components.isSuccessCode(aStatus));
      do_check_eq(aLivemark.lastModified, now);
    } );
  do_check_true(callbackCalled);
  do_check_eq(livemark.lastModified, now);
});

add_task(function test_removeLivemark_emptyObject_throws()
{
  try {
    yield PlacesUtils.livemarks.removeLivemark({});
    do_throw("Invoking removeLivemark with empty object should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_removeLivemark_noValidId_throws()
{
  try {
    yield PlacesUtils.livemarks.removeLivemark({ id: -10, guid: "test"});
    do_throw("Invoking removeLivemark with no valid id should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_removeLivemark_nonExistent_fails()
{
  let callbackCalled = false;
  try {
    yield PlacesUtils.livemarks.removeLivemark(
      { id: 1337 },
      (aStatus, aLivemark) => {
        callbackCalled = true;
        do_check_false(Components.isSuccessCode(aStatus));
        do_check_eq(aLivemark, null);
      } );
    do_throw("Removing a non-existent livemark should fail");
  }
  catch(ex) {
    do_check_true(callbackCalled);
  }
});

add_task(function test_removeLivemark_guid_succeeds()
{
  let livemark = yield PlacesUtils.livemarks.addLivemark(
    { title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI
    , guid: "234567890ABC"
  });


  do_check_eq(livemark.guid, "234567890ABC");

  yield PlacesUtils.livemarks.removeLivemark({
    id: 789, guid: "234567890ABC"
  });

  do_check_eq(PlacesUtils.bookmarks.getItemIndex(livemark.id), -1);
});

add_task(function test_removeLivemark_id_succeeds()
{
  let livemark = yield PlacesUtils.livemarks.addLivemark(
    { title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI
  });

  yield PlacesUtils.livemarks.removeLivemark({ id: livemark.id });
  do_check_eq(PlacesUtils.bookmarks.getItemIndex(livemark.id), -1);
});

add_task(function test_getLivemark_emptyObject_throws()
{
  try {
    yield PlacesUtils.livemarks.getLivemark({});
    do_throw("Invoking getLivemark with empty object should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_getLivemark_noValidId_throws()
{
  try {
    yield PlacesUtils.livemarks.getLivemark({ id: -10, guid: "test"});
    do_throw("Invoking getLivemark with no valid id should throw");
  } catch (ex) {
    do_check_eq(ex.result, Cr.NS_ERROR_INVALID_ARG);
  }
});

add_task(function test_getLivemark_nonExistentId_fails()
{
  let callbackCalled = false;
  try {
    yield PlacesUtils.livemarks.getLivemark({ id: 1234 },
      (aStatus, aLivemark) => {
        callbackCalled = true;
        do_check_false(Components.isSuccessCode(aStatus));
        do_check_eq(aLivemark, null);
      } );
    do_throw("getLivemark for a non existent id should fail");
  }
  catch(ex) {
    do_check_true(callbackCalled);
  }
});

add_task(function test_getLivemark_nonExistentGUID_fails()
{
  let callbackCalled = false;
  try {
    yield PlacesUtils.livemarks.getLivemark({ guid: "34567890ABCD" },
      (aStatus, aLivemark) => {
        callbackCalled = true;
        do_check_false(Components.isSuccessCode(aStatus));
        do_check_eq(aLivemark, null);
      } );
    do_throw("getLivemark for a non-existent guid should fail");
  }
  catch(ex) {
    do_check_true(callbackCalled);
  }
});

add_task(function test_getLivemark_guid_succeeds()
{
  yield PlacesUtils.livemarks.addLivemark(
    { title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI
    , guid: "34567890ABCD" });

  let checkLivemark = aLivemark => {
    do_check_eq(aLivemark.title, "test");
    do_check_eq(aLivemark.parentId, PlacesUtils.unfiledBookmarksFolderId);
    do_check_eq(aLivemark.index, PlacesUtils.bookmarks.getItemIndex(aLivemark.id));
    do_check_true(aLivemark.feedURI.equals(FEED_URI));
    do_check_eq(aLivemark.siteURI, null);
    do_check_eq(aLivemark.guid, "34567890ABCD");
  };

  // invalid id to check the guid wins.
  let livemark =
    yield PlacesUtils.livemarks.getLivemark({ id: 789, guid: "34567890ABCD" },
      (aStatus, aLivemark) => {
        callbackCalled = true;
        do_check_true(Components.isSuccessCode(aStatus));
        checkLivemark(aLivemark)
      } );

  do_check_true(callbackCalled);
  checkLivemark(livemark);
});

add_task(function test_getLivemark_id_succeeds()
{
  let livemark = yield PlacesUtils.livemarks.addLivemark(
    { title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI
    });

  let checkLivemark = aLivemark => {
    do_check_eq(aLivemark.title, "test");
    do_check_eq(aLivemark.parentId, PlacesUtils.unfiledBookmarksFolderId);
    do_check_eq(aLivemark.index, PlacesUtils.bookmarks.getItemIndex(aLivemark.id));
    do_check_true(aLivemark.feedURI.equals(FEED_URI));
    do_check_eq(aLivemark.siteURI, null);
    do_check_guid_for_bookmark(aLivemark.id, aLivemark.guid);
  };

  let callbackCalled = false;
  livemark = yield PlacesUtils.livemarks.getLivemark(
    { id: livemark.id },
    (aStatus, aLivemark) => {
      callbackCalled = true;
      do_check_true(Components.isSuccessCode(aStatus));
      checkLivemark(aLivemark);
    } );

  do_check_true(callbackCalled);
  checkLivemark(livemark);
});

add_task(function test_getLivemark_removeItem_contention()
{
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    });
  PlacesUtils.bookmarks.removeFolderChildren(PlacesUtils.unfiledBookmarksFolderId);
  PlacesUtils.livemarks.addLivemark({ title: "test"
                                    , parentId: PlacesUtils.unfiledBookmarksFolderId
                                    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
                                    , feedURI: FEED_URI
                                    });
  let id = PlacesUtils.bookmarks.getIdForItemAt(PlacesUtils.unfiledBookmarksFolderId,
                                                PlacesUtils.bookmarks.DEFAULT_INDEX);

  let checkLivemark = (aLivemark) => {
    do_check_eq(aLivemark.title, "test");
    do_check_eq(aLivemark.parentId, PlacesUtils.unfiledBookmarksFolderId);
    do_check_eq(aLivemark.index, PlacesUtils.bookmarks.getItemIndex(aLivemark.id));
    do_check_true(aLivemark.feedURI.equals(FEED_URI));
    do_check_eq(aLivemark.siteURI, null);
    do_check_guid_for_bookmark(aLivemark.id, aLivemark.guid);
  };

  let callbackCalled = false;
  let livemark = yield PlacesUtils.livemarks.getLivemark(
    { id: id },
    (aStatus, aLivemark) => {
      callbackCalled = true;
      do_check_true(Components.isSuccessCode(aStatus));
      checkLivemark(aLivemark);
    } );

  do_check_true(callbackCalled);
  checkLivemark(livemark);
});

add_task(function test_title_change()
{
  let livemark = yield PlacesUtils.livemarks.addLivemark(
    { title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI });

  PlacesUtils.bookmarks.setItemTitle(livemark.id, "test2");
  do_check_eq(livemark.title, "test2");
});

add_task(function test_livemark_move()
{
  let livemark = yield PlacesUtils.livemarks.addLivemark(
    { title: "test"
    , parentId: PlacesUtils.unfiledBookmarksFolderId
    , index: PlacesUtils.bookmarks.DEFAULT_INDEX
    , feedURI: FEED_URI } );

  PlacesUtils.bookmarks.moveItem(livemark.id,
                                 PlacesUtils.toolbarFolderId,
                                 PlacesUtils.bookmarks.DEFAULT_INDEX);
  do_check_eq(livemark.parentId, PlacesUtils.toolbarFolderId);
  do_check_eq(livemark.index, PlacesUtils.bookmarks.getItemIndex(livemark.id));
});

function run_test() {
  run_next_test();
}
