= 組み込み関数

Ruby には厳密な意味では関数はありませんが、((<Kernel>)) モジュールで
定義されているメソッドは (どこからでも関数形式で呼び出せるので)
他言語における関数のように使えます。これらのメソッドを再定義する
場合は他の場所への影響を考えて行なう必要があります。

--- ` str

    文字列 ((|str|)) を外部コマンドとして実行し、その出力を文字列として
    返します。このメソッドは (({`str`})) の形式で呼ばれます
    (((<%x(...)|リテラル/%記法>)) という表記によっても呼び出せます。
    詳細は ((<リテラル/コマンド出力>)) を参照してください)。

    実行したコマンドの終了ステータスは ((<組み込み変数/$?>)) で参照できます。

    コマンドの出力を得る必要がなく、単にコマンドを実行したいだけなら
    ((<組み込み関数/system>)) を使います。特に端末を制御するコマンドでは
    (({`command`})) は失敗するかもしれません。

--- Array(arg)

    ((|arg|)).to_ary か ((|arg|)).to_a を呼び出して引数を配
    列に変換した結果を返します。
    変換した結果が配列でなければ例外 ((<TypeError>)) が発生
    します。

    ((|arg|)) に to_ary, to_a のいずれのメソッドもない場合は
    一要素の配列 [((|arg|))] を返します。

--- Float(arg)

    引数を浮動小数点数(((<Float>)))に変換した結果を返します。

    整数や浮動小数点数と見なせない文字列を引数に指定した場合、例外
    ((<ArgumentError>)) が発生します。

    ((<String#to_f|String/to_f>)) も参照してください。

        p Float("10")    # => 10.0
        p Float("10e2")  # => 1000.0
        p Float("1e-2")  # => 0.01
        p Float(".1")    # => 0.1

        p Float("nan")   # => NaN         (ruby 1.7 feature: => invalid value (ArgumentError))
        p Float("INF")   # => Infinity    (ruby 1.7 feature: => invalid value (ArgumentError))
        p Float("-Inf")  # => -Infinity   (ruby 1.7 feature: => invalid value (ArgumentError))
        p Float(("10" * 1000).to_f) # => Infinity

        p Float("0xa.a") # => 10.625     # (ruby 1.7 feature: => invalid value (ArgumentError))
        p Float(" \n10") # => 10.0       # 先頭の空白は無視される
        p Float("1_0_0") # => 100.0      # `_' も数値要素とみなす
        p Float("")      # => invalid value (ArgumentError)
        p Float(nil)     # => 0.0       # ruby 1.7 feature: cannot convert nil into Float (TypeError)

        p Float(Object.new) # => cannot convert Object into Float (TypeError)

--- Integer(arg)

    引数を整数(((<Fixnum>)),((<Bignum>)))に変換した結果を返します。
    数値、文字列以外のオブジェクトに対しては (({to_i})) メソッドを
    使用します(((<ruby 1.7 feature>)): 1.7 では、(({to_int})))。
    変換した結果が整数(((<Integer>))のサブクラス)でなければ
    例外 ((<TypeError>)) が発生します。

    引数が文字列であった場合には、0x, 0b, 0 などの接頭辞に応じて
    それぞれ 16 進、2 進、8 進数として変換します。

    整数と見なせない文字列を引数に指定した場合、例外 ((<ArgumentError>))
    が発生します。

        p Integer(10.1)       # => 10
        p Integer(10.8)       # => 10
        p Integer("10")       # => 10
        p Integer("10_0_0")   # => 1000
        p Integer("10__0")    # => 100
                              # => invalid value for Integer: "10__0" (ArgumentError) (ruby 1.7 feature)
        p Integer("_10")      # => invalid value for Integer: "_10" (ArgumentError)
        p Integer("10_")      # => invalid value for Integer: "10_" (ArgumentError)
        p Integer("0d10")     # => invalid value for Integer: "0d10" (ArgumentError)
                              # => 10 (ruby 1.7 feature)
        p Integer("010")      # => 8
        p Integer("0o10")     # => invalid value for Integer: "0o10" (ArgumentError) (ruby 1.7 feature)
                              # => 8
        p Integer("0x10")     # => 16
        p Integer("0b10")     # => 2
        p Integer(" \n10\t ") # => 10
        p Integer("")         # => `Integer': invalid value for Integer: "" (ArgumentError)
        p Integer(nil)        # => 0

        p Integer(Object.new) # => cannot convert Object into Integer (TypeError)

    ((<String#hex|String/hex>)),
    ((<String#oct|String/oct>)),
    ((<String#to_i|String/to_i>)) も参照してください。

--- String(arg)

    ((|arg|)).(({to_s})) を呼び出して引数を文字列に変換した
    結果を返します。
    変換した結果が文字列でなければ例外 ((<TypeError>)) が発
    生します。
    ((|arg|)) が文字列の場合、何もせず ((|arg|)) を返します。

--- abort
--- abort(message)      ((<ruby 1.7 feature>))

    Ruby プログラムを異常終了します。((<組み込み関数/exit>)) との違いは、
    呼び出し時に ((<組み込み変数/$!>)) が (({nil})) でなければその例外の
    メッセージを標準エラー出力に出力することと、プログラムの終了ステー
    タスが EXIT_FAILURE(ほとんどの環境では1) 固定であることです。

    ((<ruby 1.7 feature>)): 引数 ((|message|)) を指定すると
    ((<SystemExit>))#((<Exception/message>)) に ((|message|)) を設定し
    て標準エラー出力に出力します。

--- at_exit { .... }

    与えられたブロックをインタプリタ終了時に実行します。(({at_exit}))
    がメソッドである点を除けば、((<制御構造/END>)) ブロックによる終了
    処理の登録と同等です。登録した処理を取り消すことはできません。
    ((<終了処理>))も参照してください。

    登録した処理を ((<Proc>)) オブジェクトで返します。

--- autoload(const_name, feature)
--- autoload?(const_name)            ((<ruby 1.8 feature>))

    定数 ((|const_name|)) を最初に参照した時に ((|feature|)) を
    ((<組み込み関数/require>)) するように設定します。((|const_name|))
    は文字列または ((<Symbol>)) で指定します。
    なお、((|const_name|)) には、"::" 演算子を含めることはできません
    (つまり、トップレベルの定数しか指定できません)。

    (({nil})) を返します。

    ((<ruby 1.8 feature>)): 任意のクラス／モジュールの定数を autoload 
    できます。やはり、((|const_name|)) に "::" 演算子を含めることはで
    きませんが、以下のように定義する事ができます。(実際には、
    ((<Module#autoload|Module/autoload>)) が呼ばれます)

        ------- /tmp/foo.rb ---------
        class Foo
          class Bar
          end
        end
        ----- end of /tmp/foo.rb ----

        class Foo
          autoload :Bar, '/tmp/foo.rb'
        end
        p Foo::Bar

    あるいは、以下のようにもできます。

        class Foo
        end
        Foo.autoload :Bar, '/tmp/foo.rb'
        p Foo::Bar

    (({autoload?})) は、autoload 定数がまだ定義されてない(ロードされて
    いない)ときにそのパス名を返します。また、ロード済みなら nil を返し
    ます。

        ------- /tmp/foo.rb ---------
        class Foo
          class Bar
          end
        end
        ----- end of /tmp/foo.rb ----

        class Foo
        end
        Foo.autoload :Bar, '/tmp/foo.rb'
        p Foo.autoload?(:Bar)         # => "/tmp/foo.rb"
        p Foo::Bar                    # => Foo::Bar
        p Foo.autoload?(:Bar)         # => nil

    以下のように、autoload したライブラリがネストした定数を定義しない場
    合、一見、正常に動作しているように見えるので注意が必要です(警告メッ
    セージが出ています)。

          ------- /tmp/bar.rb ---------
          class Bar
          end
          ----- end of /tmp/bar.rb ----

          class Foo
            autoload :Bar, '/tmp/bar.rb'
          end
          p Foo::Bar
          p Foo.autoload?(:Bar)

          # => -:4: warning: toplevel constant Bar referenced by Foo::Bar
               Bar
               nil

    これは以下のようにネストせずに定義したのと同じことです。

          class Foo
          end
          class Bar
          end
          p Foo::Bar

          # => -:5: warning: toplevel constant Bar referenced by Foo::Bar
               Bar

--- binding

    変数・メソッドなどの環境情報を含んだ ((<Binding>)) オブジェクトを
    生成して返します。通常、((<組み込み関数/eval>)) の第二引数として使います。

--- caller([level])

    ((|level|)) 段上(引数省略時は、1)の呼び出し元の情報を ((<組み込み変数/$@>))
    の形式のバックトレース(文字列の配列)として返します。トップレベルで
    は空の配列を返します。caller の戻り値を (({$@})) に代入することで
    例外の発生位置を設定できます。

        def foo
          p caller(0)
          p caller(1)
          p caller(2)
          p caller(3)
        end

        def bar
          foo
        end

        bar

        => ["-:2:in `foo'", "-:9:in `bar'", "-:12"]
           ["-:9:in `bar'", "-:12"]
           ["-:12"]
           []

    以下の関数は、caller の要素から [ファイル名, 行番号, メソッド名] 
    を取り出して返します。

        def parse_caller(at)
          if /^(.+?):(\d+)(?::in `(.*)')?/ =~ at
            file = $1
            line = $2.to_i
            method = $3
            [file, line, method]
          end
        end

        def foo
          p parse_caller(caller.first)
        end

        def bar
          foo
          p parse_caller(caller.first)
        end

        bar
        p parse_caller(caller.first)

        => ["-", 15, "bar"]
           ["-", 19, nil]
           nil

    以下は、((<組み込み変数/$DEBUG>)) が真の場合に役に立つ debug 関数
    のサンプルです。

        def debug(*args)
          p [caller.first, *args] if $DEBUG
        end

        debug "debug information"

        => ["-:5", "debug information"]

--- callcc {|cont| .... }

    ((<Continuation>)) を参照してください。

--- catch(tag) {|tag| .... }

    ブロックを実行してその値を返します。ブロックの実行中に
    ((|tag|)) と同じ名前の ((<組み込み関数/throw>)) が行われた
    場合は、その (({throw})) の第二引数を戻り値とします。

    例えば以下のコードを実行すると ((|some_process|)) は
    呼び出されず、また catch の戻り値は 10 ではなく 25 に
    なります。

      ret = catch(:exit) {
         throw :exit, 25
         some_process()
         10
      }
      p ret   #=> 25

    ネストしたループは ((<制御構造/break>)) によって一気に抜
    けることはできません。
    このような場合、catch や ((<例外|制御構造/begin>)) を使用します。

      catch(:loop1) {
        for i in 1..2
          for j in 1..2
            throw :loop1, j
          end
        end
      }

--- chop
--- chop!

    組み込み変数 ((<組み込み変数/$_>)) を最後の文字を取り除いたものにし
    ます(終端が"\r\n"であれば2文字取り除きます)。
    (({chop!})) は文字列そのものを変更しその結果を返しますが、
    取り除く文字列がなければ (({nil})) を返します。

    詳細は ((<String#chop|String/chop>)) を参照してください。
    (({$_.chop})) と関数 (({chop})) では以下の点で違いがあります。

    * (({chop})) は (({$_})) の値をコピーして、コピーの方を更新し、
      (({$_})) に((*再代入*))します。

--- chomp([rs])
--- chomp!([rs])

    組み込み変数 ((<組み込み変数/$_>)) を ((|rs|)) で指定される末尾
    の文字列を取り除いたものにします。
    (({chomp!})) は文字列そのものを変更しその結果を返しますが、
    取り除く文字列がなければ (({nil})) を返します。
    ((|rs|)) のデフォルト値は ((<組み込み変数/"$/">)) です。

    詳細は ((<String#chomp|String/chomp>)) を参照してください。
    (({$_.chomp})) と関数 (({chomp})) では以下の点で違いがあります。

    * (({chomp})) は (({$_})) の値をコピーして、コピーの方を更新し、
      (({$_})) に((*再代入*))します。

--- eval(expr[, binding[, fname[, lineno=1]]])

    文字列 ((|expr|)) を Ruby プログラムとして評価してその結果を返しま
    す。第2引数に ((<Proc>)) オブジェクトまたは ((<Binding>)) オブジェ
    クトを与えた場合、そのオブジェクトを生成したコンテキストで文字列を
    評価します。((<組み込み関数/binding>)) も参照してください。

        def foo
          a = 1
          binding
        end

        eval("p a", foo)  # => 1

    ((|fname|)) と ((|lineno|)) が与えられた場合には、ファイル
    ((|fname|)) 行番号 ((|lineno|)) に文字列があるかのように
    コンパイルされ、スタックトレースの表示などを差し替えることが
    できます。

--- exec(command)
--- exec(program[, arg1[, arg2[, ...]]])

    ((|command|)) で指定されたコマンドを実行します。プロセスの実行コー
    ドはそのコマンド(あるいは shell。後述)になるので、起動に成功した場
    合、この関数からは戻りません。起動に失敗し、ruby インタプリタに制
    御が戻った場合は、例外 ((<Errno::EXXX>)) が発生します。

    一番目の形式では ((|command|)) が shell のメタ文字
    ((({* ? {} [] <> () ~ & | \ $ ; ' ` " \n})))を含む場合、
    shell 経由で実行されます。そうでなければインタプリタから直接
    実行されます。

    二番目の形式では、常に shell を経由せずに実行されます。
    この場合には空白や shell のメタキャラクタもそのまま
    ((|program|)) の引数に渡されます。
    先頭の引数が2要素の配列であった場合、第1要素の文字列が実際に
    起動するプログラムのパスであり、第2要素が「みせかけ」のプロ
    グラム名になります。

    ((<ruby 1.8 feature>)): このとき、第1要素をフルパスで指定しなくても
    環境変数 PATH から探します。

--- exit([status])

    Rubyプログラムの実行を終了します。((|status|)) として整
    数が与えられた場合、その値を Ruby コマンドの終了ステータスとし
    ます。デフォルトの終了ステータスは 0 です。

    (({exit})) は例外 ((<SystemExit>)) を発生させ
    ることによってプログラムの実行を終了させますので、
    必要に応じて ((<rescue|制御構造/begin>)) 節で捕捉することができます。

    ((<ruby 1.8 feature>)): ((|status|))が(({true}))の場合、(({EXIT_SUCCESS}))を
    終了ステータスとします。((|status|))が(({false}))の場合、(({EXIT_FAILURE}))を
    終了ステータスとします。デフォルトの終了ステータスは(({EXIT_SUCCESS}))に
    変更されています。

--- exit!([status])

    整数 ((|status|)) を終了ステータスとして、Ruby プログラム
    の実行を終了します。デフォルトの終了ステータスは -1 です。

    (({exit!})) は (({exit})) とは違って、例外処理などは一切行ないませ
    ん。((<組み込み関数/fork>))の後、子プロセスを終了させる時などに用
    いられます。

    ((<ruby 1.8 feature>)): ((|status|))が(({true}))の場合、(({EXIT_SUCCESS}))を
    終了ステータスとします。((|status|))が(({false}))の場合、(({EXIT_FAILURE}))を
    終了ステータスとします。デフォルトの終了ステータスは(({EXIT_FAILURE}))に
    変更されています。

--- fork
--- fork { ... }

    ((<fork(2)|manual page>)) システムコールを使ってプロセスの複製を作
    ります。親プロセスでは子プロセスのプロセスIDを、子プロセスでは
    (({nil})) を返します。ブロックを指定して呼び出した場合には、生成し
    た子プロセスでブロックを評価します。

    ((<ruby 1.8.2 feature>)): fork 前に STDOUT と STDERR を
    ((<IO/flush>)) します。

--- gets([rs])
--- readline([rs])

    Ruby インタプリタ実行時に引数として与えられたファイル(なければ標準
    入力)をつなげた仮想的なファイル(組み込み変数 ((<組み込み変数/$<>))
    や ((<ARGF>)) でアクセスできる) から一行読み込んで、文字列を返しま
    す。ファイルの終りに到達した時には (({nil})) を返します。

    行の区切りは引数 ((|rs|)) で指定した文字列になります。((|rs|)) の
    デフォルト値は組み込み変数 ((<組み込み変数/"$/">)) の値です。読み込
    んだ文字列は組み込み変数 ((<組み込み変数/$_>)) にもセットされます。

    ((|rs|)) に (({nil})) を指定すると行区切りなしとみなしてファイルの内容を
    すべて読み込みます。
    空文字列 (({""})) を指定すると連続する改行を行の区切りとみなします
    (パラグラフモード)。

    (({readline})) は (({gets})) と同じ働きをしますが、
    ファイルの最後まで読むと例外 ((<EOFError>)) を発生させます。

--- global_variables

    プログラム中で定義されているグローバル変数(`$'で始まる変数)名の
    配列を返します。

    ((<組み込み関数/local_variables>)),
    ((<Object#instance_variables|Object/instance_variables>)),
    ((<Module/Module.constants>)),
    ((<Module#constants|Module/constants>)),
    ((<Module#class_variables|Module/class_variables>))
    も参照してください。

--- gsub(pattern[, replace])
--- gsub!(pattern[, replace])
--- gsub(pattern) {|matched| ... }
--- gsub!(pattern) {|matched| ... }

    組み込み変数 ((<組み込み変数/$_>)) の指す文字列内で ((|pattern|)) に
    マッチする部分を全て ((|replace|)) に置き換えた文字列を返します。
    引数 ((|replace|)) が省略された時にはイテレータとして動作し、ブロッ
    クを評価した結果で置換を行います。ブロックには引数としてマッチした
    文字列が渡されます。

       $_ = "foobar"
       p gsub(/o+/) {|m|
         m.upcase
       }
       # => "fOObar"

    (({gsub!})) は (({$_})) の指している文字列そのものを書き換えます。
    詳細は、((<String#gsub|String/gsub>)) を参照してください。
    (({String#gsub})) と関数 (({gsub})) では以下の点で違いがあります。

    * (({gsub}))メソッドは (({$_})) の値をコピーして、コピーの方を更新し、
      (({$_})) に((*再代入*))します。

--- iterator?           ((<obsolete>))
--- block_given?

    メソッドにブロックが与えられている時には真、そうでない時に偽
    を返します。

--- load(file[, priv])

    Ruby プログラム ((|file|)) をロードして実行します。
    ((|file|)) が絶対パスのときは ((|file|)) からロードします。
    ((|file|)) が相対パスのときは組み込み変数 ((<組み込み変数/$:>))
    に示されるパスを順番に探し、最初に見付かったファイルを
    ロードします。このとき、$: の要素文字列の先頭文字が
    `(({~}))' (チルダ) だと、環境変数 HOME の値に展開されます。

    省略可能な引数 ((|priv|)) が真のとき、
    ロード・実行は内部的に生成される無名モジュールを
    トップレベルとして行われ、グローバルな名前空間を汚染しません。

    ロードに成功した場合は (({true})) を返します。
    失敗した場合は例外 ((<LoadError>)) が発生します。

    [require と load の違いについて]

    require は同じファイルは一度だけしかロードしませんが、
    load は無条件にロードします。また、require は拡張子
    .rb や .so を自動的に補完しますが、load は行いません。
    require はライブラリのロード、load は設定ファイルの
    読み込みなどに使うのが典型的な用途です。

      # 使用例
      load "#{ENV['HOME']}/.myapprc"
      load "/etc/myapprc"

    なお、特定のディレクトリからファイルをロードしたい場合、
    load 'filename' とするのは不適切です。必ず絶対パスを
    使ってください。

--- local_variables

    現在のスコープで定義されているローカル変数名の配列を返します。

    ((<組み込み関数/global_variables>)),
    ((<Object#instance_variables|Object/instance_variables>)),
    ((<Module/Module.constants>)),
    ((<Module#constants|Module/constants>)),
    ((<Module#class_variables|Module/class_variables>))
    も参照してください。

--- loop { ... }

    (中断されない限り)永遠にブロックの評価を繰り返します。

--- open(file[, mode[, perm]])
--- open(file[, mode[, perm]]) {|io| ... }

    ((|file|)) をオープンして、((<File>)) オブジェクトを返します。
    ((|mode|)) は、以下の文字列か整数(((<File::Constants>)) モジュール
    の定数の論理和)を指定します。省略時は (({"r"})) が指定されたもの
    とみなします。

    * "r", RDONLY: ファイルを読み込みモードでオープンします。

    * "w", WRONLY|CREAT|TRUNC: ファイルを書き込みモードでオー
      プンします。オープン時にファイルがすでに存在していれば
      その内容を空にします。

    * "a", WRONLY|CREAT|APPEND: ファイルを書き込みモードでオー
      プンします。出力は((*常に*))ファイルの末尾に追加されま
      す
      ((-例えば、ファイルオープン中にファイルのサイズが小さ
      くなってもその末尾に出力されます。このことはログ出力な
      どでプログラムを実行したままそのログを小さくしたい場合
      に利用されます-))

    "+" があれば、ファイルは読み書き両用モード(RDWR)でオープ
    ンされます。

    * "r+": ファイルの読み書き位置は先頭にセットされます。

    * "w+": "r+" と同じですが、オープン時にファイルがすでに
      存在していればその内容を空にします。

    * "a+": "r+" と同じですが、オープン時にファイルがすでに
      存在していれば読み書き位置がファイルの末尾にセットされ
      ます。

    これらのいずれに対しても "b" フラグを("r+b"のように)つけることがで
    きます(整数なら File::BINARY)。この場合、バイナリモードでオープン
    します(ただし、システムがテキスト／バイナリでファイルを区別する場
    合に限ります)

    第 3 引数 ((|perm|)) は ((<open(2)|manual page>)) の第 3 引数と同
    じく、(({CREAT})) 時のファイルのアクセス権を整数で指定します。
    この引数は、第 2 引数が数値形式でなければ無視されます((-version 1.7
    では第 2 引数によらず指定されれば有効になります-))。
    デフォルトは 0666 です。

    ファイル名が `(({|}))' で始まる時には続く文字列をコマンドとして起
    動し、コマンドの標準入出力に対してパイプラインを生成します

    ファイル名が (({"|-"})) である時、(({open})) は Ruby の子プロセス
    を生成し、その子プロセスとの間のパイプ(((<IO>))オブジェクト)を返し
    ます。(このときの動作は、((<IO/IO.popen>)) と同じです。
    ((<File/File.open>)) にはパイプラインを生成する機能はありません)。

    ((*注意*)): Perlと異なりコマンドは常に `(({|}))' で始まります。

    (({open})) がブロックとともに呼び出された時、(({open})) はファイル
    をオープンしてブロックを実行し、ブロックの実行が終了するとファイル
    をクローズします。この場合はブロックを評価した結果を返します。つま
    り、以下のようになります。

      open(path, mode) do |f|
         ...
      end

      # 上記とほぼ同じコード
      f = open(path, mode)
      begin
         ...
      ensure
        f.close
      end

--- p(obj, [obj2, ...])

    ((|obj|)) を人間に読みやすい形で出力します。以下のコードと同じです。
    (((<Object#inspect|Object/inspect>))参照)

      print obj.inspect, "\n", obj2.inspect, "\n", ...

    (({nil})) を返します。

--- print([arg1[, arg2, ...]])

    引数を順に出力します。引数が与えられない時には変数
    ((<組み込み変数/$_>)) の値を出力します。
    文字列以外のオブジェクトが引数として与えられた場合には、
    当該オブジェクトを (({to_s})) メソッドにより文字列に変換
    してから出力します。
    ただし、(({nil})) に対しては文字列 (({"nil"})) を出力します

    変数 ((<組み込み変数/$,>)) (出力フィールドセパレータ)に (({nil})) で
    ない値がセットされている時には、各引数の間にその文字列を出力します。
    変数 ((<組み込み変数/$\>)) (出力レコードセパレータ)に (({nil})) でな
    い値がセットされている時には、最後にそれを出力します。

    (({nil})) を返します。

--- printf([port, ]format[, arg[, ...]])

    C 言語の printf と同じように、((|format|)) に従い引数を文字列に変
    換して ((<$>|組み込み変数>)) に出力します。第一引数が ((<IO>)) のサ
    ブクラスのインスタンスであった場合はそのオブジェクトに対して出力を
    行ないます((-第一引数に IO を渡すインタフェースは obsolete と明言
    されています(あらい 2002-01-07 どこで言ってたっけ？確か「デスクトッ
    プリファレンスにはこのことは載ってなかったという話だったはず)-))。
    引数を 1 つも指定しなければ何もしません
    ((-あらい 2002-01-07: IO#printfとは違う-))。

    Ruby における format 文字列の拡張については((<sprintfフォーマット>))
    の項を参照してください。

    (({nil})) を返します。

--- proc { ... }
--- lambda { ... }
--- proc
--- lambda

    与えられたブロックから手続きオブジェクト (((<Proc>)) のインスタンス)
    を生成して返します(((<Proc/Proc.new>))と同じです)。

    ブロックが指定されなければ、呼び出し元のメソッドで指定されたブロック
    を手続きオブジェクトとして返します。呼び出し元のメソッドがブロックなし
    で呼ばれると ((<ArgumentError>)) 例外が発生します。

--- putc(ch)

    文字 ((|ch|)) を ((<$>|組み込み変数>)) に出力します。
    ((|ch|)) が数字なら 0 〜 255 の範囲の対応する文字を出力
    します。((|ch|)) が文字列なら、その先頭の文字を出力します。

    ((|ch|)) を返します。

        putc("ch")
        putc(?c)
        putc(99)
        # => ccc

--- puts([obj[, obj2[, ....]]] )

    ((|obj|)) と改行を順番に ((<$>|組み込み変数>)) に出力します。
    引数がなければ改行のみを出力します。

    引数が配列の場合、その要素と改行を順に出力します。
    配列や文字列以外のオブジェクトが引数として与えられた場合には、
    当該オブジェクトを最初に (({to_ary})) により配列へ、
    次に (({to_s})) メソッドにより文字列へ変換を試みます。
    ただし、(({nil})) に対しては文字列 (({"nil"})) を出力します

    末尾が改行で終っている引数に対しては (({puts})) 自身
    は改行を出力しません。

        puts "foo", "bar\n", "baz"
        puts ""    # 改行のみ出力
        puts       # 改行のみ出力
        puts "foo"
        => foo
           bar
           baz


           foo

    (({nil})) を返します。

    ((<組み込み関数/print>)), ((<組み込み関数/warn>)) も参照。

--- raise
--- raise(exception)
--- raise(message)
--- raise(error_type, message [, backtrace])
--- fail(error_type, message [, backtrace])

    例外を発生させます。

    引数が無い場合は、同スレッドの同じブロック内で最後に rescue された
    例外オブジェクト (((<組み込み変数/$!>))) を再発生させます。そのような
    例外が存在しないときは例外 ((<RuntimeError>)) を発生させます。

        begin
          open("nonexist")
        rescue
          raise   # => `open': No such file or directory - "nonexist" (Errno::ENOENT)
        end

    引数が一つの場合、引数が文字列であれば、その文字列をメッ
    セージとする (({RuntimeError})) 例外を発生させます。引数
    が例外クラスまたは例外オブジェクトであった場合にはその例
    外を発生させます。

        raise "error message"    # => -:1: error message (RuntimeError)

        raise ArgumentError      # => -:1: ArgumentError (ArgumentError)

        raise ArgumentError.new  # => -:1: ArgumentError (ArgumentError)

    引数が二つまたは三つの場合、第一引数で指定された例外を、第二引数に
    与えたメッセージとともに発生させます。この場合、例外は例外クラスまたは
    例外オブジェクトで指定します。第三引数は例外発生時のスタックトレース
    で、((<組み込み関数/caller>)) の戻り値と同じ形式でなければいけません。

        raise ArgumentError, "error message"
        # => -:1: error message (ArgumentError)

        raise ArgumentError, "error message", ["file1:99",
                                               "file2:999:in `method'"]

        # => file1:99: error message (ArgumentError)
                      from file2:999:in `method'

    例外ではないクラスやオブジェクトを第一引数に指定した場合、実際に
    発生する例外はそのオブジェクトの (({exception})) メソッドが
    返す値になります。

        class MyException
          def exception
            ArgumentError.new
          end
        end

        raise MyException.new

        # => -:7: ArgumentError (ArgumentError)

    第二の形式で引数を指定した場合は、(({exception})) メソッ
    ドにその引数が渡されます。

        class MyException
          def exception(mesg)
            ArgumentError.new(mesg)
          end
        end

        raise MyException.new, "error message"

        # => -:7: error message (ArgumentError)

    (({exception})) メソッドは必ず例外オブジェクトを返さなければいけ
    ません。そうでない場合は ((<TypeError>)) が発生します。

    発生した例外は変数 ((<組み込み変数/$!>)) に格納されます。また例外が
    発生した時のスタックトーレスが変数 ((<組み込み変数/$@>)) に格納され
    ます。

--- rand([max=0])

    0 以上 ((|max|)) 未満の範囲の整数の乱数を発生します。ま
    だ((<組み込み関数/srand>)) が呼ばれていなければ自動的に
    (({srand})) を呼び出します。

    ((|max|)) に (({nil})) または 0 を指定すると 0 以上 1 未
    満の実数値 ((<Float>)) で乱数を与えます。

--- readlines([rs])

    コマンドライン引数として与えられたファイル(なければ標準入力) をつ
    なげた仮想的なファイル(((<組み込み変数/ARGF>))を全て読み込んで、その
    各行を要素としてもつ配列を返します。

    行の区切りは引数 ((|rs|)) で指定した文字列になります。((|rs|)) の
    デフォルト値は組み込み変数 ((<組み込み変数/"$/">)) の値です。

    ((|rs|)) に (({nil})) を指定すると行区切りなしとみなします。
    空文字列 (({""})) を指定すると連続する改行を行の区切りとみなします
    (パラグラフモード)。

--- require(feature)

    Ruby ライブラリ ((|feature|)) をロードパス ((<組み込み変数/$:>)) 上
    から探し、同じライブラリがまだロードされていなかった時だけロードし
    ます。

    Ruby ライブラリとは Ruby スクリプト ((({*.rb}))) か拡張ライブラリ
    ((({*.so}))) であり、((|feature|)) の拡張子が省略された場合はその
    両方から探します(検索順序に関しては ((<組み込み変数/$:>)) を参照して
    ください)。省略されなかった場合は指定された種別のみを探します。ま
    た拡張ライブラリの拡張子にはアーキテクチャで実際に使われる拡張子に
    関らず常に (({.so})) を用います。

    実際にライブラリをロードした時には (({true}))、既にロードされてい
    る時には (({false})) を返します。ロードに失敗した場合は、例外
    ((<LoadError>)) が発生します。ロードした ((|feature|)) の名前を(拡
    張子も含めて)、変数 ((<組み込み変数/$">)) に追加します。

--- scan(re)
--- scan(re) {|matched| ... }

    ((<組み込み変数/$_>)).((<String/scan>)) と同じです。

--- select(reads[, writes[, excepts[, timeout]]])

    ((<IO/IO.select>)) と同じです。

--- set_trace_func(trace_proc)

    Ruby インタプリタがプログラムを実行する過程で、メソッドの呼び出しや
    式の評価などのイベントが発生する度に手続きオブジェクト ((|trace_proc|))
    を実行します。標準添付の((<デバッガ|debug.rb>))、((<トレーサ|tracer.rb>))、
    ((<プロファイラ|profile.rb>))はこの組み込み関数を利用して実現されています。

    例:
        set_trace_func lambda {|event, file, line, id, binding, klass|
            # ....
        }

    ブロック引数の意味は以下の通りです。

    : event
        実行のタイプを表す、以下のいずれかの文字列。
        * "line" ... 式の評価。
        * "call" ... メソッドの呼び出し。
        * "return" ... メソッド呼び出しからのリターン。
        * "c-call" ... Cで記述されたメソッドの呼び出し。
        * "c-return" ... Cで記述されたメソッド呼び出しからのリターン。
        * "class" ... クラス定義、特異クラス定義、モジュール定義への突入。
        * "end" ... クラス定義、特異クラス定義、モジュール定義の終了。
        * "raise" ... 例外の発生。

    : file
        実行中のプログラムのソースファイル名 (文字列)。

    : line
        実行中のプログラムのソースファイル上の行番号 (整数)。

    : id
        ((|event|)) に応じ、以下のものが渡されます。
        第六ブロック引数の ((|klass|)) と対応しています。

        : line
            最後に呼び出されたメソッドを表す ((<Symbol>)) オブジェクト。
            トップレベルでは (({nil}))。
        : call/return/c-call/c-return
            呼び出された/リターンするメソッドを表す ((<Symbol>)) オブジェクト。
        : class/end
            (({nil}))。
        : raise
            最後に呼び出されたメソッドを表す ((<Symbol>)) オブジェクト。
            トップレベルでは (({nil}))。

    : binding
        実行中のプログラムのコンテキストを表す ((<Binding>)) オブジェクト。

    : klass
        ((|event|)) に応じ、以下のものが渡されます。
        第四ブロック引数の ((|id|)) と対応しています。

        : line
            最後に呼び出されたメソッドが属するクラスを表す
            ((<Class>)) オブジェクト。トップレベルでは
            (({false}))。
        : call/return/c-call/c-return
            呼び出された/リターンするメソッドが属するクラス
            を表す ((<Class>)) オブジェクト。
        : class/end
            (({false}))。
        : raise
            最後に呼び出されたメソッドが属するクラスを表す
            ((<Class>))オブジェクト。トップレベルでは
            (({false}))。

--- sleep([sec])

    ((|sec|)) 秒だけプログラムの実行を停止します。((|sec|))
    には浮動小数点数も指定できます。((|sec|)) が省略された場
    合、((({SIGALRM})) または他スレッドからの
    ((<Thread#run|Thread/run>))) などで明示的に起こさない限り永
    久にスリープします。戻り値は実際に停止していた秒数(整数)です。

--- spawn(cmd, [arg, ...])      ((<ruby 1.9 feature>))

    子プロセスの終了を待ち合わせないことを除けば、((<組み込み関数/system>))
    と同じです。生成した子プロセスのプロセスIDを返します。

--- split([sep[, limit]])

    ((|$_|)) の示す文字列をパターン ((|sep|)) によって
    ((|limit|)) 個の文字列に分割し、その配列を返します。詳細
    は ((<String#split|String/split>)) を参照してください。

--- sprintf(format ... )
--- format(format ... )

    ((|format|)) 文字列を C 言語の (({sprintf})) と同じよう
    に解釈し、引数をフォーマットした文字列を返します。
    ((|format|)) 指定子は C 言語の (({sprintf})) が受け付け
    るものとほとんど同じです。

    Ruby には整数の大きさに上限がないので、(({%b, %o, %x}))
    に負の数を与えると(左側に無限に1が続くとみなせるので)
    (({..f})) のような表示をします。絶対値に符号を付けた形式
    で出力するためには(({%+x}))、(({% x}))のように指定します。

    詳細は((<sprintfフォーマット>))を参照してください。

--- srand([seed])

    ((<組み込み関数/rand>)) の乱数の種を設定し、古い初期値を返します
    (ただし最初だけは0を返します)。初期値が省略された時には現在の時刻
    やプロセスID、srand を呼び出した回数などを元に種を作ります。

    ((<ruby 1.8 feature>)):
    1.8.3 からは、seed として unsigned long よりも大きな値を与えることが出来ます。

--- sub(pattern[, replace])
--- sub!(pattern[, replace])
--- sub(pattern) {|matched| ... }
--- sub!(pattern) {|matched| ... }

    組み込み変数 ((<組み込み変数/$_>)) の指す文字列内で ((|pattern|)) に
    マッチする最初の部分を ((|replace|)) に置き換えた文字列を返します。
    引数 ((|replace|)) が省略された時にはイテレータとして動作し、ブロッ
    クを評価した結果で置換を行います。ブロックには引数としてマッチした
    文字列が渡されます。

    (({sub!})) は (({$_})) の指している文字列そのものを書き換えます。
    詳細は、((<String#sub|String/sub>)) を参照してください。
    (({String#sub})) と関数 (({sub})) では以下の点で違いがあります。

    * (({sub}))メソッドは (({$_})) の値をコピーして、コピーの方を更新し、
      (({$_})) に((*再代入*))します。

--- syscall(num, arg ... )

    ((|num|))で指定された番号のシステムコールを実行します。
    第2引数以降をシステムコールの引数として渡します。引数は文字
    列または整数でなければなりません。

    どの数値がどのシステムコールに対応するかは、
    ((<syscall(2)|manual page>)) や
    (({/usr/include/sys/syscall.h})) を参照してください。

    システムコールの慣習に従い、((<syscall(2)|manual page>))
    が -1 を返す場合には例外 ((<Errno::EXXX>)) が発生します。
    それ以外では、返した値をそのまま数値で返します。

--- system(command)
--- system(program[, arg1[, arg2[, ...]]])

    ((|command|)) を実行して、成功した時(子プロセスが終了ステータス 0
    で終了した時)には真を、失敗した時(コマンドが実行できなかった場合も
    含む)には偽を返します。終了ステータスは変数 ((<組み込み変数/$?>)) で
    参照できます。

    コマンドを実行することができなかった場合、多くのシェルはステータス
    127 を返します。(したがって (({$?})) の数値は、0x7f00)、シェルを介
    さない場合は Ruby の子プロセスがステータス 127 で終了します。コマ
    ンドが実行できなかったのか、コマンドが失敗したのかは、普通
    (({$?})) を参照することで判別可能です。

    ((<ruby 1.9 feature>)):
    コマンドを実行できないときに例外 ((<Errno::EXXX>)) が発生します。
    シェルを介する場合は、上述の通りです。

    その他の挙動に関しては ((<組み込み関数/exec>)) を参照して
    ください。

    ((<`command`|組み込み関数/"`">)), ((<組み込み関数/open>)) も参照してくだ
    さい。

--- test(cmd, file1 [, file2])

    ファイルテストを行います。((|cmd|)) は以下に示す
    ((<リテラル/数値リテラル>))か文字列です(文字列の場合はそ
    の先頭の文字だけをコマンドとみなします)。

    * 1つの引数を取るもの

      * (({?r}))

        ファイルを実効 uid で読むことができる
      * (({?w}))

        ファイルに実効 uid で書くことができる
      * (({?x}))

        ファイルを実効 uid で実行することができる
      * (({?o}))

        ファイルの所有者が実効 uid である
      * (({?G}))

        ファイルのグループ所有者が実効 gid である
      * (({?R}))

        ファイルを実 uid で読むことができる
      * (({?W}))

        ファイルに実 uid で書くことができる
      * (({?X}))

        ファイルを実 uid で実行することができる
      * (({?O}))

        ファイルの所有者が実 uid である
      * (({?e}))

        ファイルが存在する
      * (({?z}))

        ファイルサイズが 0 である
      * (({?s}))

        ファイルサイズが 0 でない(ファイルサイズを返す)
      * (({?f}))

        ファイルはプレーンファイルである
      * (({?d}))

        ファイルはディレクトリである
      * (({?l}))

        ファイルはシンボリックリンクである
      * (({?p}))

        ファイルは名前つきパイプ(FIFO)である
      * (({?S}))

        ファイルはソケットである
      * (({?b}))

        ファイルはブロック特殊ファイルである
      * (({?c}))

        ファイルはキャラクター特殊ファイルである
      * (({?u}))

        ファイルに setuid ビットがセットされている
      * (({?g}))

        ファイルに setgid ビットがセットされている
      * (({?k}))

        ファイルに sticky ビットがセットされている

      * (({?M}))

        ファイルの最終更新時刻を返す
      * (({?A}))

        ファイルの最終アクセス時刻を返す
      * (({?C}))

        ファイルの inode 変更時刻を返す

    * 2つの引数を取るもの

      * (({?=}))

        ファイル1とファイル2の最終更新時刻が等しい
      * (({?>}))

        ファイル1の方がファイル2より最終更新時刻が新しい
      * (({?<}))

        ファイル1の方がファイル2より最終更新時刻が古い
      * (({?-}))

        ファイル1がファイル2にハードリンクされている

--- throw(tag[, value=nil])

    同じ ((|tag|)) を指定した ((<組み込み関数/catch>)) のブロックの
    終わりまで(メソッドを越えて)脱出します。もし同じ ((|tag|)) で
    待っている (({catch})) が存在しない場合は ((<NameError>)) で
    スレッドが終了します。((|tag|)) は文字列またはシンボルです。
    ((|value|)) は (({catch})) の戻り値になります。

    例:

      ret = catch(:exit) {
         throw :exit, 25
         some_process()    # 絶対に実行されない
         10
      }
      p ret   #=> 25

--- trace_var(varname, hook)
--- trace_var(varname) {|newval| .... }

    グローバル変数 ((|varname|)) への代入のフックを登録します。
    ((|varname|)) は文字列か ((<Symbol>)) で指定します。

    ここでの「グローバル変数」は、((<特殊変数|組み込み変数>))も含めた`$'
    で始まる変数のことです。

    この呼び出し以降、((|varname|)) で指定したグローバル変数に
    代入が起こると文字列または ((<Proc>)) オブジェクト ((|hook|))
    が評価されます。フックが (({Proc})) オブジェクトなら
    ブロック引数に代入された値が渡されます。またフックは複数
    登録できます。

    トレースを解除するには、((|hook|)) に (({nil})) を
    指定するか、((<組み込み関数/untrace_var>)) を用います。

    例:
        trace_var(:$v) {|val| puts "$v=#{val.inspect}" }
        $v = "foo"   #=> $v="foo"
        $v = 1       #=> $v=1

    ((|hook|)) が (({nil})) ならば、設定されていた
    ((|hook|)) の配列を返します(ブロックで登録されていれば
    ((<Proc>)) オブジェクトで返されます)
    それ以外は、(({nil})) を返します。

--- trap(signal, command)
--- trap(signal) { ... }

    ※ Ruby 1.8 以降では ((<Signal/Signal.trap>)) を推奨します。

    ((|signal|)) で指定された割り込みにたいするハンドラとして
    ((|command|)) を登録します。((|signal|)) はシグナル名の
    文字列か ((<Symbol>))、またはシグナル番号で指定します。

    ((|command|)) は文字列またはブロックで指定します。
    (({nil}))、空文字列(({""}))、(({"SIG_IGN"})) または
    (({"IGNORE"})) を指定した時は、そのシグナルを無視します
    (可能ならば)。
    (({"SIG_DFL"})) または (({"DEFAULT"})) を指定した時は、
    デフォルトの動作を行なうようになります。
    (({"EXIT"}))を指定した時は、シグナルを受け取ると((<終了処理>))を
    行ったあとステータス 0 で終了します。

    また ((|signal|)) の特別な値として 0 または (({"EXIT"}))
    を指定できます。これは「プログラムの終了時」を表します。

    いくつかのシグナルに対して、Ruby インタプリタは例外 ((<Interrupt>)) や
    ((<SignalException>)) を発生させます。このようなシグナルは例外処理によっ
    て捕捉することもできます。

        begin
          Process.kill :QUIT, $$   # 自身にSIGQUITを送信
        rescue SignalException
          puts "rescue #$!"
        end
        # => rescue SIGQUIT

    trap() により捕捉されたシグナルは例外を発生させません。

    trap は既にシグナルに対応する ((|command|)) が登録されて
    いれば、それを返します(ブロックは ((<Proc>)) オブジェク
    トとして返されます。"IGNORE" や "DEFAULT" に対しては
    (({nil})) を返します)。何も登録されていなければ
    (({nil})) を返します。

    ((<ruby 1.8 feature>)):
    trap 実行前に、"IGNORE" や "DEFAULT" が登録されていたなら、
    "IGNORE"、"DEFAULT" を返します。ruby インタプリタが知り得ない
    シグナルハンドラが登録されていた場合((-ちょっとウソ-))は (({nil})) を返します。

        p trap(:INT, "p true")     # => nil
        p trap(:INT) { }           # => "p true"
        p trap(:INT, "SIG_IGN")    # => #<Proc:0x401b1328>
        p trap(:INT, "DEFAULT")    # => nil
        p trap(:INT, "EXIT")       # => nil
        p trap(:INT, nil)          # => "EXIT"

    存在しないシグナルを指定した場合は、例外 ((<ArgumentError>)) が発
    生します((-なお、Windows (((<mswin32>)), ((<mingw32>)))では、INT
    ILL FPE SEGV TERM BREAK ABRT そして EXIT しか指定できません。-))

--- untrace_var(varname[, hook])

    グローバル変数 ((|varname|)) に関連付けられたフックを
    解除します。((|hook|)) が指定された場合にはそのフックだけを
    解除します。((|hook|)) が省略されるかまたは (({nil})) を
    与えた場合は ((|varname|)) のフックを全て解除します。
    解除されたフックの配列を返します。

    例:

      $v = nil
      trace_var(:$v) {|val| puts "$v=#{val.inspect}" }
      $v = 'str'        #=> $v="str"
      untrace_var :$v
      $v = 'str'        # なにも出力されない

--- warn(mesg)

    ((<ruby 1.8 feature>))

    ((|mesg|)) と改行を順番に ((<組み込み変数/$stderr>)) に出力します。
    組み込み変数 ((<組み込み変数/$VERBOSE>)) が (({nil})) のときは何も
    出力しません。

        $stderr.print mesg, "\n" unless $VERBOSE.nil?

    と同じです。(({nil})) を返します。
