=begin
=TestRunner.rb
A test runner UI for Fox.
=end

#<standard_header>
#
# Copyright (C) 2001 Jimmy Thrasher
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#</standard_header>

require 'fox'
require 'fox/responder'
require 'Lapidary/UI/TestRunnerMediator'
require 'Lapidary/UI/TestRunnerUtilities'

include Fox

GC.disable

module Lapidary
	module UI
		module Fox
			class TestRunner < FXMainWindow
				include Responder
				extend TestRunnerUtilities

				ID_RUN, ID_SELECT_LISTITEM, ID_QUIT = 
					(FXMainWindow::ID_LAST...(FXMainWindow::ID_LAST+3)).to_a

				def TestRunner.run(suite)
					new(suite).start
				end
				def initialize(suite)
					super(mainApp, "R2 TestRunner", nil, nil, DECOR_ALL, 20, 20)

					FXTooltip.new(mainApp)

					@suite = suite

					FXMAPFUNC(SEL_COMMAND, ID_RUN, "onRun")
					FXMAPFUNC(SEL_COMMAND, ID_SELECT_LISTITEM, "onSelectListItem")
					FXMAPFUNC(SEL_COMMAND, ID_QUIT, "onQuit")

					@red = false

					mainPanel
				end

				def onRun(sender, sel, ptr)
					@mediator.runSuite
					1
				end

				def onQuit(sender, sel, ptr)
				end

				def onSelectListItem(sender, sel, ptr)
					if (sender.retrieveItem(sender.getCurrentItem).isSelected)
						showFault(sender.retrieveItem(sender.getCurrentItem).fault)
					else
						clearFault
					end
					1
				end

				def start
					setupUI
					setupMediator
					attachToMediator
					startUI
				end
				def setupMediator
					@mediator = TestRunnerMediator.new(@suite)
					suiteName = @suite.to_s
					if ( @suite.kind_of?(Module) )
						suiteName = @suite.name
					end
					suiteNameEntry.setText(suiteName)
				end
				def setupUI
					@red = false
				end
				def attachToMediator
					@mediator.tuner.addListener(TestRunnerMediator::RESET, method(:resetUI))
					@mediator.tuner.addListener(TestRunnerMediator::FAULT_ADDED, method(:addFault))
					@mediator.tuner.addListener(TestRunnerMediator::RESULT_CHANGED, method(:resultChanged))
					@mediator.tuner.addListener(TestRunnerMediator::STATUS_CHANGED, method(:updateStatus))
				end

				def create
					super
					show(PLACEMENT_SCREEN)
				end

				def startUI
#					timer = Gtk::timeout_add(0) {
#						Gtk::timeout_remove(timer)
#						@mediator.runSuite
#					}
					mainApp.create
					mainApp.run
				end
				def stop
					mainApp.exit(0)
				end
				def resetUI(count)
					testProgressBar.setBarColor(greenStyle)
					testProgressBar.setTotal(count)
					testProgressBar.setProgress(0)
					@red = false

					runCountLabel.setText("0")
					assertionCountLabel.setText("0")
					failureCountLabel.setText("0")
					errorCountLabel.setText("0")

					faultList.clearItems
				end
				def addFault(fault)
					if ( ! @red )
						testProgressBar.setBarColor(redStyle)
						@red = true
					end
					item = FaultListItem.new(fault)
					faultList.appendItem(item)
				end
				def showFault(fault)
					rawShowFault(fault.longDisplay)
				end
				def rawShowFault(string)
					faultDetailLabel.setText(string)
#					outerDetailSubPanel.queue_resize
				end
				def clearFault
					rawShowFault("")
				end
				def resultChanged(result)
					testProgressBar.setProgress(testProgressBar.getProgress + 1)

					runCountLabel.setText(result.runs.to_s)
					assertionCountLabel.setText(result.assertions.to_s)
					failureCountLabel.setText(result.failures.to_s)
					errorCountLabel.setText(result.errors.to_s)
				end
				def updateStatus(status)
					statusEntry.getStatusline.setText(status.message)
				end

				def mainApp
					lazyInitialize(:mainApp) {
						@mainApp = FXApp.new("TestRunner", "Lapidary")
						@mainApp.init([])
					}
				end

				def mainPanel
					lazyInitialize(:mainPanel) {
						@mainPanel = FXVerticalFrame.new(self, LAYOUT_FILL_X)
						suitePanel
						progressPanel
						infoPanel
						listPanel
						detailPanel
						statusEntry
					}
				end

				def suitePanel
					lazyInitialize(:suitePanel) {
						@suitePanel = FXHorizontalFrame.new(@mainPanel, LAYOUT_SIDE_LEFT | LAYOUT_FILL_X, 0,0,0,0,10,10,10)
						FXLabel.new(@suitePanel, "Suite:", nil, LAYOUT_SIDE_LEFT | LAYOUT_FILL_X)
						suiteNameEntry
						runButton
					}
				end
				def suiteNameEntry
					lazyInitialize(:suiteNameEntry) {
						@suiteNameEntry = FXTextField.new(@suitePanel, 30)
						@suiteNameEntry.disable
					}
				end
				def runButton
					lazyInitialize(:runButton) {
						@runButton = FXButton.new(@suitePanel, "&Run\tRun the current suite", nil, self, ID_RUN)
					}
				end
				def progressPanel
					lazyInitialize(:progressPanel) {
						@progressPanel = FXHorizontalFrame.new(@mainPanel, LAYOUT_FILL_X | LAYOUT_FILL_Y, 0, 0, 0, 50, 10,10,10)
						testProgressBar
					}
				end
				def testProgressBar
					lazyInitialize(:testProgressBar) {
						@testProgressBar = FXProgressBar.new(@progressPanel, nil, 0, PROGRESSBAR_NORMAL | LAYOUT_FILL_X | LAYOUT_FILL_Y)#, 0, 0, 0, 50)
						@testProgressBar.setHeight(500)
						@testProgressBar.setBarColor(greenStyle)
					}
				end
				def greenStyle
					lazyInitialize(:greenStyle) {
						@greenStyle = FXRGBA(0,0xFF,0,0xFF) #0x00FF0000
					}
				end
				def redStyle
					lazyInitialize(:redStyle) {
						@redStyle = FXRGBA(0xFF,0,0,0xFF) #0xFF000000
					}
				end
				def infoPanel
					lazyInitialize(:infoPanel) {
						@infoPanel = FXHorizontalFrame.new(@mainPanel, 0, 0,0,0,0, 10,10,10)
						FXLabel.new(@infoPanel, "Runs:", nil, JUSTIFY_CENTER_X | LAYOUT_FILL_X)
						runCountLabel
						FXLabel.new(@infoPanel, "Assertions:", nil, JUSTIFY_CENTER_X | LAYOUT_FILL_X)
						assertionCountLabel
						FXLabel.new(@infoPanel, "Failures:", nil, JUSTIFY_CENTER_X | LAYOUT_FILL_X)
						failureCountLabel
						FXLabel.new(@infoPanel, "Errors:", nil, JUSTIFY_CENTER_X | LAYOUT_FILL_X)
						errorCountLabel
					}
				end
				def runCountLabel
					lazyInitialize(:runCountLabel) {
						@runCountLabel = FXLabel.new(@infoPanel, "0", nil, JUSTIFY_CENTER_X | LAYOUT_FILL_X)
					}
				end
				def assertionCountLabel
					lazyInitialize(:assertionCountLabel) {
						@assertionCountLabel = FXLabel.new(@infoPanel, "0", nil, JUSTIFY_CENTER_X | LAYOUT_FILL_X)
					}
				end
				def failureCountLabel
					lazyInitialize(:failureCountLabel) {
						@failureCountLabel = FXLabel.new(@infoPanel, "0", nil, JUSTIFY_CENTER_X | LAYOUT_FILL_X)
					}
				end
				def errorCountLabel
					lazyInitialize(:errorCountLabel) {
						@errorCountLabel = FXLabel.new(@infoPanel, "0", nil, JUSTIFY_CENTER_X | LAYOUT_FILL_X)
					}
				end
				def listPanel
					lazyInitialize(:listPanel) {
						@listPanel = FXHorizontalFrame.new(@mainPanel, LAYOUT_FILL_X | FRAME_SUNKEN | FRAME_THICK, 0,0,0,0, 10,10,10)
						faultList
					}
				end
				def faultList
					lazyInitialize(:faultList) {
						@faultList = FXList.new(@listPanel, 10, self, ID_SELECT_LISTITEM, LIST_SINGLESELECT | LAYOUT_FILL_X) #, 0, 0, 0, 150)
					}
				end
				def detailPanel
					lazyInitialize(:detailPanel) {
						@detailPanel = FXHorizontalFrame.new(@mainPanel, LAYOUT_FILL_X | FRAME_SUNKEN | FRAME_THICK, 0,0,0,0, 10,10,10)
						detailScrolledWindow
					}
				end
				def detailScrolledWindow
					lazyInitialize(:detailScrolledWindow) {
						@detailScrolledWindow = FXScrollWindow.new(@detailPanel, LAYOUT_FILL_X)
						faultDetailLabel
					}
				end
				def faultDetailLabel
					lazyInitialize(:faultDetailLabel) {
						@faultDetailLabel = FXLabel.new(@detailScrolledWindow, "", nil, LAYOUT_SIDE_LEFT)
						class << @faultDetailLabel
							def setText(text)
								super(text.gsub(/\n\t/, "\n" + (" " * 4)))
							end
						end
					}
				end
				def statusPanel
					lazyInitialize(:statusPanel) {
						@statusPanel = FXHorizontalFrame.new(@mainWindow)
						statusEntry
					}
				end
				def statusEntry
					lazyInitialize(:statusEntry) {
						@statusEntry = FXStatusbar.new(self, LAYOUT_SIDE_RIGHT | LAYOUT_FILL_X)
					}
				end

				def lazyInitialize(symbol)
					if ( instance_eval("@" + symbol.to_s).nil? )
						yield
					end
					return instance_eval("@" + symbol.to_s)
				end
			end

			class FaultListItem < FXListItem
				attr_reader(:fault)
				def initialize(fault)
					super(fault.shortDisplay)
					@fault = fault
				end
			end
		end
	end
end

if __FILE__ == $0
	Lapidary::UI::Fox::TestRunner.startCommandLineTest
end
