require 'runit/testcase'
require 'runit/cui/testrunner'
require 'erb'

class ERBSimpleTest < RUNIT::TestCase
  def setup
    @erb = ERB
  end

  def test_01
    _test_01(nil)
    _test_01(0)
    _test_01(1)
    _test_01(2)
    _test_01(3)
  end

  def _test_01(safe)
    erb = @erb.new("hello")
    assert_equal(erb.result, "hello")

    erb = @erb.new("hello", safe, 0)
    assert_equal(erb.result, "hello")

    erb = @erb.new("hello", safe, 1)
    assert_equal(erb.result, "hello")

    erb = @erb.new("hello", safe, 2)
    assert_equal(erb.result, "hello")

    src = <<EOS
%% hi
= hello
<% 3.times do |n| %>
% n=0
* <%= n %>
<% end %>
EOS
      
    ans = <<EOS
%% hi
= hello

% n=0
* 0

% n=0
* 1

% n=0
* 2

EOS
    erb = @erb.new(src)
    assert_equal(ans, erb.result)
    erb = @erb.new(src, safe, 0)
    assert_equal(ans, erb.result)
    erb = @erb.new(src, safe, '')
    assert_equal(ans, erb.result)

    ans = <<EOS
%% hi
= hello
% n=0
* 0% n=0
* 1% n=0
* 2
EOS
    erb = @erb.new(src, safe, 1)
    assert_equal(ans.chomp, erb.result)
    erb = @erb.new(src, safe, '>')
    assert_equal(ans.chomp, erb.result)

    ans  = <<EOS
%% hi
= hello
% n=0
* 0
% n=0
* 1
% n=0
* 2
EOS
      
    erb = @erb.new(src, safe, 2)
    assert_equal(ans, erb.result)
    erb = @erb.new(src, safe, '<>')
    assert_equal(ans, erb.result)

    ans = <<EOS
% hi
= hello

* 0

* 0

* 0

EOS
    erb = @erb.new(src, safe, '%')
    assert_equal(ans, erb.result)

    ans = <<EOS
% hi
= hello
* 0* 0* 0
EOS
    erb = @erb.new(src, safe, '%>')
    assert_equal(ans.chomp, erb.result)

    ans = <<EOS
% hi
= hello
* 0
* 0
* 0
EOS
    erb = @erb.new(src, safe, '%<>')
    assert_equal(ans, erb.result)
  end

  def test_02_safe_04
    erb = @erb.new('<%=$SAFE%>', 4)
    assert_equal(erb.result(TOPLEVEL_BINDING.taint), '4')
  end

  class Foo; end

  def test_03_def_class
    erb = @erb.new('hello')
    cls = erb.def_class
    assert_equal(Object, cls.superclass)
    assert(cls.new.respond_to?('result'))
    cls = erb.def_class(Foo)
    assert_equal(Foo, cls.superclass)
    assert(cls.new.respond_to?('result'))
    cls = erb.def_class(Object, 'erb')
    assert_equal(Object, cls.superclass)
    assert(cls.new.respond_to?('erb'))
  end

  def test_04_percent
    src = <<EOS
%n = 1
<%= n%>
EOS
    assert_equal("1\n", ERB.new(src, nil, '%').result)

    src = <<EOS
<%
%>
EOS
    ans = "\n"
    assert_equal(ans, ERB.new(src, nil, '%').result)

    src = "<%\n%>"
    ans = "\n"
    assert_equal(ans, ERB.new(src, nil, '%').result)

    src = <<EOS
<%
n = 1
%><%= n%>
EOS
    assert_equal("1\n", ERB.new(src, nil, '%').result)
  end

  class Bar; end

  def test_05_def_method
    assert(! Bar.new.respond_to?('hello'))
    Bar.module_eval do
      extend ERB::DefMethod
      def_erb_method('hello', 'hello.erb')
    end
    assert(Bar.new.respond_to?('hello'))

    assert(! Bar.new.respond_to?('hello_world'))
    erb = @erb.new('hello, world')
    Bar.module_eval do
      def_erb_method('hello_world', erb)
    end
    assert(Bar.new.respond_to?('hello_world'))    
  end
end

if __FILE__ == $0
  RUNIT::CUI::TestRunner.run(ERBSimpleTest.suite)
end
