#
# session.rb  0.2.0
#
#   author: Minero Aoki <aamine@dp.u-netsurf.ne.jp>
#

require 'socket'



class String

  def doquote
    str = self.gsub( "\n", '\\n' )
    str = str.gsub( "\r", '\\r' )
    str = str.gsub( "\t", '\\t' )
    
    return str
  end

end



module Mail

  DEBUG = false   #$DEBUG


  class Session

    Version = '0.3.0'

    def initialize( addr = 'localhost', port = nil )
      proto_initialize
      @address = addr
      @port    = port if port
      @active  = false
    end

    class << self
      def start( address = 'localhost', port = nil, *args )
        inst = new( address, port )
        ret = inst.start( *args )

        if iterator? then
          ret = yield( inst )
          inst.finish
        end
        return ret
      end
    end


    attr :address
    attr :port

    attr :socket

    attr :proto_type
    attr :proto, true

    def start( *args )
      if active? then return false end

      @socket = ProtocolSocket.open( @address, @port )
      @proto = @proto_type.new( @socket )
      do_start( *args )

      @active = true
    end

    def finish
      if @proto then
        do_finish
        @proto = nil

        return true
      else
        return false
      end
    end

    def active?() @active end

  end



  class Protocol

    def initialize( sock )
      @socket = sock
      check_reply( SuccessCode )
    end

    attr :socket, true

    def quit
      if @socket and not @socket.closed? then
        do_quit
        @socket.close
        @socket = nil
      end
    end


    private

    def check_reply( *oks )
      rep = get_reply
      oks.each do |i|
        if i === rep then
          return rep
        end
      end

      rep.error! @socket.sending
    end

    
    class ProtocolError        < StandardError   ; end
    class   ProtoSyntaxError   <   ProtocolError ; end
    class   ProtoFatalError    <   ProtocolError ; end
    class   ProtoUnknownError  <   ProtocolError ; end
    class   ProtoServerError   <   ProtocolError ; end
    class   ProtoAuthError     <   ProtocolError ; end
    class   ProtoCommandError  <   ProtocolError ; end

    class ReplyCode

      def initialize( cod, mes )
        @code = cod
        @msg  = mes
      end

      attr :code
      attr :msg

      def error!( sending )
        err, tag = Errors[ self.type ]
        mes = sprintf( <<MES, tag, @code, sending.doquote, @msg.doquote )

%s: status %s
writing string is:
%s

error message from server is:
%s
MES
        raise err, mes
      end

    end


    class SuccessCode     < ReplyCode ; end
    class ContinueCode    < SuccessCode ; end
    class ErrorCode       < ReplyCode ; end
    class SyntaxErrorCode < ErrorCode ; end
    class FatalErrorCode  < ErrorCode ; end
    class ServerBusyCode  < ErrorCode ; end
    class UnknownCode     < ReplyCode ; end


    class ReplyCode
      Errors = {
        SuccessCode     => [ ProtoUnknownError, 'unknown error' ],
        ContinueCode    => [ ProtoUnknownError, 'unknown error' ],
        ErrorCode       => [ ProtocolError, 'protocol error' ],
        SyntaxErrorCode => [ ProtoSyntaxError, 'syntax error' ],
        FatalErrorCode  => [ ProtoFatalError, 'fatal error' ],
        ServerBusyCode  => [ ProtoServerError, 'probably server busy' ],
        UnknownCode     => [ ProtoUnknownError, 'unknown error' ]
      }
    end

  end



  class ProtocolSocket

    def initialize( addr, port )
      @address = addr
      @port    = port

      @ipaddr  = ''
      @closed  = false
      @sending = ''
      @buffer  = ''

      @socket = TCPsocket.new( addr, port )
      @ipaddr = @socket.addr[3]

      @dout = Mail::DEBUG
    end

    class << self
      alias open new
    end


    attr :socket, true

    def close
      @socket.close
      @closed = true
    end

    def closed?() @closed end

    def addr() @address.dup end
    def port() @port end
    def ipaddr() @ipaddr.dup end

    attr :sending


    CRLF    = "\r\n"
    D_CRLF  = ".\r\n"
    TERMEXP = /\n|\r\n|\r/o

    READ_BLOCK = 1024 * 8


    def read( len )
      while @buffer.size < len do
        @buffer << @socket.sysread( READ_BLOCK )
      end

      return fetch_rbuf( len )
    end


    def readuntil( target )
      until idx = @buffer.index( target ) do
        @buffer << @socket.sysread( READ_BLOCK )
      end

      return fetch_rbuf( idx + target.size )
    end


    def fetch_rbuf( len )
      bsi = @buffer.size
      ret = @buffer[ 0, len ]
      @buffer = @buffer[ len, bsi - len ]

      if @dout then
        $stderr.print 'read  "'
        debugout ret
        $stderr.print "\"\n"
      end
      return ret
    end
    private :fetch_rbuf

        
    def readline
      ret = readuntil( CRLF )
      ret.chop!
      return ret
    end


    def read_pendstr
      $stderr.puts "reading pendstr" if pre = @dout ; @dout = false

      ret = ''
      lines = 0

      while (str = readuntil( CRLF )) != D_CRLF do
        str.gsub!( /\A\./o, '' )
        ret << str
      end

      $stderr.puts "read pendstr #{ret.size} bytes" if @dout = pre
      return ret
    end


    def read_pendlist
      arr = []
      str = nil
      call = iterator?

      while (str = readuntil( CRLF )) != D_CRLF do
        str.chop!
        arr.push str
        if call then
          yield str
        end
      end

      return arr
    end


    ### write


    def write( src )
      do_write_beg
      src.each("\n") do |line|
        line.gsub! TERMEXP, CRLF
        do_write_do line
      end
      return do_write_fin
    end


    def writebin( src )
      do_write_beg
      src.each("\n") do |line|
        do_write_do line
      end
      return do_write_fin
    end


    def writeline( str )
      do_write_beg
      do_write_do str
      do_write_do CRLF
      return do_write_fin
    end


    def write_pendstr( src )
      $stderr.puts "writing pendstr from #{src.type}" if pre = @dout
      @dout = false

      do_write_beg

      line = nil
      src.each("\n") do |line|
        line.gsub! TERMEXP, CRLF
        line.each( CRLF ) do |i|
          do_write_do '.' if i[0] == ?.
          do_write_do i
        end
      end
      do_write_do CRLF unless line[ -2, 2 ] == CRLF
      do_write_do D_CRLF

      wsize = do_write_fin

      $stderr.puts "wrote pendstr #{wsize} bytes" if @dout = pre
      return wsize
    end


    private


    def do_write_beg
      $stderr.print 'write "' if @dout

      @writtensize = 0
      @sending = ''
    end

    def do_write_do( arg )
      debugout arg if @dout

      if @sending.size < 128 then
        @sending << arg
      end
      if @sending.size > 128 then
        @sending[ 125, @sending.size - 125 ] = '...'
      end
      @writtensize += @socket.write( arg )
    end

    def do_write_fin
      $stderr.puts if @dout

      @socket.flush

      return @writtensize
    end


    def debugout( ret )
      while ret and tmp = ret[ 0, 50 ] do
        ret = ret[ 50, ret.size - 50 ]
        tmp = tmp.inspect
        $stderr.print tmp[ 1, tmp.size - 2 ]
      end
    end

  end

end
