
=head1 NAME

Lingua::EN::Nickname - Genealogical nickname matching (Liz=Beth)

=head1 SYNOPSIS

  use Lingua::EN::Nickname;

  # Equivalent first names?
  $score= nickname_eq( $firstn_0, $firstn_1 );

  # Full, expanded, name(s) 
  @roots= rootname( $firstn );

=head1 DESCRIPTION

Nicknames, alternate spellings, and alternate etymological derivations
make checking first name equivalence nearly impossible.
This module will tell you that 'Maggie', 'Peg', and 'Margaret' are all
probably the same name.

=head1 SOURCES

=over 4

=item * USGenWeb Project 
L<http://www.usgenweb.com/researchers/nicknames.html>

=item * TNGenWeb Project 
L<http://www.tngenweb.usit.com/franklin/frannick.htm>

=item * Chesnut Family Pages 
L<http://www.users.mis.net/~chesnut/pages/nicknams.htm>

=item * Ultimate Family Tree 
L<http://www.uftree.com/UFT/HowTos/SettingOut/nickname1.html>

=back

=head1 TODO

=over 4

=item * Hire a team of experts to provide a more scientific, 
statistically accurate Name Etymology source file.

=item * Create more phoenetically-based sub-regexes.

=item * Detect simple monosyllabic truncation nicknames, 
be less certain about them, but match more.

=item * Pay more attention to gender.

=back

=head1 AUTHOR

Brian Lalonde, E<lt>brian@webcoder.infoE<gt>

=head1 SEE ALSO

perl(1)

=cut

package Lingua::EN::Nickname;
require Exporter;
use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);
use vars qw(%root %multi %match %akin);

$VERSION=     '1.12';
@ISA=         qw(Exporter);
@EXPORT=      qw(nickname_eq nickroot);
@EXPORT_OK=   qw(nickmatch nickfollow);
%EXPORT_TAGS= 
( 
  ALL => [ @EXPORT, @EXPORT_OK ],
);

sub nickmatch($)
{
  local $_= shift;
  s/\b(\w+)/\L\u$1/g if !/[a-z]/ and length $_ > 2; s/\s+//g;
  return $match{$_} if $match{$_};
  s/([^aeiouyA-Z])(ie?|e?y)$/$1E/;
  return $match{$root{$_}} if $root{$_};
  my @root= map "$match{$_}", @{$multi{$_}};
  return unless @root;
  local $"= '|';
  return qr/@root/;
}

sub nickfollow
{ # follow path of similar names
  my $dest= shift; 
  my $history= join('|', my @history= @_);
  my $step= shift; 
  my $match;
  return 1 if $dest eq $step;
  ($match)= sort {$a<=>$b} 
    grep {defined}
    map {nickfollow($dest,$_,@history)} 
    grep !/^($history)$/, 
    map { $akin{$_} ? @{$akin{$_}} : () } 
    nickroot($step);
  return 1+$match if $match;
  return;
}

sub nickname_eq
{
  my($a,$b,$regex,$match)= map 
    {my$n=$_;$n=~s/\b(\w+)/\L\u$1/g if!/[a-z]/;$n=~s/\s+//g;$n} @_;
  return 100 if $a eq $b; # trivial case 
  return 98 if $a eq $root{$b} or $b eq $root{$a};
  return 95 if ( $regex= nickroot($b) and $a=~ /^($regex)$/ )
    or ( $regex= nickroot($a) and $b=~ /^($regex)$/ );
  return unless $regex= nickmatch $a;
  return 90 if $b=~ /$regex/;
  return int 80/(2*$match) if $match= (nickfollow($a,$b)||nickfollow($b,$a));
  return;
}

sub nickroot($)
{
  local $_= shift;
  s/\b(\w+)/\L\u$1/g if !/[a-z]/ and length $_ > 2; s/\s+//g;
  return $_ if $match{$_};
  s/(?<![aeiouyA-Z])(ie?|e?y)$/E/;
  return $root{$_} if $root{$_};
  local $"= '|';
  return( wantarray ? @{$multi{$_}} : "@{$multi{$_}}" ) if $multi{$_};
}

my $E= qr/(ie?|e?y)/; # phoenetic long 'E' for many name endings 


# Massive Data Structures 
#
# Hand-tuning is not recommended--edit the source text file
# instead, then regenerate the hashes.


%root=
(
  Abertina => 'Alberta',
  Abiah => 'Abijah',
  Abram => 'Abraham',
  Acuilla => 'Aquilla',
  Ada => 'Adaline',
  Adaline => 'Adelaide',
  Adelia => 'Adelaide',
  Adeline => 'Adelaide',
  Adeliza => 'Adelaide',
  Ado => 'Rudolphus',
  Adolf => 'Rudolphus',
  Adolphus => 'Rudolphus',
  Adoph => 'Rudolphus',
  Ag => 'Agatha',
  Aileen => 'Helena',
  AlE => 'Alice',
  Alaina => 'Elaine',
  Alan => 'Alanson',
  Albert => 'Adelbert',
  Albertine => 'Alberta',
  Alec => 'Alexander',
  Alexandra => 'Alexandria',
  Alf => 'Alfred',
  Alice => 'Elsie',
  Alicia => 'Alice',
  Aline => 'Adaline',
  Alisha => 'Alice',
  Alison => 'Alice',
  Alla => 'Alexandria',
  Alphonzo => 'Alonzo',
  Alphus => 'Alphinias',
  Amabel => 'Mehitabel',
  Amos => 'Moses',
  Ana => 'Anastasia',
  Anabelle => 'Arabella',
  Ander => 'Anderson',
  Andre => 'Anderson',
  Andria => 'Andrea',
  AngE => 'Angela',
  Angelica => 'Angela',
  Angelina => 'Angela',
  Angeline => 'Angela',
  Anna => 'Hannah',
  Annette => 'Anna',
  Anse => 'Anselm',
  Ansel => 'Anselm',
  Antoine => 'Anthony',
  Antonia => 'Antoinette',
  Antonio => 'Anthony',
  Aphinius => 'Alphinias',
  AppE => 'Appoline',
  ArE => 'Arielle',
  Ara => 'Arabella',
  Arabelle => 'Arabella',
  Arch => 'Archibald',
  ArchE => 'Archibald',
  Archelous => 'Archibald',
  ArdE => 'Ardeshir',
  ArlE => 'Arlene',
  Armanda => 'Amanda',
  ArnE => 'Arnold',
  Art => 'Arthur',
  Asa => 'Asaph',
  Asahel => 'Asaph',
  Assene => 'Asenath',
  AugE => 'Augusta',
  Augusta => 'Augusta',
  Augustina => 'Augusta',
  Augustine => 'Augustus',
  Aurilla => 'Aurelia',
  Azarich => 'Azariah',
  Aze => 'Azariah',
  Bab => 'Barbara',
  BabbE => 'Barbara',
  Babs => 'Barbara',
  BaisE => 'Elizabeth',
  BaissE => 'Elizabeth',
  BaldE => 'Archibald',
  Baldo => 'Archibald',
  Barb => 'Barbara',
  BarbE => 'Barbara',
  BarbarE => 'Barbara',
  BarberE => 'Barbara',
  BarnE => 'Barnabas',
  Barnard => 'Barnabas',
  Bart => 'Bartholomew',
  Bartel => 'Bartholomew',
  Barth => 'Bartholomew',
  Basil => 'Bazaleel',
  Bat => 'Bartholomew',
  Bear => 'Barry',
  Becca => 'Rebecca',
  Beck => 'Rebecca',
  BeckE => 'Rebecca',
  Bede => 'Obedience',
  BeedE => 'Obedience',
  Bela => 'William',
  BenjE => 'Benjamin',
  Bennett => 'Benedict',
  BernE => 'Barnabas',
  Bernard => 'Barnabas',
  Bess => 'Elizabeth',
  BessE => 'Elizabeth',
  Beth => 'Elizabeth',
  Bethia => 'Elizabeth',
  BetsE => 'Elizabeth',
  BettE => 'Elizabeth',
  Bev => 'Beverly',
  BeverlE => 'Beverly',
  Biah => 'Abijah',
  Bias => 'Tobias',
  Biel => 'Abiel',
  Bige => 'Abijah',
  Bill => 'William',
  BillE => 'William',
  Bird => 'Albert',
  BirdE => 'Roberta',
  BitsE => 'Elizabeth',
  Bo => 'Boetius',
  Bob => 'Robert',
  BonnE => 'Bonita',
  BrE => 'Bridget',
  Brad => 'Bradford',
  BradE => 'Broderick',
  BradlE => 'Bradford',
  Bree => 'Aubrey',
  BridgE => 'Bridget',
  Bridgit => 'Bedelia',
  Brina => 'Sabrina',
  BrodE => 'Broderick',
  Bryan => 'Brian',
  Bryant => 'Brian',
  Buck => 'Charles',
  Burt => 'Egbert',
  CaddE => 'Caroline',
  Cager => 'Micajah',
  CammE => 'Camille',
  CandE => 'Candace',
  Car => 'Charlotte',
  Carl => 'Charles',
  Carlotta => 'Charlotte',
  Carol => 'Caroline',
  Carole => 'Caroline',
  Carolina => 'Caroline',
  Carolyn => 'Caroline',
  Casper => 'Jasper',
  Cassandra => 'Alexandria',
  CathE => 'Katherine',
  Catherine => 'Katherine',
  Cathleen => 'Katherine',
  Ceall => 'Lucille',
  CecelE => 'Cecilia',
  CecilE => 'Cecilia',
  Cecilia => 'Sheila',
  Ced => 'Cedrick',
  Celina => 'Selina',
  Cene => 'Cyrenius',
  Cenia => 'Laodicia',
  Chad => 'Charles',
  Chan => 'Chauncy',
  Char => 'Charlotte',
  CharlE => 'Charles',
  Charlotta => 'Lotta',
  Charlotte => 'Lotta',
  Chat => 'Charity',
  Cheryl => 'Sharon',
  CheslE => 'Chesley',
  Chet => 'Chesley',
  Chick => 'Charles',
  ChristE => 'Christiana',
  Christian => 'Christopher',
  Christina => 'Christiana',
  Christine => 'Christiana',
  Christopher => 'Christian',
  Chuck => 'Charles',
  Cibyl => 'Sibbilla',
  Cil => 'Priscilla',
  Cille => 'Lucille',
  Ciller => 'Priscilla',
  Cinthia => 'Cynthia',
  Claas => 'Nicholas',
  Claes => 'Nicholas',
  Claire => 'Clarissa',
  Clara => 'Clarissa',
  Clarice => 'Clarissa',
  Clarinda => 'Clarissa',
  Cleat => 'Cleatus',
  Clo => 'Chloe',
  Clum => 'Columbus',
  Con => 'Conrad',
  Cono => 'Cornelius',
  Cora => 'Corinne',
  CordE => 'Cordelia',
  CornE => 'Cornelia',
  Court => 'Courtney',
  CreasE => 'Lucretia',
  Crece => 'Lucretia',
  Crese => 'Lucretia',
  Cris => 'Christiana',
  CrissE => 'Christiana',
  Cristina => 'Christiana',
  Curg => 'Lecurgus',
  DacE => 'Candace',
  Dahl => 'Dalton',
  DaisE => 'Margaret',
  Dal => 'Dalton',
  Damaris => 'Demerias',
  DanE => 'Danielle',
  Danelle => 'Danielle',
  Danial => 'Daniel',
  Daph => 'Daphne',
  DaphE => 'Daphne',
  DarE => 'Darwin',
  DarrE => 'Darlene',
  DavE => 'David',
  Dave => 'David',
  Day => 'David',
  DeannE => 'Geraldine',
  Deb => 'Deborah',
  DebbE => 'Deborah',
  Debbe => 'Deborah',
  Debi => 'Deborah',
  Debora => 'Deborah',
  Debra => 'Deborah',
  Delbert => 'Adelbert',
  DellE => 'Deliverance',
  Delpha => 'Philadelphia',
  Delphia => 'Philadelphia',
  Delphina => 'Adelphia',
  Demaris => 'Demerias',
  DennE => 'Dennis',
  Dennison => 'Dennis',
  DensE => 'Prudence',
  Denys => 'Denise',
  Denyse => 'Denise',
  DesE => 'Desiree',
  DesirE => 'Desiree',
  DesrE => 'Desiree',
  Desree => 'Desiree',
  Dewayne => 'Duane',
  Di => 'Diana',
  Diane => 'Diana',
  Dickon => 'Richard',
  Dickson => 'Richard',
  Dilbert => 'Delbert',
  DillE => 'Deliverance',
  Dimmis => 'Demerias',
  Dina => 'Geraldine',
  Dirch => 'Derrick',
  Dite => 'Aphrodite',
  Ditus => 'Aphrodite',
  Dob => 'Robert',
  Dobbin => 'Robert',
  Dock => 'Zadock',
  Doda => 'Dorothy',
  Dode => 'Dorothy',
  Dolf => 'Rudolphus',
  DollE => 'Dorothy',
  Dom => 'Dominic',
  Don => 'Donald',
  Dona => 'Caldonia',
  Donia => 'Fredonia',
  DonnE => 'Donald',
  Donna => 'Fredonia',
  Dorinda => 'Dorothy',
  Doris => 'Dorothy',
  Dorothea => 'Dorothy',
  Dortha => 'Dorothy',
  DosE => 'Eudoris',
  Dosia => 'Theodosia',
  DossE => 'Eudoris',
  Dot => 'Dorothy',
  Dotha => 'Dorothy',
  DottE => 'Dorothy',
  Doug => 'Douglas',
  Dunk => 'Duncan',
  Dwane => 'Duane',
  Dwayne => 'Duane',
  Dyce => 'Aphrodite',
  Dyche => 'Aphrodite',
  Eben => 'Ebenezer',
  Ec => 'Alexander',
  EddE => 'Edward',
  Edgar => 'Edward',
  Edith => 'Adaline',
  Edmund => 'Edward',
  EdnE => 'Edith',
  Edna => 'Edith',
  EffE => 'Euphemia',
  Elaina => 'Elaine',
  Eleanor => 'Helena',
  Eleck => 'Alexander',
  Electa => 'Electra',
  Elena => 'Helena',
  Elenor => 'Leonora',
  Elenora => 'Eleanor',
  Elic => 'Alexandria',
  Elicia => 'Alice',
  Elinamifia => 'Eleanor',
  Eliphal => 'Eliphalet',
  Elis => 'Elizabeth',
  Elisha => 'Alice',
  Elissa => 'Elizabeth',
  Ellender => 'Helena',
  Ellis => 'Alice',
  Ells => 'Elwood',
  Elnora => 'Eleanor',
  Eloise => 'Lousie',
  Elouise => 'Lousie',
  Em => 'Emeline',
  Emanuel => 'Manuel',
  Emeline => 'Emeline',
  Emiline => 'Emeline',
  Emm => 'Emeline',
  EmmE => 'Emeline',
  Emma => 'Emeline',
  Emmer => 'Emeline',
  Epaphroditius => 'Aphrodite',
  Epaphroditus => 'Aphrodite',
  Eph => 'Ephraim',
  Eppa => 'Aphrodite',
  Eric => 'Derrick',
  Erin => 'Aaron',
  Erma => 'Emeline',
  ErnE => 'Earnest',
  Erna => 'Ernestine',
  Ernest => 'Earnest',
  Erwin => 'Irwin',
  Essa => 'Vanessa',
  Estelle => 'Estella',
  Esther => 'Hester',
  Euy => 'Eugenia',
  Ev => 'Evelyn',
  Eva => 'Evaline',
  Evelina => 'Evelyn',
  Evelyn => 'Evelyn',
  FallE => 'Eliphalet',
  Fan => 'Frances',
  FannE => 'Frances',
  FannnE => 'Nathaniel',
  Fate => 'Lafayette',
  Fay => 'Faith',
  Felicia => 'Felicity',
  Fena => 'Euphrosina',
  Fenee => 'Euphrosina',
  FerbE => 'Pharaba',
  FerdE => 'Ferdinand',
  Fidelia => 'Bedelia',
  Field => 'Winfield',
  FinnE => 'Phineas',
  Finnius => 'Alphinias',
  Flick => 'Felicity',
  Flo => 'Florence',
  Flora => 'Florence',
  Floss => 'Florence',
  FlossE => 'Florence',
  Fran => 'Frances',
  Frank => 'Francis',
  Frankisek => 'Francis',
  Franklin => 'Francis',
  Franz => 'Francis',
  Fredric => 'Frederick',
  Fredricka => 'Frederica',
  Frish => 'Frederick',
  Fritz => 'Frederick',
  Frona => 'Sophronia',
  Fronia => 'Sophronia',
  Gabe => 'Gabriel',
  Gabriella => 'Gabrielle',
  Gail => 'Abigail',
  GatsE => 'Augustus',
  GatsbE => 'Augustus',
  GattE => 'Gertrude',
  Gay => 'Gerhardt',
  GenE => 'Eugenia',
  GencE => 'Genevieve',
  Geoff => 'Jefferson',
  GeoffrE => 'Jefferson',
  Georgiana => 'Georgia',
  GerE => 'Geraldine',
  Gert => 'Gertrude',
  GertE => 'Gertrude',
  Gib => 'Gilbert',
  Gil => 'Gilbert',
  Ginger => 'Virginia',
  GinnE => 'Virginia',
  GlorE => 'Gloria',
  Green => 'Greenberry',
  Greg => 'Gregory',
  Gregg => 'Gregory',
  Greta => 'Margaret',
  Gretta => 'Margaret',
  Grissel => 'Griselda',
  Gum => 'Montgomery',
  GusE => 'Augusta',
  Gustus => 'Augustus',
  Gwen => 'Gwendolyn',
  HallE => 'Mahalla',
  Ham => 'Hamilton',
  Hamp => 'Hamilton',
  Hans => 'John',
  HappE => 'Karonhappuck',
  Harman => 'Herman',
  HattE => 'Harriet',
  Hebsabeth => 'Hepsabah',
  HeidE => 'Adelaide',
  Heide => 'Adelaide',
  Helene => 'Helena',
  HelmE => 'Wilhelmina',
  Hen => 'Henry',
  Hence => 'Henry',
  Hephsibah => 'Hepsabah',
  HepsE => 'Hepsabah',
  Hepsabel => 'Hepsabah',
  Hepsibah => 'Hepsabah',
  Herb => 'Herbert',
  HermE => 'Hermione',
  Hermoine => 'Hermione',
  HessE => 'Hester',
  Hez => 'Hezekiah',
  Hiel => 'Jehiel',
  HilE => 'Hiram',
  HittE => 'Mehitabel',
  Hob => 'Robert',
  Hobkin => 'Robert',
  Hodge => 'Roger',
  Hodgekin => 'Roger',
  HonE => 'Honora',
  Hop => 'Hopkins',
  Hopp => 'Hopkins',
  Horatio => 'Horace',
  HorrE => 'Horace',
  HosE => 'Hosea',
  HowE => 'Howard',
  Hub => 'Hubert',
  Hugo => 'Hubert',
  HumE => 'Posthuma',
  Ian => 'John',
  Ib => 'Isabella',
  IggE => 'Ignatius',
  Ike => 'Isaac',
  Immanuel => 'Emanuel',
  Ina => 'Lavinia',
  IndE => 'India',
  Inez => 'Agnes',
  Irving => 'Irvin',
  Isabel => 'Isabella',
  Isabelle => 'Isabella',
  Ivan => 'John',
  IzzE => 'Isidore',
  Jaap => 'Jacob',
  Jack => 'John',
  Jacklin => 'Jacqueline',
  Jacklyn => 'Jacqueline',
  Jackson => 'John',
  Jaclin => 'Jacqueline',
  Jaclyn => 'Jacqueline',
  Jacobus => 'Jacob',
  Jacque => 'Jacqueline',
  JakE => 'Jacqueline',
  Jake => 'Jacob',
  Jameson => 'James',
  JanE => 'Jane',
  Janet => 'Jane',
  Janett => 'Jane',
  Janette => 'Jane',
  Janice => 'Jane',
  Janiece => 'Jane',
  Janyce => 'Jane',
  Jaques => 'John',
  Jay => 'Jacob',
  Jayce => 'Jane',
  Jayhugh => 'Jehu',
  Jc => 'Jane',
  Jeanne => 'Jane',
  Jed => 'Jedediah',
  Jedidiah => 'Jedediah',
  Jeff => 'Jefferson',
  JeffrE => 'Jefferson',
  Jem => 'James',
  Jemma => 'Jemima',
  JeremE => 'Jeremiah',
  Jereme => 'Jeremiah',
  Jess => 'Jessica',
  Jesse => 'Jessica',
  Jill => 'Julia',
  Jim => 'James',
  JimmE => 'James',
  JinE => 'Virginia',
  JincE => 'Jane',
  Jo => 'Josephine',
  Joann => 'Joanna',
  Joanna => 'Jane',
  Joannah => 'Joanna',
  Joanne => 'Joanna',
  Jock => 'John',
  Joe => 'Joseph',
  Johanna => 'Joanna',
  Johannah => 'Joanna',
  John => 'Jonathan',
  Jorge => 'George',
  JosE => 'Josephine',
  Josefa => 'Joseph',
  Josepha => 'Josephine',
  Josephine => 'Pheney',
  Josh => 'Joshua',
  Joy => 'Joyce',
  Jr => 'Junior',
  Jud => 'Judson',
  JulE => 'Julia',
  Jule => 'Julias',
  Julian => 'Julias',
  Juliet => 'Julia',
  Julius => 'Julias',
  JunE => 'Junior',
  June => 'Junior',
  Justus => 'Justin',
  KC => 'Casey',
  KarE => 'Caroline',
  Karel => 'Charles',
  Karl => 'Charles',
  Karon => 'Karonhappuck',
  KasE => 'Casey',
  Kat => 'Katherine',
  KatE => 'Katherine',
  Katarina => 'Katherine',
  Kate => 'Katherine',
  KathE => 'Katherine',
  Kathe => 'Katherine',
  Katherina => 'Katherine',
  Kathleen => 'Katherine',
  Kathryn => 'Katherine',
  Kay => 'Katherine',
  Ken => 'Kenneth',
  Kendall => 'Kenneth',
  Kendrick => 'Kenneth',
  Kenj => 'Kendra',
  KenjE => 'Kendra',
  KennE => 'Kenneth',
  Kent => 'Kenneth',
  KerE => 'Caroline',
  KerstE => 'Christiana',
  Kester => 'Christopher',
  KezzE => 'Keziah',
  Kiah => 'Hezekiah',
  Kid => 'Keziah',
  Kim => 'Kimberly',
  KimberlE => 'Kimberly',
  KimmE => 'Kimberly',
  KingslE => 'King',
  Kingston => 'King',
  KissE => 'Calista',
  KitsE => 'Katherine',
  KittE => 'Katherine',
  Kizza => 'Keziah',
  Knowell => 'Noel',
  Kris => 'Christiana',
  KristE => 'Christiana',
  Kristine => 'Christiana',
  Kurt => 'Curtis',
  Kurtis => 'Curtis',
  Ky => 'Hezekiah',
  Kym => 'Kimberly',
  KymberlE => 'Kimberly',
  LR => 'Leroy',
  LaffE => 'Lafayette',
  LainE => 'Elaine',
  Lan => 'Yulan',
  LannE => 'Roland',
  Lanna => 'Eleanor',
  Lanson => 'Alanson',
  LarrE => 'Lawrence',
  Lars => 'Lawrence',
  LaurE => 'Lauryn',
  Laurence => 'Lawrence',
  Lavina => 'Lavinia',
  LawrE => 'Lawrence',
  Lazar => 'Eleazer',
  Lb => 'Littleberry',
  LeafE => 'Relief',
  Leafa => 'Relief',
  Lecta => 'Electra',
  Leet => 'Philetus',
  LeffE => 'Lafayette',
  Lem => 'Lemuel',
  Len => 'Leonard',
  Lenhart => 'Leonard',
  LennE => 'Leonard',
  Leo => 'Leonard',
  Leonora => 'Eleanor',
  LessE => 'Celeste',
  Lester => 'Leslie',
  Lettice => 'Letitia',
  LevE => 'Aleva',
  Leve => 'Aleva',
  Lewis => 'Louis',
  Lexa => 'Alexandria',
  Lias => 'Elias',
  Lib => 'Elizabeth',
  Liba => 'Libuse',
  Lidia => 'Linda',
  Lig => 'Elijah',
  Lige => 'Elijah',
  LilE => 'Lillian',
  Lila => 'Delilah',
  LillE => 'Lillian',
  Lillah => 'Lillian',
  Lineau => 'Leonard',
  Linette => 'Linda',
  Link => 'Lincoln',
  Lise => 'Elizabeth',
  Lish => 'Alice',
  Lissa => 'Melissa',
  Little => 'Littleberry',
  LivE => 'Olivia',
  Livia => 'Olivia',
  Liz => 'Elizabeth',
  Lizabeth => 'Elizabeth',
  LizzE => 'Elizabeth',
  Lloyd => 'Floyd',
  LodE => 'Melody',
  Loenore => 'Leonora',
  Lois => 'Lousie',
  Lola => 'Delores',
  LonnE => 'Lawrence',
  Lonson => 'Alanson',
  Lonzo => 'Alonzo',
  Loren => 'Lorenzo',
  Lorinda => 'Laurinda',
  Lorne => 'Lawrence',
  Lotta => 'Charlotte',
  Louann => 'Luann',
  Louanne => 'Luann',
  Louetta => 'Lousie',
  Louie => 'Louis',
  Louisa => 'Lousie',
  Louvina => 'Lavinia',
  Louvinia => 'Lavinia',
  Luanne => 'Luann',
  Lucas => 'Lucias',
  Lucia => 'Luciana',
  Lucinda => 'Cynthia',
  Lula => 'Luella',
  Lum => 'Columbus',
  Lura => 'Lurana',
  LydE => 'Linda',
  LyddE => 'Linda',
  Lyn => 'Belinda',
  Lynette => 'Linda',
  Mabel => 'Mehitabel',
  Mac => 'Malcolm',
  MaddE => 'Madeline',
  Madeleine => 'Madeline',
  Mae => 'Mary',
  Magdalen => 'Magdelina',
  MaisE => 'Margaret',
  Mal => 'Malcolm',
  MalachE => 'Malcolm',
  MamE => 'Mary',
  Manda => 'Amanda',
  Manuel => 'Emanuel',
  Marcia => 'Mary',
  MargE => 'Margaret',
  Margaret => 'Gretchen',
  Margaretha => 'Margaret',
  Margaretta => 'Margaret',
  Margauerite => 'Margarita',
  Marge => 'Margaret',
  MargerE => 'Margaret',
  Margo => 'Margaret',
  Mariah => 'Mary',
  Marianna => 'Maryanne',
  Marica => 'Mary',
  Marietta => 'Mary',
  Marilyn => 'Mary',
  Marion => 'Mary',
  Maris => 'Demerias',
  MarjorE => 'Margaret',
  Mark => 'Marcus',
  Mart => 'Martha',
  Marv => 'Marvin',
  Maryanna => 'Maryanne',
  Mate => 'Mary',
  Mathilda => 'Matilda',
  Matt => 'Matthew',
  Matthias => 'Matthew',
  MaurE => 'Maurice',
  Maura => 'Mary',
  Maureen => 'Mary',
  Mave => 'Mavine',
  MaverE => 'Mavine',
  May => 'Mary',
  Medora => 'Dorothy',
  Mees => 'Bartholomew',
  Meg => 'Margaret',
  Megan => 'Margaret',
  Mehetabel => 'Mehitabel',
  Mehetable => 'Mehitabel',
  Mehitable => 'Mehitabel',
  Melchizedek => 'Zadock',
  Melia => 'Amelia',
  Mell => 'Mildred',
  Mellia => 'Carmellia',
  Merle => 'Merlin',
  Merlyn => 'Merlin',
  Mert => 'Myrtle',
  Mervyn => 'Marvin',
  Meta => 'Margaret',
  Metta => 'Margaret',
  Meus => 'Bartholomew',
  Mia => 'Marianna',
  Micah => 'Michael',
  Michael => 'Mitchell',
  Mick => 'Michael',
  MiddE => 'Madeline',
  Midge => 'Margaret',
  Mike => 'Michael',
  Mima => 'Jemima',
  Minerva => 'Manerva',
  Mitch => 'Mitchell',
  Mitchell => 'Michael',
  Mock => 'Democrates',
  Moira => 'Mary',
  Moll => 'Mary',
  Mona => 'Ramona',
  Montesque => 'Montgomery',
  MorrE => 'Seymour',
  Morris => 'Maurice',
  Mort => 'Mortimer',
  Mose => 'Moses',
  Moses => 'Amos',
  Moss => 'Moses',
  MossE => 'Maurice',
  Mur => 'Muriel',
  Myrt => 'Myrtle',
  MyrtE => 'Myrtle',
  NabbE => 'Abigail',
  Nace => 'Ignatius',
  Nada => 'Nadine',
  Nadia => 'Nadezhda',
  Nana => 'Anna',
  Nance => 'Nancy',
  Nap => 'Napoleon',
  NappE => 'Napoleon',
  Nat => 'Nathaniel',
  Natius => 'Ignatius',
  Neal => 'Cornelius',
  Neil => 'Cornelius',
  Nelia => 'Cornelia',
  Nelle => 'Cornelia',
  Nels => 'Nelson',
  NeppE => 'Penelope',
  NessE => 'Agnes',
  Net => 'Antoinette',
  Netta => 'Antoinette',
  Neva => 'Genevieve',
  Newt => 'Newton',
  Nib => 'Isabella',
  NibbE => 'Isabella',
  NicE => 'Vernisee',
  Nicodemus => 'Nicholas',
  Nicole => 'Nicolena',
  Nicolene => 'Nicolena',
  Nicolina => 'Nicolena',
  Niel => 'Cornelius',
  NikkE => 'Nicolena',
  Nina => 'Enedina',
  Nita => 'Juanita',
  Noah => 'Manoah',
  NollE => 'Olivia',
  NonE => 'Joanna',
  Norah => 'Honora',
  NorbE => 'Norbert',
  NorrE => 'Honora',
  Nowell => 'Noel',
  Odo => 'Odell',
  Olive => 'Olivia',
  Olph => 'Rudolphus',
  OnnE => 'Iona',
  OphE => 'Theophilus',
  Orlando => 'Roland',
  Orolia => 'Caroline',
  Orphelia => 'Ophelia',
  OssE => 'Oswald',
  Oswald => 'Waldo',
  Ote => 'Othello',
  Otis => 'Othello',
  OzzE => 'Oswald',
  PaddE => 'Patrick',
  Pam => 'Pamela',
  Parmelia => 'Amelia',
  ParsunE => 'Parthenia',
  PasoonE => 'Parthenia',
  Patricia => 'Patrick',
  Paula => 'Paulina',
  Pauline => 'Paulina',
  PeddE => 'Experience',
  Peg => 'Margaret',
  PeggE => 'Margaret',
  PenE => 'Philipina',
  PennE => 'Penelope',
  PercE => 'Percival',
  Perce => 'Percival',
  Permelia => 'Amelia',
  PerrE => 'Pelegrine',
  Pete => 'Peter',
  Peter => 'Patrick',
  Phelia => 'Ophelia',
  Phena => 'Tryphena',
  Pheobe => 'Tryphena',
  Pherbia => 'Pharaba',
  Pheriba => 'Pharaba',
  Phililpa => 'Philipina',
  PhillE => 'Adelphia',
  Phineas => 'Alphinias',
  Phoebe => 'Philipina',
  PhoenE => 'Tryphena',
  PhosE => 'Tryphosia',
  Phyllis => 'Philinda',
  Pip => 'Philip',
  Pleas => 'Pleasant',
  Ples => 'Pleasant',
  PokE => 'Pocahontas',
  PonE => 'Napoleon',
  Pres => 'Prescott',
  PrissE => 'Priscilla',
  Pru => 'Prudence',
  PrudE => 'Prudence',
  Prue => 'Prudence',
  Quil => 'Aquilla',
  QuillE => 'Aquilla',
  Quilla => 'Aquilla',
  Quinn => 'Quince',
  Quint => 'Quince',
  Rae => 'Rachel',
  Raech => 'Rachel',
  Raff => 'Raphael',
  Ralph => 'Raphael',
  Rana => 'Lorraine',
  Randall => 'Randolph',
  Reba => 'Rebecca',
  Refina => 'Rufina',
  Reg => 'Reginald',
  Renaldo => 'Reginald',
  Renius => 'Cyrenius',
  Reynold => 'Reginald',
  RhodE => 'Rhodella',
  Rhoda => 'Rhodella',
  Rian => 'Brian',
  Ricka => 'Frederica',
  Rob => 'Robert',
  Robin => 'Robert',
  RoddE => 'Rodney',
  Rodger => 'Roger',
  Roge => 'Roger',
  Roland => 'Orlando',
  Rolf => 'Rudolphus',
  RollE => 'Roland',
  Ronna => 'Veronica',
  Rosabella => 'Isabella',
  Rosable => 'Rosabella',
  Rosalinda => 'Rosalyn',
  Roseann => 'Roseanne',
  Roseanna => 'Roseanne',
  Rowland => 'Roland',
  Rox => 'Roseanne',
  RoxE => 'Roseanne',
  Roxane => 'Roseanne',
  Roxanna => 'Roseanne',
  Roxanne => 'Roseanne',
  Roy => 'Leroy',
  RubE => 'Reuben',
  Rube => 'Reuben',
  RudE => 'Rudolphus',
  Rudolph => 'Rudolphus',
  Rupert => 'Robert',
  Russ => 'Russell',
  RustE => 'Russell',
  Ryan => 'Brian',
  Rye => 'Zachariah',
  Sabe => 'Isabella',
  Sabra => 'Isabella',
  SadE => 'Sarah',
  SallE => 'Sarah',
  Salmon => 'Solomon',
  Samantha => 'Samuel',
  SammE => 'Samuel',
  Samson => 'Sampson',
  Sandra => 'Alexandria',
  Sara => 'Sarah',
  Saul => 'Solomon',
  Scott => 'Prescott',
  ScottE => 'Prescott',
  Sean => 'Shaun',
  SenE => 'Eseneth',
  Sene => 'Asenath',
  Serena => 'Sabrina',
  Serene => 'Cyrenius',
  Seymore => 'Seymour',
  Shane => 'Shaun',
  Shar => 'Sharon',
  SharE => 'Sharon',
  SharrE => 'Sharon',
  Sharyn => 'Sharon',
  Shawn => 'Shaun',
  Shayne => 'Shaun',
  Shel => 'Sheldon',
  Shelton => 'Sheldon',
  Sher => 'Sheridan',
  SherE => 'Sharon',
  Sheron => 'Sharon',
  Sheryl => 'Sharon',
  Sheryn => 'Sharon',
  Shirl => 'Shirley',
  SibbE => 'Sibbilla',
  Sibbell => 'Sibbilla',
  Sibyl => 'Sibbilla',
  Sid => 'Sidney',
  Sig => 'Sigismund',
  Sigmund => 'Sigismund',
  Silver => 'Sylvester',
  Silvester => 'Sylvester',
  Sim => 'Simeon',
  Simon => 'Simeon',
  Sion => 'Simeon',
  Sis => 'Frances',
  SlE => 'Sylvester',
  SmittE => 'Smith',
  Sol => 'Solomon',
  SollE => 'Solomon',
  SophE => 'Sophronia',
  Sophia => 'Sophronia',
  Stacia => 'Eustacia',
  Stella => 'Estella',
  Steph => 'Stephen',
  Steve => 'Stephen',
  Steven => 'Stephen',
  SuchE => 'Susannah',
  Sue => 'Susannah',
  SullE => 'Sullivan',
  SurrE => 'Sarah',
  SusE => 'Susannah',
  Susan => 'Susannah',
  Susannah => 'Hannah',
  SuzE => 'Susannah',
  Suzanne => 'Susannah',
  Swene => 'Cyrenius',
  Sy => 'Sylvester',
  Sybrina => 'Sabrina',
  Syd => 'Sidney',
  SydnE => 'Sidney',
  Syl => 'Sylvester',
  Sylvanus => 'Sylvester',
  TabbE => 'Tabitha',
  TallE => 'Natalie',
  Tamarra => 'Tamara',
  TammE => 'Tamara',
  Tamzine => 'Thomasine',
  Tash => 'Tasha',
  TashE => 'Tasha',
  Tave => 'Octavia',
  Tavia => 'Octavia',
  TellE => 'Aristotle',
  Temera => 'Tamara',
  TempE => 'Temperance',
  TennE => 'Tennessee',
  TensE => 'Hortense',
  TentE => 'Content',
  Terence => 'Terrance',
  Teresa => 'Theresa',
  TessE => 'Theresa',
  Tessa => 'Theresa',
  Thad => 'Thaddeus',
  Than => 'Nathaniel',
  Thea => 'Althea',
  ThenE => 'Parthenia',
  Theodora => 'Theodosia',
  Theodorick => 'Theodore',
  Theodrick => 'Theodore',
  Therese => 'Theresa',
  Thias => 'Matthew',
  Thirsa => 'Theresa',
  Thom => 'Thomas',
  Thomasa => 'Thomasine',
  Thriza => 'Theresa',
  Thursa => 'Theresa',
  Thys => 'Matthew',
  Tiah => 'Azariah',
  TibbE => 'Isabella',
  TicE => 'Theresa',
  Tick => 'Felicity',
  Tilda => 'Matilda',
  Tim => 'Timothy',
  TimmE => 'Timothy',
  TippE => 'Tipton',
  TipsE => 'Tipton',
  Titia => 'Letitia',
  TobE => 'Tobias',
  Tobe => 'Tobias',
  Tom => 'Thomas',
  TommE => 'Thomas',
  TorE => 'Victoria',
  TorrE => 'Victoria',
  TracE => 'Theresa',
  Tricia => 'Patricia',
  Trina => 'Katherine',
  Trish => 'Patricia',
  Trisha => 'Beatrice',
  Trix => 'Beatrice',
  TrudE => 'Gertrude',
  Val => 'Valentina',
  Valeda => 'Valentina',
  ValerE => 'Valentina',
  VallE => 'Valentina',
  VangE => 'Evangeline',
  VannE => 'Vanessa',
  Vanna => 'Vanessa',
  VergE => 'Virginia',
  Verna => 'Laverne',
  VessE => 'Sylvester',
  Vest => 'Sylvester',
  Vester => 'Sylvester',
  Vet => 'Sylvester',
  Vic => 'Victor',
  VicE => 'Lewvisa',
  Vick => 'Victor',
  VinE => 'Lavinia',
  Vince => 'Vincent',
  Vinson => 'Vincent',
  Viola => 'Violet',
  Virg => 'Virgil',
  VirgE => 'Virginia',
  Virginia => 'Jane',
  Volodia => 'Vladimir',
  VonnE => 'Veronica',
  WaitE => 'Waitstill',
  Waldo => 'Oswald',
  WallE => 'Wallace',
  Walt => 'Walter',
  Wat => 'Walter',
  Webb => 'Webster',
  WendE => 'Gwendolyn',
  Wenefred => 'Winifred',
  Wes => 'Wesley',
  Wib => 'Wilber',
  WilE => 'William',
  Wilber => 'Gilbert',
  Wilbur => 'Wilber',
  Wilhelm => 'William',
  Willis => 'William',
  Wilma => 'Wilhelmina',
  Winifred => 'Jennifer',
  Winnet => 'Winifred',
  Wint => 'Winton',
  Wood => 'Woodrow',
  Wyncha => 'Lavinia',
  Xan => 'Alexandria',
  Xena => 'Christiana',
  Xina => 'Christiana',
  Yolonda => 'Yolanda',
  Yul => 'Yulan',
  Zach => 'Zachariah',
  ZachE => 'Zachariah',
  ZacharE => 'Zachariah',
  Zacharias => 'Zachariah',
  ZadE => 'Isaiah',
  ZaddE => 'Arzada',
  Zadock => 'Melchizedek',
  Zay => 'Isaiah',
  Zel => 'Zelphia',
  Zella => 'Zelphia',
  ZelphE => 'Zelphia',
  Zeph => 'Zepaniah',
  ZollE => 'Solomon',
  Zubiah => 'Azubah',
);

%multi=
(
  Ab => [qw<Abel Abiel Abigail Abijah Abijah Abner Absalom>],
  AbbE => [qw<Abigail Abner Absalom>],
  Abe => [qw<Abel Abraham>],
  Ad => [qw<Adam Adelbert Rudolphus>],
  AddE => [qw<Adaline Adelaide Adelphia Agatha>],
  Ade => [qw<Adam Adelbert>],
  Adela => [qw<Adaline Adelaide>],
  Adele => [qw<Adelaide Adelphia>],
  AggE => [qw<Agatha Agnes Augusta>],
  Agnes => [qw<Agatha Nancy>],
  Al => [qw<Alanson Albert Alderick Alexander Alfred Alonzo>],
  Alex => [qw<Alexander Alexandria>],
  AlfE => [qw<Alfred Alfreda>],
  AllE => [qw<Aileen Alberta Alice Almena>],
  AmE => [qw<Amalia Amelia>],
  Ance => [qw<Anderson Anselm>],
  AndE => [qw<Alexander Anderson Andrea Andrew>],
  Ann => [qw<Agnes Anna Antoinette Luann Marianna Maryanne Nancy Roseanne>],
  AnnE => [qw<Anna Luann Maryanne>],
  Anne => [qw<Anna Luann Marianna Maryanne>],
  ArrE => [qw<Arabella Armena>],
  August => [qw<Augusta Augustus>],
  Bar => [qw<Barbara Barry>],
  Bea => [qw<Beatrice Blanche>],
  Bell => [qw<Arabella Belinda William>],
  Bella => [qw<Arabella Isabella Mehitabel>],
  Belle => [qw<Arabella Belinda Isabella Rosabella>],
  Ben => [qw<Benedict Benjamin>],
  BennE => [qw<Benedict Benjamin>],
  BerrE => [qw<Barry Greenberry Littleberry>],
  Bert => [qw<Adelbert Albert Alberta Bertha Bertram Delbert Egbert Elbertson Gilbert Herbert Hubert Norbert Roberta>],
  BertE => [qw<Alberta Bertha Roberta>],
  BiddE => [qw<Bridget Obedience>],
  BobbE => [qw<Barbara Robert Roberta>],
  BridE => [qw<Bertha Bridget>],
  Cal => [qw<Caleb Calvin>],
  CallE => [qw<Caldonia California Calpurnia Caroline>],
  Cam => [qw<Cameron Campbell>],
  CarrE => [qw<Caroline Karonhappuck>],
  Cass => [qw<Alexandria Caswell>],
  CassE => [qw<Alexandria Caroline Katherine>],
  Celia => [qw<Cecilia Celeste>],
  Chris => [qw<Christian Christiana Christopher>],
  Cilla => [qw<Cecilia Priscilla>],
  CindE => [qw<Cynthia Luciana Lucinda>],
  CissE => [qw<Cecilia Clarissa Frances Priscilla>],
  Clair => [qw<Clarence Clarissa>],
  Clare => [qw<Clarence Clarissa>],
  Clem => [qw<Clement Clementine>],
  Cliff => [qw<Clifford Clifton>],
  ConnE => [qw<Conrad Constance Cornelius>],
  Curt => [qw<Courtney Curtis>],
  Cy => [qw<Cyrenius Cyrus>],
  Dan => [qw<Daniel Sheridan>],
  DannE => [qw<Daniel Sheridan>],
  Dee => [qw<Audrey Delores Dorothy>],
  DeeDee => [qw<Deidre Nadine>],
  Del => [qw<Adelbert Delbert>],
  Delia => [qw<Adaline Bedelia Bridget Cordelia Fidelia>],
  Dell => [qw<Adaline Adelaide Adelphia Delilah Delores Rhodella>],
  Della => [qw<Adelaide Delilah Deliverance Delores Rhodella>],
  Diah => [qw<Jedediah Obadiah Zedediah>],
  DicE => [qw<Diana Edith Eurydice Laodicia>],
  Dick => [qw<Melchizedek Richard Zadock>],
  DodE => [qw<Delores Dorothy>],
  Dolph => [qw<Randolph Rudolphus>],
  Dora => [qw<Dorothy Eudora Isidora Pandora Theodosia>],
  Drew => [qw<Andrew Woodrow>],
  Dyer => [qw<Jedediah Obadiah Zedediah>],
  Eb => [qw<Abel Ebenezer>],
  EbbE => [qw<Abel Ebenezer>],
  Ed => [qw<Edmund Edward Edwin>],
  EdE => [qw<Adam Edith>],
  Eileen => [qw<Aileen Helena>],
  ElE => [qw<Alice Elias Elijah>],
  Elaine => [qw<Eleanor Helena>],
  Elbert => [qw<Adelbert Albert Alberta Elbertson>],
  Eliza => [qw<Elizabeth Lousie>],
  Elizabeth => [qw<Elsie Isabella>],
  EllE => [qw<Alexandria Eleanor Elmira Elwood>],
  Ella => [qw<Eleanor Gabrielle Helena Luella>],
  Ellen => [qw<Eleanor Helena>],
  ElsE => [qw<Alice Elizabeth>],
  EmilE => [qw<Amelia Emeline>],
  EssE => [qw<Estella Hester>],
  EttE => [qw<Carthaette Henrietta>],
  Etta => [qw<Carthaette Henrietta Loretta>],
  Eve => [qw<Evaline Evelyn Genevieve>],
  Ez => [qw<Ezekiel Ezra>],
  Fina => [qw<Josephine Rufina>],
  Ford => [qw<Bradford Clifford Tilford>],
  FrankE => [qw<Francis Veronica>],
  Fred => [qw<Alfred Frederick Wilfred Winifred>],
  Freda => [qw<Alfreda Frederica>],
  FreddE => [qw<Alfred Alfreda Frederica Frederick Winifred>],
  Frieda => [qw<Alfreda Frederica>],
  FronE => [qw<Sophronia Veronica>],
  GabbE => [qw<Gabriel Gabrielle>],
  Gee => [qw<Georgia Jehu>],
  Gen => [qw<Genevieve Virginia>],
  Gene => [qw<Eugene Eugenia>],
  GerrE => [qw<Gerald Geraldine>],
  Gina => [qw<Regina Virginia>],
  Gus => [qw<Augusta Augustus>],
  GussE => [qw<Augusta Augustus>],
  Hal => [qw<Harold Henry Howard>],
  Hank => [qw<Harold Henrietta Henry>],
  Hannah => [qw<Anna Joanna Susannah>],
  HarrE => [qw<Harold Henry>],
  Helen => [qw<Aileen Elaine Eleanor Helena>],
  Heloise => [qw<Lois Lousie>],
  HettE => [qw<Henrietta Hester Mehitabel>],
  Hugh => [qw<Hubert Jehu>],
  Hy => [qw<Hezekiah Hiram>],
  IssE => [qw<Isabella Isidora>],
  JackE => [qw<Jacqueline John>],
  JamE => [qw<Benjamin James>],
  Jan => [qw<Jane John>],
  Jane => [qw<Angela Joanna Virginia>],
  Jean => [qw<Genevieve Jane Joanna John>],
  JennE => [qw<Eugenia Genevieve Jane Jennifer Virginia>],
  JerrE => [qw<Gerald Geraldine Jeremiah>],
  JessE => [qw<Jane Jennet Jessica>],
  JinsE => [qw<Genevieve Jane>],
  Joan => [qw<Jane Joanna>],
  JodE => [qw<Joanna Joseph Josephine>],
  Joey => [qw<Joseph Josephine>],
  JohnnE => [qw<John Jonathan>],
  Jonathan => [qw<Jonathan Nathaniel>],
  Jos => [qw<Joseph Joshua Josiah>],
  JudE => [qw<Judah Judith>],
  Karen => [qw<Karonhappuck Katherine>],
  Kit => [qw<Christian Christopher Katherine>],
  Laura => [qw<Laurinda Loretta>],
  Lee => [qw<Elias Leonora Leroy Levi Shirley>],
  Left => [qw<Eliphalet Lafayette>],
  Lena => [qw<Adaline Aileen Angela Arlene Caroline Darlene Evaline Helena Katherine Madeline Magdelina Selina>],
  Leon => [qw<Leonard Leonidas Lionel Napoleon>],
  Les => [qw<Leslie Lester>],
  LettE => [qw<Charlotte Letitia>],
  LibbE => [qw<Elizabeth Libuse>],
  LiddE => [qw<Elizabeth Linda>],
  Lil => [qw<Delilah Lillian>],
  Lina => [qw<Emeline Paulina>],
  LindE => [qw<Celinda Linda Lyndon Melinda>],
  Linda => [qw<Belinda Celinda Melinda Philinda Rosalyn>],
  Lisa => [qw<Alice Elizabeth Melissa>],
  Liza => [qw<Adelaide Elizabeth>],
  LollE => [qw<Charlotte Delores Lillian>],
  Lon => [qw<Alonzo Lawrence Zebulon>],
  LorrE => [qw<Lauryn Lawrence Loretta Lorraine>],
  LottE => [qw<Carlotta Charlotte Lotta>],
  Lou => [qw<Luann Louis Lousie Lucille Lucinda Marylou>],
  Louise => [qw<Lois Lousie>],
  Lu => [qw<Lucille Lucinda Luella>],
  LucE => [qw<Luciana Lucille Lucinda>],
  Luke => [qw<Lucias Luthor>],
  Lulu => [qw<Luann Lousie Marylou>],
  LyndE => [qw<Linda Lyndon>],
  Lynn => [qw<Caroline Celinda Linda Lyndon>],
  Lynne => [qw<Belinda Melinda>],
  Madge => [qw<Madeline Magdelina Margaret>],
  Magda => [qw<Madeline Magdelina>],
  MaggE => [qw<Madeline Margaret>],
  Maida => [qw<Madeline Magdelina>],
  MandE => [qw<Amanda Miranda>],
  MannE => [qw<Emanuel Manuel>],
  MarE => [qw<Marianna Marylou Mary Maryanne Maureen Miriam Mitzi>],
  Maria => [qw<Marianna Mary>],
  Marian => [qw<Marianna Mary>],
  Marianne => [qw<Marianna Maryanne>],
  MartE => [qw<Martha Martin>],
  Mat => [qw<Martha Matilda>],
  MattE => [qw<Martha Matilda>],
  Maud => [qw<Madeline Matilda>],
  Max => [qw<Maximillian Maxine>],
  Mel => [qw<Amelia Melinda Melissa>],
  MellE => [qw<Amelia Melinda Permelia>],
  Mena => [qw<Almena Armena Philomena>],
  MercE => [qw<Mary Mercedes>],
  Merv => [qw<Marvin Mervyn>],
  MickE => [qw<Michael Michelle>],
  MillE => [qw<Amelia Armilda Camille Emeline Melissa Mildred Millicent Permelia>],
  MimE => [qw<Jemima Mary Mildred Miriam>],
  Mina => [qw<Mindwell Minerva Wilhelmina>],
  MindE => [qw<Arminda Melinda>],
  MinnE => [qw<Almina Mary Minerva Wilhelmina>],
  Mira => [qw<Elmira Miranda>],
  MissE => [qw<Melissa Millicent>],
  MittE => [qw<Mehitabel Submit>],
  MitzE => [qw<Mary Miriam>],
  MollE => [qw<Amalia Amelia Martha Mary>],
  MontE => [qw<Lamont Montgomery>],
  MorE => [qw<Maurice Seymour>],
  Myra => [qw<Almira Elmira>],
  Naldo => [qw<Reginald Roland>],
  Nan => [qw<Anna Hannah Nancy>],
  NancE => [qw<Agnes Anna>],
  NannE => [qw<Anna Hannah Nancy>],
  Nate => [qw<Ignatius Nathaniel>],
  Nathan => [qw<Jonathan Nathaniel>],
  NattE => [qw<Asenath Natalie Nathaniel>],
  Ned => [qw<Edmund Edward Edwin>],
  NeelE => [qw<Cornelia Cornelius>],
  Nell => [qw<Eleanor Helena>],
  NellE => [qw<Cornelia Eleanor Helena>],
  NervE => [qw<Manerva Minerva>],
  Nerva => [qw<Manerva Minerva>],
  Nessa => [qw<Agnes Vanessa>],
  NettE => [qw<Antoinette Henrietta Jane Juanita Natalie Ninell Pernetta>],
  Nick => [qw<Dominic Nicholas>],
  NickE => [qw<Nicholas Nicolena>],
  Nora => [qw<Eleanor Honora Leonora>],
  ObE => [qw<Obadiah Oberon>],
  Obe => [qw<Obadiah Oberon>],
  Obed => [qw<Obadiah Obedience>],
  OllE => [qw<Oliver Olivia>],
  Ora => [qw<Aurelia Aurilla>],
  Orilla => [qw<Aurelia Aurilla>],
  Pat => [qw<Patience Patricia Patrick>],
  Pate => [qw<Patrick Peter>],
  PatsE => [qw<Martha Patricia Patrick>],
  PattE => [qw<Martha Matilda Parthenia Patience Patricia>],
  PhenE => [qw<Josephine Parthenia Tryphena>],
  Phidelia => [qw<Bedelia Fidelia>],
  Phil => [qw<Philetus Philip>],
  PollE => [qw<Mary Paulina>],
  Puss => [qw<Philadelphia Prudence>],
  RandE => [qw<Miranda Randolph>],
  Ray => [qw<Rachel Raymond Regina>],
  ReenE => [qw<Irene Maureen Sabrina>],
  ReggE => [qw<Regina Reginald>],
  Rena => [qw<Irene Maureen Sabrina>],
  RennE => [qw<Irene Reginald>],
  Retta => [qw<Henrietta Loretta Loretta>],
  Riah => [qw<Azariah Uriah>],
  Rich => [qw<Alderick Richard>],
  RichE => [qw<Alderick Derrick Richard>],
  Rick => [qw<Cedrick Derrick Eric Richard>],
  RickE => [qw<Broderick Cedrick Eric Richard>],
  RillE => [qw<Aurelia Aurilla>],
  Rilla => [qw<Avarilla Gabrilla Serilla>],
  Rita => [qw<Jerita Margaret Margarita>],
  RobbE => [qw<Robert Roberta>],
  Rod => [qw<Rodney Roger>],
  Rollo => [qw<Roland Rudolphus>],
  Ron => [qw<Aaron Cameron Roland Veronica>],
  RonnE => [qw<Aaron Cameron Roland Veronica>],
  RosE => [qw<Euphrosina Roseanne>],
  Rosa => [qw<Rosabella Rosalyn>],
  Rose => [qw<Rosabella Rosalyn Roseanne>],
  Roz => [qw<Rosabella Rosalyn Roseanne>],
  Sal => [qw<Salvador Sarah Solomon>],
  Sam => [qw<Sampson Samuel>],
  SandE => [qw<Alexander Alexandria>],
  Sasha => [qw<Alexander Alexandria>],
  ShellE => [qw<Michelle Rachel Sheldon>],
  SherrE => [qw<Charlotte Shirley>],
  Si => [qw<Cyrus Josiah Silas Simeon Sylvester>],
  Silla => [qw<Drusilla Priscilla>],
  SonnE => [qw<Anderson Jefferson Judson>],
  StacE => [qw<Anastasia Eustacia>],
  SukE => [qw<Sarah Susannah>],
  Susanna => [qw<Anna Susannah>],
  Tad => [qw<Thaddeus Theodore>],
  Ted => [qw<Edmund Edward Theodore>],
  TeddE => [qw<Edward Theodore>],
  TerrE => [qw<Terrance Theresa>],
  Tess => [qw<Elizabeth Theresa>],
  Theo => [qw<Theodore Theodosia Theophilus>],
  TillE => [qw<Matilda Temperance Tilford>],
  Tina => [qw<Augusta Christiana Ernestine Katherine>],
  Tish => [qw<Letitia Patricia>],
  TonE => [qw<Anthony Antoinette Clifton Sheldon>],
  TrissE => [qw<Beatrice Theresa>],
  TrixE => [qw<Beatrice Patricia>],
  Van => [qw<Sullivan Vanessa>],
  Vi => [qw<Violet Vivian>],
  VickE => [qw<Veronica Victoria>],
  Vin => [qw<Calvin Vincent>],
  Vina => [qw<Lavinia Melvina>],
  VinnE => [qw<Calvin Lavinia Vincent>],
  Will => [qw<Wilber Wilfred William>],
  WillE => [qw<Wilda Wilfred Wilhelmina William>],
  Win => [qw<Edwin Winfield>],
  WinnE => [qw<Edwina Winfield Winifred>],
  WoodE => [qw<Elwood Woodrow>],
  Zeb => [qw<Zebedee Zebulon>],
  Zed => [qw<Zadock Zedediah>],
  Zeke => [qw<Ezekiel Isaac Zachariah>],
);

%match=
(
  Aaron => qr/^((Aaron|Erin|Ron(n$E)?))$/, # Ron Ronn$E Erin
  Abel => qr/^((Ab(el?)?|Eb(b$E)?))$/, # Abe Ab Eb Ebb$E
  Abiel => qr/^((Ab(iel)?|Biel))$/, # Biel Ab
  Abigail => qr/^((Ab(b$E|igail)?|Gail|Nabb$E))$/, # Ab Abb$E Gail Nabb$E
  Abijah => qr/^((Ab(i(ah|jah))?|Bi(ah|ge)))$/, # Ab Bige Abiah Ab Biah
  Abner => qr/^(Ab(b$E|ner)?)$/, # Ab Abb$E
  Abraham => qr/^(Ab(e|ra(ham|m)))$/, # Abe Abram
  Absalom => qr/^(Ab(b$E|salom)?)$/, # Ab Abb$E
  Adaline => qr/^((A(d(a(line)?|d$E|ela)|line)|Del(ia|l)|Edith|Lena))$/, # Ada Adela Aline Edith Add$E Dell Delia Lena
  Adam => qr/^((Ad(am|e)?|Ed$E))$/, # Ade Ed$E Ad
  Adelaide => qr/^((Ad(aline|d$E|el(a(ide)?|e|i(a|ne|za)))|Della?|Heid($E|e)|Liza))$/, # Add$E Adela Adaline Adeline Della Heide Adele Dell Heid$E Adeliza Liza Adelia
  Adelbert => qr/^((A(d(e(lbert)?)?|lbert)|Bert|Del(bert)?|Elbert))$/, # Albert Del Delbert Bert Ad Ade Elbert
  Adelphia => qr/^((Ad(d$E|el(e|phia))|Del(l|phina)|Phill$E))$/, # Adele Add$E Dell Delphina Phill$E
  Agatha => qr/^(A(dd$E|g(atha|g$E|nes)?))$/, # Add$E Ag Agg$E Agnes
  Agnes => qr/^((A(g(g$E|nes)|nn)|Inez|N(anc$E|ess($E|a))))$/, # Agg$E Ann Nanc$E Ness$E Inez Nessa
  Aileen => qr/^((A(ileen|ll$E)|Eileen|Helen|Lena))$/, # Helen All$E Lena Eileen
  Alanson => qr/^((Al(an(son)?)?|L(anson|onson)))$/, # Al Lanson Alan Lonson
  Albert => qr/^((Al(bert)?|B(ert|ird)|Elbert))$/, # Al Bert Bird Elbert
  Alberta => qr/^((A(bertina|l(bert(a|ine)|l$E))|Bert($E)?|Elbert))$/, # Abertina Albertine All$E Bert Bert$E Elbert
  Alderick => qr/^((Al(derick)?|Rich($E)?))$/, # Al Rich Rich$E
  Aleva => qr/^((Aleva|Lev($E|e)))$/, # Leve Lev$E
  Alexander => qr/^((A(l(e(c|x(ander)?))?|nd$E)|E(c|leck)|Sa(nd$E|sha)))$/, # Alec Alex And$E Ec Sand$E Al Eleck Sasha
  Alexandria => qr/^((Al(ex(andr(a|ia))?|la)|Cass($E|andra)?|El(ic|l$E)|Lexa|Sa(nd($E|ra)|sha)|Xan))$/, # Alla Ell$E Elic Sandra Alexandra Lexa Xan Alex Sand$E Sasha Cassandra Cass Cass$E
  Alfred => qr/^((Al(f($E|red)?)?|Fred(d$E)?))$/, # Al Alf Alf$E Fred Fredd$E
  Alfreda => qr/^((Alf($E|reda)|Fr(ed(a|d$E)|ieda)))$/, # Alf$E Frieda Fredd$E Freda
  Alice => qr/^((Al($E|i(c(e|ia)|s(ha|on))|l$E)|El($E|i(cia|sha)|lis|s$E)|Lis[ah]))$/, # Alicia Elicia Els$E All$E Lisa Alison Al$E El$E Alisha Elisha Lish Ellis
  Almena => qr/^((Al(l$E|mena)|Mena))$/, # All$E Mena
  Almina => qr/^((Almina|Minn$E))$/, # Minn$E
  Almira => qr/^((Almira|Myra))$/, # Myra
  Alonzo => qr/^((Al(onzo|phonzo)?|Lon(zo)?))$/, # Al Alphonzo Lon Lonzo
  Alphinias => qr/^((A(lph(inias|us)|phinius)|Finnius|Phineas))$/, # Alphus Aphinius Phineas Finnius
  Althea => qr/^((Althea|Thea))$/, # Thea
  Amalia => qr/^((Am($E|alia)|Moll$E))$/, # Am$E Moll$E
  Amanda => qr/^((A(manda|rmanda)|Mand($E|a)))$/, # Armanda Manda Mand$E
  Amelia => qr/^((Am($E|elia)|Emil$E|M(el(ia|l$E)?|ill$E|oll$E)|P(armelia|ermelia)))$/, # Am$E Emil$E Mel Melia Mill$E Moll$E Parmelia Mell$E Permelia
  Amos => qr/^((Amos|Moses))$/, # Moses
  Anastasia => qr/^((Ana(stasia)?|Stac$E))$/, # Ana Stac$E
  Anderson => qr/^((An(ce|d($E|er(son)?|re))|Sonn$E))$/, # Ander And$E Sonn$E Ance Andre
  Andrea => qr/^(And($E|r(ea|ia)))$/, # Andria And$E
  Andrew => qr/^((And($E|rew)|Drew))$/, # And$E Drew
  Angela => qr/^((Ang($E|el(a|i(ca|n[ae])))|Jane|Lena))$/, # Angelica Angelina Angeline Ang$E Jane Lena
  Anna => qr/^((Ann($E|a|e(tte)?)?|Hannah|Nan(a|c$E|n$E)?|Susanna))$/, # Ann Anne Ann$E Hannah Nan Nann$E Nanc$E Nana Susanna Annette
  Anselm => qr/^(An(ce|se(lm?)?))$/, # Anse Ance Ansel
  Anthony => qr/^((Ant(hony|o(ine|nio))|Ton$E))$/, # Ton$E Antonio Antoine
  Antoinette => qr/^((An(n|to(inette|nia))|Net(t($E|a))?|Ton$E))$/, # Antonia Net Nett$E Ann Ton$E Netta
  Aphrodite => qr/^((Aphrodite|D(it(e|us)|yc(e|he))|Ep(aphrodit(ius|us)|pa)))$/, # Dite Ditus Dyce Dyche Eppa Epaphroditus Epaphroditius
  Appoline => qr/^(App($E|oline))$/, # App$E
  Aquilla => qr/^((A(cuilla|quilla)|Quil(l($E|a))?))$/, # Quil Quill$E Acuilla Quilla
  Arabella => qr/^((A(nabelle|r(a(bell[ae])?|r$E))|Bell[ae]?))$/, # Ara Bella Belle Arabelle Anabelle Bell Arr$E
  Archibald => qr/^((Arch($E|elous|ibald)?|Bald($E|o)))$/, # Arch$E Bald$E Baldo Arch Archelous
  Ardeshir => qr/^(Ard($E|eshir))$/, # Ard$E
  Arielle => qr/^(Ar($E|ielle))$/, # Ar$E
  Aristotle => qr/^((Aristotle|Tell$E))$/, # Tell$E
  Arlene => qr/^((Arl($E|ene)|Lena))$/, # Lena Arl$E
  Armena => qr/^((Ar(mena|r$E)|Mena))$/, # Arr$E Mena
  Armilda => qr/^((Armilda|Mill$E))$/, # Mill$E
  Arminda => qr/^((Arminda|Mind$E))$/, # Mind$E
  Arnold => qr/^(Arn($E|old))$/, # Arn$E
  Arthur => qr/^(Art(hur)?)$/, # Art
  Arzada => qr/^((Arzada|Zadd$E))$/, # Zadd$E
  Asaph => qr/^(Asa(hel|ph)?)$/, # Asa Asahel
  Asenath => qr/^((As(enath|sene)|Natt$E|Sene))$/, # Assene Sene Natt$E
  Aubrey => qr/^((Aubrey|Bree))$/, # Bree
  Audrey => qr/^((Audrey|Dee))$/, # Dee
  Augusta => qr/^((A(gg$E|ug($E|ust(a|ina)?))|Gus($E|s$E)?|Tina))$/, # Augustina Agg$E Aug$E Gus$E Guss$E Tina Gus August Augusta
  Augustus => qr/^((August(ine|us)?|G(ats($E|b$E)|us(s$E|tus)?)))$/, # Augustine Gus Gustus Gats$E Gatsb$E Guss$E August
  Aurelia => qr/^((Aur(elia|illa)|Or(a|illa)|Rill$E))$/, # Aurilla Orilla Ora Rill$E
  Aurilla => qr/^((Aurilla|Or(a|illa)|Rill$E))$/, # Ora Rill$E Orilla
  Avarilla => qr/^((Avarilla|Rilla))$/, # Rilla
  Azariah => qr/^((Az(ari(ah|ch)|e)|Riah|Tiah))$/, # Aze Riah Azarich Tiah
  Azubah => qr/^((Azubah|Zubiah))$/, # Zubiah
  Barbara => qr/^(B(a(b(b$E|s)?|r(b($E|ar($E|a)|er$E)?)?)|obb$E))$/, # Babs Barb Barb$E Bobb$E Bab Babb$E Barbar$E Barber$E Bar
  Barnabas => qr/^(B(arn($E|a(bas|rd))|ern($E|ard)))$/, # Barn$E Barnard Bernard Bern$E
  Barry => qr/^(B(ar(ry)?|e(ar|rr$E)))$/, # Berr$E Bar Bear
  Bartholomew => qr/^((Ba(rt(el|h(olomew)?)?|t)|Me(es|us)))$/, # Bart Bartel Barth Bat Mees Meus
  Bazaleel => qr/^(Ba(sil|zaleel))$/, # Basil
  Beatrice => qr/^((Bea(trice)?|Tri(s(ha|s$E)|x($E)?)))$/, # Bea Trisha Triss$E Trix$E Trix
  Bedelia => qr/^((B(edelia|ridgit)|Delia|Fidelia|Phidelia))$/, # Delia Bridgit Fidelia Phidelia
  Belinda => qr/^((Bel(inda|le?)|L(inda|yn(ne)?)))$/, # Bell Belle Linda Lynne Lyn
  Benedict => qr/^(Ben(edict|n($E|ett))?)$/, # Bennett Ben Benn$E
  Benjamin => qr/^((Ben(j($E|amin)|n$E)?|Jam$E))$/, # Ben Benj$E Benn$E Jam$E
  Bertha => qr/^(B(ert($E|ha)?|rid$E))$/, # Bert$E Brid$E Bert
  Bertram => qr/^(Bert(ram)?)$/, # Bert
  Beverly => qr/^(Bev(erl($E|y))?)$/, # Beverl$E Bev
  Blanche => qr/^(B(ea|lanche))$/, # Bea
  Boetius => qr/^(Bo(etius)?)$/, # Bo
  Bonita => qr/^(Bon(ita|n$E))$/, # Bonn$E
  Bradford => qr/^((Brad(ford|l$E)?|Ford))$/, # Brad Ford Bradl$E
  Brian => qr/^((Br(ian|yant?)|R(ian|yan)))$/, # Bryan Bryant Ryan Rian
  Bridget => qr/^((B(idd$E|r($E|id($E|g($E|et))))|Delia))$/, # Bidd$E Brid$E Br$E Delia Bridg$E
  Broderick => qr/^((Br(ad$E|od($E|erick))|Rick$E))$/, # Rick$E Brad$E Brod$E
  Caldonia => qr/^((Cal(donia|l$E)|Dona))$/, # Call$E Dona
  Caleb => qr/^(Cal(eb)?)$/, # Cal
  California => qr/^(Cal(ifornia|l$E))$/, # Call$E
  Calista => qr/^((Calista|Kiss$E))$/, # Kiss$E
  Calpurnia => qr/^(Cal(l$E|purnia))$/, # Call$E
  Calvin => qr/^((Cal(vin)?|Vin(n$E)?))$/, # Cal Vin Vinn$E
  Cameron => qr/^((Cam(eron)?|Ron(n$E)?))$/, # Cam Ronn$E Ron
  Camille => qr/^((Cam(ille|m$E)|Mill$E))$/, # Camm$E Mill$E
  Campbell => qr/^(Cam(pbell)?)$/, # Cam
  Candace => qr/^((Cand($E|ace)|Dac$E))$/, # Cand$E Dac$E
  Carlotta => qr/^((Carlotta|Lott$E))$/, # Lott$E
  Carmellia => qr/^((Carmellia|Mellia))$/, # Mellia
  Caroline => qr/^((Ca(dd$E|ll$E|r(ol(e|in[ae]|yn)?|r$E)|ss$E)|K(ar$E|er$E)|L(ena|ynn)|Orolia))$/, # Cadd$E Carol Carole Carr$E Cass$E Lynn Carolyn Call$E Carolina Lena Orolia Ker$E Kar$E
  Carthaette => qr/^((Carthaette|Ett($E|a)))$/, # Etta Ett$E
  Casey => qr/^((Casey|K(C|as$E)))$/, # Kas$E KC
  Caswell => qr/^(Cas(s|well))$/, # Cass
  Cecilia => qr/^(C(e(c(el$E|il($E|ia))|lia)|i(lla|ss$E)))$/, # Celia Ciss$E Cecil$E Cilla Cecel$E
  Cedrick => qr/^((Ced(rick)?|Rick($E)?))$/, # Ced Rick Rick$E
  Celeste => qr/^((Cel(este|ia)|Less$E))$/, # Celia Less$E
  Celinda => qr/^((Celinda|L(ind($E|a)|ynn)))$/, # Linda Lind$E Lynn
  Charity => qr/^(Cha(rity|t))$/, # Chat
  Charles => qr/^((Buck|C(arl|h(a(d|rl($E|es))|ick|uck))|Kar(el|l)))$/, # Buck Carl Chad Charl$E Chick Chuck Karl Karel
  Charlotte => qr/^((C(ar(lotta)?|har(lotte)?)|L(ett$E|o(ll$E|tt($E|a)))|Sherr$E))$/, # Car Carlotta Lett$E Lotta Lott$E Char Sherr$E Loll$E
  Chauncy => qr/^(Cha(n|uncy))$/, # Chan
  Chesley => qr/^(Che(sl($E|ey)|t))$/, # Chesl$E Chet
  Chloe => qr/^(C(hloe|lo))$/, # Clo
  Christian => qr/^((Chris(t(ian|opher))?|Kit))$/, # Chris Christopher Kit
  Christiana => qr/^((C(hris(t($E|i(ana|n[ae])))?|ris(s$E|tina)?)|K(erst$E|ris(t($E|ine))?)|Tina|X(ena|ina)))$/, # Christine Chris Criss$E Christ$E Tina Xina Christina Kris Kerst$E Xena Cristina Cris Krist$E Kristine
  Christopher => qr/^((Chris(t(ian|opher))?|K(ester|it)))$/, # Chris Christian Kester Kit
  Clarence => qr/^(Cla(ir|re(nce)?))$/, # Clair Clare
  Clarissa => qr/^(C(iss$E|la(ire?|r(a|e|i(ce|nda|ssa)))))$/, # Clarinda Clarice Clair Clara Clare Ciss$E Claire
  Cleatus => qr/^(Cleat(us)?)$/, # Cleat
  Clement => qr/^(Clem(ent)?)$/, # Clem
  Clementine => qr/^(Clem(entine)?)$/, # Clem
  Clifford => qr/^((Cliff(ord)?|Ford))$/, # Cliff Ford
  Clifton => qr/^((Clif(f|ton)|Ton$E))$/, # Cliff Ton$E
  Columbus => qr/^((C(lum|olumbus)|Lum))$/, # Lum Clum
  Conrad => qr/^(Con(n$E|rad)?)$/, # Con Conn$E
  Constance => qr/^(Con(n$E|stance))$/, # Conn$E
  Content => qr/^((Content|Tent$E))$/, # Tent$E
  Cordelia => qr/^((Cord($E|elia)|Delia))$/, # Cord$E Delia
  Corinne => qr/^(Cor(a|inne))$/, # Cora
  Cornelia => qr/^((Corn($E|elia)|Ne(el$E|l(ia|l($E|e)))))$/, # Corn$E Neel$E Nelle Nell$E Nelia
  Cornelius => qr/^((Co(n(n$E|o)|rnelius)|N(e(al|el$E|il)|iel)))$/, # Neil Neel$E Cono Conn$E Niel Neal
  Courtney => qr/^(C(ourt(ney)?|urt))$/, # Court Curt
  Curtis => qr/^((Curt(is)?|Kurt(is)?))$/, # Curt Kurtis Kurt
  Cynthia => qr/^((C(in(d$E|thia)|ynthia)|Lucinda))$/, # Cind$E Lucinda Cinthia
  Cyrenius => qr/^((C(ene|y(renius)?)|Renius|S(erene|wene)))$/, # Cene Cy Renius Serene Swene
  Cyrus => qr/^((Cy(rus)?|Si))$/, # Cy Si
  Dalton => qr/^(Da(hl|l(ton)?))$/, # Dahl Dal
  Daniel => qr/^(Dan(i(al|el)|n$E)?)$/, # Dan Dann$E Danial
  Danielle => qr/^(Dan($E|elle|ielle))$/, # Dan$E Danelle
  Daphne => qr/^(Daph($E|ne)?)$/, # Daph Daph$E
  Darlene => qr/^((Dar(lene|r$E)|Lena))$/, # Lena Darr$E
  Darwin => qr/^(Dar($E|win))$/, # Dar$E
  David => qr/^(Da(v($E|e|id)|y))$/, # Dav$E Dave Day
  Deborah => qr/^(Deb(b($E|e)|i|orah?|ra)?)$/, # Deb Debb$E Debora Debbe Debi Debra
  Deidre => qr/^(De(eDee|idre))$/, # DeeDee
  Delbert => qr/^((Bert|D(el(bert)?|ilbert)))$/, # Bert Del Dilbert
  Delilah => qr/^((Del(ilah|la?)|Lila?))$/, # Dell Della Lil Lila
  Deliverance => qr/^(D(el(iverance|l($E|a))|ill$E))$/, # Della Dell$E Dill$E
  Delores => qr/^((D(e(e|l(la?|ores))|od$E)|Lol(a|l$E)))$/, # Dee Dod$E Lola Dell Loll$E Della
  Demerias => qr/^((D(amaris|em(aris|erias)|immis)|Maris))$/, # Demaris Maris Dimmis Damaris
  Democrates => qr/^((Democrates|Mock))$/, # Mock
  Denise => qr/^(Den(ise|yse?))$/, # Denys Denyse
  Dennis => qr/^(Denn($E|is(on)?))$/, # Denn$E Dennison
  Derrick => qr/^((D(errick|irch)|Eric|Ric(h$E|k)))$/, # Eric Rick Rich$E Dirch
  Desiree => qr/^(Des($E|ir($E|ee)|r($E|ee)))$/, # Des$E Desree Desr$E Desir$E
  Diana => qr/^(Di(an[ae]|c$E)?)$/, # Diane Di Dic$E
  Dominic => qr/^((Dom(inic)?|Nick))$/, # Dom Nick
  Donald => qr/^(Don(ald|n$E)?)$/, # Don Donn$E
  Dorothy => qr/^((D(ee|o(d($E|a|e)|ll$E|r(a|i(nda|s)|oth(ea|y)|tha)|t(ha|t$E)?))|Medora))$/, # Dee Doll$E Dot Dotha Dott$E Dode Dod$E Dortha Doda Dora Dorothea Dorinda Doris Medora
  Douglas => qr/^(Doug(las)?)$/, # Doug
  Drusilla => qr/^((Drusilla|Silla))$/, # Silla
  Duane => qr/^(D(ewayne|uane|wa(ne|yne)))$/, # Dewayne Dwayne Dwane
  Duncan => qr/^(Dun(can|k))$/, # Dunk
  Earnest => qr/^(E(arnest|rn($E|est)))$/, # Ernest Ern$E
  Ebenezer => qr/^(Eb(b$E|en(ezer)?)?)$/, # Eb Eben Ebb$E
  Edith => qr/^((Dic$E|Ed($E|ith|n($E|a))))$/, # Dic$E Ed$E Edna Edn$E
  Edmund => qr/^((Ed(mund)?|Ned|Ted))$/, # Ed Ned Ted
  Edward => qr/^((Ed(d$E|gar|mund|ward)?|Ned|Ted(d$E)?))$/, # Ed Ned Ted Edgar Edmund Edd$E Tedd$E
  Edwin => qr/^((Ed(win)?|Ned|Win))$/, # Ed Ned Win
  Edwina => qr/^((Edwina|Winn$E))$/, # Winn$E
  Egbert => qr/^((B(ert|urt)|Egbert))$/, # Bert Burt
  Elaine => qr/^((Alaina|Elain[ae]|Helen|Lain$E))$/, # Helen Lain$E Alaina Elaina
  Elbertson => qr/^((Bert|Elbert(son)?))$/, # Elbert Bert
  Eleanor => qr/^((El(aine|e(anor|nora)|inamifia|l($E|a|en)|nora)|Helen|L(anna|eonora)|N(ell($E)?|ora)))$/, # Elaine Ella Ellen Helen Lanna Leonora Nell Nell$E Nora Ell$E Elenora Elinamifia Elnora
  Eleazer => qr/^((Eleazer|Lazar))$/, # Lazar
  Electra => qr/^((Elect(a|ra)|Lecta))$/, # Electa Lecta
  Elias => qr/^((El($E|ias)|L(ee|ias)))$/, # El$E Lee Lias
  Elijah => qr/^((El($E|ijah)|Lige?))$/, # El$E Lige Lig
  Eliphalet => qr/^((Eliphal(et)?|Fall$E|Left))$/, # Left Eliphal Fall$E
  Elizabeth => qr/^((B(ais($E|s$E)|e(ss($E)?|t(h(ia)?|s$E|t$E))|its$E)|El(i(s(sa)?|za(beth)?)|s$E)|Li(b(b$E)?|dd$E|s[ae]|z(a(beth)?|z$E)?)|Tess))$/, # Bess Bess$E Beth Bett$E Elis Elissa Els$E Libb$E Liz Liza Lizabeth Lizz$E Tess Bits$E Eliza Lidd$E Lisa Lise Lib Bets$E Bethia Bais$E Baiss$E
  Elmira => qr/^((El(l$E|mira)|M(ira|yra)))$/, # Mira Myra Ell$E
  Elsie => qr/^((Alice|El(izabeth|sie)))$/, # Alice Elizabeth
  Elwood => qr/^((El(l($E|s)|wood)|Wood$E))$/, # Ells Ell$E Wood$E
  Emanuel => qr/^((Emanuel|Immanuel|Man(n$E|uel)))$/, # Mann$E Manuel Immanuel
  Emeline => qr/^((E(m(eline|il($E|ine)|m($E|a|er)?)?|rma)|Lina|Mill$E))$/, # Emmer Emm$E Lina Mill$E Em Emma Emeline Emiline Erma Emm Emil$E
  Enedina => qr/^((Enedina|Nina))$/, # Nina
  Ephraim => qr/^(Eph(raim)?)$/, # Eph
  Eric => qr/^((Eric|Rick($E)?))$/, # Rick Rick$E
  Ernestine => qr/^((Ern(a|estine)|Tina))$/, # Erna Tina
  Eseneth => qr/^((Eseneth|Sen$E))$/, # Sen$E
  Estella => qr/^((Es(s$E|tell[ae])|Stella))$/, # Ess$E Stella Estelle
  Eudora => qr/^((Dora|Eudora))$/, # Dora
  Eudoris => qr/^((Dos($E|s$E)|Eudoris))$/, # Dos$E Doss$E
  Eugene => qr/^((Eugene|Gene))$/, # Gene
  Eugenia => qr/^((Eu(genia|y)|Gen($E|e)|Jenn$E))$/, # Euy Gene Gen$E Jenn$E
  Euphemia => qr/^(E(ff$E|uphemia))$/, # Eff$E
  Euphrosina => qr/^((Euphrosina|Fen(a|ee)|Ros$E))$/, # Fenee Fena Ros$E
  Eurydice => qr/^((Dic$E|Eurydice))$/, # Dic$E
  Eustacia => qr/^((Eustacia|Stac($E|ia)))$/, # Stacia Stac$E
  Evaline => qr/^((Ev(a(line)?|e)|Lena))$/, # Eva Eve Lena
  Evangeline => qr/^((Evangeline|Vang$E))$/, # Vang$E
  Evelyn => qr/^(Ev(e(l(ina|yn))?)?)$/, # Ev Eve Evelina Evelyn
  Experience => qr/^((Experience|Pedd$E))$/, # Pedd$E
  Ezekiel => qr/^((Ez(ekiel)?|Zeke))$/, # Ez Zeke
  Ezra => qr/^(Ez(ra)?)$/, # Ez
  Faith => qr/^(Fa(ith|y))$/, # Fay
  Felicity => qr/^((F(elici(a|ty)|lick)|Tick))$/, # Flick Tick Felicia
  Ferdinand => qr/^(Ferd($E|inand))$/, # Ferd$E
  Fidelia => qr/^((Delia|Fidelia|Phidelia))$/, # Phidelia Delia
  Florence => qr/^(Flo(r(a|ence)|ss($E)?)?)$/, # Flo Flora Floss Floss$E
  Floyd => qr/^((Floyd|Lloyd))$/, # Lloyd
  Frances => qr/^((Ciss$E|F(an(n$E)?|ran(ces)?)|Sis))$/, # Fan Fann$E Fran Sis Ciss$E
  Francis => qr/^(Fran(cis|k($E|isek|lin)?|z))$/, # Frank Frank$E Frankisek Franklin Franz
  Frederica => qr/^((Fr(ed(a|d$E|erica|ricka)|ieda)|Ricka))$/, # Fredricka Freda Fredd$E Ricka Frieda
  Frederick => qr/^(Fr(ed(d$E|erick|ric)?|i(sh|tz)))$/, # Fred Fredric Fredd$E Frish Fritz
  Fredonia => qr/^((Don(ia|na)|Fredonia))$/, # Donia Donna
  Gabriel => qr/^(Gab(b$E|e|riel))$/, # Gabe Gabb$E
  Gabrielle => qr/^((Ella|Gab(b$E|riell[ae])))$/, # Ella Gabb$E Gabriella
  Gabrilla => qr/^((Gabrilla|Rilla))$/, # Rilla
  Genevieve => qr/^((Eve|Gen(c$E|evieve)?|J(e(an|nn$E)|ins$E)|Neva))$/, # Genc$E Jins$E Jenn$E Eve Jean Gen Neva
  George => qr/^((George|Jorge))$/, # Jorge
  Georgia => qr/^(Ge(e|orgia(na)?))$/, # Georgiana Gee
  Gerald => qr/^((Ger(ald|r$E)|Jerr$E))$/, # Jerr$E Gerr$E
  Geraldine => qr/^((D(eann$E|ina)|Ger($E|aldine|r$E)|Jerr$E))$/, # Dina Gerr$E Jerr$E Deann$E Ger$E
  Gerhardt => qr/^(G(ay|erhardt))$/, # Gay
  Gertrude => qr/^((G(att$E|ert($E|rude)?)|Trud$E))$/, # Gatt$E Gert$E Trud$E Gert
  Gilbert => qr/^((Bert|Gi(b|l(bert)?)|Wilber))$/, # Bert Gib Gil Wilber
  Gloria => qr/^(Glor($E|ia))$/, # Glor$E
  Greenberry => qr/^((Berr$E|Green(berry)?))$/, # Berr$E Green
  Gregory => qr/^(Greg(g|ory)?)$/, # Greg Gregg
  Gretchen => qr/^((Gretchen|Margaret))$/, # Margaret
  Griselda => qr/^(Gris(elda|sel))$/, # Grissel
  Gwendolyn => qr/^((Gwen(dolyn)?|Wend$E))$/, # Gwen Wend$E
  Hamilton => qr/^(Ham(ilton|p)?)$/, # Ham Hamp
  Hannah => qr/^((Anna|Hannah|Nan(n$E)?|Susannah))$/, # Anna Nan Nann$E Susannah
  Harold => qr/^(Ha(l|nk|r(old|r$E)))$/, # Hal Harr$E Hank
  Harriet => qr/^(Ha(rriet|tt$E))$/, # Hatt$E
  Helena => qr/^((Aileen|E(ileen|l(aine|e(anor|na)|l(a|en(der)?)))|Helen[ae]?|Lena|Nell($E)?))$/, # Helen Aileen Eileen Elaine Eleanor Ellen Lena Nell Nell$E Helene Ella Elena Ellender
  Henrietta => qr/^((Ett($E|a)|H(ank|e(nrietta|tt$E))|Nett$E|Retta))$/, # Etta Hank Hett$E Nett$E Retta Ett$E
  Henry => qr/^(H(a(l|nk|rr$E)|en(ce|ry)?))$/, # Hal Hank Harr$E Hen Hence
  Hepsabah => qr/^(He(bsabeth|p(hsibah|s($E|ab(ah|el)|ibah))))$/, # Heps$E Hepsibah Hephsibah Hebsabeth Hepsabel
  Herbert => qr/^((Bert|Herb(ert)?))$/, # Bert Herb
  Herman => qr/^(H(arman|erman))$/, # Harman
  Hermione => qr/^(Herm($E|ione|oine))$/, # Hermoine Herm$E
  Hester => qr/^((Es(s$E|ther)|He(s(s$E|ter)|tt$E)))$/, # Esther Hess$E Hett$E Ess$E
  Hezekiah => qr/^((H(ez(ekiah)?|y)|K(iah|y)))$/, # Hez Ky Hy Kiah
  Hiram => qr/^(H(i(l$E|ram)|y))$/, # Hy Hil$E
  Honora => qr/^((Hon($E|ora)|Nor(ah?|r$E)))$/, # Hon$E Nora Norah Norr$E
  Hopkins => qr/^(Hop(kins|p)?)$/, # Hop Hopp
  Horace => qr/^(Hor(a(ce|tio)|r$E))$/, # Horatio Horr$E
  Hortense => qr/^((Hortense|Tens$E))$/, # Tens$E
  Hosea => qr/^(Hos($E|ea))$/, # Hos$E
  Howard => qr/^(H(al|ow($E|ard)))$/, # Hal How$E
  Hubert => qr/^((Bert|Hu(b(ert)?|g[ho])))$/, # Hugh Hugo Bert Hub
  Ignatius => qr/^((Ig(g$E|natius)|Na(ce|t(e|ius))))$/, # Igg$E Nace Nate Natius
  India => qr/^(Ind($E|ia))$/, # Ind$E
  Iona => qr/^((Iona|Onn$E))$/, # Onn$E
  Irene => qr/^((Irene|Re(en$E|n(a|n$E))))$/, # Rena Renn$E Reen$E
  Irvin => qr/^(Irving?)$/, # Irving
  Irwin => qr/^((Erwin|Irwin))$/, # Erwin
  Isaac => qr/^((I(ke|saac)|Zeke))$/, # Ike Zeke
  Isabella => qr/^((Bell[ae]|Elizabeth|I(b|s(abel(l[ae])?|s$E))|Nib(b$E)?|Rosabella|Sab(e|ra)|Tibb$E))$/, # Isabelle Bella Belle Elizabeth Ib Iss$E Nib Sabe Sabra Tibb$E Isabel Rosabella Nibb$E
  Isaiah => qr/^((Isaiah|Za(d$E|y)))$/, # Zad$E Zay
  Isidora => qr/^((Dora|Is(idora|s$E)))$/, # Dora Iss$E
  Isidore => qr/^(I(sidore|zz$E))$/, # Izz$E
  Jacob => qr/^(Ja(ap|cob(us)?|ke|y))$/, # Jake Jaap Jay Jacobus
  Jacqueline => qr/^(Ja(c(k($E|l(in|yn))|l(in|yn)|que(line)?)|k$E))$/, # Jack$E Jak$E Jacque Jaclyn Jacklyn Jaclin Jacklin
  James => qr/^(J(am($E|es(on)?)|em|im(m$E)?))$/, # Jam$E Jim Jameson Jem Jimm$E
  Jane => qr/^((J(a(n($E|e(t(te?)?)?|i(ce|ece)|yce)?|yce)|c|e(an(ne)?|nn$E|ss$E)|in(c$E|s$E)|oan(na)?)|Nett$E|Virginia))$/, # Janet Jan$E Jess$E Jenn$E Joan Joanna Virginia Janette Jean Jinc$E Jins$E Jeanne Nett$E Jan Janett Janice Janyce Jayce Jc Janiece
  Jasper => qr/^((Casper|Jasper))$/, # Casper
  Jedediah => qr/^((D(iah|yer)|Jed(ediah|idiah)?))$/, # Diah Dyer Jed Jedidiah
  Jefferson => qr/^((Geoff(r$E)?|Jeff(erson|r$E)?|Sonn$E))$/, # Jeff Sonn$E Jeffr$E Geoffr$E Geoff
  Jehiel => qr/^((Hiel|Jehiel))$/, # Hiel
  Jehu => qr/^((Gee|Hugh|J(ayhugh|ehu)))$/, # Jayhugh Gee Hugh
  Jemima => qr/^((Jem(ima|ma)|Mim($E|a)))$/, # Jemma Mim$E Mima
  Jennet => qr/^(Je(nnet|ss$E))$/, # Jess$E
  Jennifer => qr/^((Jenn($E|ifer)|Winifred))$/, # Winifred Jenn$E
  Jeremiah => qr/^(Jer(em($E|e|iah)|r$E))$/, # Jerem$E Jerr$E Jereme
  Jerita => qr/^((Jerita|Rita))$/, # Rita
  Jessica => qr/^(Jess($E|e|ica)?)$/, # Jess Jesse Jess$E
  Joanna => qr/^((Hannah|J(ane|ean|o(an(n(ah?|e)?)?|d$E|hannah?))|Non$E))$/, # Jane Jean Joan Jod$E Johanna Non$E Joannah Johannah Hannah Joann Joanne
  John => qr/^((Hans|I(an|van)|J(a(ck($E|son)?|n|ques)|ean|o(ck|hn(n$E)?))))$/, # Hans Ian Ivan Jack Jan Jean Jaques Jock Jack$E Johnn$E Jackson
  Jonathan => qr/^((Jo(hn(n$E)?|nathan)|Nathan))$/, # John Johnn$E Jonathan Nathan
  Joseph => qr/^(Jo(d$E|ey?|s(e(fa|ph))?))$/, # Joe Joey Jos Jod$E Josefa
  Josephine => qr/^((Fina|Jo(d$E|ey|s($E|eph(a|ine)))?|Phen$E))$/, # Josepha Phen$E Jo Joey Jos$E Fina Jod$E
  Joshua => qr/^(Jos(h(ua)?)?)$/, # Josh Jos
  Josiah => qr/^((Jos(iah)?|Si))$/, # Jos Si
  Joyce => qr/^(Joy(ce)?)$/, # Joy
  Juanita => qr/^((Juanita|N(ett$E|ita)))$/, # Nita Nett$E
  Judah => qr/^(Jud($E|ah))$/, # Jud$E
  Judith => qr/^(Jud($E|ith))$/, # Jud$E
  Judson => qr/^((Jud(son)?|Sonn$E))$/, # Jud Sonn$E
  Julia => qr/^(J(ill|ul($E|i(a|et))))$/, # Jill Jul$E Juliet
  Julias => qr/^(Jul(e|i(a[ns]|us)))$/, # Julius Julian Jule
  Junior => qr/^(J(r|un($E|e|ior)))$/, # Jr June Jun$E
  Justin => qr/^(Just(in|us))$/, # Justus
  Karonhappuck => qr/^((Carr$E|Happ$E|Kar(en|on(happuck)?)))$/, # Carr$E Karen Karon Happ$E
  Katherine => qr/^((Ca(ss$E|th($E|erine|leen))|K(a(ren|t($E|arina|e|h($E|e(rin[ae])?|leen|ryn))?|y)|it(s$E|t$E)?)|Lena|T(ina|rina)))$/, # Katarina Kathleen Cath$E Kat Kits$E Kitt$E Kate Kat$E Kath$E Kit Kay Trina Tina Lena Cathleen Catherine Cass$E Karen Kathryn Katherina Kathe
  Kendra => qr/^(Ken(dra|j($E)?))$/, # Kenj Kenj$E
  Kenneth => qr/^(Ken(d(all|rick)|n($E|eth)|t)?)$/, # Ken Kenn$E Kendall Kendrick Kent
  Keziah => qr/^(K(ez(iah|z$E)|i(d|zza)))$/, # Kezz$E Kizza Kid
  Kimberly => qr/^(K(im(berl($E|y)|m$E)?|ym(berl$E)?))$/, # Kimberl$E Kymberl$E Kim Kym Kimm$E
  King => qr/^(King(s(l$E|ton))?)$/, # Kingston Kingsl$E
  Lafayette => qr/^((Fate|L(af(ayette|f$E)|ef(f$E|t))))$/, # Fate Left Leff$E Laff$E
  Lamont => qr/^((Lamont|Mont$E))$/, # Mont$E
  Laodicia => qr/^((Cenia|Dic$E|Laodicia))$/, # Dic$E Cenia
  Laurinda => qr/^(L(aur(a|inda)|orinda))$/, # Lorinda Laura
  Lauryn => qr/^(L(aur($E|yn)|orr$E))$/, # Laur$E Lorr$E
  Laverne => qr/^((Laverne|Verna))$/, # Verna
  Lavinia => qr/^((Ina|L(avin(a|ia)|ouvin(a|ia))|Vin($E|a|n$E)|Wyncha))$/, # Vin$E Wyncha Lavina Ina Vina Vinn$E Louvina Louvinia
  Lawrence => qr/^(L(a(r(r$E|s)|urence|wr($E|ence))|o(n(n$E)?|r(ne|r$E))))$/, # Larr$E Lars Laurence Lawr$E Lon Lorr$E Lonn$E Lorne
  Lecurgus => qr/^((Curg|Lecurgus))$/, # Curg
  Lemuel => qr/^(Lem(uel)?)$/, # Lem
  Leonard => qr/^(L(e(n(hart|n$E)?|o(n(ard)?)?)|ineau))$/, # Leo Leon Len Lenn$E Lineau Lenhart
  Leonidas => qr/^(Leon(idas)?)$/, # Leon
  Leonora => qr/^((Elenor|L(e(e|onora)|oenore)|Nora))$/, # Loenore Elenor Nora Lee
  Leroy => qr/^((L(R|e(e|roy))|Roy))$/, # Lee Roy LR
  Leslie => qr/^(Les(lie|ter)?)$/, # Les Lester
  Lester => qr/^(Les(ter)?)$/, # Les
  Letitia => qr/^((Let(itia|t($E|ice))|Ti(sh|tia)))$/, # Lett$E Lettice Titia Tish
  Levi => qr/^(Le(e|vi))$/, # Lee
  Lewvisa => qr/^((Lewvisa|Vic$E))$/, # Vic$E
  Libuse => qr/^(Lib(a|b$E|use))$/, # Liba Libb$E
  Lillian => qr/^(L(il($E|l($E|ah|ian))?|oll$E))$/, # Lil Lil$E Lill$E Loll$E Lillah
  Lincoln => qr/^(Lin(coln|k))$/, # Link
  Linda => qr/^(L(i(d(d$E|ia)|n(d($E|a)|ette))|y(d($E|d$E)|n(d$E|ette|n))))$/, # Lind$E Lynn Lynd$E Lynette Linette Lyd$E Lidia Lydd$E Lidd$E
  Lionel => qr/^(L(eon|ionel))$/, # Leon
  Littleberry => qr/^((Berr$E|L(b|ittle(berry)?)))$/, # Berr$E Lb Little
  Lois => qr/^((Heloise|Lo(is|uise)))$/, # Heloise Louise
  Lorenzo => qr/^(Loren(zo)?)$/, # Loren
  Loretta => qr/^((Loretta|Retta))$/, # Retta
  Lorraine => qr/^((Lorr($E|aine)|Rana))$/, # Lorr$E Rana
  Lotta => qr/^((Charlott[ae]|Lott($E|a)))$/, # Lott$E Charlotte Charlotta
  Louis => qr/^(L(ewis|ou(i[es])?))$/, # Lewis Lou Louie
  Lousie => qr/^((El(iza|o(ise|uise))|Heloise|L(o(is|u(etta|is[ae]|sie)?)|ulu)))$/, # Eloise Elouise Louise Heloise Eliza Lois Lou Lulu Louisa Louetta
  Luann => qr/^((Ann($E|e)?|L(ou(anne?)?|u(anne?|lu))))$/, # Luanne Louanne Louann Lulu Lou Ann Anne Ann$E
  Luciana => qr/^((Cind$E|Luc($E|ia(na)?)))$/, # Lucia Luc$E Cind$E
  Lucias => qr/^(Lu(c(as|ias)|ke))$/, # Lucas Luke
  Lucille => qr/^((C(eall|ille)|L(ou|u(c($E|ille))?)))$/, # Ceall Cille Luc$E Lu Lou
  Lucinda => qr/^((Cind$E|L(ou|u(c($E|inda))?)))$/, # Cind$E Luc$E Lu Lou
  Lucretia => qr/^((Cre(as$E|ce|se)|Lucretia))$/, # Creas$E Crese Crece
  Luella => qr/^((Ella|Lu(ella|la)?))$/, # Ella Lu Lula
  Lurana => qr/^(Lura(na)?)$/, # Lura
  Luthor => qr/^(Lu(ke|thor))$/, # Luke
  Lyndon => qr/^(L(ind$E|yn(d($E|on)|n)))$/, # Lynn Lynd$E Lind$E
  Madeline => qr/^((Lena|M(a(d(d$E|el(eine|ine)|ge)|g(da|g$E)|ida|ud)|idd$E)))$/, # Lena Madd$E Madge Magg$E Maud Midd$E Magda Maida Madeleine
  Magdelina => qr/^((Lena|Ma(dge|gd(a(len)?|elina)|ida)))$/, # Lena Madge Magda Magdalen Maida
  Mahalla => qr/^((Hall$E|Mahalla))$/, # Hall$E
  Malcolm => qr/^(Ma(c|l(ach$E|colm)?))$/, # Mac Mal Malach$E
  Manerva => qr/^((M(anerva|inerva)|Nerv($E|a)))$/, # Minerva Nerva Nerv$E
  Manoah => qr/^((Manoah|Noah))$/, # Noah
  Manuel => qr/^((Emanuel|Man(n$E|uel)))$/, # Emanuel Mann$E
  Marcus => qr/^(Mar(cus|k))$/, # Mark
  Margaret => qr/^((Dais$E|Gret(a|ta)|M(a(dge|gg$E|is$E|r(g($E|aret(ha|ta)?|e(r$E)?|o)|jor$E))|e(g(an)?|t(a|ta))|idge)|Peg(g$E)?|Rita))$/, # Dais$E Greta Madge Magg$E Mais$E Marge Margo Peg Meg Metta Midge Pegg$E Margaretha Meta Gretta Rita Marger$E Marjor$E Marg$E Margaretta Megan
  Margarita => qr/^((Marga(rita|uerite)|Rita))$/, # Rita Margauerite
  Marianna => qr/^((Anne?|M(ar($E|ia(n(n[ae])?)?)|ia)))$/, # Marian Ann Marianne Anne Mar$E Maria Mia
  Martha => qr/^((M(a(rt($E|ha)?|t(t$E)?)|oll$E)|Pat(s$E|t$E)))$/, # Mart Mart$E Mat Matt$E Pats$E Patt$E Moll$E
  Martin => qr/^(Mart($E|in))$/, # Mart$E
  Marvin => qr/^(M(arv(in)?|erv(yn)?))$/, # Marv Merv Mervyn
  Mary => qr/^((M(a(e|m$E|r($E|cia|i(a[hn]?|ca|etta|lyn|on)|y)|te|ur(a|een)|y)|erc$E|i(m$E|nn$E|tz$E)|o(ira|ll($E)?))|Poll$E))$/, # Mae Mam$E Marietta Marion Maureen May Merc$E Minn$E Mitz$E Moll$E Poll$E Moll Mim$E Mate Maura Moira Marilyn Maria Mariah Marian Mar$E Marcia Marica
  Maryanne => qr/^((Ann($E|e)?|Mar($E|iann[ae]|yann[ae])))$/, # Marianne Maryanna Marianna Mar$E Ann Anne Ann$E
  Marylou => qr/^((L(ou|ulu)|Mar($E|ylou)))$/, # Mar$E Lulu Lou
  Matilda => qr/^((Ma(t(hilda|ilda|t$E)?|ud)|Patt$E|Til(da|l$E)))$/, # Mat Tilda Patt$E Till$E Matt$E Maud Mathilda
  Matthew => qr/^((Matt(h(ew|ias))?|Th(ias|ys)))$/, # Matt Matthias Thias Thys
  Maureen => qr/^((Ma(r$E|ureen)|Re(en$E|na)))$/, # Mar$E Reen$E Rena
  Maurice => qr/^(M(aur($E|ice)|o(r($E|ris)|ss$E)))$/, # Maur$E Morris Moss$E Mor$E
  Mavine => qr/^(Mav(e(r$E)?|ine))$/, # Maver$E Mave
  Maximillian => qr/^(Max(imillian)?)$/, # Max
  Maxine => qr/^(Max(ine)?)$/, # Max
  Mehitabel => qr/^((Amabel|Bella|H(ett$E|itt$E)|M(abel|eh(etab(el|le)|itab(el|le))|itt$E)))$/, # Hett$E Mabel Mitt$E Hitt$E Bella Mehetable Mehitable Mehetabel Amabel
  Melchizedek => qr/^((Dick|Melchizedek|Zadock))$/, # Dick Zadock
  Melinda => qr/^((L(ind($E|a)|ynne)|M(el(inda|l$E)?|ind$E)))$/, # Linda Lind$E Mel Mind$E Lynne Mell$E
  Melissa => qr/^((Lis(a|sa)|M(el(issa)?|i(ll$E|ss$E))))$/, # Lisa Mel Mill$E Miss$E Lissa
  Melody => qr/^((Lod$E|Melody))$/, # Lod$E
  Melvina => qr/^((Melvina|Vina))$/, # Vina
  Mercedes => qr/^(Merc($E|edes))$/, # Merc$E
  Merlin => qr/^(Merl(e|in|yn))$/, # Merlyn Merle
  Mervyn => qr/^(Merv(yn)?)$/, # Merv
  Micajah => qr/^((Cager|Micajah))$/, # Cager
  Michael => qr/^(Mi(c(ah|hael|k($E)?)|ke|tchell))$/, # Mick$E Mike Mitchell Micah Mick
  Michelle => qr/^((Mic(helle|k$E)|Shell$E))$/, # Mick$E Shell$E
  Mildred => qr/^(M(ell|i(l(dred|l$E)|m$E)))$/, # Mell Mill$E Mim$E
  Millicent => qr/^(Mi(ll($E|icent)|ss$E))$/, # Mill$E Miss$E
  Mindwell => qr/^(Min(a|dwell))$/, # Mina
  Minerva => qr/^((Min(a|erva|n$E)|Nerv($E|a)))$/, # Mina Minn$E Nerva Nerv$E
  Miranda => qr/^((M(and$E|ira(nda)?)|Rand$E))$/, # Mand$E Mira Rand$E
  Miriam => qr/^(M(ar$E|i(m$E|riam|tz$E)))$/, # Mar$E Mitz$E Mim$E
  Mitchell => qr/^(Mi(chael|tch(ell)?))$/, # Michael Mitch
  Mitzi => qr/^(M(ar$E|itzi))$/, # Mar$E
  Montgomery => qr/^((Gum|Mont($E|esque|gomery)))$/, # Gum Mont$E Montesque
  Mortimer => qr/^(Mort(imer)?)$/, # Mort
  Moses => qr/^((Amos|Mos(es?|s)))$/, # Amos Mose Moss
  Muriel => qr/^(Mur(iel)?)$/, # Mur
  Myrtle => qr/^(M(ert|yrt($E|le)?))$/, # Mert Myrt Myrt$E
  Nadezhda => qr/^(Nad(ezhda|ia))$/, # Nadia
  Nadine => qr/^((DeeDee|Nad(a|ine)))$/, # DeeDee Nada
  Nancy => qr/^((A(gnes|nn)|Nan(c[ey]|n$E)?))$/, # Agnes Ann Nan Nance Nann$E
  Napoleon => qr/^((Leon|Nap(oleon|p$E)?|Pon$E))$/, # Leon Nap Pon$E Napp$E
  Natalie => qr/^((N(at(alie|t$E)|ett$E)|Tall$E))$/, # Nett$E Natt$E Tall$E
  Nathaniel => qr/^((Fannn$E|Jonathan|Nat(e|han(iel)?|t$E)?|Than))$/, # Jonathan Nat Natt$E Nathan Nate Than Fannn$E
  Nelson => qr/^(Nels(on)?)$/, # Nels
  Newton => qr/^(Newt(on)?)$/, # Newt
  Nicholas => qr/^((Cla(as|es)|Nic(holas|k($E)?|odemus)))$/, # Nick Nick$E Claas Claes Nicodemus
  Nicolena => qr/^(Ni(c(k$E|ol(e(n[ae])?|ina))|kk$E))$/, # Nicole Nikk$E Nick$E Nicolene Nicolina
  Ninell => qr/^(N(ett$E|inell))$/, # Nett$E
  Noel => qr/^((Knowell|No(el|well)))$/, # Nowell Knowell
  Norbert => qr/^((Bert|Norb($E|ert)))$/, # Bert Norb$E
  Obadiah => qr/^((D(iah|yer)|Ob($E|adiah|ed?)))$/, # Diah Dyer Obe Obed Ob$E
  Obedience => qr/^((B(e(de|ed$E)|idd$E)|Obed(ience)?))$/, # Bidd$E Obed Bede Beed$E
  Oberon => qr/^(Ob($E|e(ron)?))$/, # Obe Ob$E
  Octavia => qr/^((Octavia|Tav(e|ia)))$/, # Tave Tavia
  Odell => qr/^(Od(ell|o))$/, # Odo
  Oliver => qr/^(Ol(iver|l$E))$/, # Oll$E
  Olivia => qr/^((Liv($E|ia)|Noll$E|Ol(iv(e|ia)|l$E)))$/, # Livia Olive Oll$E Noll$E Liv$E
  Ophelia => qr/^((O(phelia|rphelia)|Phelia))$/, # Phelia Orphelia
  Orlando => qr/^((Orlando|Roland))$/, # Roland
  Oswald => qr/^((O(s(s$E|wald)|zz$E)|Waldo))$/, # Oss$E Waldo Ozz$E
  Othello => qr/^(Ot(e|hello|is))$/, # Otis Ote
  Pamela => qr/^(Pam(ela)?)$/, # Pam
  Pandora => qr/^((Dora|Pandora))$/, # Dora
  Parthenia => qr/^((P(a(r(sun$E|thenia)|soon$E|tt$E)|hen$E)|Then$E))$/, # Parsun$E Pasoon$E Phen$E Patt$E Then$E
  Patience => qr/^(Pat(ience|t$E)?)$/, # Pat Patt$E
  Patricia => qr/^((Pat(ricia|s$E|t$E)?|T(ish|ri(cia|sh|x$E))))$/, # Pat Pats$E Patt$E Tricia Trix$E Trish Tish
  Patrick => qr/^(P(a(dd$E|t(e|ric(ia|k)|s$E)?)|eter))$/, # Padd$E Pat Pats$E Peter Patricia Pate
  Paulina => qr/^((Lina|P(aul(a|in[ae])|oll$E)))$/, # Paula Poll$E Lina Pauline
  Pelegrine => qr/^(Pe(legrine|rr$E))$/, # Perr$E
  Penelope => qr/^((Nepp$E|Pen(elope|n$E)))$/, # Nepp$E Penn$E
  Percival => qr/^(Perc($E|e|ival))$/, # Perce Perc$E
  Permelia => qr/^((M(ell$E|ill$E)|Permelia))$/, # Mell$E Mill$E
  Pernetta => qr/^((Nett$E|Pernetta))$/, # Nett$E
  Peter => qr/^(P(ate|eter?))$/, # Pete Pate
  Pharaba => qr/^((Ferb$E|Ph(araba|er(bia|iba))))$/, # Ferb$E Pherbia Pheriba
  Pheney => qr/^((Josephine|Pheney))$/, # Josephine
  Philadelphia => qr/^((Delph(a|ia)|P(hiladelphia|uss)))$/, # Delphia Delpha Puss
  Philetus => qr/^((Leet|Phil(etus)?))$/, # Leet Phil
  Philinda => qr/^((Linda|Ph(ilinda|yllis)))$/, # Linda Phyllis
  Philip => qr/^(P(hil(ip)?|ip))$/, # Phil Pip
  Philipina => qr/^(P(en$E|h(ili(lpa|pina)|oebe)))$/, # Phoebe Pen$E Phililpa
  Philomena => qr/^((Mena|Philomena))$/, # Mena
  Phineas => qr/^((Finn$E|Phineas))$/, # Finn$E
  Pleasant => qr/^(Ple(as(ant)?|s))$/, # Ples Pleas
  Pocahontas => qr/^(Po(cahontas|k$E))$/, # Pok$E
  Posthuma => qr/^((Hum$E|Posthuma))$/, # Hum$E
  Prescott => qr/^((Pres(cott)?|Scott($E)?))$/, # Pres Scott Scott$E
  Priscilla => qr/^((Ci(l(l(a|er))?|ss$E)|Pris(cilla|s$E)|Silla))$/, # Cil Cilla Ciller Priss$E Ciss$E Silla
  Prudence => qr/^((Dens$E|P(ru(d($E|ence)|e)?|uss)))$/, # Dens$E Prud$E Prue Pru Puss
  Quince => qr/^(Quin(ce|n|t))$/, # Quint Quinn
  Rachel => qr/^((Ra(chel|e(ch)?|y)|Shell$E))$/, # Rae Ray Raech Shell$E
  Ramona => qr/^((Mona|Ramona))$/, # Mona
  Randolph => qr/^((Dolph|Rand($E|all|olph)))$/, # Dolph Randall Rand$E
  Raphael => qr/^(Ra(ff|lph|phael))$/, # Raff Ralph
  Raymond => qr/^(Ray(mond)?)$/, # Ray
  Rebecca => qr/^((Bec(ca|k($E)?)|Reb(a|ecca)))$/, # Beck$E Reba Becca Beck
  Regina => qr/^((Gina|R(ay|eg(g$E|ina))))$/, # Gina Ray Regg$E
  Reginald => qr/^((Naldo|Re(g(g$E|inald)?|n(aldo|n$E)|ynold)))$/, # Reg Regg$E Reynold Renaldo Naldo Renn$E
  Relief => qr/^((Leaf($E|a)|Relief))$/, # Leafa Leaf$E
  Reuben => qr/^(R(euben|ub($E|e)))$/, # Rube Rub$E
  Rhodella => qr/^((Della?|Rhod($E|a|ella)))$/, # Rhoda Rhod$E Dell Della
  Richard => qr/^((Dick(on|son)?|Ric(h($E|ard)?|k($E)?)))$/, # Dick Rich Rick Rich$E Dickon Dickson Rick$E
  Robert => qr/^((Bob(b$E)?|Dob(bin)?|Hob(kin)?|R(ob(b$E|ert|in)?|upert)))$/, # Dob Dobbin Bob Bobb$E Rob Robin Rupert Hob Hobkin Robb$E
  Roberta => qr/^((B(ert($E)?|ird$E|obb$E)|Rob(b$E|erta)))$/, # Bert Bobb$E Robb$E Bert$E Bird$E
  Rodney => qr/^(Rod(d$E|ney)?)$/, # Rod Rodd$E
  Roger => qr/^((Hodge(kin)?|Ro(d(ger)?|ger?)))$/, # Hodge Hodgekin Rodger Roge Rod
  Roland => qr/^((Lann$E|Naldo|Orlando|Ro(l(and|l($E|o))|n(n$E)?|wland)))$/, # Orlando Rowland Lann$E Rollo Roll$E Ron Ronn$E Naldo
  Rosabella => qr/^((Belle|Ro(s(a(b(ella|le))?|e)|z)))$/, # Rosable Belle Rosa Rose Roz
  Rosalyn => qr/^((Linda|Ro(s(a(l(inda|yn))?|e)|z)))$/, # Rosalinda Rosa Rose Linda Roz
  Roseanne => qr/^((Ann|Ro(s($E|e(ann[ae]?)?)|x($E|an(e|n[ae]))?|z)))$/, # Roseann Rose Ann Roz Ros$E Roseanna Rox Rox$E Roxanne Roxanna Roxane
  Rudolphus => qr/^((Ad(o(l(f|phus)|ph)?)?|Dol(f|ph)|Olph|R(ol(f|lo)|ud($E|olph(us)?))))$/, # Adoph Adolf Dolf Dolph Rolf Rollo Rud$E Rudolph Olph Adolphus Ad Ado
  Rufina => qr/^((Fina|R(efina|ufina)))$/, # Refina Fina
  Russell => qr/^(Rus(s(ell)?|t$E))$/, # Russ Rust$E
  Sabrina => qr/^((Brina|Re(en$E|na)|S(abrina|erena|ybrina)))$/, # Brina Sybrina Reen$E Rena Serena
  Salvador => qr/^(Sal(vador)?)$/, # Sal
  Sampson => qr/^(Sam(pson|son)?)$/, # Samson Sam
  Samuel => qr/^(Sam(antha|m$E|uel)?)$/, # Sam Samm$E Samantha
  Sarah => qr/^(S(a(d$E|l(l$E)?|rah?)|u(k$E|rr$E)))$/, # Sad$E Sal Sall$E Sara Surr$E Suk$E
  Selina => qr/^((Celina|Lena|Selina))$/, # Celina Lena
  Serilla => qr/^((Rilla|Serilla))$/, # Rilla
  Seymour => qr/^((Mor($E|r$E)|Seymo(re|ur)))$/, # Mor$E Morr$E Seymore
  Sharon => qr/^((Cheryl|Sh(ar($E|on|r$E|yn)?|er($E|on|y[ln]))))$/, # Sharyn Sharr$E Shar Shar$E Sher$E Sheron Sheryn Sheryl Cheryl
  Shaun => qr/^(S(ean|ha(ne|un|wn|yne)))$/, # Sean Shawn Shane Shayne
  Sheila => qr/^((Cecilia|Sheila))$/, # Cecilia
  Sheldon => qr/^((Shel(don|l$E|ton)?|Ton$E))$/, # Shell$E Ton$E Shelton Shel
  Sheridan => qr/^((Dan(n$E)?|Sher(idan)?))$/, # Dan Dann$E Sher
  Shirley => qr/^((Lee|Sh(err$E|irl(ey)?)))$/, # Lee Sherr$E Shirl
  Sibbilla => qr/^((Cibyl|Sib(b($E|ell|illa)|yl)))$/, # Sibyl Cibyl Sibb$E Sibbell
  Sidney => qr/^(S(id(ney)?|yd(n$E)?))$/, # Sid Sydn$E Syd
  Sigismund => qr/^(Sig(ismund|mund)?)$/, # Sigmund Sig
  Silas => qr/^(Si(las)?)$/, # Si
  Simeon => qr/^(Si(m(eon|on)?|on)?)$/, # Sim Simon Si Sion
  Smith => qr/^(Smit(h|t$E))$/, # Smitt$E
  Solomon => qr/^((S(a(l(mon)?|ul)|ol(l$E|omon)?)|Zoll$E))$/, # Sal Salmon Sol Soll$E Zoll$E Saul
  Sophronia => qr/^((Fron($E|a|ia)|Soph($E|ia|ronia)))$/, # Frona Fronia Sophia Fron$E Soph$E
  Stephen => qr/^(Ste(ph(en)?|ven?))$/, # Steve Steven Steph
  Submit => qr/^((Mitt$E|Submit))$/, # Mitt$E
  Sullivan => qr/^((Sull($E|ivan)|Van))$/, # Sull$E Van
  Susannah => qr/^((Hannah|Su(ch$E|e|k$E|s($E|an(nah?)?)|z($E|anne))))$/, # Hannah Sue Susan Sus$E Suzanne Such$E Susanna Suk$E Suz$E
  Sylvester => qr/^((S(i(lve(r|ster))?|l$E|y(l(v(anus|ester))?)?)|Ve(s(s$E|t(er)?)|t)))$/, # Si Sy Syl Sl$E Vet Vest Vester Vess$E Silvester Sylvanus Silver
  Tabitha => qr/^(Tab(b$E|itha))$/, # Tabb$E
  Tamara => qr/^(T(am(ar(a|ra)|m$E)|emera))$/, # Tamm$E Temera Tamarra
  Tasha => qr/^(Tash($E|a)?)$/, # Tash Tash$E
  Temperance => qr/^(T(emp($E|erance)|ill$E))$/, # Temp$E Till$E
  Tennessee => qr/^(Tenn($E|essee))$/, # Tenn$E
  Terrance => qr/^(Ter(ence|r($E|ance)))$/, # Terence Terr$E
  Thaddeus => qr/^(T(ad|had(deus)?))$/, # Tad Thad
  Theodore => qr/^(T(ad|ed(d$E)?|heo(d(or(e|ick)|rick))?))$/, # Ted Theodrick Theodorick Tad Theo Tedd$E
  Theodosia => qr/^((Do(ra|sia)|Theo(do(ra|sia))?))$/, # Dosia Theo Theodora Dora
  Theophilus => qr/^((Oph$E|Theo(philus)?))$/, # Oph$E Theo
  Theresa => qr/^(T(e(r(esa|r$E)|ss($E|a)?)|h(eres[ae]|irsa|riza|ursa)|ic$E|r(ac$E|iss$E)))$/, # Therese Terr$E Tess Tess$E Thursa Tic$E Trac$E Triss$E Thriza Teresa Thirsa Tessa
  Thomas => qr/^(T(hom(as)?|om(m$E)?))$/, # Thom Tom Tomm$E
  Thomasine => qr/^(T(amzine|homas(a|ine)))$/, # Thomasa Tamzine
  Tilford => qr/^((Ford|Til(ford|l$E)))$/, # Till$E Ford
  Timothy => qr/^(Tim(m$E|othy)?)$/, # Tim Timm$E
  Tipton => qr/^(Tip(p$E|s$E|ton))$/, # Tipp$E Tips$E
  Tobias => qr/^((Bias|Tob($E|e|ias)))$/, # Tobe Tob$E Bias
  Tryphena => qr/^((Ph(e(n($E|a)|obe)|oen$E)|Tryphena))$/, # Phena Phoen$E Phen$E Pheobe
  Tryphosia => qr/^((Phos$E|Tryphosia))$/, # Phos$E
  Uriah => qr/^((Riah|Uriah))$/, # Riah
  Valentina => qr/^(Val(e(da|ntina|r$E)|l$E)?)$/, # Val Vall$E Valer$E Valeda
  Vanessa => qr/^((Essa|Nessa|Van(essa|n($E|a))?))$/, # Nessa Van Vann$E Vanna Essa
  Vernisee => qr/^((Nic$E|Vernisee))$/, # Nic$E
  Veronica => qr/^((Fr(ank$E|on$E)|Ron(n($E|a))?|V(eronica|ick$E|onn$E)))$/, # Frank$E Ronn$E Ronna Vonn$E Fron$E Ron Vick$E
  Victor => qr/^(Vic(k|tor)?)$/, # Vic Vick
  Victoria => qr/^((Tor($E|r$E)|Vic(k$E|toria)))$/, # Torr$E Tor$E Vick$E
  Vincent => qr/^(Vin(ce(nt)?|n$E|son)?)$/, # Vin Vince Vinn$E Vinson
  Violet => qr/^(Vi(ol(a|et))?)$/, # Viola Vi
  Virgil => qr/^(Virg(il)?)$/, # Virg
  Virginia => qr/^((G(en|in(a|ger|n$E))|J(ane|enn$E|in$E)|V(erg$E|irg($E|inia))))$/, # Ginger Ginn$E Jane Jenn$E Jin$E Virg$E Gen Verg$E Gina
  Vivian => qr/^(Vi(vian)?)$/, # Vi
  Vladimir => qr/^(V(ladimir|olodia))$/, # Volodia
  Waitstill => qr/^(Wait($E|still))$/, # Wait$E
  Waldo => qr/^((Oswald|Waldo))$/, # Oswald
  Wallace => qr/^(Wall($E|ace))$/, # Wall$E
  Walter => qr/^(Wa(lt(er)?|t))$/, # Wat Walt
  Webster => qr/^(Web(b|ster))$/, # Webb
  Wesley => qr/^(Wes(ley)?)$/, # Wes
  Wilber => qr/^(Wi(b|l(b(er|ur)|l)))$/, # Wib Will Wilbur
  Wilda => qr/^(Wil(da|l$E))$/, # Will$E
  Wilfred => qr/^((Fred|Wil(fred|l($E)?)))$/, # Fred Will$E Will
  Wilhelmina => qr/^((Helm$E|Min(a|n$E)|Wil(helmina|l$E|ma)))$/, # Helm$E Mina Minn$E Will$E Wilma
  William => qr/^((B(el[al]|ill($E)?)|Wil($E|helm|l($E|i(am|s))?)))$/, # Bill Will Will$E Bill$E Bell Bela Wil$E Wilhelm Willis
  Winfield => qr/^((Field|Win(field|n$E)?))$/, # Field Win Winn$E
  Winifred => qr/^((Fred(d$E)?|W(enefred|in(ifred|n($E|et)))))$/, # Fredd$E Winn$E Winnet Wenefred Fred
  Winton => qr/^(Wint(on)?)$/, # Wint
  Woodrow => qr/^((Drew|Wood($E|row)?))$/, # Wood Drew Wood$E
  Yolanda => qr/^(Yol(anda|onda))$/, # Yolonda
  Yulan => qr/^((Lan|Yul(an)?))$/, # Lan Yul
  Zachariah => qr/^((Rye|Z(ach($E|ar($E|ia[hs]))?|eke)))$/, # Zach Zacharias Zachar$E Zeke Zach$E Rye
  Zadock => qr/^((D(ick|ock)|Melchizedek|Z(adock|ed)))$/, # Dick Dock Melchizedek Zed
  Zebedee => qr/^(Zeb(edee)?)$/, # Zeb
  Zebulon => qr/^((Lon|Zeb(ulon)?))$/, # Lon Zeb
  Zedediah => qr/^((D(iah|yer)|Zed(ediah)?))$/, # Zed Diah Dyer
  Zelphia => qr/^(Zel(la|ph($E|ia))?)$/, # Zel Zella Zelph$E
  Zepaniah => qr/^(Zep(aniah|h))$/, # Zeph
);

%akin=
(
  Aaron => [qw<Ronald>],
  Abigail => [qw<Tabitha>],
  Adaline => [qw<Adelaide>],
  Adelaide => [qw<Elizabeth>],
  Adelphia => [qw<Adeline>],
  Aileen => [qw<Eileen Helen>],
  Albert => [qw<Adelbert>],
  Alice => [qw<Elicia>],
  Amelia => [qw<Mildred>],
  Anna => [qw<Maryanne Christina Antoinette Roseanne Hannah>],
  Arabella => [qw<Isabella Mehitabel>],
  Arnold => [qw<Ronald>],
  Bedelia => [qw<Delia Cordelia Adelaide>],
  Belinda => [qw<Melinda Linda Philinda>],
  Broderick => [qw<Richard>],
  Clarissa => [qw<Belinda Melinda Philinda>],
  Cordelia => [qw<Adelaide Bedelia Fedelia>],
  Edmund => [qw<Edward Sigismund>],
  Edwin => [qw<Edward>],
  Edwina => [qw<Edward>],
  Elaine => [qw<Eleanor>],
  Eleanor => [qw<Honora>],
  Elizabeth => [qw<Alice Alisha>],
  Elsie => [qw<Elizabeth>],
  Emeline => [qw<Amelia>],
  Eric => [qw<Derrick>],
  Eudora => [qw<Dorothy>],
  Fidelia => [qw<Delia Cordelia Adelaide>],
  Gabrilla => [qw<Gabrielle>],
  Genevieve => [qw<Jane Virginia>],
  Gerald => [qw<Jeremiah>],
  Hannah => [qw<Joanna Susannah Anna>],
  Harold => [qw<Henry>],
  Helena => [qw<Eleanor>],
  Henry => [qw<Harold>],
  Hepsabah => [qw<Elizabeth Rosabella>],
  Honora => [qw<Eleanor>],
  Isabella => [qw<Arabella Rosabella>],
  Jane => [qw<Genevieve Virginia Joanna>],
  Jefferson => [qw<Jeffery>],
  Jehu => [qw<John>],
  Jennifer => [qw<Winifred>],
  Jerita => [qw<Margarita Rosarita>],
  Jessica => [qw<Jane Virginia>],
  Joanna => [qw<Anna Hannah Suzannah>],
  John => [qw<Johnathan>],
  Jonathan => [qw<Nathaniel John>],
  Josiah => [qw<Joseph>],
  Leonora => [qw<Eleanor>],
  Linda => [qw<Belinda Melinda Philinda>],
  Loretta => [qw<Lorraine>],
  Lousie => [qw<Loretta>],
  Luann => [qw<Anna Louise>],
  Luciana => [qw<Lucinda Cynthia>],
  Lucinda => [qw<Cynthia>],
  Madeline => [qw<Magdalena>],
  Magdelina => [qw<Madeline>],
  Margaret => [qw<Mary Margarita>],
  Margarita => [qw<Rosarita>],
  Marianna => [qw<Mary Anne>],
  Martha => [qw<Patricia>],
  Mary => [qw<Maryanne Martha Margaret>],
  Maryanne => [qw<Mary Anna>],
  Marylou => [qw<Mary Louise>],
  Maureen => [qw<Mary>],
  Melinda => [qw<Belinda Linda Philinda>],
  Mindwell => [qw<Wilhelmina Minerva>],
  Montgomery => [qw<Lamont>],
  Nancy => [qw<Ann>],
  Nathaniel => [qw<Johnathan>],
  Obadiah => [qw<Jedediah>],
  Patience => [qw<Patricia>],
  Patrick => [qw<Peter>],
  Philinda => [qw<Belinda Linda Philinda>],
  Rodney => [qw<Roger Reginald Ronald>],
  Roger => [qw<Rodney>],
  Roland => [qw<Orlando Reginald>],
  Rosabella => [qw<Isabella Roseanne Rosarita Rosalyn>],
  Rosalyn => [qw<Rose Linda>],
  Roseanne => [qw<Rosabella Rosalyn Rosarita Anna>],
  Sabrina => [qw<Selina Serilla>],
  Samuel => [qw<Samson>],
  Selina => [qw<Sabrina>],
  Theodosia => [qw<Theodore>],
  Tryphena => [qw<Tryphosia>],
  Tryphosia => [qw<Tryphenia>],
  Virginia => [qw<Jane Genevieve>],
  Wilber => [qw<Gilbert>],
  Winfield => [qw<Winifred>],
  Zedediah => [qw<Jedediah Obediah>],
);


1