#############################################################################
#
# PerlCryptLib.ph
#
# This include file contains:
#
#     - CONSTANTs declaration;
#     - ENUMerated constants declaration;
#     - SUBroutine definitions for those cryptlib macros defined in original 
#       C code as preprocessed instructions;
#     - translations for data-structure used in cryptlib API.
#
#############################################################################


#############################################################################
#
#   C O N S T A N T S   A N D   E N U M E R A T I O N S
#
#############################################################################

	use constant {
					_CRYPTLIB_DEFINED	=> 1,
					CRYPTLIB_VERSION	=> 3220
	};

	##### Algorithm and Object Types
	use enum qw	(
					CRYPT_ALGO_NONE=0
					CRYPT_ALGO_DES
					CRYPT_ALGO_3DES
					CRYPT_ALGO_IDEA
					CRYPT_ALGO_CAST
					CRYPT_ALGO_RC2
					CRYPT_ALGO_RC4
					CRYPT_ALGO_RC5
					CRYPT_ALGO_AES
					CRYPT_ALGO_BLOWFISH
					CRYPT_ALGO_SKIPJACK
					CRYPT_ALGO_DH=100
					CRYPT_ALGO_RSA
					CRYPT_ALGO_DSA
					CRYPT_ALGO_ELGAMAL
					CRYPT_ALGO_KEA
					CRYPT_ALGO_MD2=200
					CRYPT_ALGO_MD4
					CRYPT_ALGO_MD5
					CRYPT_ALGO_SHA
					CRYPT_ALGO_RIPEMD160
					CRYPT_ALGO_SHA2
					CRYPT_ALGO_HMAC_MD5=300
					CRYPT_ALGO_HMAC_SHA
					CRYPT_ALGO_HMAC_RIPEMD160
					CRYPT_ALGO_LAST
	);

	## In order that we can scan through a range of algorithms with
	#   cryptQueryCapability(), we define the following boundary points for
	#   each algorithm class ##
	use constant {
					CRYPT_ALGO_FIRST_CONVENTIONAL	=> CRYPT_ALGO_DES,
					CRYPT_ALGO_LAST_CONVENTIONAL	=> CRYPT_ALGO_DH - 1,
					CRYPT_ALGO_FIRST_PKC			=> CRYPT_ALGO_DH,
					CRYPT_ALGO_LAST_PKC				=> CRYPT_ALGO_MD2 - 1,
					CRYPT_ALGO_FIRST_HASH			=> CRYPT_ALGO_MD2,
					CRYPT_ALGO_LAST_HASH			=> CRYPT_ALGO_HMAC_MD5 - 1,
					CRYPT_ALGO_FIRST_MAC			=> CRYPT_ALGO_HMAC_MD5,
					CRYPT_ALGO_LAST_MAC				=> CRYPT_ALGO_HMAC_MD5 + 99
	};

	## Block cipher modes ##
	use enum qw	(
					CRYPT_MODE_NONE=0
					CRYPT_MODE_ECB
					CRYPT_MODE_CBC
					CRYPT_MODE_CFB
					CRYPT_MODE_OFB
					CRYPT_MODE_LAST
	);


	use enum qw	(
					CRYPT_KEYSET_NONE=0
					CRYPT_KEYSET_FILE
					CRYPT_KEYSET_HTTP
					CRYPT_KEYSET_LDAP
					CRYPT_KEYSET_ODBC
					CRYPT_KEYSET_DATABASE
					CRYPT_KEYSET_PLUGIN
					CRYPT_KEYSET_ODBC_STORE
					CRYPT_KEYSET_DATABASE_STORE
					CRYPT_KEYSET_PLUGIN_STORE
					CRYPT_KEYSET_LAST
	);

	## Device subtypes ##
	use enum qw	(
					CRYPT_DEVICE_NONE=0
					CRYPT_DEVICE_FORTEZZA
					CRYPT_DEVICE_PKCS11
					CRYPT_DEVICE_CRYPTOAPI
					CRYPT_DEVICE_LAST
	);

	## Certificate subtypes ##
	use enum qw	(
					CRYPT_CERTTYPE_NONE=0
					CRYPT_CERTTYPE_CERTIFICATE
					CRYPT_CERTTYPE_ATTRIBUTE_CERT
					CRYPT_CERTTYPE_CERTCHAIN
					CRYPT_CERTTYPE_CERTREQUEST
					CRYPT_CERTTYPE_REQUEST_CERT
					CRYPT_CERTTYPE_REQUEST_REVOCATION
					CRYPT_CERTTYPE_CRL
					CRYPT_CERTTYPE_CMS_ATTRIBUTES
					CRYPT_CERTTYPE_RTCS_REQUEST
					CRYPT_CERTTYPE_RTCS_RESPONSE
					CRYPT_CERTTYPE_OCSP_REQUEST
					CRYPT_CERTTYPE_OCSP_RESPONSE
					CRYPT_CERTTYPE_PKIUSER
					CRYPT_CERTTYPE_LAST
	);

	## Envelope/data format subtypes ##
	use enum qw	(
					CRYPT_FORMAT_NONE=0
					CRYPT_FORMAT_AUTO
					CRYPT_FORMAT_CRYPTLIB
					CRYPT_FORMAT_CMS
					CRYPT_FORMAT_SMIME
					CRYPT_FORMAT_PGP
					CRYPT_FORMAT_LAST
	);
	use constant {
					CRYPT_FORMAT_PKCS7		=> CRYPT_FORMAT_CMS
	};

	## Session subtypes ##
	use enum qw	(
					CRYPT_SESSION_NONE=0
					CRYPT_SESSION_SSH
					CRYPT_SESSION_SSH_SERVER
					CRYPT_SESSION_SSL
					CRYPT_SESSION_SSL_SERVER
					CRYPT_SESSION_RTCS
					CRYPT_SESSION_RTCS_SERVER
					CRYPT_SESSION_OCSP
					CRYPT_SESSION_OCSP_SERVER
					CRYPT_SESSION_TSP
					CRYPT_SESSION_TSP_SERVER
					CRYPT_SESSION_CMP
					CRYPT_SESSION_CMP_SERVER
					CRYPT_SESSION_SCEP
					CRYPT_SESSION_SCEP_SERVER
					CRYPT_SESSION_CERTSTORE_SERVER
					CRYPT_SESSION_LAST
	);

	## User subtypes ##
	use enum qw	(
					CRYPT_USER_NONE=0
					CRYPT_USER_NORMAL
					CRYPT_USER_SO
					CRYPT_USER_CA
					CRYPT_USER_LAST
	);


	#############################################################################
	# Attribute Types
	#############################################################################

	## Attribute types.  These are arranged in the following order:
	#
	#	PROPERTY	- Object property
	#	ATTRIBUTE	- Generic attributes
	#	OPTION		- Global or object-specific config.option
	#	CTXINFO		- Context-specific attribute
	#	CERTINFO	- Certificate-specific attribute
	#	KEYINFO		- Keyset-specific attribute
	#	DEVINFO		- Device-specific attribute
	#	ENVINFO		- Envelope-specific attribute
	#	SESSINFO	- Session-specific attribute
	#	USERINFO	- User-specific attribute
	##

	use enum qw	(
					CRYPT_ATTRIBUTE_NONE=0
					CRYPT_PROPERTY_FIRST
					CRYPT_PROPERTY_HIGHSECURITY
					CRYPT_PROPERTY_OWNER
					CRYPT_PROPERTY_FORWARDCOUNT
					CRYPT_PROPERTY_LOCKED
					CRYPT_PROPERTY_USAGECOUNT
					CRYPT_PROPERTY_NONEXPORTABLE
					CRYPT_PROPERTY_LAST
					CRYPT_GENERIC_FIRST
					CRYPT_ATTRIBUTE_ERRORTYPE
					CRYPT_ATTRIBUTE_ERRORLOCUS
					CRYPT_ATTRIBUTE_INT_ERRORCODE
					CRYPT_ATTRIBUTE_INT_ERRORMESSAGE
					CRYPT_ATTRIBUTE_CURRENT_GROUP
					CRYPT_ATTRIBUTE_CURRENT
					CRYPT_ATTRIBUTE_CURRENT_INSTANCE
					CRYPT_ATTRIBUTE_BUFFERSIZE
					CRYPT_GENERIC_LAST
					CRYPT_OPTION_FIRST=100
					CRYPT_OPTION_INFO_DESCRIPTION
					CRYPT_OPTION_INFO_COPYRIGHT
					CRYPT_OPTION_INFO_MAJORVERSION
					CRYPT_OPTION_INFO_MINORVERSION
					CRYPT_OPTION_INFO_STEPPING
					CRYPT_OPTION_ENCR_ALGO
					CRYPT_OPTION_ENCR_HASH
					CRYPT_OPTION_ENCR_MAC
					CRYPT_OPTION_PKC_ALGO
					CRYPT_OPTION_PKC_KEYSIZE
					CRYPT_OPTION_SIG_ALGO
					CRYPT_OPTION_SIG_KEYSIZE
					CRYPT_OPTION_KEYING_ALGO
					CRYPT_OPTION_KEYING_ITERATIONS
					CRYPT_OPTION_CERT_SIGNUNRECOGNISEDATTRIBUTES
					CRYPT_OPTION_CERT_VALIDITY
					CRYPT_OPTION_CERT_UPDATEINTERVAL
					CRYPT_OPTION_CERT_COMPLIANCELEVEL
					CRYPT_OPTION_CERT_REQUIREPOLICY
					CRYPT_OPTION_CMS_DEFAULTATTRIBUTES
					CRYPT_OPTION_KEYS_LDAP_OBJECTCLASS
					CRYPT_OPTION_KEYS_LDAP_OBJECTTYPE
					CRYPT_OPTION_KEYS_LDAP_FILTER
					CRYPT_OPTION_KEYS_LDAP_CACERTNAME
					CRYPT_OPTION_KEYS_LDAP_CERTNAME
					CRYPT_OPTION_KEYS_LDAP_CRLNAME
					CRYPT_OPTION_KEYS_LDAP_EMAILNAME
					CRYPT_OPTION_DEVICE_PKCS11_DVR01
					CRYPT_OPTION_DEVICE_PKCS11_DVR02
					CRYPT_OPTION_DEVICE_PKCS11_DVR03
					CRYPT_OPTION_DEVICE_PKCS11_DVR04
					CRYPT_OPTION_DEVICE_PKCS11_DVR05
					CRYPT_OPTION_DEVICE_PKCS11_HARDWAREONLY
					CRYPT_OPTION_NET_SOCKS_SERVER
					CRYPT_OPTION_NET_SOCKS_USERNAME
					CRYPT_OPTION_NET_HTTP_PROXY
					CRYPT_OPTION_NET_CONNECTTIMEOUT
					CRYPT_OPTION_NET_READTIMEOUT
					CRYPT_OPTION_NET_WRITETIMEOUT
					CRYPT_OPTION_MISC_ASYNCINIT
					CRYPT_OPTION_MISC_SIDECHANNELPROTECTION
					CRYPT_OPTION_CONFIGCHANGED
					CRYPT_OPTION_SELFTESTOK
					CRYPT_OPTION_LAST
					CRYPT_CTXINFO_FIRST=1000
					CRYPT_CTXINFO_ALGO
					CRYPT_CTXINFO_MODE
					CRYPT_CTXINFO_NAME_ALGO
					CRYPT_CTXINFO_NAME_MODE
					CRYPT_CTXINFO_KEYSIZE
					CRYPT_CTXINFO_BLOCKSIZE
					CRYPT_CTXINFO_IVSIZE
					CRYPT_CTXINFO_KEYING_ALGO
					CRYPT_CTXINFO_KEYING_ITERATIONS
					CRYPT_CTXINFO_KEYING_SALT
					CRYPT_CTXINFO_KEYING_VALUE
					CRYPT_CTXINFO_KEY
					CRYPT_CTXINFO_KEY_COMPONENTS
					CRYPT_CTXINFO_IV
					CRYPT_CTXINFO_HASHVALUE
					CRYPT_CTXINFO_LABEL
					CRYPT_CTXINFO_LAST
					CRYPT_CERTINFO_FIRST=2000
					CRYPT_CERTINFO_SELFSIGNED
					CRYPT_CERTINFO_IMMUTABLE
					CRYPT_CERTINFO_XYZZY
					CRYPT_CERTINFO_CERTTYPE
					CRYPT_CERTINFO_FINGERPRINT
					CRYPT_CERTINFO_FINGERPRINT_SHA
					CRYPT_CERTINFO_CURRENT_CERTIFICATE
					CRYPT_CERTINFO_CURRENT_EXTENSION
					CRYPT_CERTINFO_CURRENT_FIELD
					CRYPT_CERTINFO_CURRENT_COMPONENT
					CRYPT_CERTINFO_TRUSTED_USAGE
					CRYPT_CERTINFO_TRUSTED_IMPLICIT
					CRYPT_CERTINFO_SIGNATURELEVEL
					CRYPT_CERTINFO_VERSION
					CRYPT_CERTINFO_SERIALNUMBER
					CRYPT_CERTINFO_SUBJECTPUBLICKEYINFO
					CRYPT_CERTINFO_CERTIFICATE
					CRYPT_CERTINFO_CACERTIFICATE
					CRYPT_CERTINFO_ISSUERNAME
					CRYPT_CERTINFO_VALIDFROM
					CRYPT_CERTINFO_VALIDTO
					CRYPT_CERTINFO_SUBJECTNAME
					CRYPT_CERTINFO_ISSUERUNIQUEID
					CRYPT_CERTINFO_SUBJECTUNIQUEID
					CRYPT_CERTINFO_CERTREQUEST
					CRYPT_CERTINFO_THISUPDATE
					CRYPT_CERTINFO_NEXTUPDATE
					CRYPT_CERTINFO_REVOCATIONDATE
					CRYPT_CERTINFO_REVOCATIONSTATUS
					CRYPT_CERTINFO_CERTSTATUS
					CRYPT_CERTINFO_DN
					CRYPT_CERTINFO_PKIUSER_ID
					CRYPT_CERTINFO_PKIUSER_ISSUEPASSWORD
					CRYPT_CERTINFO_PKIUSER_REVPASSWORD
	);
	use constant {
					CRYPT_OPTION_SMIME_DEFAULTATTRIBUTES	=> CRYPT_OPTION_CMS_DEFAULTATTRIBUTES,
					CRYPT_CERTINFO_FINGERPRINT_MD5			=> CRYPT_CERTINFO_FINGERPRINT,
					CRYPT_CERTINFO_USERCERTIFICATE			=> CRYPT_CERTINFO_CERTIFICATE
	};

	use enum (
					'CRYPT_CERTINFO_COUNTRYNAME=' . (CRYPT_CERTINFO_FIRST + 100),
					qw	(
							CRYPT_CERTINFO_STATEORPROVINCENAME
							CRYPT_CERTINFO_LOCALITYNAME
							CRYPT_CERTINFO_ORGANIZATIONNAME
							CRYPT_CERTINFO_ORGANIZATIONALUNITNAME
							CRYPT_CERTINFO_COMMONNAME
							CRYPT_CERTINFO_OTHERNAME_TYPEID
							CRYPT_CERTINFO_OTHERNAME_VALUE
							CRYPT_CERTINFO_RFC822NAME
							CRYPT_CERTINFO_DNSNAME
							CRYPT_CERTINFO_DIRECTORYNAME
							CRYPT_CERTINFO_EDIPARTYNAME_NAMEASSIGNER
							CRYPT_CERTINFO_EDIPARTYNAME_PARTYNAME
							CRYPT_CERTINFO_UNIFORMRESOURCEIDENTIFIER
							CRYPT_CERTINFO_IPADDRESS
							CRYPT_CERTINFO_REGISTEREDID
						)
	);
	use constant {
							CRYPT_CERTINFO_ORGANISATIONNAME 		=> CRYPT_CERTINFO_ORGANIZATIONNAME,
							CRYPT_CERTINFO_ORGANISATIONALUNITNAME	=> CRYPT_CERTINFO_ORGANIZATIONALUNITNAME,
							CRYPT_CERTINFO_EMAIL 					=> CRYPT_CERTINFO_RFC822NAME
	};

	use enum (
					'CRYPT_CERTINFO_CHALLENGEPASSWORD=' . (CRYPT_CERTINFO_FIRST + 200),
					qw	(
							CRYPT_CERTINFO_CRLEXTREASON
							CRYPT_CERTINFO_KEYFEATURES
							CRYPT_CERTINFO_AUTHORITYINFOACCESS
							CRYPT_CERTINFO_AUTHORITYINFO_RTCS
							CRYPT_CERTINFO_AUTHORITYINFO_OCSP
							CRYPT_CERTINFO_AUTHORITYINFO_CAISSUERS
							CRYPT_CERTINFO_AUTHORITYINFO_CERTSTORE
							CRYPT_CERTINFO_AUTHORITYINFO_CRLS
							CRYPT_CERTINFO_BIOMETRICINFO
							CRYPT_CERTINFO_BIOMETRICINFO_TYPE
							CRYPT_CERTINFO_BIOMETRICINFO_HASHALGO
							CRYPT_CERTINFO_BIOMETRICINFO_HASH
							CRYPT_CERTINFO_BIOMETRICINFO_URL
							CRYPT_CERTINFO_QCSTATEMENT
							CRYPT_CERTINFO_QCSTATEMENT_SEMANTICS
							CRYPT_CERTINFO_QCSTATEMENT_REGISTRATIONAUTHORITY
							CRYPT_CERTINFO_OCSP_NONCE
							CRYPT_CERTINFO_OCSP_RESPONSE
							CRYPT_CERTINFO_OCSP_RESPONSE_OCSP
							CRYPT_CERTINFO_OCSP_NOCHECK
							CRYPT_CERTINFO_OCSP_ARCHIVECUTOFF
							CRYPT_CERTINFO_SUBJECTINFOACCESS
							CRYPT_CERTINFO_SUBJECTINFO_CAREPOSITORY
							CRYPT_CERTINFO_SUBJECTINFO_TIMESTAMPING
							CRYPT_CERTINFO_SIGG_DATEOFCERTGEN
							CRYPT_CERTINFO_SIGG_PROCURATION
							CRYPT_CERTINFO_SIGG_PROCURE_COUNTRY
							CRYPT_CERTINFO_SIGG_PROCURE_TYPEOFSUBSTITUTION
							CRYPT_CERTINFO_SIGG_PROCURE_SIGNINGFOR
							CRYPT_CERTINFO_SIGG_MONETARYLIMIT
							CRYPT_CERTINFO_SIGG_MONETARY_CURRENCY
							CRYPT_CERTINFO_SIGG_MONETARY_AMOUNT
							CRYPT_CERTINFO_SIGG_MONETARY_EXPONENT
							CRYPT_CERTINFO_SIGG_RESTRICTION
							CRYPT_CERTINFO_STRONGEXTRANET
							CRYPT_CERTINFO_STRONGEXTRANET_ZONE
							CRYPT_CERTINFO_STRONGEXTRANET_ID
							CRYPT_CERTINFO_SUBJECTDIRECTORYATTRIBUTES
							CRYPT_CERTINFO_SUBJECTDIR_TYPE
							CRYPT_CERTINFO_SUBJECTDIR_VALUES
							CRYPT_CERTINFO_SUBJECTKEYIDENTIFIER
							CRYPT_CERTINFO_KEYUSAGE
							CRYPT_CERTINFO_PRIVATEKEYUSAGEPERIOD
							CRYPT_CERTINFO_PRIVATEKEY_NOTBEFORE
							CRYPT_CERTINFO_PRIVATEKEY_NOTAFTER
							CRYPT_CERTINFO_SUBJECTALTNAME
							CRYPT_CERTINFO_ISSUERALTNAME
							CRYPT_CERTINFO_BASICCONSTRAINTS
							CRYPT_CERTINFO_CA
							CRYPT_CERTINFO_PATHLENCONSTRAINT
							CRYPT_CERTINFO_CRLNUMBER
							CRYPT_CERTINFO_CRLREASON
							CRYPT_CERTINFO_HOLDINSTRUCTIONCODE
							CRYPT_CERTINFO_INVALIDITYDATE
							CRYPT_CERTINFO_DELTACRLINDICATOR
							CRYPT_CERTINFO_ISSUINGDISTRIBUTIONPOINT
							CRYPT_CERTINFO_ISSUINGDIST_FULLNAME
							CRYPT_CERTINFO_ISSUINGDIST_USERCERTSONLY
							CRYPT_CERTINFO_ISSUINGDIST_CACERTSONLY
							CRYPT_CERTINFO_ISSUINGDIST_SOMEREASONSONLY
							CRYPT_CERTINFO_ISSUINGDIST_INDIRECTCRL
							CRYPT_CERTINFO_CERTIFICATEISSUER
							CRYPT_CERTINFO_NAMECONSTRAINTS
							CRYPT_CERTINFO_PERMITTEDSUBTREES
							CRYPT_CERTINFO_EXCLUDEDSUBTREES
							CRYPT_CERTINFO_CRLDISTRIBUTIONPOINT
							CRYPT_CERTINFO_CRLDIST_FULLNAME
							CRYPT_CERTINFO_CRLDIST_REASONS
							CRYPT_CERTINFO_CRLDIST_CRLISSUER
							CRYPT_CERTINFO_CERTIFICATEPOLICIES
							CRYPT_CERTINFO_CERTPOLICYID
							CRYPT_CERTINFO_CERTPOLICY_CPSURI
							CRYPT_CERTINFO_CERTPOLICY_ORGANIZATION
							CRYPT_CERTINFO_CERTPOLICY_NOTICENUMBERS
							CRYPT_CERTINFO_CERTPOLICY_EXPLICITTEXT
							CRYPT_CERTINFO_POLICYMAPPINGS
							CRYPT_CERTINFO_ISSUERDOMAINPOLICY
							CRYPT_CERTINFO_SUBJECTDOMAINPOLICY
							CRYPT_CERTINFO_AUTHORITYKEYIDENTIFIER
							CRYPT_CERTINFO_AUTHORITY_KEYIDENTIFIER
							CRYPT_CERTINFO_AUTHORITY_CERTISSUER
							CRYPT_CERTINFO_AUTHORITY_CERTSERIALNUMBER
							CRYPT_CERTINFO_POLICYCONSTRAINTS
							CRYPT_CERTINFO_REQUIREEXPLICITPOLICY
							CRYPT_CERTINFO_INHIBITPOLICYMAPPING
							CRYPT_CERTINFO_EXTKEYUSAGE
							CRYPT_CERTINFO_EXTKEY_MS_INDIVIDUALCODESIGNING
							CRYPT_CERTINFO_EXTKEY_MS_COMMERCIALCODESIGNING
							CRYPT_CERTINFO_EXTKEY_MS_CERTTRUSTLISTSIGNING
							CRYPT_CERTINFO_EXTKEY_MS_TIMESTAMPSIGNING
							CRYPT_CERTINFO_EXTKEY_MS_SERVERGATEDCRYPTO
							CRYPT_CERTINFO_EXTKEY_MS_ENCRYPTEDFILESYSTEM
							CRYPT_CERTINFO_EXTKEY_SERVERAUTH
							CRYPT_CERTINFO_EXTKEY_CLIENTAUTH
							CRYPT_CERTINFO_EXTKEY_CODESIGNING
							CRYPT_CERTINFO_EXTKEY_EMAILPROTECTION
							CRYPT_CERTINFO_EXTKEY_IPSECENDSYSTEM
							CRYPT_CERTINFO_EXTKEY_IPSECTUNNEL
							CRYPT_CERTINFO_EXTKEY_IPSECUSER
							CRYPT_CERTINFO_EXTKEY_TIMESTAMPING
							CRYPT_CERTINFO_EXTKEY_OCSPSIGNING
							CRYPT_CERTINFO_EXTKEY_DIRECTORYSERVICE
							CRYPT_CERTINFO_EXTKEY_ANYKEYUSAGE
							CRYPT_CERTINFO_EXTKEY_NS_SERVERGATEDCRYPTO
							CRYPT_CERTINFO_EXTKEY_VS_SERVERGATEDCRYPTO_CA
							CRYPT_CERTINFO_FRESHESTCRL
							CRYPT_CERTINFO_FRESHESTCRL_FULLNAME
							CRYPT_CERTINFO_FRESHESTCRL_REASONS
							CRYPT_CERTINFO_FRESHESTCRL_CRLISSUER
							CRYPT_CERTINFO_INHIBITANYPOLICY
							CRYPT_CERTINFO_NS_CERTTYPE
							CRYPT_CERTINFO_NS_BASEURL
							CRYPT_CERTINFO_NS_REVOCATIONURL
							CRYPT_CERTINFO_NS_CAREVOCATIONURL
							CRYPT_CERTINFO_NS_CERTRENEWALURL
							CRYPT_CERTINFO_NS_CAPOLICYURL
							CRYPT_CERTINFO_NS_SSLSERVERNAME
							CRYPT_CERTINFO_NS_COMMENT
							CRYPT_CERTINFO_SET_HASHEDROOTKEY
							CRYPT_CERTINFO_SET_ROOTKEYTHUMBPRINT
							CRYPT_CERTINFO_SET_CERTIFICATETYPE
							CRYPT_CERTINFO_SET_MERCHANTDATA
							CRYPT_CERTINFO_SET_MERID
							CRYPT_CERTINFO_SET_MERACQUIRERBIN
							CRYPT_CERTINFO_SET_MERCHANTLANGUAGE
							CRYPT_CERTINFO_SET_MERCHANTNAME
							CRYPT_CERTINFO_SET_MERCHANTCITY
							CRYPT_CERTINFO_SET_MERCHANTSTATEPROVINCE
							CRYPT_CERTINFO_SET_MERCHANTPOSTALCODE
							CRYPT_CERTINFO_SET_MERCHANTCOUNTRYNAME
							CRYPT_CERTINFO_SET_MERCOUNTRY
							CRYPT_CERTINFO_SET_MERAUTHFLAG
							CRYPT_CERTINFO_SET_CERTCARDREQUIRED
							CRYPT_CERTINFO_SET_TUNNELING
							CRYPT_CERTINFO_SET_TUNNELINGFLAG
							CRYPT_CERTINFO_SET_TUNNELINGALGID
						)
	);
	use constant {
					CRYPT_CERTINFO_AUTHORITY 			=> CRYPT_CERTINFO_CA,
					CRYPT_CERTINFO_SET_TUNNELLING 		=> CRYPT_CERTINFO_SET_TUNNELING,
					CRYPT_CERTINFO_SET_TUNNELLINGFLAG 	=> CRYPT_CERTINFO_SET_TUNNELINGFLAG,
					CRYPT_CERTINFO_SET_TUNNELLINGALGID 	=> CRYPT_CERTINFO_SET_TUNNELINGALGID
	};

	## S/MIME attributes ##
	use enum (
					'CRYPT_CERTINFO_CMS_CONTENTTYPE=' . (CRYPT_CERTINFO_FIRST + 500),
					qw	(
							CRYPT_CERTINFO_CMS_MESSAGEDIGEST
							CRYPT_CERTINFO_CMS_SIGNINGTIME
							CRYPT_CERTINFO_CMS_COUNTERSIGNATURE
							CRYPT_CERTINFO_CMS_SIGNINGDESCRIPTION
							CRYPT_CERTINFO_CMS_SMIMECAPABILITIES
							CRYPT_CERTINFO_CMS_SMIMECAP_3DES
							CRYPT_CERTINFO_CMS_SMIMECAP_AES
							CRYPT_CERTINFO_CMS_SMIMECAP_CAST128
							CRYPT_CERTINFO_CMS_SMIMECAP_IDEA
							CRYPT_CERTINFO_CMS_SMIMECAP_RC2
							CRYPT_CERTINFO_CMS_SMIMECAP_RC5
							CRYPT_CERTINFO_CMS_SMIMECAP_SKIPJACK
							CRYPT_CERTINFO_CMS_SMIMECAP_DES
							CRYPT_CERTINFO_CMS_SMIMECAP_PREFERSIGNEDDATA
							CRYPT_CERTINFO_CMS_SMIMECAP_CANNOTDECRYPTANY
							CRYPT_CERTINFO_CMS_RECEIPTREQUEST
							CRYPT_CERTINFO_CMS_RECEIPT_CONTENTIDENTIFIER
							CRYPT_CERTINFO_CMS_RECEIPT_FROM
							CRYPT_CERTINFO_CMS_RECEIPT_TO
							CRYPT_CERTINFO_CMS_SECURITYLABEL
							CRYPT_CERTINFO_CMS_SECLABEL_CLASSIFICATION
							CRYPT_CERTINFO_CMS_SECLABEL_POLICY
							CRYPT_CERTINFO_CMS_SECLABEL_PRIVACYMARK
							CRYPT_CERTINFO_CMS_SECLABEL_CATTYPE
							CRYPT_CERTINFO_CMS_SECLABEL_CATVALUE
							CRYPT_CERTINFO_CMS_MLEXPANSIONHISTORY
							CRYPT_CERTINFO_CMS_MLEXP_ENTITYIDENTIFIER
							CRYPT_CERTINFO_CMS_MLEXP_TIME
							CRYPT_CERTINFO_CMS_MLEXP_NONE
							CRYPT_CERTINFO_CMS_MLEXP_INSTEADOF
							CRYPT_CERTINFO_CMS_MLEXP_INADDITIONTO
							CRYPT_CERTINFO_CMS_CONTENTHINTS
							CRYPT_CERTINFO_CMS_CONTENTHINT_DESCRIPTION
							CRYPT_CERTINFO_CMS_CONTENTHINT_TYPE
							CRYPT_CERTINFO_CMS_EQUIVALENTLABEL
							CRYPT_CERTINFO_CMS_EQVLABEL_POLICY
							CRYPT_CERTINFO_CMS_EQVLABEL_CLASSIFICATION
							CRYPT_CERTINFO_CMS_EQVLABEL_PRIVACYMARK
							CRYPT_CERTINFO_CMS_EQVLABEL_CATTYPE
							CRYPT_CERTINFO_CMS_EQVLABEL_CATVALUE
							CRYPT_CERTINFO_CMS_SIGNINGCERTIFICATE
							CRYPT_CERTINFO_CMS_SIGNINGCERT_ESSCERTID
							CRYPT_CERTINFO_CMS_SIGNINGCERT_POLICIES
							CRYPT_CERTINFO_CMS_SIGNATUREPOLICYID
							CRYPT_CERTINFO_CMS_SIGPOLICYID
							CRYPT_CERTINFO_CMS_SIGPOLICYHASH
							CRYPT_CERTINFO_CMS_SIGPOLICY_CPSURI
							CRYPT_CERTINFO_CMS_SIGPOLICY_ORGANIZATION
							CRYPT_CERTINFO_CMS_SIGPOLICY_NOTICENUMBERS
							CRYPT_CERTINFO_CMS_SIGPOLICY_EXPLICITTEXT
							CRYPT_CERTINFO_CMS_SIGTYPEIDENTIFIER
							CRYPT_CERTINFO_CMS_SIGTYPEID_ORIGINATORSIG
							CRYPT_CERTINFO_CMS_SIGTYPEID_DOMAINSIG
							CRYPT_CERTINFO_CMS_SIGTYPEID_ADDITIONALATTRIBUTES
							CRYPT_CERTINFO_CMS_SIGTYPEID_REVIEWSIG
							CRYPT_CERTINFO_CMS_NONCE
							CRYPT_CERTINFO_SCEP_MESSAGETYPE
							CRYPT_CERTINFO_SCEP_PKISTATUS
							CRYPT_CERTINFO_SCEP_FAILINFO
							CRYPT_CERTINFO_SCEP_SENDERNONCE
							CRYPT_CERTINFO_SCEP_RECIPIENTNONCE
							CRYPT_CERTINFO_SCEP_TRANSACTIONID
							CRYPT_CERTINFO_CMS_SPCAGENCYINFO
							CRYPT_CERTINFO_CMS_SPCAGENCYURL
							CRYPT_CERTINFO_CMS_SPCSTATEMENTTYPE
							CRYPT_CERTINFO_CMS_SPCSTMT_INDIVIDUALCODESIGNING
							CRYPT_CERTINFO_CMS_SPCSTMT_COMMERCIALCODESIGNING
							CRYPT_CERTINFO_CMS_SPCOPUSINFO
							CRYPT_CERTINFO_CMS_SPCOPUSINFO_NAME
							CRYPT_CERTINFO_CMS_SPCOPUSINFO_URL
							CRYPT_CERTINFO_LAST
							CRYPT_KEYINFO_FIRST=3000
							CRYPT_KEYINFO_QUERY
							CRYPT_KEYINFO_QUERY_REQUESTS
							CRYPT_KEYINFO_LAST
							CRYPT_DEVINFO_FIRST=4000
							CRYPT_DEVINFO_INITIALISE
							CRYPT_DEVINFO_AUTHENT_USER
							CRYPT_DEVINFO_AUTHENT_SUPERVISOR
							CRYPT_DEVINFO_SET_AUTHENT_USER
							CRYPT_DEVINFO_SET_AUTHENT_SUPERVISOR
							CRYPT_DEVINFO_ZEROISE
							CRYPT_DEVINFO_LOGGEDIN
							CRYPT_DEVINFO_LABEL
							CRYPT_DEVINFO_LAST
							CRYPT_ENVINFO_FIRST=5000
							CRYPT_ENVINFO_DATASIZE
							CRYPT_ENVINFO_COMPRESSION
							CRYPT_ENVINFO_CONTENTTYPE
							CRYPT_ENVINFO_DETACHEDSIGNATURE
							CRYPT_ENVINFO_SIGNATURE_RESULT
							CRYPT_ENVINFO_MAC
							CRYPT_ENVINFO_PASSWORD
							CRYPT_ENVINFO_KEY
							CRYPT_ENVINFO_SIGNATURE
							CRYPT_ENVINFO_SIGNATURE_EXTRADATA
							CRYPT_ENVINFO_RECIPIENT
							CRYPT_ENVINFO_PUBLICKEY
							CRYPT_ENVINFO_PRIVATEKEY
							CRYPT_ENVINFO_PRIVATEKEY_LABEL
							CRYPT_ENVINFO_ORIGINATOR
							CRYPT_ENVINFO_SESSIONKEY
							CRYPT_ENVINFO_HASH
							CRYPT_ENVINFO_TIMESTAMP
							CRYPT_ENVINFO_KEYSET_SIGCHECK
							CRYPT_ENVINFO_KEYSET_ENCRYPT
							CRYPT_ENVINFO_KEYSET_DECRYPT
							CRYPT_ENVINFO_LAST
							CRYPT_SESSINFO_FIRST=6000
							CRYPT_SESSINFO_ACTIVE
							CRYPT_SESSINFO_CONNECTIONACTIVE
							CRYPT_SESSINFO_USERNAME
							CRYPT_SESSINFO_PASSWORD
							CRYPT_SESSINFO_PRIVATEKEY
							CRYPT_SESSINFO_KEYSET
							CRYPT_SESSINFO_AUTHRESPONSE
							CRYPT_SESSINFO_SERVER_NAME
							CRYPT_SESSINFO_SERVER_PORT
							CRYPT_SESSINFO_SERVER_FINGERPRINT
							CRYPT_SESSINFO_CLIENT_NAME
							CRYPT_SESSINFO_CLIENT_PORT
							CRYPT_SESSINFO_SESSION
							CRYPT_SESSINFO_NETWORKSOCKET
							CRYPT_SESSINFO_VERSION
							CRYPT_SESSINFO_REQUEST
							CRYPT_SESSINFO_RESPONSE
							CRYPT_SESSINFO_CACERTIFICATE
							CRYPT_SESSINFO_TSP_MSGIMPRINT
							CRYPT_SESSINFO_CMP_REQUESTTYPE
							CRYPT_SESSINFO_CMP_PKIBOOT
							CRYPT_SESSINFO_CMP_PRIVKEYSET
							CRYPT_SESSINFO_SSH_CHANNEL
							CRYPT_SESSINFO_SSH_CHANNEL_TYPE
							CRYPT_SESSINFO_SSH_CHANNEL_ARG1
							CRYPT_SESSINFO_SSH_CHANNEL_ARG2
							CRYPT_SESSINFO_SSH_CHANNEL_ACTIVE
							CRYPT_SESSINFO_LAST
							CRYPT_USERINFO_FIRST=7000
							CRYPT_USERINFO_PASSWORD
							CRYPT_USERINFO_CAKEY_CERTSIGN
							CRYPT_USERINFO_CAKEY_CRLSIGN
							CRYPT_USERINFO_CAKEY_RTCSSIGN
							CRYPT_USERINFO_CAKEY_OCSPSIGN
							CRYPT_USERINFO_LAST
						)
	);
	use constant {
					CRYPT_DEVINFO_INITIALIZE	=> CRYPT_DEVINFO_INITIALISE,
					CRYPT_DEVINFO_ZEROIZE		=> CRYPT_DEVINFO_ZEROISE,
					CRYPT_ATTRIBUTE_LAST		=> CRYPT_USERINFO_LAST
	};


	#############################################################################
	# Attribute Subtypes and Related Values
	#############################################################################

	## Flags for the X.509 keyUsage extension ##
	use enum qw	(	BITMASK:
					CRYPT_KEYUSAGE_NONE=0
					CRYPT_KEYUSAGE_DIGITALSIGNATURE=1
					CRYPT_KEYUSAGE_NONREPUDIATION
					CRYPT_KEYUSAGE_KEYENCIPHERMENT
					CRYPT_KEYUSAGE_DATAENCIPHERMENT
					CRYPT_KEYUSAGE_KEYAGREEMENT
					CRYPT_KEYUSAGE_KEYCERTSIGN
					CRYPT_KEYUSAGE_CRLSIGN
					CRYPT_KEYUSAGE_ENCIPHERONLY
					CRYPT_KEYUSAGE_DECIPHERONLY
					CRYPT_KEYUSAGE_LAST
	);

	## X.509 cRLReason and cryptlib cRLExtReason codes ##
	use enum qw	(
					CRYPT_CRLREASON_UNSPECIFIED=0
					CRYPT_CRLREASON_KEYCOMPROMISE
					CRYPT_CRLREASON_CACOMPROMISE
					CRYPT_CRLREASON_AFFILIATIONCHANGED
					CRYPT_CRLREASON_SUPERSEDED
					CRYPT_CRLREASON_CESSATIONOFOPERATION
					CRYPT_CRLREASON_CERTIFICATEHOLD
					CRYPT_CRLREASON_REMOVEFROMCRL=8
					CRYPT_CRLREASON_PRIVILEGEWITHDRAWN
					CRYPT_CRLREASON_AACOMPROMISE
					CRYPT_CRLREASON_LAST
					CRYPT_CRLREASON_NEVERVALID=20
					CRYPT_CRLEXTREASON_LAST
	);


	## X.509 CRL reason flags.  These identify the same thing as the cRLReason
	#   codes but allow for multiple reasons to be specified.  Note that these
	#   don't follow the X.509 naming since in that scheme the enumerated types
	#   and bitflags have the same names
	##
	use enum qw	(	BITMASK:
					CRYPT_CRLREASONFLAG_UNUSED=1
					CRYPT_CRLREASONFLAG_KEYCOMPROMISE
					CRYPT_CRLREASONFLAG_CACOMPROMISE
					CRYPT_CRLREASONFLAG_AFFILIATIONCHANGED
					CRYPT_CRLREASONFLAG_SUPERSEDED
					CRYPT_CRLREASONFLAG_CESSATIONOFOPERATION
					CRYPT_CRLREASONFLAG_CERTIFICATEHOLD
					CRYPT_CRLREASONFLAG_LAST
	);

	## X.509 CRL holdInstruction codes ##
	use enum qw	(
					CRYPT_HOLDINSTRUCTION_NONE=0
					CRYPT_HOLDINSTRUCTION_CALLISSUER
					CRYPT_HOLDINSTRUCTION_REJECT
					CRYPT_HOLDINSTRUCTION_PICKUPTOKEN
					CRYPT_HOLDINSTRUCTION_LAST
	);

	## Certificate checking compliance levels ##

	use enum qw	(
					CRYPT_COMPLIANCELEVEL_OBLIVIOUS=0
					CRYPT_COMPLIANCELEVEL_REDUCED
					CRYPT_COMPLIANCELEVEL_STANDARD
					CRYPT_COMPLIANCELEVEL_PKIX_PARTIAL
					CRYPT_COMPLIANCELEVEL_PKIX_FULL
					CRYPT_COMPLIANCELEVEL_LAST
	);


	## Flags for the Netscape netscape-cert-type extension ##
	use enum qw	(	BITMASK:
					CRYPT_NS_CERTTYPE_SSLCLIENT=1
					CRYPT_NS_CERTTYPE_SSLSERVER
					CRYPT_NS_CERTTYPE_SMIME
					CRYPT_NS_CERTTYPE_OBJECTSIGNING
					CRYPT_NS_CERTTYPE_RESERVED
					CRYPT_NS_CERTTYPE_SSLCA
					CRYPT_NS_CERTTYPE_SMIMECA
					CRYPT_NS_CERTTYPE_OBJECTSIGNINGCA
					CRYPT_NS_CERTTYPE_LAST
	);

	## Flags for the SET certificate-type extension ##
	use enum qw	(	BITMASK:
					CRYPT_SET_CERTTYPE_CARD=1
					CRYPT_SET_CERTTYPE_MER
					CRYPT_SET_CERTTYPE_PGWY
					CRYPT_SET_CERTTYPE_CCA
					CRYPT_SET_CERTTYPE_MCA
					CRYPT_SET_CERTTYPE_PCA
					CRYPT_SET_CERTTYPE_GCA
					CRYPT_SET_CERTTYPE_BCA
					CRYPT_SET_CERTTYPE_RCA
					CRYPT_SET_CERTTYPE_ACQ
					CRYPT_SET_CERTTYPE_LAST
	);

	## CMS contentType values ##
	use enum qw	(
					CRYPT_CONTENT_NONE=0
					CRYPT_CONTENT_DATA
					CRYPT_CONTENT_SIGNEDDATA
					CRYPT_CONTENT_ENVELOPEDDATA
					CRYPT_CONTENT_SIGNEDANDENVELOPEDDATA
					CRYPT_CONTENT_DIGESTEDDATA
					CRYPT_CONTENT_ENCRYPTEDDATA
					CRYPT_CONTENT_COMPRESSEDDATA
					CRYPT_CONTENT_TSTINFO
					CRYPT_CONTENT_SPCINDIRECTDATACONTEXT
					CRYPT_CONTENT_RTCSREQUEST
					CRYPT_CONTENT_RTCSRESPONSE
					CRYPT_CONTENT_RTCSRESPONSE_EXT
					CRYPT_CONTENT_LAST
	);

	## ESS securityClassification codes ##
	use enum qw	(
					CRYPT_CLASSIFICATION_UNMARKED=0
					CRYPT_CLASSIFICATION_UNCLASSIFIED
					CRYPT_CLASSIFICATION_RESTRICTED
					CRYPT_CLASSIFICATION_CONFIDENTIAL
					CRYPT_CLASSIFICATION_SECRET
					CRYPT_CLASSIFICATION_TOP_SECRET
					CRYPT_CLASSIFICATION_LAST=255
	);

	## RTCS certificate status ##
	use enum qw	(
					CRYPT_CERTSTATUS_VALID=0
					CRYPT_CERTSTATUS_NOTVALID
					CRYPT_CERTSTATUS_NONAUTHORITATIVE
					CRYPT_CERTSTATUS_UNKNOWN
	);

	## OCSP revocation status ##
	use enum qw	(
					CRYPT_OCSPSTATUS_NOTREVOKED=0
					CRYPT_OCSPSTATUS_REVOKED
					CRYPT_OCSPSTATUS_UNKNOWN
	);

	## The amount of detail to include in signatures when signing certificate objects ##
	use enum qw	(
					CRYPT_SIGNATURELEVEL_NONE=0
					CRYPT_SIGNATURELEVEL_SIGNERCERT
					CRYPT_SIGNATURELEVEL_ALL
					CRYPT_SIGNATURELEVEL_LAST
	);

	## The certificate export format type, which defines the format in which a
	#   certificate object is exported
	##
	use enum qw	(
					CRYPT_CERTFORMAT_NONE=0
					CRYPT_CERTFORMAT_CERTIFICATE
					CRYPT_CERTFORMAT_CERTCHAIN
					CRYPT_CERTFORMAT_TEXT_CERTIFICATE
					CRYPT_CERTFORMAT_TEXT_CERTCHAIN
					CRYPT_CERTFORMAT_XML_CERTIFICATE
					CRYPT_CERTFORMAT_XML_CERTCHAIN
					CRYPT_CERTFORMAT_LAST
	);

	## CMP request types ##
	use enum qw	(
					CRYPT_REQUESTTYPE_NONE=0
					CRYPT_REQUESTTYPE_INITIALISATION
					CRYPT_REQUESTTYPE_CERTIFICATE
					CRYPT_REQUESTTYPE_KEYUPDATE
					CRYPT_REQUESTTYPE_REVOCATION
					CRYPT_REQUESTTYPE_PKIBOOT
					CRYPT_REQUESTTYPE_LAST
	);
	use constant {
					CRYPT_REQUESTTYPE_INITIALIZATION	=> CRYPT_REQUESTTYPE_INITIALISATION
	};

	## Key ID types ##
	use enum qw	(
					CRYPT_KEYID_NONE=0
					CRYPT_KEYID_NAME
					CRYPT_KEYID_URI
					CRYPT_KEYID_LAST
	);
	use constant {
					CRYPT_KEYID_EMAIL	=> CRYPT_KEYID_URI	## Synonym: owner email addr.##
	};

	## The encryption object types ##
	use enum qw	(
					CRYPT_OBJECT_NONE=0
					CRYPT_OBJECT_ENCRYPTED_KEY
					CRYPT_OBJECT_PKCENCRYPTED_KEY
					CRYPT_OBJECT_KEYAGREEMENT
					CRYPT_OBJECT_SIGNATURE
					CRYPT_OBJECT_LAST
	);

	## Object/attribute error type information ##
	use enum qw	(
					CRYPT_ERRTYPE_NONE=0
					CRYPT_ERRTYPE_ATTR_SIZE
					CRYPT_ERRTYPE_ATTR_VALUE
					CRYPT_ERRTYPE_ATTR_ABSENT
					CRYPT_ERRTYPE_ATTR_PRESENT
					CRYPT_ERRTYPE_CONSTRAINT
					CRYPT_ERRTYPE_ISSUERCONSTRAINT
					CRYPT_ERRTYPE_LAST
	);

	## Cert store management action type ##
	use enum qw	(
					CRYPT_CERTACTION_NONE=0
					CRYPT_CERTACTION_CREATE
					CRYPT_CERTACTION_CONNECT
					CRYPT_CERTACTION_DISCONNECT
					CRYPT_CERTACTION_ERROR
					CRYPT_CERTACTION_ADDUSER
					CRYPT_CERTACTION_DELETEUSER
					CRYPT_CERTACTION_REQUEST_CERT
					CRYPT_CERTACTION_REQUEST_RENEWAL
					CRYPT_CERTACTION_REQUEST_REVOCATION
					CRYPT_CERTACTION_CERT_CREATION
					CRYPT_CERTACTION_CERT_CREATION_COMPLETE
					CRYPT_CERTACTION_CERT_CREATION_DROP
					CRYPT_CERTACTION_CERT_CREATION_REVERSE
					CRYPT_CERTACTION_RESTART_CLEANUP
					CRYPT_CERTACTION_RESTART_REVOKE_CERT
					CRYPT_CERTACTION_ISSUE_CERT
					CRYPT_CERTACTION_ISSUE_CRL
					CRYPT_CERTACTION_REVOKE_CERT
					CRYPT_CERTACTION_EXPIRE_CERT
					CRYPT_CERTACTION_CLEANUP
					CRYPT_CERTACTION_LAST
	);


	#############################################################################
	# General Constants
	#############################################################################

	use constant {
					## The maximum user key size - 2048 bits ##
					CRYPT_MAX_KEYSIZE	=> 256,
					## The maximum IV size - 256 bits ##
					CRYPT_MAX_IVSIZE	=> 32,
					## The maximum public-key component size - 4096 bits ##
					CRYPT_MAX_PKCSIZE	=> 512,
					## The maximum hash size - 256 bits ##
					CRYPT_MAX_HASHSIZE	=> 32,
					## The maximum size of a text string (e.g.key owner name) ##
					CRYPT_MAX_TEXTSIZE	=> 64,
					## A magic value indicating that the default setting for this parameter should be used ##
					CRYPT_USE_DEFAULT	=> -10,
					## A magic value for unused parameters ##
					CRYPT_UNUSED	=> -11,
					## Whether the PKC key is a public or private key ##
					CRYPT_KEYTYPE_PRIVATE	=> 0,
					CRYPT_KEYTYPE_PUBLIC	=> 1,
					## The type of information polling to perform to get random seed information ##
					CRYPT_RANDOM_FASTPOLL	=> -10,
					CRYPT_RANDOM_SLOWPOLL	=> -11,
					## Cursor positioning codes for certificate/CRL extensions ##
					CRYPT_CURSOR_FIRST		=> -20,
					CRYPT_CURSOR_PREVIOUS	=> -21,
					CRYPT_CURSOR_NEXT		=> -22,
					CRYPT_CURSOR_LAST		=> -23
	};

	## Keyset open options ##
	use enum qw	(
					CRYPT_KEYOPT_NONE=0
					CRYPT_KEYOPT_READONLY
					CRYPT_KEYOPT_CREATE
					CRYPT_KEYOPT_LAST
	);

	use constant {
					### The various cryptlib objects - these are just integer handles ##
					CRYPT_CERTIFICATE	=> 0,
					CRYPT_CONTEXT		=> 0,
					CRYPT_DEVICE		=> 0,
					CRYPT_ENVELOPE		=> 0,
					CRYPT_KEYSET		=> 0,
					CRYPT_SESSION		=> 0,
					CRYPT_USER			=> 0,
					### Sometimes we don't know the exact type of a cryptlib object, so we use a
					##   generic handle type to identify it
					###
					CRYPT_HANDLE		=> 0
	};


	#############################################################################
	# Status Codes
	#############################################################################

	use constant {
					## No error in function call ##
					CRYPT_OK	=> 0,	## No error ##
					## Error in parameters passed to function ##
					CRYPT_ERROR_PARAM1	=> -1,	## Bad argument, parameter 1 ##
					CRYPT_ERROR_PARAM2	=> -2,	## Bad argument, parameter 2 ##
					CRYPT_ERROR_PARAM3	=> -3,	## Bad argument, parameter 3 ##
					CRYPT_ERROR_PARAM4	=> -4,	## Bad argument, parameter 4 ##
					CRYPT_ERROR_PARAM5	=> -5,	## Bad argument, parameter 5 ##
					CRYPT_ERROR_PARAM6	=> -6,	## Bad argument, parameter 6 ##
					CRYPT_ERROR_PARAM7	=> -7,	## Bad argument, parameter 7 ##
					## Errors due to insufficient resources ##
					CRYPT_ERROR_MEMORY		=> -10,	## Out of memory ##
					CRYPT_ERROR_NOTINITED	=> -11,	## Data has not been initialised ##
					CRYPT_ERROR_INITED		=> -12,	## Data has already been init'd ##
					CRYPT_ERROR_NOSECURE	=> -13,	## Opn.not avail.at requested sec.level ##
					CRYPT_ERROR_RANDOM		=> -14,	## No reliable random data available ##
					CRYPT_ERROR_FAILED		=> -15,	## Operation failed ##
					## Security violations ##
					CRYPT_ERROR_NOTAVAIL	=> -20,	## This type of opn.not available ##
					CRYPT_ERROR_PERMISSION	=> -21,	## No permiss.to perform this operation ##
					CRYPT_ERROR_WRONGKEY	=> -22,	## Incorrect key used to decrypt data ##
					CRYPT_ERROR_INCOMPLETE	=> -23,	## Operation incomplete/still in progress ##
					CRYPT_ERROR_COMPLETE	=> -24,	## Operation complete/can't continue ##
					CRYPT_ERROR_TIMEOUT		=> -25,	## Operation timed out before completion ##
					CRYPT_ERROR_INVALID		=> -26,	## Invalid/inconsistent information ##
					CRYPT_ERROR_SIGNALLED	=> -27,	## Resource destroyed by extnl.event ##
					## High-level function errors ##
					CRYPT_ERROR_OVERFLOW	=> -30,	## Resources/space exhausted ##
					CRYPT_ERROR_UNDERFLOW	=> -31,	## Not enough data available ##
					CRYPT_ERROR_BADDATA		=> -32,	## Bad/unrecognised data format ##
					CRYPT_ERROR_SIGNATURE	=> -33,	## Signature/integrity check failed ##
					## Data access function errors ##
					CRYPT_ERROR_OPEN		=> -40,	## Cannot open object ##
					CRYPT_ERROR_READ		=> -41,	## Cannot read item from object ##
					CRYPT_ERROR_WRITE		=> -42,	## Cannot write item to object ##
					CRYPT_ERROR_NOTFOUND	=> -43,	## Requested item not found in object ##
					CRYPT_ERROR_DUPLICATE	=> -44,	## Item already present in object ##
					## Data enveloping errors ##
					CRYPT_ENVELOPE_RESOURCE	=> -50	## Need resource to proceed ##
	};



#############################################################################
#
# STRUCTURES TO HOLD LOW-LEVEL COMPONENTS
#
#############################################################################

	use constant {
		CRYPT_QUERY_INFO	=>	{},
		CRYPT_OBJECT_INFO	=>	{},
		CRYPT_PKCINFO_RSA	=>	{
									## Status information
									isPublicKey => 0,						## Whether this is a public or private key
									## Public components
									n           => ' ' x CRYPT_MAX_PKCSIZE,	## Modulus
									nLen        => 0,						## Length of modulus in bits
									e           => ' ' x CRYPT_MAX_PKCSIZE, ## Public exponent
									eLen        => 0,						## Length of public exponent in bits
									## Private components
									d           => ' ' x CRYPT_MAX_PKCSIZE, ## Private exponent
									dLen        => 0,						## Length of private exponent in bits
									p           => ' ' x CRYPT_MAX_PKCSIZE, ## Prime factor 1
									pLen        => 0,						## Length of prime factor 1 in bits
									q           => ' ' x CRYPT_MAX_PKCSIZE, ## Prime factor 2
									qLen        => 0,						## Length of prime factor 2 in bits
									u           => ' ' x CRYPT_MAX_PKCSIZE, ## Mult.inverse of q, mod p
									uLen        => 0,						## Length of private exponent in bits
									e1          => ' ' x CRYPT_MAX_PKCSIZE, ## Private exponent 1 (PKCS)
									e1Len       => 0,						## Length of private exponent in bits
									e2          => ' ' x CRYPT_MAX_PKCSIZE, ## Private exponent 2 (PKCS)
									e2Len       => 0						## Length of private exponent in bits
								},
		CRYPT_PKCINFO_DLP	=>	{
									## Status information
									isPublicKey => 0,   ## Whether this is a public or private key
									## Public components
									p           => ' ' x CRYPT_MAX_PKCSIZE, ## Prime modulus
									pLen        => 0,						## Length of prime modulus in bits
									q           => ' ' x CRYPT_MAX_PKCSIZE, ## Prime divisor
									qLen        => 0,						## Length of prime divisor in bits
									g           => ' ' x CRYPT_MAX_PKCSIZE, ## h^( ( p - 1 ) / q ) mod p
									gLen        => 0,						## Length of g in bits
									y           => ' ' x CRYPT_MAX_PKCSIZE, ## Public random integer
									yLen        => 0,						## Length of public integer in bits
									## Private components
									x           => ' ' x CRYPT_MAX_PKCSIZE, ## Private random integer
									xLen        => 0						## Length of private integer in bits
								}
	};



#############################################################################
#
# MACROS TO EXAMINE RETURN VALUES
#
#############################################################################

	sub cryptStatusError($) {
		my $status = shift;
		return ( $status < CRYPT_OK );
	}

	sub cryptStatusOK($) {
		my $status = shift;
		return ( $status == CRYPT_OK );
	}



#############################################################################
#
# MACROS TO MANAGE LOW-LEVEL COMPONENTS
#
#############################################################################

	sub cryptInitComponents(;$$@) {
		die "Usage: cryptInitComponents(\$componentInfo, \$componentKeyType)\n" if scalar(@_) != 2;
		my ($componentInfo, $componentKeyType) = @_;
		$componentInfo->{isPublicKey} = ( $componentKeyType == CRYPT_KEYTYPE_PUBLIC ? 1 : 0 );
		return CRYPT_OK;
	}

	sub cryptDestroyComponents(;$@) {
		die "Usage: cryptDestroyComponents(\$componentInfo)\n" if scalar(@_) != 1;
		my $componentInfo = shift;
		undef $componentInfo;
		return CRYPT_OK;
	}

	sub cryptSetComponent(;$$$$@) {
		die "Usage: cryptSetComponent(\$componentInfo, \$element, \$source, \$length)\nFor more info see README file." if scalar(@_) != 4;
		my ($componentInfo, $element, $source, $length) = @_;
		$componentInfo->{$element} = join('', $source, ("\0" x (CRYPT_MAX_PKCSIZE - length($source))) );
		$componentInfo->{$element.'Len'} = $length;
		return CRYPT_OK;
	}

	sub cryptFinalizeComponents(;$\$\$@) {
		die "Usage: cryptFinalizeComponents(\$componentInfo, \$blob, \$size)\nFor more info see README file." if scalar(@_) != 3;
		my ($componentInfo, $blob, $size) = @_;
		my @rsaFields = qw(isPublicKey n nLen e eLen d dLen p pLen q qLen u uLen e1 e1Len e2 e2Len);
		my @dlpFields = qw(isPublicKey p pLen q qLen g gLen y yLen x xLen);
		my @values = ();
		map {
				my $field = $_;
				my $value = $componentInfo->{$field};
				if ( $field =~ /Len$/  ||  $field eq 'isPublicKey' ) {
					$value = sprintf("%08x", $componentInfo->{$field});
					$value = pack("H8", join('', reverse($value =~ /(..)/g))); # inverte l'ordine dei byte e impacka
				}
				push @values, $value;
				$$size += length($values[$#values]);
			} ( scalar(keys(%{$componentInfo})) == scalar(@rsaFields) ? @rsaFields : @dlpFields );
		$$blob = join('', @values);
		return CRYPT_OK;
	}


1;
