#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <stdarg.h>
#include <errno.h>

#include "libcryptsetup.h"
#include "internal.h"

struct safe_allocation {
	size_t	size;
	char	data[1];
};

static char *error;

void set_error_va(const char *fmt, va_list va)
{
	int bufsize;

	bufsize = fmt ? (strlen(fmt) + 1) : 0;
	if (bufsize < 128)
		bufsize = 128;

	if (error)
		free(error);
	if (!fmt) {
		error = NULL;
		return;
	}

	error = malloc(bufsize);

	for(;;) {
		int n;

		n = vsnprintf(error, bufsize, fmt, va);

		if (n >= 0 && n < bufsize)
			break;

		if (n >= 0)
			bufsize = n + 1;
		else
			bufsize *= 2;

		error = realloc(error, bufsize);
	}
}

void set_error(const char *fmt, ...)
{
	va_list va;

	va_start(va, fmt);
	set_error_va(fmt, va);
	va_end(va);
}

const char *get_error(void)
{
	return error;
}

void *safe_alloc(size_t size)
{
	struct safe_allocation *alloc;

	if (!size)
		return NULL;

	alloc = malloc(size + offsetof(struct safe_allocation, data));
	if (!alloc)
		return NULL;

	alloc->size = size;

	return &alloc->data;
}

void safe_free(void *data)
{
	struct safe_allocation *alloc;

	if (!data)
		return;

	alloc = data - offsetof(struct safe_allocation, data);

	memset(data, 0, alloc->size);

	alloc->size = 0x55aa55aa;
	free(alloc);
}

void *safe_realloc(void *data, size_t size)
{
	void *new_data;

	new_data = safe_alloc(size);

	if (new_data && data) {
		struct safe_allocation *alloc;

		alloc = data - offsetof(struct safe_allocation, data);

		if (size > alloc->size)
			size = alloc->size;

		memcpy(new_data, data, size);
	}

	safe_free(data);
	return new_data;
}

char *safe_strdup(const char *s)
{
	char *s2 = safe_alloc(strlen(s) + 1);

	if (!s2)
		return NULL;

	return strcpy(s2, s);
}

/* Credits go to Michal's padlock patches for this alignment code */

static void *aligned_malloc(char **base, int size, int alignment) 
{
	char *ptr;

	ptr  = malloc(size + alignment);
	if(ptr == NULL) return NULL;

	*base = ptr;
	if(alignment > 1 && ((long)ptr & (alignment - 1))) {
		ptr += alignment - ((long)(ptr) & (alignment - 1));
	}
	return ptr;
}

ssize_t write_blockwise(int fd, const void *orig_buf, size_t count) 
{
	char *padbuf; char *base;
	char *buf = (char *)orig_buf;
	int hangover = count % SECTOR_SIZE;
	int solid = count - hangover;
	int r;

	padbuf = aligned_malloc(&base, SECTOR_SIZE, SECTOR_SIZE);
	if(padbuf == NULL) return -ENOMEM;

	while(solid) {
		memcpy(padbuf, buf, SECTOR_SIZE);
		r = write(fd, padbuf, SECTOR_SIZE);
		if(r < 0 || r != SECTOR_SIZE) goto out;

		solid -= SECTOR_SIZE;
		buf += SECTOR_SIZE;
	}
	if(hangover) {
		r = read(fd,padbuf,SECTOR_SIZE);
		if(r < 0 || r != SECTOR_SIZE) goto out;

		lseek(fd,-SECTOR_SIZE,SEEK_CUR);
		memcpy(padbuf,buf,hangover);

		r = write(fd,padbuf, SECTOR_SIZE);
		if(r < 0 || r != SECTOR_SIZE) goto out;
		buf += hangover;
	}
 out:
	free(base);
	return (buf-(char *)orig_buf)?(buf-(char *)orig_buf):r;

}


ssize_t read_blockwise(int fd, void *orig_buf, size_t count) {
	char *base; char *padbuf;
	char *buf = (char *)orig_buf;
	int r;
	int step;

	padbuf = aligned_malloc(&base, SECTOR_SIZE, SECTOR_SIZE);
	if(padbuf == NULL) return -ENOMEM;

	while(count) {
		r = read(fd,padbuf,SECTOR_SIZE);
		if(r < 0) goto out;
		
		step = count<SECTOR_SIZE?count:SECTOR_SIZE;
		memcpy(buf,padbuf,step);
		buf += step;
		count -= step;
	}
 out:
	free(base); 
	return (buf-(char *)orig_buf)?(buf-(char *)orig_buf):r;
}
