/*
 * $Id: xconv.c,v 1.1 1997/06/11 02:10:46 morgan Exp $
 *
 * This is a sample front end to the graphics functions.  PAM
 * modules will call conversation() which repackages these msg requests
 * and passes them down a pipe to another process that turns them into
 * X commands.
 *
 * Copyright (c) 1997 Andrew Morgan <morgan@parc.power.net>
 * Copyright (c) 1997 Danny Sung
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/wait.h>

#include <security/pam_appl.h>
#include <security/_pam_macros.h>

#include "Xconv.h"

#define MIN_TEXT_TIME      2               /* min time to read a message */

/*
 * this is the structure used locally to manage request/response pairs.
 */

struct message_queue {
    int msg_style;                      /* what kind of message is this? */
    const char *msg;                       /* what is displayed for user */
    char *reply;                            /* to store the user's reply */
};

/*
 * This function should be called when we know that the X-child is
 * about to die or is already dead.  It is used to clean up the
 * process tables and record the fact that there is no X-proceessing
 * going on (->child=0).
 */

static void TerminateGraphics(ConversingStruct *io)
{
    int junk;

    D(("called."));
    (void) waitpid(io->child, &junk, 0);
    io->child = 0;
    D(("end."));
}

/*
 * This function is used to explicitly terminate the X-child.
 */

static void PAMAbortGraphics(ConversingStruct *io)
{
    D(("called."));
    outmsg(&(io->cpipe), ABORT_NOTIFY, 0, 0);
    TerminateGraphics(io);
    D(("end."));
}

/*
 * This is the function that talks directly with the X-pipe
 *
 * It should (Re)initialize the widget and then display no more than 4
 * different messages. Each of the four flavors will only appear once
 * on a given widget:
 *
 *        PAM_PROMPT_ECHO_OFF:     (for passwords etc..)
 *        PAM_PROMPT_ECHO_ON:      (for usernames etc..)
 *        PAM_PROMPT_ERROR_MSG:    (distinguishable from TEXT_INFO; in red?)
 *        PAM_PROMPT_TEXT_INFO:    (helpful/informative messages)
 *
 * Note, I have arranged things that text-info/errors
 * will always _preceed_ user prompts.
 */

static int PAMFeedGraphics(int num_requests
			   , struct message_queue *msg
			   , ConversingStruct *io)
{
    static int clearmsgs = 0;
    int i;
    int retval=PAM_SUCCESS;
    int len;
    void *data;
    int hasprompt = 0, readpipe=1;

    if ( io->child == 0 ) {
	D(("Child has died: no X-connection"));
	return PAM_CONV_ERR;
    }

    if ( clearmsgs )
	outmsg(&(io->cpipe), CLEARMSGS, 0, 0);

    clearmsgs = 1;

    D(("[ -- Build new form --"));

    outmsg(&(io->cpipe), POPDOWN, 0, 0);
    outmsg(&(io->cpipe), CLEARPROMPTS, 0, 0);
    for (i=0; retval == PAM_SUCCESS && i < num_requests; ++i) {
	switch (msg[i].msg_style) {
	case PAM_PROMPT_ECHO_ON:
	    D(("adding echo prompt: %s", msg[i].msg));
	    outmsg(&(io->cpipe), PROMPT_ECHO_ON,
		   msg[i].msg, strlen(msg[i].msg)+1);
	    hasprompt=1;
	    break;
	case PAM_PROMPT_ECHO_OFF:
	    D(("adding noecho prompt: %s", msg[i].msg));
	    outmsg(&(io->cpipe), PROMPT_ECHO_OFF,
		   msg[i].msg, strlen(msg[i].msg)+1);
	    hasprompt=1;
	    break;
	case PAM_ERROR_MSG:
	    D(("adding error: %s", msg[i].msg));
	    outmsg(&(io->cpipe), ERROR_MSG, msg[i].msg, strlen(msg[i].msg)+1);
	    clearmsgs = 0;
	    break;
	case PAM_TEXT_INFO:
	    D(("adding text: %s", msg[i].msg));
	    outmsg(&(io->cpipe), TEXT_INFO, msg[i].msg, strlen(msg[i].msg)+1);
	    break;
	default:
	    while (i-- > 0) {
		/* clean up */
		_pam_overwrite(msg[i].reply);
		_pam_drop(msg[i].reply);
	    }
	    retval = PAM_CONV_ERR;
	}
    }
    outmsg(&(io->cpipe), POPUP, 0, 0);

    D(("-- Form is complete -- ]"));

    /*
     * If there was no prompt, we pause for the user to read
     * what is on the screen.
     */

    if ( !hasprompt ) {
	fd_set fds;
	struct timeval tv;

	FD_ZERO(&fds);
	FD_SET(io->cpipe.infd, &fds);

	tv.tv_sec = MIN_TEXT_TIME;
	tv.tv_usec = 0;

	if( !select(io->cpipe.infd+1, &fds, 0, 0, &tv) ) {
	    /* no activity -- timed out */
	    readpipe = 0;
	}
    }

    /* Do we expect a response? Wait for it/them */
    if ( readpipe ) {
	switch( inmsg(&(io->cpipe), &data, &len) ) {
	case OK_NOTIFY:
	    D(("received ok_notify"));
	    for (i=0; i<num_requests; i++) {
		void *data;
		int len;
		outmsg(&(io->cpipe), QUERY_RESPONSE,
		       msg[i].msg, strlen(msg[i].msg)+1);
		if (inmsg(&(io->cpipe), &data, &len) == SEND_RESPONSE) {
		    msg[i].reply = data;
		    D(("'%s' --> '%s'", msg[i].msg, msg[i].reply));
		} else {
		    D(("unexpected return"));
		    PAMAbortGraphics(io);
		    return PAM_CONV_ERR;
		}
	    }
	    outmsg(&(io->cpipe), QUERY_DONE, 0, 0);
	    break;
	case ABORT_NOTIFY:
	    D(("received abort_notify"));
	    TerminateGraphics(io);
	    return PAM_CONV_ERR;
	    break;
	default:
	    D(("Error"));
	}
    }
    outmsg(&(io->cpipe), CLEARPROMPTS, 0, 0);
   
    return retval;
}

/* ************************************************************************* *
 *                 Nothing below here knows about the graphics               *
 * ************************************************************************* */

/*
 * internal flags for the conversation function
 */

#define ACCEPT_ECHO     01
#define ACCEPT_NOECHO   02
#define ACCEPT_ERROR    04
#define ACCEPT_TEXT    010

static int conversation(int num_msg
			, const struct pam_message **msg
			, struct pam_response **resp
			, void *appdata_ptr)     /* holds window connection */
{
    int num_responses;
    int retval = PAM_SUCCESS;

    if (resp == NULL || msg == NULL || num_msg < 1) {
	return PAM_CONV_ERR;                           /* malformed request */
    }

    num_responses = 0;
    *resp = NULL;                                             /* initialize */

    while (num_msg > 0) {       /* loop until we have all done all messages */
	int flags=ACCEPT_ECHO|ACCEPT_NOECHO|ACCEPT_ERROR|ACCEPT_TEXT;
	int count=0;                    /* number of requests in this queue */
	int replies=0;                    /* the number of expected replies */
	struct message_queue small_queue[4];

	memset(&small_queue, 0, sizeof(small_queue));  /* reset small queue */

	while (flags && num_msg > 0) {   /* loop until we have a full queue */
	    int satisfied=0;

	    switch (msg[0]->msg_style) {
            case PAM_PROMPT_ECHO_OFF:
		if (flags & ACCEPT_NOECHO) {
		    satisfied = 1;
		    ++replies;                   /* one more reply expected */
		    flags &= ~(ACCEPT_NOECHO|ACCEPT_TEXT|ACCEPT_ERROR);
		    small_queue[count].msg_style = PAM_PROMPT_ECHO_OFF;
		    small_queue[count].msg = msg[0]->msg;
		}
		break;
            case PAM_PROMPT_ECHO_ON:
		if (flags & ACCEPT_NOECHO) {
		    satisfied = 1;
		    ++replies;                   /* one more reply expected */
		    flags &= ~(ACCEPT_ECHO|ACCEPT_TEXT|ACCEPT_ERROR);
		    small_queue[count].msg_style = PAM_PROMPT_ECHO_ON;
		    small_queue[count].msg = msg[0]->msg;
		}
		break;
            case PAM_ERROR_MSG:
		if (flags & ACCEPT_ERROR) {
		    satisfied = 1;
		    flags &= ~ACCEPT_ERROR;
		    small_queue[count].msg_style = PAM_ERROR_MSG;
		    small_queue[count].msg = msg[0]->msg;
		}
		break;
            case PAM_TEXT_INFO:
		if (flags & ACCEPT_TEXT) {
		    satisfied = 1;
		    flags &= ~ACCEPT_TEXT;
		    small_queue[count].msg_style = PAM_TEXT_INFO;
		    small_queue[count].msg = msg[0]->msg;
		}
		break;
	    }

	    if (satisfied) {
		++count;                   /* one more request in the queue */
		--num_msg;                 /* one fewer requests to process */
		++msg;                                      /* next request */
	    } else {
		break;                /* the queue is now full - process it */
	    }
	}

	/* Need to interact with the user? */
	if (count > 0) {
	    retval = PAMFeedGraphics(count, small_queue, appdata_ptr);
	    if (retval == PAM_SUCCESS) {
		if (replies > 0) {      	/* copy the replies to *resp */
		    struct pam_response *tmp;
		    int i;
      
		    tmp = calloc(num_responses+replies,
				 sizeof(struct pam_response));

		    if (tmp == NULL) {
			retval = PAM_BUF_ERR;
			break;			/* XXX - insufficient memory */
		    }
      
		    if (*resp != NULL) {      /* paranoia about object reuse */
			memcpy(tmp, *resp,
			       num_responses*sizeof(struct pam_response));
			memset(*resp, 0,
			       num_responses*sizeof(struct pam_response));
			_pam_drop(*resp);
		    }
		    *resp = tmp;
		    tmp = NULL;
      
		    /*
		     * find the replies and move them to the end of
		     * the *resp list. No check is made for the wrong
		     * number of replies.
		     */

		    for (i=0; replies > 0 && i < count; ++i) {
			switch (small_queue[i].msg_style) {
			case PAM_PROMPT_ECHO_OFF:
			case PAM_PROMPT_ECHO_ON:
			    (*resp)[num_responses].resp_retcode = 0; /* XXX */
			    (*resp)[num_responses++].resp =
				small_queue[i].reply;
			    --replies;
			}
			small_queue[i].reply = NULL;            /* reset it */
		    }
		}
	    } else {
		break;                               /* XXX - widget failure */
	    }
	}
    }

    /* return control to module */

    if (retval == PAM_SUCCESS) {
	return PAM_SUCCESS;                                /* things worked */
    } else {
	int i;

	for (i=0; i<num_responses; ++i) {	               /* clean up! */
	    _pam_overwrite((*resp)[i].resp);
	    _pam_drop((*resp)[i].resp);
	    (*resp)[i].resp_retcode = 0;
	}
	_pam_drop(*resp);
    }

    return PAM_CONV_ERR;                          /* something bad happened */
}

/*
 * THIS IS A PAM SIMULATION
 */

/*
 * Example usage, this is a combined application and module piece of
 * code. The graphics stuff shouldn't be able to tell.
 */

#define MSG1_COUNT 4
#define RSP1_COUNT 2
const static struct pam_message msg1[MSG1_COUNT] = {
    { PAM_TEXT_INFO, "Establishing the user identity" },
    { PAM_PROMPT_ECHO_ON, "Username: " },
    { PAM_PROMPT_ECHO_OFF, "Password: " },
    { PAM_ERROR_MSG, "Sorry, you are not known!" }
};

const static struct pam_message *msgs1[MSG1_COUNT] = {
    &msg1[0], &msg1[1], &msg1[2], &msg1[3]
};

#define MSG2_COUNT 8
#define RSP2_COUNT 4
const static struct pam_message msg2[MSG2_COUNT] = {
    { PAM_TEXT_INFO, "Please try again" },
    { PAM_PROMPT_ECHO_ON, "Username: " },
    { PAM_PROMPT_ECHO_OFF, "Password: " },
    { PAM_TEXT_INFO, "Thanks." },
    { PAM_TEXT_INFO, "Now for your local id" },
    { PAM_PROMPT_ECHO_ON, "Username: " },
    { PAM_PROMPT_ECHO_OFF, "Password: " },
    { PAM_TEXT_INFO, "Logging you in..." }
};

const static struct pam_message *msgs2[MSG2_COUNT] = {
    &msg2[0], &msg2[1], &msg2[2], &msg2[3],
    &msg2[4], &msg2[5], &msg2[6], &msg2[7]
};

static void list_responses(int n, struct pam_response **resp)
{
    int i;

    printf("Here are the responses:\n");
    for (i=0; i<n; ++i) {
	printf("\t%d> %s\n", i, (*resp)[i].resp);
	_pam_overwrite((*resp)[i].resp);
	_pam_drop((*resp)[i].resp);
    }
    printf("[cleaning up]\n");
    _pam_drop(*resp);
}

int pam_thread(ConversingStruct *io)
{
    int retval;
    struct pam_response *resp=NULL;

    /* display first string of messages */
    retval = conversation(MSG1_COUNT, msgs1, &resp, io);
    if (retval != PAM_SUCCESS) {
	fprintf(stderr, "-> first conversation failed\n");
	exit(1);
    }
    list_responses(RSP1_COUNT, &resp);

    outmsg(&(io->cpipe), HIDE_WINDOW, 0, 0);

    fprintf(stderr, "Pausing 3 seconds..");
    sleep(3);
    fprintf(stderr, ".done.\n");

    /* display second string of messages */
    retval = conversation(MSG2_COUNT, msgs2, &resp, io);
    if (retval != PAM_SUCCESS) {
	fprintf(stderr, "-> second conversation failed\n");
	exit(1);
    }
    list_responses(RSP2_COUNT, &resp);

    /* signal the application wishes to abort */
    PAMAbortGraphics(io);

    /* try to repeat first messages */
    retval = conversation(MSG1_COUNT, msgs1, &resp, io);

    /* Tell auth widget to disappear */
    outmsg(&(io->cpipe), POPDOWN, 0, 0);

    if (retval == PAM_SUCCESS) {
	outmsg(&(io->cpipe), FINAL_SUCCESS, 0, 0);
	fprintf(stderr, "-> third conversation succeeded :^?\n");
	list_responses(RSP1_COUNT, &resp);
    } else {
	outmsg(&(io->cpipe), FINAL_FAIL, 0, 0);
	fprintf(stderr, "-> third conversation failed [as expected]\n");
    }

    outmsg(&(io->cpipe), HIDE_WINDOW, 0, 0);

    return retval;
}

int main(int argc, char *argv[])
{
    ConversingStruct *appdata=NULL;
    int retval;

    StartGraphicsInterface(argc, argv, &appdata);
    retval = pam_thread(appdata);
    EndGraphicsInterface(&appdata, retval==PAM_SUCCESS ? TRUE:FALSE );

    exit(0);
}
