#!/bin/bash -
## -----------------------------------------------------------------------
##   
##   Copyright 2000-2002 H. Peter Anvin - All Rights Reserved
##
##   This program is free software; you can redistribute it and/or modify
##   it under the terms of the GNU General Public License as published by
##   the Free Software Foundation, Inc., 675 Mass Ave, Cambridge MA 02139,
##   USA; either version 2 of the License, or (at your option) any later
##   version; incorporated herein by reference.
##
## -----------------------------------------------------------------------
## $Id: make.sh,v 1.81 2002/11/07 04:15:53 root Exp $

##
## Main make script for SuperRescue CD
##

if [ -z "$1" ]; then
    echo "Usage: $0 version" 1>&2
    exit 1
fi

BUILDER=`cat Builder 2>/dev/null`
if [ -z "$BUILDER" ]; then
    host=`hostname --fqdn 2>/dev/null`
    echo "Please create the Builder file first, for example:" 1>&2
    echo "  echo 'Your Name Here <$LOGNAME@$host>' > Builder" 1>&2
    exit 1
fi

# Missing trees?
if [ ! -d usr -o ! -d root ]; then
    echo "Missing usr and/or root tree!" 1>&2
    exit 1
fi

set -xe

DATE=`date`
BUILDDIRS='rootpatch irtree isolinux'
READMES='README README-BUILD README-NETWORK CHANGES'
MAKES='make.sh makeinit.sh'

umask 022

# This allows the user to override the default binaries
NCPUS=`grep -c '^processor' /proc/cpuinfo`
NJOBS=`expr $NCPUS + 2`
[ -z "$MKISOFS" ] && MKISOFS=mkisofs
[ -z "$MKZFTREE" ] && MKZFTREE="mkzftree -p$NJOBS"

# Note on trees: the iso image is formed from the directories
# usr.zf (from usr), isotree.zf (from isotree), and nozftree
# (the latter, of course, which is not compressed.)

# Initial cleanup
if [ -e work ]; then
  mv -f work work.$$
  mkdir -p work
  ( mv work.$$/usr.zf work/usr.zf.old || \
    mv work.$$/usr.zf.old work/usr.zf.old ) 2>/dev/null
  ( mv work.$$/isotree.zf work/isotree.zf.old || \
    mv work.$$/isotree.zf.old work/isotree.zf.old ) 2>/dev/null
  rm -rf work.$$ &
fi
rm -f  superrescue-$1.iso superrescue-$1.iso.gz buildtree-$1.tar.gz
mkdir -p work/isotree work/nozftree work/nozftree/boot work/nozftree/isolinux
mkdir -p work/roottree work/buildtree

# Make initrd
./makeinit.sh

# Set up boot files
cd work/nozftree/isolinux
  cp -a ../../../usr/boot/vmlinuz* .
  cp -a ../../../usr/lib/syslinux/isolinux.bin .
  cp ../../initrd.gz .
  cp -a ../../../isolinux/. .
  # Turn symlinks into hard links
  for file in *; do
    if [ -L "$file" ]; then
      target=`ls -l --color=no "$file" | cut -d\> -f2- | cut -c2-`
      rm -f "$file"
      ln "$target" "$file"
    fi
  done
  # For message files, substitute version number
  for file in *.msg; do
    if [ -f ../../../isolinux/$file ]; then
	sed -e "s/@@VERSION@@/$1/g" < ../../../isolinux/$file > $file
    fi
  done
cd ../../..

# This is a list of items in the root which will get symlinked to
# the usr directory
symlink_in_root='bin sbin lib etc/codepages etc/termcap etc/lynx.cfg* etc/makedev.d var/lib/rpm'

# Generate root tree.
cp -a root/. work/roottree/. 
# Remove things which are already symlinked
(
  cd work/roottree
  for l in $symlink_in_root; do
    if [ -L "$l" -a -e ../usr/root/"$l" ]; then
      rm -f "$l"
      cp -a ../usr/root/"$l" "$l"
    fi
  done
)
# Cleanup of root which is hard to do at tree-creation time..
cp -af rootpatch/. work/roottree/.
find work/roottree/var/lock work/roottree/var/run work/roottree/var/cache \
    -type f -print0 | xargs -0rt rm -f
find work/roottree/var/log -type f -exec cp /dev/null '{}' \;
rm -rf work/roottree/work
# Global cleanup of files...
find work/roottree usr work/isotree isolinux \
    \( -name \*~ -or -name \#\* -or -name .journal \) \
	-print0 | xargs -0rt rm -f
# Global cleanup of directories...
find work/roottree usr work/isotree isolinux \
    \( -name rr_moved -o -name .rr_moved -o -name lost+found \
       -o -name \*.rpmsave -o -name \*.rpmnew \) \
	-print0 | xargs -0rt rm -rf
rm -f work/roottree/.bash_history work/roottree/root/.bash_history work/roottree/home/*/.bash_history
rm -f work/roottree/.history work/roottree/root/.history work/roottree/home/*/.history
rm -f work/roottree/.autofsck
rm -rf work/roottree/.ssh work/roottree/root/.ssh work/roottree/home/*/.ssh
rm -rf work/roottree/.netscape work/roottree/root/.netscape work/roottree/home/*/.netscape
rm -f work/roottree/.rpmmacros work/roottree/root/.rpmmacros work/roottree/home/*/.rpmmacros
rm -f work/roottree/.TWM-errors work/roottree/root/.TWM-errors work/roottree/home/*/.TWM-errors
rm -f work/roottree/var/lib/random-seed
rm -f work/roottree/var/lib/rpm/__db*
rm -f work/roottree/etc/*- work/roottree/etc/dhcpc/*
# We don't know what the CD-ROM device is going to be
rm -f work/roottree/dev/cdrom
# We definitely don't want inetd running by default
rm -f work/roottree/etc/rc.d/rc?.d/S??inet
# tunnelv doesn't work right.  Kill it.
rm -f work/roottree/etc/rc.d/rc?.d/S??tunnelv
# NFS locking has possible remote root exploit
rm -f work/roottree/etc/rc.d/rc?.d/S??nfslock
rm -rf work/roottree/tmp work/roottree/var/tmp
# build links are not likely to be correct
rm -f work/roottree/lib/modules/*/build
# Create empty /tmp and /var/tmp
mkdir -p work/roottree/tmp work/roottree/var/tmp
chown root.root work/roottree/tmp work/roottree/var/tmp
chmod 1777 work/roottree/tmp work/roottree/var/tmp

# Some things don't really need to be in the root, now when the initrd
# can mount /usr for us, thanks to pivot_root
for ent in $symlink_in_root; do
    graft=`dirname "$ent"`
    tail=`basename "$ent"`
    dots=`dirname /"$ent" | sed -e 's:^/::' -e 's:[^/]\+:..:g' -e 's:\.\.$:\.\./:'`
    if [ ! -L work/roottree/$ent ]; then
	if [ -d work/roottree/$ent -o -f work/roottree/$ent ]; then
	    mkdir -p work/isotree/root/$graft
	    mv work/roottree/$ent work/isotree/root/$ent
	    ( cd work/roottree/$graft && ln -s ${dots}usr/root/$ent . )
	fi
    fi
done
# Create links for things in moved root directories that use relative
# paths to find /usr
( cd work/isotree/root && ln -s .. ./usr )

# Make sure we don't have conflict between isotree/root and usr/root
( cd work/isotree && find root -not -type d -print0 ) | \
    ( cd usr && xargs -0rt rm -f )

# Create root tarball
( cd work/roottree && tar cfp - . | gzip -9 > ../nozftree/root.tar.gz )

# Generate distribution copy of build tree
# (Be careful when to use "cp" versus "cp -a" here!)
mkdir -p work/buildtree/root work/buildtree/usr
cp -a $BUILDDIRS work/buildtree
cp $MAKES $READMES work/buildtree

# Create build tree tarball
find work/buildtree -type d -name RCS -print0 | \
    xargs -0rt rm -rf
( cd work/buildtree && tar cfp ../nozftree/buildtree-$1.tar . )
( cd work/nozftree && gzip -9 buildtree-$1.tar && \
  ln -s buildtree-$1.tar.gz buildtree.tar.gz )

# Copy buildtree tarball to the root directory; for separate upload.
cp work/nozftree/buildtree-$1.tar.gz .

# Create version number and buildinfo files
echo "$1" > work/nozftree/SuperRescue-2
cat > work/nozftree/buildinfo.txt <<EOF
SuperRescue $1
Build by $BUILDER on $DATE
EOF

# Install README files
cp $READMES work/nozftree

# Compression test file (highly compressible)
cat > work/isotree/compression-test-12345 <<EOF
If you can read this, your kernel has zisofs/RockRidge compression support
enabled.
EOF
havebytes=`wc -c < work/isotree/compression-test-12345`
needbytes=`expr 12345 - $havebytes`
dd if=/dev/zero bs=$needbytes count=1 >> work/isotree/compression-test-12345

# Things that must not be in the usr tree
rm -f usr/root.tar.gz usr/buildtree*.tar.gz
rm -rf usr/isolinux usr/boot/isoboot.img usr/boot/isoboot.cat
rm -f usr/SuperRescue-2 usr/hpa-mega-rescue-disk-1
rm -f usr/buildinfo.txt usr/compression-test-12345
( cd usr && rm -f $READMES )

# Create compressed trees.
time $MKZFTREE -C work/usr.zf.old usr work/usr.zf
rm -rf work/usr.zf.old 2>/dev/null
time $MKZFTREE -C work/isotree.zf.old work/isotree work/isotree.zf
rm -rf work/isotree.zf.old 2>/dev/null

# Generate ISO filesystem.
rm -f mkisofs.log
$MKISOFS -z -v -v -o superrescue-$1.iso \
    -b isolinux/isolinux.bin -c isolinux/isoboot.cat \
    -no-emul-boot -boot-load-size 4 -boot-info-table \
    -l -J -R -hide-rr-moved \
    -A SuperRescue -P "$BUILDER" -V "SuperRescue $1" \
    work/nozftree work/isotree.zf work/usr.zf \
    > mkisofs.log 2>&1

# Generate compressed version
mkdir -p work/gzdir
ln -f superrescue-$1.iso work/gzdir/superrescue-$1.iso
( cd work/gzdir && gzip -9f superrescue-$1.iso )
mv work/gzdir/superrescue-$1.iso.gz .
rm -rf work/gzdir
# gzip seems to fsck up permissions somehow...
chmod 444 superrescue-$1.iso superrescue-$1.iso.gz
