/************************************************************************/
/* File		dataindex.h						*/
/*									*/
/* Purpose	This C++ header file contains the class definition for	*/
/*		the DataIndexRec class, the DataIndexHeader class,	*/
/*		and the DataIndex class. The DataIndex class manages	*/
/*		ObjectFile class record numbers. Each record number is	*/
/*		stored in the DataIndex file using a unique key. The	*/
/*		DataIndex file maintains the unique keys using a	*/
/*		red/black tree structure. The DataIndexRec class and	*/
/*		the DataIndexHeader class are classes that support the	*/
/*		DataIndex class.					*/
/*									*/
/* Author	This C++ header file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ header file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Sunday, May 26, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#ifndef OBJECTINDEX_H			// If not defined...
#define OBJECTINDEX_H			// Then process the header file.

#define DEBUG

/* Headers	The following headers are required by dataindex.h.	*/

#include "utilitymodule.h"		// UtilityModule class.
#include "objectfile.h"			// ObjectFile class.

/* Constants	The values of the following constants contain the	*/
/*		version information for the DataIndex library.		*/

const int DataIndexVersion	= 0;	// Library version number.
const int DataIndexRevision	= 0;	// Library revision number.
const int DataIndexUpdata	= 0;	// Library update number.

/* Constant	The following constant defines the default DataIndex	*/
/*		file mode. This file mode is used to create a file if	*/
/*		no mode is specified.					*/

const int DataIndexMode		= PhysicalOSOwnRead | PhysicalOSOwnWrite |
				  PhysicalOSGrpRead | PhysicalOSGrpWrite;

/* Constant	The following constant defines the default DataIndex	*/
/*		file flag. This file flag is used to open files if no	*/
/*		flag is specified.					*/

const int DataIndexFlag		= FileReadWrite;

/* Constant	The following constant defines the number of array	*/
/*		elements to allocate when expanding the stacks that are	*/
/*		used by the DataIndex class functions.			*/

const int DataIndexArrayBlock	= 5;

/* Constants	The following constants define the valid DataIndex	*/
/*		error codes that are used by the DataIndex member	*/
/*		functions.						*/

const int DataIndexErrors	= 19;	// Number of error messages.
const int DataIndexNoError	= 0;	// No error.
const int DataIndexBadMake	= 1;	// Could not make index file.
const int DataIndexBadOpen	= 2;	// Could not open index file.
const int DataIndexBadClose	= 3;	// Could not close index file.
const int DataIndexBadIndex	= 4;	// Index is corrupted.
const int DataIndexBadAdd	= 5;	// Could not add key.
const int DataIndexBadDel	= 6;	// Could not remove key.
const int DataIndexNoInsert	= 7;	// Could not insert new key.
const int DataIndexNoDelete	= 8;	// Could not delete old key.
const int DataIndexBadRead	= 9;	// Could not read index record.
const int DataIndexBadWrite	= 10;	// Could not write index record.
const int DataIndexKeyDup	= 11;	// Key already exists.
const int DataIndexKeyNot	= 12;	// Key does not exist.
const int DataIndexLocateInsert	= 13;	// No locate insertion point.
const int DataIndexLocateDelete	= 14;	// No locate delete point.
const int DataIndexBadFind	= 15;	// Could not find key.
const int DataIndexBadNext	= 16;	// Could not find next key.
const int DataIndexBadPrev	= 17;	// Could not find prev key.
const int DataIndexBadChange	= 18;	// Could not change key's value.

/* Typedef	The following typedef defines the Red and Black colors	*/
/*		that are used by the DataIndex class. The colors are	*/
/*		used to keep the tree balanced.				*/

typedef bool RBColor;
const RBColor RBColorRed	= true;
const RBColor RBColorBlack	= false;

/* Class	The DataIndexRec class defines the record structure	*/
/*		that is used by the DataIndex class. Each record	*/
/*		contains a key, a value, a color, a left node, and a	*/
/*		right node.						*/

class DataIndexRec
   {
      public:
         DataIndexRec();
	 DataIndexRec(const DataIndexRec&);
	 ~DataIndexRec();
	 int Record_Size(void);
	 status Read_From_File(File&, const int);
	 status Write_To_File(File&, const int);
	 String Get_Record_Key(void) const;
	 void Set_Record_Key(const char*);
	 void Set_Record_Key(const String&);
	 FilePointer Get_Value(void) const;
	 void Set_Value(const FilePointer);
	 RBColor Get_Color(void) const;
	 void Set_Color(const RBColor);
	 FilePointer Get_Left_Node(void) const;
	 void Set_Left_Node(const FilePointer);
	 FilePointer Get_Right_Node(void) const;
	 void Set_Right_Node(const FilePointer);

      private:
         String		itskey;
         FilePointer	itsvalue;
	 RBColor	itscolor;
	 FilePointer	itsleftnode;
	 FilePointer	itsrightnode;
   };

/* Class	The DataIndexHeader class defines the header structure	*/
/*		that is written to the beginning of a DataIndex file.	*/
/*		The ObjectFile class defines this as a user header.	*/

class DataIndexHeader
   {
      public:
         DataIndexHeader();
	 ~DataIndexHeader();
	 void Get_Header_Message(char*);
	 void Set_Header_Message(const char*);
	 condition Is_Index_Locked(void) const;
	 void Set_Locked_Index_State(const condition);
	 void Increment_Node_Count(void);
	 void Decrement_Node_Count(void);
	 int Get_Node_Count(void) const;
	 void Set_Node_Count(const int);
	 FilePointer Get_Root_Node(void) const;
	 void Set_Root_Node(const FilePointer);

      private:
	 char			itsmessage[64];
	 unsigned int		itslockedflag:1;
         int			itsnodes;
	 FilePointer		itsrootnode;
   };

/* Class	The following class defines the DataIndex class. The	*/
/*		DataIndex class manages record numbers using a unique	*/
/*		key. The keys are maintained in a red/black tree.	*/

class DataIndex : public UtilityModule
   {
      public:
         DataIndex();
	 ~DataIndex();
	 status Create_DataIndex_File(const char*,
	    const int = DataIndexMode);
	 status Create_DataIndex_File(const String&,
	    const int = DataIndexMode);
	 status Open_DataIndex_File(const char*,
	    const int = DataIndexFlag);
	 status Open_DataIndex_File(const String&,
	    const int = DataIndexFlag);
	 status Close_DataIndex_File(void);
	 status Add_Key(const char*, const FilePointer);
	 status Add_Key(const String&, const FilePointer);
	 status Change_Value(const char*, const FilePointer);
	 status Change_Value(const String&, const FilePointer);
	 status Remove_Key(const char*);
	 status Remove_Key(const String&);
	 status Find_Key(const char*, FilePointer&);
	 status Find_Key(const String&, FilePointer&);
	 status Get_First_Key(String&, FilePointer&);
	 status Get_Next_Key(String&, FilePointer&);
	 status Get_Last_Key(String&, FilePointer&);
	 status Get_Previous_Key(String&, FilePointer&);
	 #ifdef DEBUG
	 status Verify_Tree(void);
	 status Write_Tree(void);
	 #endif

      private:
         FilePointer			itsdeletedpointer;
	 DataIndexRec			itsdeletedobject;
	 int				itsindex;
	 int				itsnextindex;
	 int				itsprevindex;
	 int				itsstacksize;
	 int				itsnextstacksize;
	 int				itsprevstacksize;
	 relation*			itsdifferences;
	 FilePointer*			itspointers;
	 DataIndexRec*			itsobjects;
	 FilePointer			itsnextnode;
	 DataIndexRec*			itsnextobjects;
	 FilePointer			itsprevnode;
	 DataIndexRec*			itsprevobjects;
	 String				itsfilename;
         DataIndexHeader		itsheader;
         ObjectFile<DataIndexRec>	itsindexfile;

	 void expand_stack(void);
	 void expand_nextstack(void);
	 void expand_prevstack(void);
	 void push_record(DataIndexRec&, FilePointer, relation);
	 void push_nextrecord(DataIndexRec&);
	 void push_prevrecord(DataIndexRec&);
	 status read_index_record(DataIndexRec&, FilePointer);
	 status write_index_record(DataIndexRec&, FilePointer);

	 status case_one_delete(void);
	 status case_two_delete(DataIndexRec&, FilePointer);
	 status case_three_delete(DataIndexRec&, FilePointer);
	 status insert_case_one_both_sides(DataIndexRec&, FilePointer);
	 status insert_case_two_right_side(DataIndexRec&, FilePointer);
	 status insert_case_three_right_side(DataIndexRec&, FilePointer&);
	 status insert_case_two_left_side(DataIndexRec&, FilePointer);
	 status insert_case_three_left_side(DataIndexRec&, FilePointer&);
	 status ensure_right_side_black(DataIndexRec&, FilePointer&);
	 status delete_case_one_both_sides(DataIndexRec&, FilePointer);
	 status delete_case_two_right_side(DataIndexRec&, FilePointer);
	 status delete_case_three_right_side(DataIndexRec&, FilePointer&);
	 status ensure_left_side_black(DataIndexRec&, FilePointer&);
	 status delete_case_two_left_side(DataIndexRec&, FilePointer);
	 status delete_case_three_left_side(DataIndexRec&, FilePointer&);
	 status locate_insertion_point(const char*, FilePointer&);
	 status locate_deletion_point(const char*);
	 status insert_key(const char*, const FilePointer);
	 status delete_key(void);
	 status rebalance_after_insert(void);
	 status rebalance_after_delete(void);

	 #ifdef DEBUG
	 void recursive_verify_tree(FilePointer, status&, int&, int&);
	 void recursive_write_tree(FilePointer node, status&);
	 #endif
   };

#endif					// OBJECTINDEX_H
