/* Source Installer, Copyright (c) 2005-2006 Claudio Fontana

 gui.c - interface fundamentals

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org
*/

#include "global.h"
#include "gui.h"
#include "menu.h"
#include "dialog.h"
#include "events.h"

#include "about.h"
#include "pref.h"
#include "greeting.h"
#include "edit.h"
#include "toolbar.h"
#include "console.h"
#include "package_info.h"
#include "package_list.h"
#include "text.h"
#include "action.h"
#include "status.h"

GtkWidget* w_main[W_MAIN_N];
GtkAccelGroup* acc;

void gui_splash(void) {
  GtkWidget* splash;
  w_main[W_SPLASH] = gtk_window_new(GTK_WINDOW_POPUP);
  splash = gtk_image_new_from_file(DATADIR "/images/splash.png");

  gtk_window_set_decorated(GTK_WINDOW(w_main[W_SPLASH]), FALSE);
  gtk_container_add(GTK_CONTAINER(w_main[W_SPLASH]), splash);
  gtk_window_set_position(GTK_WINDOW(w_main[W_SPLASH]),
			  GTK_WIN_POS_CENTER_ALWAYS);

  gtk_widget_show_all(w_main[W_SPLASH]);
  gui_doevents(0);
}


void gui_unsplash(void) {
  gtk_widget_destroy(w_main[W_SPLASH]);
}

static void gui_icon_init(void) {
  /*
  GdkPixbuf* sizes[5] = { 0, 0, 0, 0, 0 };
  char* filenames[5] = { DATADIR "/images/icon16x16.png",
			 DATADIR "/images/icon24x24.png",
			 DATADIR "/images/icon32x32.png",
			 DATADIR "/images/icon48x48.png",
			 0 };
  int i;
  GList* icon_list = NULL;

  for (i = 0; filenames[i]; i++) {
    sizes[i] = gdk_pixbuf_new_from_file(filenames[i], NULL);
    if (sizes[i])
      icon_list = g_list_append(icon_list, sizes[i]);
  }
  
  if (icon_list)
    gtk_window_set_default_icon_list(icon_list);
  */

  GdkPixbuf* buf;
  if ((buf = gdk_pixbuf_new_from_file(DATADIR "/images/icon48x48.png", NULL)))
    gtk_window_set_default_icon(buf);
}

void gui_init(void) {
  GtkWidget* left, *right;
  printf("gui_init\n");

  gui_icon_init();

  text_init();			/* initialize text tags */
  w_main[W_MAIN] = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(w_main[W_MAIN]), "GNU Source Installer");

  acc = gtk_accel_group_new();
  gtk_window_add_accel_group(GTK_WINDOW(w_main[W_MAIN]), acc);

  about_create();		/* about dialog */
  pref_create();		/* preferences dialog */

  w_main[W_MAIN_CONTENT] = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(w_main[W_MAIN]), w_main[W_MAIN_CONTENT]);
  
  menu_create();		/* create and populate menubar */
  gtk_box_pack_start(GTK_BOX(w_main[W_MAIN_CONTENT]), w_menu[W_MENU], FALSE, FALSE, 0);

  toolbar_create();		/* actions toolbar dialog */
  gtk_box_pack_start(GTK_BOX(w_main[W_MAIN_CONTENT]), w_toolbar[W_TOOLBAR], FALSE, FALSE, 0);
  
  /* pane_vert */
  w_main[W_MAIN_PANE] = gtk_hpaned_new();
  left = gtk_vbox_new(FALSE, 0); right = gtk_vbox_new(FALSE, 0);

  gtk_paned_pack1(GTK_PANED(w_main[W_MAIN_PANE]), left, FALSE, FALSE);
  gtk_paned_pack2(GTK_PANED(w_main[W_MAIN_PANE]), right, TRUE, TRUE);
  
  gtk_container_add(GTK_CONTAINER(w_main[W_MAIN_CONTENT]), w_main[W_MAIN_PANE]);
  
  console_create();		/* pseudo-console */
  gtk_box_pack_start(GTK_BOX(right), w_console[W_CONSOLE], TRUE, TRUE, 0);

  package_info_create();	/* package information */
  gtk_box_pack_start(GTK_BOX(right), w_package_info[W_PACKAGE_INFO], TRUE, TRUE, 0);

  greeting_create();
  gtk_box_pack_start(GTK_BOX(right), w_greeting[W_GREETING], TRUE, TRUE, 0);

  package_list_create();	/* package list */
  gtk_box_pack_start(GTK_BOX(left), w_package_list[W_PACKAGE_LIST_SCR], TRUE, TRUE, 0);

  status_create();		/* progress status */
  gtk_box_pack_start(GTK_BOX(w_main[W_MAIN_CONTENT]), w_status[W_STATUS], FALSE, FALSE, 0);

  gui_set_default_size();
}

void gui_show(void) {
  gtk_widget_show_all(w_main[W_MAIN]);
  status_hide();
}

void gui_set_default_size(void) {
  gint width, height;
  GdkScreen* screen;
  printf("gui_set_default_size\n");

  screen = gdk_screen_get_default();

  width = gdk_screen_get_width(screen) * 2 / 3;
  height = gdk_screen_get_height(screen) / 2;

  gtk_window_set_default_size(GTK_WINDOW(w_main[W_MAIN]), width, height);
  gtk_window_resize(GTK_WINDOW(w_main[W_MAIN]), width, height);
}

void gui_maybe_quit(void) {
  int quit = 1;
  char* title = "Program exit";

  if (state.actions_locked) {
    quit = dialog_confirm(title, "Action in progress!",
		       "A quit request has been detected, but currently "
		       "executing action is not complete.\n\n"
		       "Are you sure that you want to quit?", 0);
  }
  
  if (quit) {
    gui_destroy(NULL, NULL);
  }
}

/* gui update functions */

void gui_update(void) {
  /* update GUI basing on current state */

  menu_update();
  toolbar_update();
  package_list_update();
  /* cursor_update(); */
}

/* process pending events */

void gui_doevents(char* unused) {
  while (g_main_context_iteration(NULL, FALSE));
}

/* GUI actions */

void gui_action_add(GtkWidget* w, gpointer data) {
  char* filename, *packagename;
  printf("gui_action_add\n");

  filename = packagename = 0;

  if (dialog_input_package2("Choose package or directory to add",
			    &filename, &packagename)) {
    /* printf("filename: %s\npackname: %s\n", filename, packagename); */
    action_add(filename, packagename);
  }

  if (filename) free(filename);
  if (packagename) free(packagename);
}

void gui_action_remove(GtkWidget* w, gpointer data) {
  printf("gui_action_remove\n");

  if (dialog_confirm("Remove package", "Completely remove package?", srcinst_get_name(state.package), "will be uninstalled, and its stored source removed. Really proceed?"))
    action_remove();
}

void gui_action_install(GtkWidget* w, gpointer data) {
  int reconf = 1;
  printf("gui_action_install\n");

  if (srcinst_get_configure_string(state.package)) {
    if (!dialog_confirm("Install package", "Reconfigure package?", "This package is already configured, and you might want to keep the same configuration. Do you want to reconfigure the package?", 0))
      reconf = 0;
  }

  action_install(reconf);
}

void gui_action_uninstall(GtkWidget* w, gpointer data) {
  printf("gui_action_uninstall\n");

  if (dialog_confirm("Uninstall package", "Uninstall package?", srcinst_get_name(state.package), "installed files will be deleted; really proceed?"))
    action_uninstall();
}

void gui_action_upgrade(GtkWidget* w, gpointer data) {
  char* filename, *packagename;
  printf("gui_action_upgrade\n");

  filename = packagename = 0;

  if (srcinst_is_installed(state.package) &&
      srcinst_get_installed_files(state.package)->count == 0) {
    dialog_warning("Upgrade package", "Not allowed",
		   "The safe-upgrade action is not possible for non-conforming"
		   "packages. Instead, remove the previous version, then"
		   "install the new one.", 0);
    return;
  }

  if (dialog_input_package2("Choose replacement package or directory",
			    &filename, &packagename)) {
    /* printf("filename: %s\npackname: %s\n", filename, packagename); */
    char* txt; int c;
    txt = srcinst_strjoin(filename, " will be built, and if successful, current package ", srcinst_get_name(state.package), " will be replaced by the new version. Proceed?", 0);
    c = dialog_confirm("Upgrade package", "Upgrade current package?", txt, 0);
    free(txt);
    if (c)
      action_upgrade(filename, packagename);
  }

  if (filename) free(filename);
  if (packagename) free(packagename);
}

void gui_action_about(GtkWidget* widget, gpointer data) {
  printf("gui_action_about\n");
  about_run();
}

void gui_action_pref(GtkWidget* widget, gpointer data) {
  printf("gui_action_pref\n");
  pref_show();
}

void gui_action_stop(GtkWidget* widget, gpointer data) {
  printf("gui_action_stop\n");
  srcinst_interrupt();
}

void gui_action_export_all(GtkWidget* widget, gpointer data) {
  char* filename; SRCINST_EXPORT fmt;
  printf("gui_action_export_all\n");

  filename = dialog_input_export_info("Export all packages information", 
				      0, &fmt);
  if (filename) {
    action_export_info(0, filename, fmt);
    free(filename);
  }
}

void gui_action_export_info(GtkWidget* widget, gpointer data) {
  char* filename; SRCINST_EXPORT fmt;
  printf("gui_action_export_info\n");

  filename = dialog_input_export_info("Export package information", 
				      srcinst_get_name(state.package), &fmt);
  if (filename) {
    action_export_info(state.package, filename, fmt);
    free(filename);
  }
}

void gui_action_export_bin(GtkWidget* widget, gpointer data) {
  char* filename;
  printf("gui_action_export_bin\n");

  filename = dialog_input_export_bin("Export as binary package",
				     srcinst_get_name(state.package),
				     state.o.src_compress
				     );
  if (filename)
    action_export_bin(filename, state.o.src_compress);
}

void gui_action_savelog(GtkWidget* w, gpointer unused_data) {
  char* filename = 0; char* title = "Save console log";
  GtkWidget* dialog; gint response;
  printf("gui_action_savelog\n");

  dialog = gtk_file_chooser_dialog_new(title, GTK_WINDOW(w_main[W_MAIN]),
				       GTK_FILE_CHOOSER_ACTION_SAVE,
				       GTK_STOCK_SAVE, GTK_RESPONSE_OK,
				       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				       NULL);

  gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), "console.log");

  gtk_widget_show_all(dialog);
  response = gtk_dialog_run(GTK_DIALOG(dialog));

  if (response == GTK_RESPONSE_OK)
    filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));

  gtk_widget_destroy(dialog);
  
  if (filename) {
    if (console_savelog(filename)) {
      dialog_message(title, "Console log saved.", "The console log has been saved as", filename);
    } else {
      dialog_warning(title, "Error saving console log", "Console log save could NOT be completed successfully", 0);
    }
    free(filename);
  }
}

void gui_action_default_size(GtkWidget* unused_w, gpointer unused_d) {
  printf("gui_action_default_size\n");
  gui_set_default_size();
}

/* gui events */

gboolean gui_delete_event(GtkWidget* w, GdkEvent* e, gpointer data) {
  /* If you return FALSE in the "delete_event" signal handler,
   * GTK will emit the "destroy" signal. Returning TRUE means
   * you don't want the window to be destroyed. */
  printf("gui_delete_event\n");
  gui_maybe_quit();
  return TRUE;
}

void gui_destroy(GtkWidget* w, gpointer data) {
  printf("gui_destroy\n");
  exit(0);
}
