/* Source Installer, Copyright (c) 2005-2006 Claudio Fontana

 srcinst.c - main interface to source installer library implementation

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org
*/

#include "src_stdinc.h"
#include "package_info.h"
#include "actions.h"
#include "srcexport.h"

/* interruptible action */
#define SRCINST_INTR_ACTION(action)                   \
    _srcinst_state.sigint_seen = 0;                   \
    rv = action;                                      \
    if (rv != SRCINST_ERR_OK)                         \
      if (_srcinst_state.sigint_seen)                 \
	rv = SRCINST_ERR_INTR;                        \

struct _srcinst_state _srcinst_state;

/* static auxiliary functions */

static void _init_state(void);
static int _init_pathnames(void);
static void _free_pathnames(void);
static int _lock_user(void);
static void _unlock_user(void);
static int _register_signals(void);
static int _unregister_signals(void);
static void _handle_SIGINT(int s);

#ifdef SA_SIGINFO
static void _handle_SIGINT_2(int s, siginfo_t* t, void* p);
#endif

/* for really bad problems, like heap errors and such. Otherwise, functions
   will report the error and return to the calling code. */

void srcinst_exit(int errcode) {
  srcinst_fini();

  if (fclose(stdout) != 0) {
    errcode = 0xff;
  }

  exit(errcode);
}

/* initialize the library. This is the first srcinst function to call.
   See srcinst.h for FLAGS.
 */

int srcinst_init(unsigned int flags) {
  _init_state();
  
  if (!_init_pathnames())
    return 0;

  _init_preferences(&_srcinst_state.preferences);
  _init_package_list(&_srcinst_state.packages);
  
  if (!srcinst_check_dirs()) {
    /* check/ensure existence of user directory structure */
    goto err_proc;
  }

  if (!_lock_user()) {
    goto err_proc;
  }

  if (!_clear_builddirs()) {
    /* clear files in build directories from a previous interrupted run */
    goto err_proc;
  }
  
  if (srcinst_file_type(_srcinst_state.files[SRCINST_FILE_CFG]) ==
      SRCINST_TYPE_NONE) {
    if (!_save_preferences(&_srcinst_state.preferences)) {
      srcinst_warning(SRCINST_ERR_CORE, "could not write preferences", 
		      _srcinst_state.files[SRCINST_FILE_CFG]);
      goto err_proc;
    }

  } else if (!_load_preferences(&_srcinst_state.preferences)) {
    srcinst_warning(SRCINST_ERR_CORE, "could not load preferences", 0);
    goto err_proc;
  }
  
  if (!_register_signals()) {
    srcinst_warning(SRCINST_ERR_CORE, "could not register signal handlers", 0);
    goto err_proc;
  }

  if (!_load_package_list(&_srcinst_state.packages, (flags & SRCINST_FLAG_LOAD_ONDEMAND) ? 0 : -1)) {
    srcinst_warning(SRCINST_ERR_CORE, "could not load package list", 0);
    goto err_proc;
  }

  return _srcinst_state.initialized = 1;

 err_proc:
  _free_package_list(&_srcinst_state.packages);
  _free_preferences(&_srcinst_state.preferences);
  _free_pathnames();

  return _srcinst_state.initialized = 0;
}

static void _init_state(void) {
  memset(&_srcinst_state, 0, sizeof(struct _srcinst_state));
  _srcinst_state.beat_delay = 50000;
}

/* The SIGINT handler just accounts the number of signals detected in the
   state, and then calls the previous handler if present.
   However, if the process is currently waiting for another process
   (see srcinst_wait() in srcspawn.c), that process is also sent a SIGINT.
 */

static void _handle_SIGINT(int s) {
  _srcinst_state.sigint_seen++;
  
  if (_srcinst_state.waiting_for) {
    kill(srcinst_get_spawn_pid(_srcinst_state.waiting_for), SIGINT);
  }

  if (_srcinst_state.sa_old_SIGINT.sa_handler != SIG_IGN &&
      _srcinst_state.sa_old_SIGINT.sa_handler != SIG_DFL) {
    _srcinst_state.sa_old_SIGINT.sa_handler(s);
  }
}

/* if extended signal handlers are supported, must provide another
   implementation */

#ifdef SA_SIGINFO
static void _handle_SIGINT_2(int s, siginfo_t* t, void* p) {
  _srcinst_state.sigint_seen++;

  if (_srcinst_state.waiting_for) {
    kill(srcinst_get_spawn_pid(_srcinst_state.waiting_for), SIGINT);
  }

  if ((void (*)(int))_srcinst_state.sa_old_SIGINT.sa_sigaction != SIG_IGN &&
      (void (*)(int))_srcinst_state.sa_old_SIGINT.sa_sigaction != SIG_DFL) {
    _srcinst_state.sa_old_SIGINT.sa_sigaction(s, t, p);
  }
}
#endif

static int _register_signals(void) {
  struct sigaction sa;
  memset(&sa, 0, sizeof(struct sigaction));

  /* SIGINT */

#ifdef SA_SIGINFO
  if (sigaction(SIGINT, 0, &_srcinst_state.sa_old_SIGINT) != 0)
    return 0;

  if (_srcinst_state.sa_old_SIGINT.sa_flags & SA_SIGINFO) {
    sa.sa_sigaction = &_handle_SIGINT_2;
  } else {
    sa.sa_handler = &_handle_SIGINT;
  }
#else
  sa.sa_handler = &_handle_SIGINT;
#endif
  
  if (sigaction(SIGINT, &sa, &_srcinst_state.sa_old_SIGINT) != 0)
    return 0;
  
  return 1;
}


/* restore previous handlers */

static int _unregister_signals(void) {
  if (sigaction(SIGINT, &_srcinst_state.sa_old_SIGINT, 0) != 0)
    return 0;

  return 1;
}

/* raise an interrupt signal in current process */

void srcinst_interrupt(void) {
  raise(SIGINT);
}

/* register callbacks for capturing process output, or regaining
   control during process execution without needing threads */

void srcinst_register_stdout_callback(srcinst_callback f) {
  if (!f) {
    _fini_callback_list(&_srcinst_state.stdout_callbacks);

  } else {
    _add_callback_list(&_srcinst_state.stdout_callbacks, f);
  }
}

void srcinst_unregister_stdout_callback(srcinst_callback f) {
  if (!f) {
    _fini_callback_list(&_srcinst_state.stdout_callbacks);

  } else {
    _remove_callback_list(&_srcinst_state.stdout_callbacks, f);
  }
}

void srcinst_register_stderr_callback(srcinst_callback f) {
  if (!f) {
    _fini_callback_list(&_srcinst_state.stderr_callbacks);

  } else {
    _add_callback_list(&_srcinst_state.stderr_callbacks, f);
  }
}

void srcinst_unregister_stderr_callback(srcinst_callback f) {
  if (!f) {
    _fini_callback_list(&_srcinst_state.stderr_callbacks);

  } else {
    _remove_callback_list(&_srcinst_state.stderr_callbacks, f);
  }
}

void srcinst_register_spawn_callback(srcinst_callback f) {
  if (!f) {
    _fini_callback_list(&_srcinst_state.spawn_callbacks);

  } else {
    _add_callback_list(&_srcinst_state.spawn_callbacks, f);
  }
}

void srcinst_unregister_spawn_callback(srcinst_callback f) {
  if (!f) {
    _fini_callback_list(&_srcinst_state.spawn_callbacks);

  } else {
    _remove_callback_list(&_srcinst_state.spawn_callbacks, f);
  }
}

void srcinst_register_beat_callback(srcinst_callback f) {
  if (!f) {
    _fini_callback_list(&_srcinst_state.beat_callbacks);

  } else {
    _add_callback_list(&_srcinst_state.beat_callbacks, f);
  }
}

void srcinst_unregister_beat_callback(srcinst_callback f) {
  if (!f) {
    _fini_callback_list(&_srcinst_state.beat_callbacks);

  } else {
    _remove_callback_list(&_srcinst_state.beat_callbacks, f);
  }
}

long srcinst_get_beat_delay(void) {
  return _srcinst_state.beat_delay;
}

void srcinst_set_beat_delay(long delay) {
  _srcinst_state.beat_delay = delay;
}

/* returns 1 if ok (user was not locked, and user is now successfully locked),
   or 0 if lock could not be acquired. */

static int _lock_user(void) {
  int fd; char pid_str[64];
  char* lockfile; struct flock fd_lock;

  lockfile = _srcinst_state.files[SRCINST_FILE_LOCK];
  memset(&fd_lock, 0, sizeof(fd_lock));
  fd_lock.l_type = F_WRLCK; fd_lock.l_whence = SEEK_SET;
  fd_lock.l_start = fd_lock.l_len = 0;
  
  if ((fd = open(lockfile, O_RDWR | O_CREAT | O_TRUNC, 0700)) == -1) {
    srcinst_warning(SRCINST_ERR_LOCK, "cannot open lockfile", lockfile);
    return 0;
  }

  if (fcntl(fd, F_SETLK, &fd_lock) != 0) {

    srcinst_warning(SRCINST_ERR_LOCK, "could not lock file", lockfile);
    if (fcntl(fd, F_GETLK, &fd_lock) != 0 || fd_lock.l_type == F_UNLCK) {
      close(fd); return 0;
    }
    
    sprintf(pid_str, "%d", (int)(fd_lock.l_pid));
    srcinst_warning(SRCINST_ERR_LOCK, "seems to be held by process", pid_str);
    close(fd); return 0;
  }
  
  sprintf(pid_str, "%d\n", (int)getpid());
  write(fd, pid_str, strlen(pid_str));
  _srcinst_state.user_locked = 1;
  _srcinst_state.lockfile_fd = fd;
  return 1;
}

static void _unlock_user(void) {
  char* fname; fname = _srcinst_state.files[SRCINST_FILE_LOCK];

  if (close(_srcinst_state.lockfile_fd) != 0) {
    srcinst_warning(SRCINST_ERR_LOCK, "error closing lockfile", fname);
  }
}

int srcinst_check_dirs(void) {
  int i; int dirs[] = {
    SRCINST_DIR_CFG,
    SRCINST_DIR_PACK, SRCINST_DIR_SRC,
    SRCINST_DIR_BUILD, SRCINST_DIR_DEST, -1
  };
  
  for (i = 0; dirs[i] != -1; i++) {
    int t;
    char* dirname = _srcinst_state.dirs[dirs[i]];
    t = srcinst_file_type(dirname);

    if (t != SRCINST_TYPE_NONE) {

      if (t != SRCINST_TYPE_DIR || !srcinst_file_access(dirname, "rwx")) {
	srcinst_warning(SRCINST_ERR_CORE, "cannot access directory", dirname);
	return 0;
      }

    } else if (mkdir(dirname, 0755) != 0 || !srcinst_chown_user(dirname)) {
      srcinst_warning(SRCINST_ERR_CORE, "cannot create directory", dirname);
      return 0;
    }
  }
  
  return 1;
}

void srcinst_fini(void) {
  if (!_srcinst_state.initialized) {
    /* if library is not initialized there is nothing to do */
    return;
  }

  if (_srcinst_state.user_locked)
    _unlock_user();

  _unregister_signals();

  _free_package_list(&_srcinst_state.packages);
  _free_preferences(&_srcinst_state.preferences);
  _free_pathnames();
  _init_state();		/* restore initial state */
}


static int _init_pathnames(void) {
  char buffer[SRCINST_BUFSIZE]; char* h;
  memset(_srcinst_state.files, 0, sizeof(char*) * SRCINST_FILE_N);
  memset(_srcinst_state.dirs, 0, sizeof(char*) * SRCINST_DIR_N);

  if (!(h = getenv("HOME"))) {
    srcinst_warning(SRCINST_ERR_CORE, "cannot find env variable HOME", 0);
    return 0;
  }

  if (!getcwd(buffer, SRCINST_BUFSIZE)) {
    srcinst_warning(SRCINST_ERR_CORE, "could not get current directory", "_init_pathnames");
    _srcinst_state.dirs[SRCINST_DIR_RUN] = 0;
  } else {
    _srcinst_state.dirs[SRCINST_DIR_RUN] = srcinst_strdup(buffer);
  }
  _srcinst_state.dirs[SRCINST_DIR_RUNBUILD] = 0;
  _srcinst_state.dirs[SRCINST_DIR_CFG] = srcinst_fnjoin(h, ".sourceinstall");
  _srcinst_state.dirs[SRCINST_DIR_PACK] = srcinst_fnjoin(_srcinst_state.dirs[SRCINST_DIR_CFG], "packages");
  _srcinst_state.dirs[SRCINST_DIR_SRC] = srcinst_fnjoin(_srcinst_state.dirs[SRCINST_DIR_CFG], "src");
  _srcinst_state.dirs[SRCINST_DIR_BUILD] = srcinst_fnjoin(_srcinst_state.dirs[SRCINST_DIR_CFG], "build");
  _srcinst_state.dirs[SRCINST_DIR_DEST] = srcinst_fnjoin(_srcinst_state.dirs[SRCINST_DIR_CFG], "install-destdir");
  
  /* _srcinst_state.dirs[SRCINST_DIR_ICON] = something; */

  _srcinst_state.files[SRCINST_FILE_CFG] = srcinst_fnjoin(_srcinst_state.dirs[SRCINST_DIR_CFG], "sourceinstallrc");
  _srcinst_state.files[SRCINST_FILE_LOCK] = srcinst_fnjoin(_srcinst_state.dirs[SRCINST_DIR_CFG], ".sourceinstall_lock");
  return 1;
}

static void _free_pathnames(void) {
  int i;

  for (i = 0; i < SRCINST_FILE_N; i++) {
    if (_srcinst_state.files[i])
      { free(_srcinst_state.files[i]); _srcinst_state.files[i] = 0; }
  }

  for (i = 0; i < SRCINST_DIR_N; i++) {
    if (_srcinst_state.dirs[i])
      { free(_srcinst_state.dirs[i]); _srcinst_state.dirs[i] = 0; }
  }
}

char* srcinst_get_dir(int idx) {
  if (idx >= SRCINST_DIR_N || idx < 0)
    return 0;

  return _srcinst_state.dirs[idx];
}

char* srcinst_get_file(int idx) {
  if (idx >= SRCINST_FILE_N || idx < 0)
    return 0;

  return _srcinst_state.files[idx];
}

void srcinst_list_packages(struct srcinst_string_list* list) {
  int i; struct _package_element* this;

  list->count = _srcinst_state.packages.count;
  list->strings = srcinst_malloc(sizeof(char*) * list->count);
  
  for (this = _srcinst_state.packages.first, i = 0; this; this = this->next, i++) {
    list->strings[i] = srcinst_strdup(this->info->name);
  }
}

/* open existing package called NAME */

SRCINST_PHANDLE srcinst_open_package(char* name) {
  struct _package_element* e;
  e = _search_package_list(&_srcinst_state.packages, name);

  if (!e)
    return 0;

  e->info->onclose = SRCINST_ONCLOSE_NOP;

  if (!e->info->loaded)
    if (!_load_package_info(e->info))
      return 0;

  return e;
}

/* NAME may be (char*)0 to mean 'assign default'. */

SRCINST_PHANDLE srcinst_new_package(char* name) {
  struct _package_element* e; struct _package_info* info;

  if (name && _search_package_list(&_srcinst_state.packages, name)) {
    /* a package with the same name already exists! */
    return 0;
  }

  e = srcinst_malloc(sizeof(struct _package_element));
  info = srcinst_malloc(sizeof(struct _package_info));
  _init_package_info(info, name);
  info->onclose = SRCINST_ONCLOSE_ADD;
  _init_package_element(e, info);
  return e;
}

/* newname can be (char*)0 here too */

SRCINST_PHANDLE srcinst_upgrade_package(char* newname, char* oldname) {
  SRCINST_PHANDLE newh, oldh;
  struct _package_element* olde, *newe;

  if (!(oldh = srcinst_open_package(oldname)))
    return 0;
  
  olde = oldh;

  if (!olde->info->installed) {
    return 0;
  }

  _detach_package_list(&_srcinst_state.packages, olde);

  if (!(newh = srcinst_new_package(newname)))
    goto err_proc;

  newe = newh;
  newe->info->upgrading_from = olde;
  
  newe->info->long_description = olde->info->long_description ?
    strdup(olde->info->long_description) : 0;

  newe->info->description = olde->info->description ?
    strdup(olde->info->description) : 0;

  return newh;

 err_proc:
  assert(_add_package_element(&_srcinst_state.packages, olde));
  return 0;
}

int srcinst_close_package(SRCINST_PHANDLE h) {
  struct _package_element* e;
  struct _package_info* info;
  int rv = 1;
  e = h; info = e->info;

  if (info->build_ready) {
    SRCINST_ERR result;
    if ((result = _action_fini_build(info)) != SRCINST_ERR_OK) {
      srcinst_warning(result, "could not finalize build on package close", 0); 
    }
  } else {
    if (!_clear_builddirs()) {
      srcinst_warning(SRCINST_ERR_ACTION, "could not clear build dirs "
		      "on package close", 0);
    }
  }

  if (info->onclose & SRCINST_ONCLOSE_ADD) {

    if ((!info->upgrading_from &&
	 (info->installed || info->source_location)) ||
	(info->upgrading_from && info->installed)) {

      /* add package successful */
      assert(_add_package_element(&_srcinst_state.packages, e));
      info->loaded = 1;
      /* the action that lead to ->installed or ->source_location must
	 have set ONCLOSE_UPDATE */
      assert(info->onclose & SRCINST_ONCLOSE_UPDATE);
      
      if (info->upgrading_from) {
	_action_upgrade(info->upgrading_from->info, info);
	_unlink_package_info(info->upgrading_from->info);

	_free_package_element(info->upgrading_from);
	free(info->upgrading_from);
	info->upgrading_from = 0;
      }
      
    } else {
      /* failed to add new package */
      if (info->upgrading_from) {
	assert(_add_package_element(&_srcinst_state.packages,
				    e->info->upgrading_from));
      }
      _free_package_element(e); free(e);
      return 1;
    }
  }

  if (info->onclose & SRCINST_ONCLOSE_UPDATE) {
    rv = _update_package(&_srcinst_state.packages, e);
  }

  info->onclose = SRCINST_ONCLOSE_NOP;
  return rv;
}

SRCINST_ERR srcinst_init_build(SRCINST_PHANDLE h, char* filename, char* subdir) {
  SRCINST_ERR rv; struct _package_element* e;
  e = h;

  SRCINST_INTR_ACTION(_action_init_build(e->info, filename, subdir));
  return rv;
}

SRCINST_ERR srcinst_chdir_build(SRCINST_PHANDLE h, char* subdir) {
  SRCINST_ERR rv; struct _package_element* e;
  e = h;

  SRCINST_INTR_ACTION(_action_chdir_build(e->info, subdir));
  return rv;
}

SRCINST_ERR srcinst_fini_build(SRCINST_PHANDLE h) {
  SRCINST_ERR rv; struct _package_element* e;
  e = h;

  SRCINST_INTR_ACTION(_action_fini_build(e->info));
  return rv;
}

SRCINST_ERR srcinst_configure(SRCINST_PHANDLE h, char* prefix, struct srcinst_string_list* argv) {
  SRCINST_ERR rv; struct _package_element* e;
  e = h;

  SRCINST_INTR_ACTION(_action_configure(e->info, prefix, argv));

  return rv;
}

SRCINST_ERR srcinst_find_configure(SRCINST_PHANDLE h, struct srcinst_string_list* scripts) {
  SRCINST_ERR rv; struct _package_element* e;
  e = h;
  SRCINST_INTR_ACTION(_action_find_configure(e->info, scripts));
  return rv;
}

SRCINST_ERR srcinst_detect_autoconf(SRCINST_PHANDLE h, float* version) {
  SRCINST_ERR rv; struct _package_element* e;
  e = h;
  SRCINST_INTR_ACTION(_action_detect_autoconf(e->info, version));
  return rv;
}

SRCINST_ERR srcinst_detect_configure_opt(SRCINST_PHANDLE h, struct srcinst_configure_sets* sets) {
  SRCINST_ERR rv; struct _package_element* e;
  e = h;
  
  SRCINST_INTR_ACTION(_action_detect_configure_opt(e->info, sets));
  return rv;
}

void srcinst_free_configure_opt(struct srcinst_configure_sets* sets) {
  _action_free_configure_opt(sets);
}

SRCINST_MFEATS srcinst_detect_makefile_features(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  
  return (SRCINST_MFEATS)_action_detect_makefile_features(e->info);
}

int srcinst_is_feature_supported(SRCINST_MFEATS mf, SRCINST_FEATURE idx) {
  struct _makefile_features* m;
  m = mf;
  return m->features[idx].supported;
}

char* srcinst_get_makefile(SRCINST_MFEATS mf) {
  struct _makefile_features* m;
  m = mf;
  return m->makefile;
}

void srcinst_free_makefile_features(SRCINST_MFEATS mf) {
  struct _makefile_features* m;
  m = mf;
  _free_makefile_features(m);
}

SRCINST_ERR srcinst_build(SRCINST_PHANDLE h, SRCINST_MFEATS mf) {
  SRCINST_ERR rv;
  struct _package_element* e;
  struct _makefile_features* m;

  e = h; m = mf;
  SRCINST_INTR_ACTION(_action_build(e->info, m));
  return rv;
}

SRCINST_ERR srcinst_test_install(SRCINST_PHANDLE h, SRCINST_MFEATS mf, int strip) {
  SRCINST_ERR rv;
  struct _package_element* e;
  struct _makefile_features* m;

  e = h; m = mf;
  SRCINST_INTR_ACTION(_action_test_install(e->info, m, strip));
  return rv;
}

int srcinst_get_conflict_list(SRCINST_PHANDLE h, struct srcinst_string_list* list) {
  struct _package_element* e;

  e = h;
  return _action_get_conflict_list(e->info, list);
}
  
SRCINST_ERR srcinst_install_binary(SRCINST_PHANDLE h, SRCINST_MFEATS mf, int strip) {
  SRCINST_ERR rv;
  struct _package_element* e;
  struct _makefile_features* m;
  e = h; m = mf;

  SRCINST_INTR_ACTION(_action_install_binary(e->info, m, strip));
  return rv;
}

/* parameter MF is unused, but it's helpful because it makes it clear
   that only configured source, with Makefiles ready, must be installed */

SRCINST_ERR srcinst_install_source(SRCINST_PHANDLE h, SRCINST_MFEATS mf, SRCINST_COMP format) {
  SRCINST_ERR rv;
  struct _package_element* e;
  /* struct _makefile_features* m; */
  e = h; /* m = mf; */

  SRCINST_INTR_ACTION(_action_install_source(e->info, format));
  return rv;
}

SRCINST_ERR srcinst_uninstall_binary(SRCINST_PHANDLE h) {
  SRCINST_ERR rv; struct _package_element* e;
  e = h;

  SRCINST_INTR_ACTION(_action_uninstall_binary(e->info));
  return rv;
}

SRCINST_ERR srcinst_uninstall_source(SRCINST_PHANDLE h) {
  SRCINST_ERR rv; struct _package_element* e;
  e = h;

  SRCINST_INTR_ACTION(_action_uninstall_source(e->info));
  return rv;
}

SRCINST_ERR srcinst_export_info(SRCINST_PHANDLE h, char* filename, SRCINST_EXPORT fmt) {
  SRCINST_ERR rv; struct _package_element* e;
  e = h;

  SRCINST_INTR_ACTION(_action_export_info(e ? e->info :
					  (struct _package_info*)0, filename, fmt));
  return rv;
}

SRCINST_ERR srcinst_export_binary(SRCINST_PHANDLE h, char* filename, SRCINST_COMP fmt) {
  SRCINST_ERR rv; struct _package_element* e;
  e = h;

  SRCINST_INTR_ACTION(_action_export_binary(e->info, filename, fmt));
  return rv;
}

SRCINST_ERR srcinst_rename_package(SRCINST_PHANDLE h, char* newname) {
  SRCINST_ERR rv; struct _package_element* e;
  e = h;

  SRCINST_INTR_ACTION(_action_rename_package(e, newname));
  return rv;
}

/* change the log setting */

void srcinst_log_stdout(int do_log) {
  _srcinst_state.log_stdout = do_log ? 1 : 0;
}

void srcinst_log_stderr(int do_log) {
  _srcinst_state.log_stderr = do_log ? 1 : 0;
}

void srcinst_log_spawn(int do_log) {
  _srcinst_state.log_spawn = do_log ? 1 : 0;
}

/* pointers always return internal strings or data; do not change or free. */
char* srcinst_get_pref_prefix(void) {
  return _srcinst_state.preferences.prefix;
}

char srcinst_get_pref_keep_source(void) {
  return _srcinst_state.preferences.keep_source;
}

char srcinst_get_pref_strip(void) {
  return _srcinst_state.preferences.strip;
}

char srcinst_get_pref_quiet(void) {
  return _srcinst_state.preferences.quiet;
}

char srcinst_get_pref_install(void) {
  return _srcinst_state.preferences.install;
}

SRCINST_COMP srcinst_get_pref_src_compress(void) {
  return _srcinst_state.preferences.src_compress;
}

/* the following have only sense with GUIs */

char* srcinst_get_pref_add_directory(void) {
  return _srcinst_state.preferences.add_directory;
}

char srcinst_get_pref_manual_conf(void) {
  return _srcinst_state.preferences.manual_configure;
}

/* now the 'set' functions. Call srcinst_save_pref() afterwards */

void srcinst_set_pref_prefix(char* prefix) {
  struct _preferences* p;
  p = &_srcinst_state.preferences;

  if (p->prefix) {
    free(p->prefix);
  }
  p->prefix = prefix ? srcinst_strdup(prefix) : 0;
}

void srcinst_set_pref_keep_source(int keep_source) {
  struct _preferences* p;
  p = &_srcinst_state.preferences;

  p->keep_source = keep_source;
}

void srcinst_set_pref_strip(int strip) {
  struct _preferences* p;
  p = &_srcinst_state.preferences;

  p->strip = strip;
}

void srcinst_set_pref_quiet(int quiet) {
  struct _preferences *p;
  p = &_srcinst_state.preferences;
  
  p->quiet = quiet;
}

void srcinst_set_pref_install(int install) {
  struct _preferences* p;
  p = &_srcinst_state.preferences;

  p->install = install;
}

void srcinst_set_pref_src_compress(SRCINST_COMP format) {
  struct _preferences* p;
  p = &_srcinst_state.preferences;

  p->src_compress = format;
}

/* the following have only sense with GUIs */

void srcinst_set_pref_add_directory(char* add_directory) {
  struct _preferences* p;
  p = &_srcinst_state.preferences;

  if (p->add_directory) {
    free(p->add_directory);
  }
  p->add_directory = add_directory ? srcinst_strdup(add_directory) : 0;
}

void srcinst_set_pref_manual_conf(int manual_conf) {
  struct _preferences* p;
  p = &_srcinst_state.preferences;

  p->manual_configure = manual_conf;
}

/* Save the preferences! */

int srcinst_save_pref(void) {
  return _save_preferences(&_srcinst_state.preferences);
}

/* Access package information */

char* srcinst_get_name(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return e->info->name;
}

char* srcinst_get_descr(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return e->info->description;
}

char* srcinst_get_descr_long(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return e->info->long_description;
}

int srcinst_set_descr(SRCINST_PHANDLE h, char* descr) {
  struct _package_element* e;
  e = h;
  if (e->info->description) {
    free(e->info->description);
  }
  e->info->description = descr ? srcinst_strdup(descr) : 0;
  e->info->onclose |= SRCINST_ONCLOSE_UPDATE;
  return 1;
}

int srcinst_set_descr_long(SRCINST_PHANDLE h, char* descr_long) {
  struct _package_element* e;
  e = h;
  if (e->info->long_description) {
    free(e->info->long_description);
  }
  e->info->long_description = descr_long ? srcinst_strdup(descr_long) : 0;
  e->info->onclose |= SRCINST_ONCLOSE_UPDATE;
  return 1;
}

char* srcinst_get_configure_string(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return e->info->configured;
}

char* srcinst_get_prefix(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return e->info->prefix;
}

int srcinst_is_source_available(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return e->info->source_location != (char*)0;
}

off_t srcinst_get_source_size(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return e->info->source_size;
}

char* srcinst_get_source_location(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return e->info->source_location;
}

char* srcinst_get_packdir(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return e->info->packdir;
}

char* srcinst_get_build_subdir(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return e->info->build_subdir;
}

int srcinst_is_installed(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return e->info->installed;
}

off_t srcinst_get_installed_size(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return e->info->installed_size;
}

struct srcinst_string_list* srcinst_get_installed_files(SRCINST_PHANDLE h) {
  struct _package_element* e;
  e = h;
  return &(e->info->installed_files);
}

