/* Source Installer, Copyright (c) 2005 Claudio Fontana

 srcinst.h - main interface to source installer library

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org

*/

#ifndef SRCINST_H
#define SRCINST_H

/* this is the only file that needs to be included in your programs */
/* requires previous inclusion of <stdlib.h> and <stdio.h> */
/* NB for GNU/Linux, SUN Solaris and similar:
     watch for LARGEFILE issues. The large file setting in your programs
     must match the setting you built the library with.
     In a nutshell, use AC_SYS_LARGEFILE (or the system-specific compiler
     macros) in your programs if you built libsrcinst with large file support
     enabled (default), or you _will experience random malfunctions!_
*/

/* flags for srcinst_init() */
#define SRCINST_FLAG_DEFAULT 0x0000
#define SRCINST_FLAG_LOAD_ONDEMAND 0x0001 /* load packages only when needed */

#include <srcinst/string_list.h>
#include <srcinst/srcalloc.h>
#include <srcinst/srcerror.h>
#include <srcinst/srcmisc.h>
#include <srcinst/srcregex.h>
#include <srcinst/srcspawn.h>
#include <srcinst/srcparser.h>
#include <srcinst/srccfgopt.h>

/* the first function to call is the following.
   It returns 1 if initialization was successful, 0 otherwise */

int srcinst_init(unsigned int);

/* this is the corresponding finalization function. REMEMBER TO CALL IT!
   A good thing to do is, just after init, atexit(srcinst_fini);
*/

void srcinst_fini(void);

/* You do not really need the following function. It just closes stdout and
   exits with errcode, or -1 if stdout close fails. Calls srcinst_fini if
   necessary. */
void srcinst_exit(int errcode);	


/* call the following function to raise a SIGINT signal in the process.
   Of course you can call raise(SIGINT) yourself.
   If waiting for a process, sends SIGINT to that process too.
 */
void srcinst_interrupt(void);


/* use callbacks to perform particular actions on spawned processes'
   output or errors. You can register multiple of these.
   Note that if you pass 0, ALL CALLBACKS ARE UNREGISTERED.
 */

typedef void (*srcinst_callback)(char*);

/* use these callbacks to capture process output and errors */
void srcinst_register_stdout_callback(srcinst_callback f);
void srcinst_register_stderr_callback(srcinst_callback f);

/* use this one to capture the command line of a process as it is spawned */
void srcinst_register_spawn_callback(srcinst_callback f);

/* use this one to be called every "fixed" time interval in microseconds.
   Useful to update your GUI, or show progress. */
void srcinst_register_beat_callback(srcinst_callback f);

/* unregister the callbacks using these. Pass the same pointer as before,
   and the function will be unregistered.
   Like before, if you pass 0, ALL CALLBACKS ARE UNREGISTERED
 */
void srcinst_unregister_stdout_callback(srcinst_callback f);
void srcinst_unregister_stderr_callback(srcinst_callback f);
void srcinst_unregister_spawn_callback(srcinst_callback f);
void srcinst_unregister_beat_callback(srcinst_callback f);

/* get/set the delay in microseconds or the beat.
   Default value is 100000 microseconds, equal to 0.1 seconds.
   See srcinst_register_beat_callback above. */

long srcinst_get_beat_delay(void);
void srcinst_set_beat_delay(long delay);

/* the following function is automatically called on initialization. It 
   checks that the needed directories in the user HOME exist, and creates them
   if needed. You might need to call this manually at some key points, but
   generally you won't need it. */
int srcinst_check_dirs(void);

/* these are the codes of the internal sourceinstall directories.
   SRCINST_DIR_PREFIX and SRCINST_DIR_ICON are reserved for future use.
   SRCINST_DIR_RUN is set to the process current directory during library init.
   SRCINST_DIR_RUNBUILD is set to the current directory during build init.
     Afer build fini, current directory is set back to this value.
   SRCINST_DIR_CFG is ~/.sourceinstall
   
   All others are subdirectories of SRCINST_DIR_CFG:
   BUILD (build) where the builds occur
   DEST (install-destdir) where the test installation is targeted
   PACK (packages) where the information about the packages is kept. */

typedef enum { 
  SRCINST_DIR_RUN, SRCINST_DIR_PREFIX, SRCINST_DIR_CFG,
  SRCINST_DIR_SRC, SRCINST_DIR_BUILD, SRCINST_DIR_DEST,
  SRCINST_DIR_PACK,
  SRCINST_DIR_ICON,
  SRCINST_DIR_RUNBUILD,
  SRCINST_DIR_N			/* _do not use_, counter only */

} SRCINST_DIR;

/* These are internal files.
   SRCINST_FILE_CFG is $SRCINST_DIR_CFG/sourceinstallrc - main config file
   SRCINST_FILE_LOCK is $SRCINST_DIR_CFG/.sourceinstall_lock - user lock file
*/   

typedef enum {
  SRCINST_FILE_CFG, SRCINST_FILE_LOCK,
  SRCINST_FILE_N		/* _do not use_, counter only */

} SRCINST_FILE;

char* srcinst_get_dir(int idx);	  /* get an internal directory (see codes^)*/
char* srcinst_get_file(int idx);  /* get an internal filename */

void srcinst_log_stdout(int do_log); /* en/dis-able log of processes' output */
void srcinst_log_stderr(int do_log); /* en/dis-able log of processes' errors */
void srcinst_log_spawn(int do_log); /* en/dis-able log of spawned command line */

/* fetch a new string list of known packages. */
void srcinst_list_packages(struct srcinst_string_list*);

/* opaque handle to a package. You will need one to perform 
   operations on a package. */
typedef void* SRCINST_PHANDLE;

/* obtain a handle to a package by opening a package by NAME */
SRCINST_PHANDLE srcinst_open_package(char* name);

/* create a new package. NAME can be (char*)0, to mean "use default name".
   This is the way new packages are added.
   If a package with the same NAME exists, fails and returns 0.
 */
SRCINST_PHANDLE srcinst_new_package(char* name);

/* release a package after messing with it. DO IT ALWAYS or your changes will not
   be guaranteed to be flushed. Watch for a 0 (error) return value. */
int srcinst_close_package(SRCINST_PHANDLE);

/* before building anything, after opening or creating the package,
   you can manually call this function to prepare the build.

   If the package source code is already installed, it will be automatically
   called if needed, but specifying it manually is better, because you
   can manually specify when the build starts (and thus look at the package
   contents yourself), provide a build SUBDIR to chdir to before building.
   SOURCE is the source code to initialize with. It can be of course
   srcinst_get_source_location, but also the SOURCE you want to build.
   pass a SUBDIR of (char*)0 to mean USE CURRENT.
 */

SRCINST_ERR srcinst_init_build(SRCINST_PHANDLE, char* source, char* subdir);

/* change working subdir of a build, without needing to fini it and init it
   again. Useful to initialize, detect build subdirectories, and then change
   builddir accordingly. Pass 0 as SUBDIR to mean NO SUBDIR (top sourcedir) */
SRCINST_ERR srcinst_chdir_build(SRCINST_PHANDLE, char* subdir);

/* you do not need to finalize the build (it is performed at package close).
   However, you might want to decide to finalize the build at a specific
   point in your procedure, so here is the function anyway. */
SRCINST_ERR srcinst_fini_build(SRCINST_PHANDLE);

/* between calls to srcinst_init_build and srcinst_fini_build it is
   strongly recommended to AVOID ANY CHANGES TO PROCESS CURRENT DIRECTORY */



/* Now the actions you can run on packages. */

/* Note in the following action that the PREFIX is specified separately,
   without '--prefix='. ARGV contains a list of the desired other configure opts.
   ARGV can contain 0 strings, but PREFIX _must_ be a real string.
   Note that srcinst_configure alone has no sense, and it is recommended to run
   a srcinst_install action before closing the package.

 */

/* find configure script, or configure-generating scripts, recursing directories.
   SCRIPTS is initialized and filled with found scripts on SRCINST_ERR_OK.
   If SRCINST_ERR_ACTION, it was impossible to parse the package directories.
 */

SRCINST_ERR srcinst_find_configure(SRCINST_PHANDLE h, struct srcinst_string_list* scripts);

/* here's the real configuration action */

SRCINST_ERR srcinst_configure(SRCINST_PHANDLE h, char* prefix, struct srcinst_string_list* argv); 

/* analyze configure script in current directory (or autogen one)
   and detect autoconf VERSION.
   If not an autoconf-generated script, version is set to 0.0
   On error, SRCINST_ERR is != SRCINST_ERR_OK of course */
SRCINST_ERR srcinst_detect_autoconf(SRCINST_PHANDLE h, float* version);

/* detect supported configure options. Declare a struct like this:
 
   SRCINST_ERR result;  
   struct srcinst_configure_sets option_sets;

   result = srcinst_detect_configure_opt(h, &option_sets);
   ...
   srcinst_free_configure_opt(&option_sets);
*/

SRCINST_ERR srcinst_detect_configure_opt(SRCINST_PHANDLE h, struct srcinst_configure_sets* sets);

void srcinst_free_configure_opt(struct srcinst_configure_sets* sets);

typedef void* SRCINST_MFEATS;

typedef enum {
  SRCINST_FEATURE_DESTDIR, 
  SRCINST_FEATURE_INSTALL_ROOT,
  SRCINST_FEATURE_UNINSTALL,
  SRCINST_FEATURE_STRIP,
  SRCINST_FEATURE_DESCR,
  SRCINST_FEATURE_LONG_DESCR,
  SRCINST_FEATURE_N

} SRCINST_FEATURE;

/* want to know about supported makefile features? Call this after
   configuration, and you won't be sorry!
   The function returns an opaque type SRCINST_MFEATS.
   Example:

    SRCINST_MFEATS my_feats;
    int destdir;

    my_feats = srcinst_detect_makefile_features(h)
    if (!my_feats) { 
       ...error...
    }

    destdir = srcinst_is_feature_supported(my_feats, SRCINST_FEATURE_DESTDIR);
    srcinst_free_makefile_features(h);
   
*/

SRCINST_MFEATS srcinst_detect_makefile_features(SRCINST_PHANDLE h);

/* check whether the feature is supported */
int srcinst_is_feature_supported(SRCINST_MFEATS mf, SRCINST_FEATURE f);

/* remember to call this following function after you're done! */
void srcinst_free_makefile_features(SRCINST_MFEATS mf);

/* use this one to perform proper installation. STRIP tries to use the
   makefile install-strip target if supported, otherwise it's ignored.

   KEEP_BINARY: whether binary installation should be performed.
   KEEP_SOURCE: whether source should be stored. If set, store in FORMAT
   compression format (use an arbitrary value, SRCINST_COMP_UNKNOWN or 0,
   what you want, if KEEP_SOURCE is 0).
   If FORCE is set, the function will not fail even if DESTDIR nor INSTALL_ROOT
   are supported. However, info about installed files will be missing in this
   case.

   It is recommended to call srcinst_configure before installation.
 */

SRCINST_ERR srcinst_install(SRCINST_PHANDLE h, int keep_binary, int keep_source, int strip, int force, SRCINST_COMP format);

/* use this to uninstall a package. srcinst will first try make uninstall
   if the source is available and the uninstall target is supported.
   Then it will use internal information to remove eventual left-overs. */

SRCINST_ERR srcinst_uninstall_binary(SRCINST_PHANDLE h);

/* use the following function to remove the stored source code */

SRCINST_ERR srcinst_uninstall_source(SRCINST_PHANDLE h);

/* call the following function to export information about the package.
   h can be the special value (SRCINST_PHANDLE)0 to mean 'export them all'.
   Otherwise, it should be the handle of an opened package.
   Note that (SRCINST_PHANDLE)0 is not valid with LST format 
   (installed-files list only).
   Also, LST format requires the package to be installed, and its files known.
   For the xml format see the included package_set.dtd
   It should be trivial to figure out though.
 */
SRCINST_ERR srcinst_export_info(SRCINST_PHANDLE h, char* filename, SRCINST_EXPORT format);

/* following function exports a binary package with the installed files.
   The binary package is a tar archive, compressed in FORMAT, with all the
   files with the original attributes. Very alpha stuff, not tested,
   use at own risk. Will incidentally be handled correctly by the Slackware
   tools. */

SRCINST_ERR srcinst_export_binary(SRCINST_PHANDLE h, char* filename, SRCINST_COMP format);

/* rename currently open package to the new name: newname . Use a valid
   filename string for newname. Avoid spaces and fancy chars. */
SRCINST_ERR srcinst_rename_package(SRCINST_PHANDLE h, char* newname);

/* Access preferences */
/* always return internal strings or data; do not change pointed data or free. */

char* srcinst_get_pref_prefix(void); /* preferred prefix for installation */
char srcinst_get_pref_keep_source(void); /* keep source code after install ? */
char srcinst_get_pref_strip(void);	/* strip binaries ? */
char srcinst_get_pref_install(void); /* 'add a package' also means install ? */
SRCINST_COMP srcinst_get_pref_src_compress(void); /* preferred compression fmt */

/* for GUIs only */
char* srcinst_get_pref_add_directory(void); /* default add directory */
char srcinst_get_pref_manual_conf(void); /* configure manually ? */

/* change preferences */

void srcinst_set_pref_prefix(char* new_prefix);
void srcinst_set_pref_strip(int strip);
void srcinst_set_pref_install(int install);
void srcinst_set_pref_keep_source(int keep_source);
void srcinst_set_pref_src_compress(SRCINST_COMP format);
void srcinst_set_pref_add_directory(char* new_adddir);
void srcinst_set_pref_manual_conf(int manual_conf);

/* save current preferences */
int srcinst_save_pref(void);

/* Access package information */
/* always return internal strings or data; do not change pointed data or free. */

char* srcinst_get_name(SRCINST_PHANDLE h); /* get package name */

char* srcinst_get_descr(SRCINST_PHANDLE h); /* get package description */
/* set description: returns 1 on success, 0 on failure. */
int srcinst_set_descr(SRCINST_PHANDLE h, char* descr);

char* srcinst_get_descr_long(SRCINST_PHANDLE h); /* get long description */
/* set long description: returns 1 on success, 0 on failure. */
int srcinst_set_descr_long(SRCINST_PHANDLE h, char* descr_long);

/* get current 'configured as' string */
char* srcinst_get_configure_string(SRCINST_PHANDLE h);

/* get current prefix */
char* srcinst_get_prefix(SRCINST_PHANDLE h);

int srcinst_is_source_available(SRCINST_PHANDLE h); /* is source available? */
off_t srcinst_get_source_size(SRCINST_PHANDLE h); /*  */

char* srcinst_get_source_location(SRCINST_PHANDLE h);

/* get basename of initial package build directory */
char* srcinst_get_packdir(SRCINST_PHANDLE h);

/* get build subdirectory */
char* srcinst_get_build_subdir(SRCINST_PHANDLE h);

int srcinst_is_installed(SRCINST_PHANDLE h);
off_t srcinst_get_installed_size(SRCINST_PHANDLE h);

/* returns a pointer to an internal string list; do not change or free. */
struct srcinst_string_list* srcinst_get_installed_files(SRCINST_PHANDLE h);

#endif /* SRCINST_H */
