"======================================================================
|
|   UnixStream (raw file access)
|
|   $Revision: 1.6.2$
|   $Date: 1999/08/31 11:23:18$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1990, 91, 92, 94, 95, 99 Free Software Foundation, Inc.
| Written by Steve Byrne.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"

PositionableStream subclass: #UnixStream
		   instanceVariableNames: 'fd'
		   classVariableNames: ''
		   poolDictionaries: ''
		   category: 'Sockets-Protocols'
!
    
UnixStream comment:
'I provide an interface to Unix file descriptors, so that Smalltalk methods
may perform most of the I/O operations that C programs can.  I am quite proud
of my contribution to the GNU Smalltalk project, and look forward to serving
the project better in the future.'
!


UnixStream
	defineCFunc: 'open'
	withSelectorArgs: 'open: aFileName flags: flagsInteger 
			   mode: anInteger'
	returning: #int
	args: #(string int int)
!

UnixStream
	defineCFunc: 'close'
	withSelectorArgs: 'close: fileDescriptor'
	returning: #int
	args: #(int)
!

UnixStream
	defineCFunc: 'read'
	withSelectorArgs: 'read: fileDescriptor into: buf bytes: anInteger'
	returning: #int
	args: #(int byteArray int)
!

UnixStream
	defineCFunc: 'read'
	withSelectorArgs: 'read: fileDescriptor into: buf chars: anInteger'
	returning: #int
	args: #(int stringOut int)
!

UnixStream
	defineCFunc: 'write'
	withSelectorArgs: 'write: fileDescriptor from: buf bytes: anInteger'
	returning: #int
	args: #(int byteArray int)
!

UnixStream
	defineCFunc: 'lseek'
	withSelectorArgs: 'lseek: fileDescriptor offset: anInteger 
			   whence: cObject'
	returning: #int
	args: #(int int int)
!

"======================================================================
|
|   Type specific I/O routines
|
 ======================================================================"

UnixStream
	defineCFunc: 'readChar'
	withSelectorArgs: 'readChar: fileDescriptor'
	returning: #smalltalk
	args: #(int)
!

UnixStream
	defineCFunc: 'readUChar'
	withSelectorArgs: 'readUChar: fileDescriptor'
	returning: #smalltalk
	args: #(int)
!

UnixStream
	defineCFunc: 'readShort'
	withSelectorArgs: 'readShort: fileDescriptor'
	returning: #smalltalk
	args: #(int)
!

UnixStream
	defineCFunc: 'readUShort'
	withSelectorArgs: 'readUShort: fileDescriptor'
	returning: #smalltalk
	args: #(int)
!

UnixStream
	defineCFunc: 'readLong'
	withSelectorArgs: 'readLong: fileDescriptor'
	returning: #smalltalk
	args: #(int)
!

UnixStream
	defineCFunc: 'readULong'
	withSelectorArgs: 'readULong: fileDescriptor'
	returning: #smalltalk
	args: #(int)
!

UnixStream
	defineCFunc: 'readFloat'
	withSelectorArgs: 'readFloat: fileDescriptor'
	returning: #smalltalk
	args: #(int)
!

UnixStream
	defineCFunc: 'readDouble'
	withSelectorArgs: 'readDouble: fileDescriptor'
	returning: #smalltalk
	args: #(int)
!


UnixStream
	defineCFunc: 'writeChar'
	withSelectorArgs: 'write: fileDescriptor char: aChar'
	returning: #void
	args: #(int char)
!

UnixStream
	defineCFunc: 'writeShort'
	withSelectorArgs: 'write: fileDescriptor short: aShort'
	returning: #void
	args: #(int int)
!

UnixStream
	defineCFunc: 'writeLong'
	withSelectorArgs: 'write: fileDescriptor long: aLong'
	returning: #void
	args: #(int long)
!

UnixStream
	defineCFunc: 'writeDouble'
	withSelectorArgs: 'write: fileDescriptor double: aDouble'
	returning: #void
	args: #(int double)
!





!UnixStream class methodsFor: 'instance creation'!

open: fileName dir: anInteger
    ^self open: fileName dir: anInteger mode: 0
!

open: fileName dir: anInteger mode: intMode
    ^self new init: fileName dir: anInteger mode: intMode
!

on: fd
    ^self new initFd: fd
!!



!UnixStream class methodsFor: 'constants'!

seekSet
    ^0
!

seekCurr
    ^1
!

seekEnd
    ^2
!

readOnly
    ^0
!

writeOnly
    ^1
!

readWrite
    ^2
!

"Other modifiers (like O_APPEND) may be added in the future as needs warrant"
!



!UnixStream methodsFor: 'basic accessing'!

close
    ^self close: fd
!

read: byteArray
    | val |
    ^self read: fd into: byteArray bytes: byteArray size
!

read: byteArray numBytes: anInteger
    | val |
    ^self read: fd into: byteArray bytes: anInteger
!

read: string numChars: anInteger
    ^self read: fd into: string chars: anInteger
!

write: byteArray
    ^self write: byteArray numBytes: byteArray size
!

write: byteArray numBytes: anInteger
    ^self write: fd from: byteArray bytes: anInteger
!

tell
    ^self lseek: fd offset: 0 whence: 1   "Curr"
!

position
    ^1 + (self lseek: fd offset: 0 whence: 1)   "Curr"
!

position: anInteger
    ^self lseek: fd offset: anInteger - 1 whence: 0 "Set"
!

ioctl: number arg: randomArg
    ^self ioctl: fd request: number arg: randomArg
!

readChar
    ^self readChar: fd
!

readUChar
    ^self readUChar: fd
!

readShort
    ^self readShort: fd
!

readUShort
    ^self readUShort: fd
!

readLong
    ^self readLong: fd
!

readULong
    ^self readULong: fd
!

readFloat
    ^self readFloat: fd
!

readDouble
    ^self readDouble: fd
!

writeChar: aChar
    ^self write: fd char: aChar
!

writeShort: aShort		"really an integer"
    ^self write: fd short: aShort
!

writeLong: aLong
    ^self write: fd long: aLong
!

writeDouble: aDouble
    ^self write: fd double: aDouble
! !



!UnixStream methodsFor: 'accessing'!

readString: numChars
    | string numRead str |
    string := String new: numChars.
    (numRead := self read: string) <= 0 "failed for some reason"
	ifTrue: [ File checkError. ^nil ].

    numRead = numChars ifTrue: [ ^string ].
    ^(String new: numRead)
        replaceFrom: 1 to: numRead with: string startingAt: 1;
        yourself
!

readBytes: numBytes
    | byteArray numRead str |
    byteArray := ByteArray new: numBytes.
    (numRead := self read: byteArray) <= 0 "failed for some reason"
	ifTrue: [ File checkError. ^nil ].

    numRead = numBytes ifTrue: [ ^byteArray ].
    ^(ByteArray new: numRead)
        replaceFrom: 1 to: numRead with: byteArray startingAt: 1;
        yourself
!

next
    ^self readChar
!

next: anInteger
    ^self readString: anInteger
!


nextPut: aValue
    self writeChar: aValue
!

contents
    | curPos result size |
    curPos := self tell.
    size := self lseek: fd offset: 0 whence: 2. "To end"
    self lseek: fd offset: 0 whence: 0.
    result := self readString: size.
    self lseek: fd offset: curPos whence: 0.
    ^curPos
!

atEnd
    "Poor man's atEnd function"
    | curPos size |
    curPos := self tell.
    size := self lseek: fd offset: 0 whence: 2. "To end"
    self lseek: fd offset: curPos whence: 0.
    ^curPos >= size
!

size
    "Poor man's size function"
    | curPos size |
    curPos := self tell.
    size := self lseek: fd offset: 0 whence: 2. "To end"
    self lseek: fd offset: curPos whence: 0.
    ^size
!

skip: anInteger
    "Skip n bytes on the file.  N can be positive or negative"
    ^self lseek: fd offset: anInteger whence: 1 "Cur"
!!



!UnixStream methodsFor: 'private'!

fd
    ^fd
!

init: fileName dir: anInteger mode: intMode
    fd := self open: fileName flags: anInteger mode: intMode.
    access := anInteger + 1.
    fd < 0 ifTrue: [ ^nil ]
!

initFd: anFd
    fd := anFd.
    access := 3.
!!

