"======================================================================
|
|   Smalltalk parser definitions.
|
|   $Revision: 1.6.2$
|   $Date: 1999/08/31 11:23:18$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1990, 91, 92, 94, 95, 99 Free Software Foundation, Inc.
| Written by Steve Byrne.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"


Object subclass: #STParser
       instanceVariableNames: 'parseErrorBlock lex'
       classVariableNames: ''
       poolDictionaries: 'VMOtherConstants'
       category: 'System-Compiler'
!

STParser comment:
'My full name is Smalltalk ''Recursive-Descent'' Parser. If you want to
parse some Smalltalk code, ask me.'!

!STParser class methodsFor: 'instance creation'!

on: aFileName
    ^self new init: (STTok on: aFileName)
!

onStream: aStream
    ^self new init: (STTok onStream: aStream)
! !


!STParser methodsFor: 'private'!

init: aStream
    lex := aStream.
    parseErrorBlock := [ :file :line :msg |
        ^self error: ('%4%1:%2: %3'
            bindWith: file
            with: line printString
            with: msg
            with: Character nl asString)
    ].
! !

!STParser methodsFor: 'tidyness'!

close
    lex close
!!

!STParser methodsFor: 'parsing'!

parseErrorBlock: aBlock
    parseErrorBlock := aBlock
!

parseSmalltalk
    [ self searchMethodListHeader ]
        whileTrue: [ self parseMethodDefinitionList ].
!

" Need some cool exception mechanism, like a return from a block or something"

parseMethodDefinitionList
    "Called after first !, expecting a set of bang terminated
     method definitions, followed by a bang"

    | startPos endPos node selector body source |

    [   lex atEnd or: [ lex peek isSTBangTok ] ] whileFalse: [

	startPos := lex position.
	selector := self parseSelector.
	body := self parseMethodBody.
	endPos := lex position.
	source := lex isFileStream
            ifFalse: [ lex stream copyFrom: startPos to: endPos ]
            ifTrue: [ FileSegment
        	on: lex stream name
        	startingAt: startPos
        	for: endPos - startPos + 1 ].

	node := STMethodDefinitionNode
            new: selector
	    body: body
	    source: source.

	self record: node source.
	self compile: node
    ].
    lex next.
    self endMethodList
!

searchMethodListHeader
    " Parses the stuff to be executed until a
        ! <class expression> methodsFor: <category string> ! "

    | startPos endPos node selector body source |

    selector := STSelectorNode selector: #Doit args: #().

    [   lex atEnd ifTrue: [ ^false ].

        startPos := lex position.
        body := self parseMethodBody.
        endPos := lex position.
        source := lex isFileStream
            ifFalse: [ lex stream copyFrom: startPos to: endPos ]
            ifTrue: [ FileSegment
                on: lex stream name
                startingAt: startPos
                for: endPos - startPos + 1 ].

	node := STMethodDefinitionNode
            new: selector
            body: body
            source: source.

	self record: source.
	self evaluate: node
    ]   whileFalse: [ ].
    ^true
!

parseMethodDefinition
    " parse a method definition, from selector, to terminating bang "
!
    
parseSelector
    | t sel | 
    t := lex peek.
    t isSTIdentifierTok ifTrue: 
	[ ^STSelectorNode selector: lex next value args: #()].
    
    t isSTKeywordTok ifTrue: [ ^self parseKeywordSelector ].
    (self isBinaryOperator: t) ifTrue: 
	[ sel := lex next.
	  t := lex next.
	  t isSTIdentifierTok ifFalse: 
	      [ ^self parseError: 'expected identifier to follow binary op' ].
	  ^STSelectorNode selector: sel value args: (Array with: t value) ].
    ^self parseError: 'invalid method selector'
!

parseKeywordSelector
    | t selector args | 
    selector := ''.
    args := OrderedCollection new: 0.
    [ t := lex peek.
      t isSTKeywordTok ] whileTrue: 
	  [
	   selector := selector, lex next value.
	   t := lex next.
	   t isSTIdentifierTok ifFalse:
	       [ ^self parseError: 'expected identifer after keyword' ].
	   args add: t value.
	   ].
    ^STSelectorNode selector: selector args: args
!

parseMethodBody
    | t temporaries primitiveIndex statements | 
    t := lex peek.
    t isSTVerticalBarTok
	ifTrue: [ temporaries := self parseTemporaries.
		  t := lex peek ]
	ifFalse: [ temporaries := #() ].
    t isSTPrimitiveStartTok
	ifTrue: [ lex next.	"gobble primitive start"
		  primitiveIndex := self parsePrimitive.
		  ].
    
    statements := self parseStatements.
    lex next.		"gobble method terminating bang"

    ^STMethodBodyNode temps: temporaries primIndex: primitiveIndex
		      stmts: statements
!


parseTemporaries
    "Parses 
         | << <name> <name> ... | >> 
     and returns the list of names"
    | t temps |
    temps := OrderedCollection new: 0.
    lex next.	"gobble vertical bar"
    [ t := lex peek.
      t isSTVerticalBarTok ] whileFalse: 
	  [ temps add: self parseVariableName ].
    lex next.		"gobble vertical bar"
    ^temps
!

parseVariableName
    | id |
    id := lex next.
    id isSTIdentifierTok
	ifFalse: [ ^self parseError: 'expected identifier' ].

    VMSpecialIdentifiers at: id value ifAbsent: [ ^id value ].
    ^self parseError: 'invalid variable name - ', id value
!

parsePrimitive
    | int t |
    int := lex next.
    (int isSTLiteralTok and: [ int value isMemberOf: Integer ])
	ifFalse: [ ^self parseError: 'primitive: must be followed by integer literal' ].
    
    t := lex next.

    (t isSTBinopTok and: [ t value = '>' ])
	ifFalse: [ ^self parseError: 'invalid terminator for primitive:, expecting ''>''' ].

    ^int value
!



parseIdentifierNode: aString

    VMSpecialIdentifiers at: aString ifPresent: [ :code |
        ^STSpecialIdentifierNode id: code
    ].
    ^STVariableNode id: aString
!

parsePrimary
    | t expr |
    t := lex peek.
    t isSTIdentifierTok ifTrue: [ ^self parseIdentifierNode: lex next value ].
    t isSTLiteralTok ifTrue: [ ^STConstNode value: lex next value].

    t isSTSharpTok ifTrue: [ lex next.
				       ^self parseSharpConstant ].

    t isSTOpenBracketTok ifTrue: 
	[ lex next.		"gobble"
	  ^self parseBlock ].

    t isSTOpenParenTok ifTrue: 
	[ lex next.		"gobble"
	  expr := self parseExpression.
	  expr isNil 
	      ifTrue: [ ^self parseError: 'Missing parenthesized expression' ].
	  t := lex next.
	  t isSTCloseParenTok
	      ifTrue: [ ^STExpressionNode expression: expr ]
	      ifFalse: [ ^self parseError: 'Expecting close paren' ] ].

    (t isSTCloseBracketTok or: [ t isSTBangTok ])
	ifTrue: [ ^nil ].
	 
    ^self parseError: 'Unrecognized expression'
!


parseSharpConstant
    " Called at
         # << id or ( lit lit lit ) or [ byte byte byte ] >> "
    | t lit |
    t := lex next.
    (self isSymbolId: t) ifTrue: [ ^STConstNode value: t value asSymbol ].
    t isSTOpenBracketTok ifTrue: [
        lit := self
            parseArrayLitTo: STCloseBracketTok
            on: (WriteStream on: (ByteArray new: 30))
            errorIf: #isGoodByteArrayElement:.
        ^STConstNode value: lit
    ].
    t isSTOpenParenTok ifTrue: [
        lit := self
            parseArrayLitTo: STCloseParenTok
            on: (WriteStream on: (Array new: 30))
            errorIf: #isGoodArrayElement:.
        ^STConstNode value: lit
    ].
    ^self parseError: 'Unrecognized literal'
!

parseArrayLitTo: closeTok on: aStream errorIf: errorSel
    "Called at
         ( << id or binop or number or char or string
              or ( array ) or [ bytes ] or empty ) >>"
    | t value |
    [ t := lex next.
      t isMemberOf: closeTok ] whileFalse: [
          value := self parseArrayElement: t.
          (self perform: errorSel with: value)
              ifFalse: [ ^self parseError: 'Bad array literal' ].
          aStream nextPut: value
    ].
    ^aStream contents
!

parseArrayElement: t

    t isSTLiteralTok ifTrue: [ ^t value ].
    (self isSymbolId: t) ifTrue: [ ^t value asSymbol ].

    t isSTOpenParenTok ifTrue: [
        ^self
            parseArrayLitTo: STCloseParenTok
            on: (WriteStream on: (Array new: 30))
            errorIf: #isGoodArrayElement:
    ].
    t isSTOpenBracketTok ifTrue: [
        ^self
            parseArrayLitTo: STCloseBracketTok
            on: (WriteStream on: (ByteArray new: 30))
            errorIf: #isGoodByteArrayElement:
    ].
    ^self parseError: 'Bad array literal'
!
    

parseBlock
    " Called at
     [ << block_identifiers ... | temporaries statements ] >> 
     "
    | t identifiers temporaries statements |
    t := lex peek.
    t isSTColonTok
	ifTrue: [ identifiers := self parseBlockIdentifiers.
		  t := lex peek ].
    t isSTVerticalBarTok
	ifTrue: [ temporaries := self parseTemporaries.
		  t := lex peek ].

    statements := self parseStatements.
    t := lex next.
    t isSTCloseBracketTok
	ifFalse: [ ^self parseError: 'bad block syntax' ].
    ^STBlockNode
        parameters: identifiers
        temporaries: temporaries
        statements: statements
!

parseBlockIdentifiers
    "Called at
      [ << :blockparam :blockparam ... | >> "
    | t identifiers |
    identifiers := OrderedCollection new: 0.
    [ t := lex next.
      t isSTVerticalBarTok ] whileFalse:
	  [ t isSTColonTok
		ifFalse: [ ^self parseError: 'Bad block param syntax' ].
	    t := lex next.
	    t isSTIdentifierTok
		ifFalse: [ ^self parseError: 'Bad block param syntax' ].
	    identifiers add: t value
	    ].
    ^identifiers
!
    
parseStatements
    | expression t list done |
    list := OrderedCollection new: 0.
    done := false.
    [ done ] whileFalse:
	[ t := lex peek.
	  t isSTUpArrowTok
	      ifTrue: [ lex next. "skip ^"
			list add: self parseReturnNode.
			done := true ]
	      ifFalse: [ expression := self parseExpression.
			 expression isNil
			     ifTrue: [ done := true ]
			     ifFalse: [ list add: expression.
					t := lex peek.
					t isSTDotTok
					    ifTrue: [ lex next "gobble it" ]
					    ifFalse: [ done := true ]
					    ]
			     ]
	      ].
    ^list
!

parseReturnNode
    | expression |
    expression := self parseExpression.
    (lex peek) isSTDotTok
	ifTrue: [ lex next ].
    ^STReturnNode expression: expression
!


parseExpression
    | expr assigns t |
    assigns := OrderedCollection new: 0.
    [ expr := self parseSimpleExpression.
      t := lex peek.
      t isSTAssignTok ] whileTrue:
	  [ lex next.
	    (expr isMemberOf: STVariableNode)
		ifFalse: [ ^self parseError: 'Invalid assignment variable' ].
	    assigns add: expr ].
    expr isNil
	ifTrue: [ ^nil ]
	ifFalse: [ ^STExpressionNode assign: assigns expression: expr ]
!

parseSimpleExpression
    ^self parseCascadedExpr
!

parseCascadedExpr
    | message expression |
    expression := self parsePrimary.
    "This feels like it should be recursive, but it's not currently"
    [ message := self parseKeywordMessage.
      message notNil ]
	whileTrue: [ expression := STMessageNode receiver: expression
					   message: message ].
    ^self parseCascadedMessage: expression.
!

parseCascadedMessage: expression
    | t message cascadeList | 
    cascadeList := OrderedCollection new: 0.
    
    [ t := lex peek.
      t isSTSemiTok ]
	whileTrue: 
	    [ lex next.	"gobble semicolon"
	      message := self parseKeywordMessage.
	      message isNil 
		  ifTrue: [ ^self parseError: 'Unfinished cascaded expression' ].
		  
	      cascadeList addLast: message.
	].

    cascadeList size == 0
	ifTrue: [ ^expression ]
	ifFalse: [ ^STCascadeNode expression: expression cascade: cascadeList ]
!
    

parseKeywordMessage
    | t selector exprs |
    t := lex peek.
    t isNil ifTrue: [ ^nil ].
    t isSTKeywordTok
	ifTrue: [ selector := WriteStream on: (String new: 15).
		  exprs := OrderedCollection new.
		  [ t := lex peek. 
		    t isSTKeywordTok ] whileTrue:
			[ selector nextPutAll: lex next value.
			  exprs add: self parseBinopExpr ].
		  ^STKeywordNode selector: selector contents
			       expressions: exprs ]
	ifFalse: [ ^self parseBinaryMessage ]
!    

parseBinopExpr
    | t receiver message |
    receiver := self parseUnaryExpr.
    ^self parseBinaryMessage: receiver
!

parseBinaryMessage: expression
    | message |
    message := self parseBinaryMessage.
    message notNil
	ifTrue: [ ^self parseBinaryMessage:
		      (STMessageNode receiver: expression
				   message: message) ]
	ifFalse: [ ^expression ]
!

parseBinaryMessage
    | t operand |
    t := lex peek.
    (self isBinaryOperator: t)
	ifTrue: [ lex next.
		  operand := self parseUnaryExpr.
		  ^STBinaryNode selector: t value
			      expression: operand ]
	ifFalse: [ ^self parseUnaryMessage ]
!


parseUnaryExpr
    | receiver |
    receiver := self parsePrimary.
    ^self parseUnaryMessage: receiver
!

parseUnaryMessage: expression
    | message |
    message := self parseUnaryMessage.
    message notNil
	ifTrue: [ ^self parseUnaryMessage:
		      (STMessageNode receiver: expression
				   message: message) ]
	ifFalse: [ ^expression ]
!

parseUnaryMessage
    | t | 
    t := lex peek.
    t isSTIdentifierTok
	ifTrue: [ ^STUnaryNode selector: lex next value ]
	ifFalse: [ ^nil ]
! !


!STParser methodsFor: 'private'!

parseError: str
    | file |
    "Transcript nextPutAll: '
Parse Error encountered
=======================

Scan so far was: 
'.
    Transcript nextPutAll: lex resetRecording.
    Transcript nl."
    file := lex isFileStream
        ifTrue: [ lex stream name ]
        ifFalse: [ '(a String)' ].

    parseErrorBlock
        value: file
        value: lex line
        value: str
!

isGoodByteArrayElement: val
    ^val isSmallInteger and: [ (val bitAnd: 255) = val ]
!

isGoodArrayElement: val
    ^true
!

isSymbolId: t
    ^t isSTIdentifierTok
	or: [ t isSTKeywordTok
		  or: [ t isSTSymbolKeywordTok
			    or: [ self isBinaryOperator: t ] ] ]
!


isBinaryOperator: token
    ^token isSTBinopTok or: [ token isSTVerticalBarTok ]
! !

!STParser methodsFor: 'overrides'!

record: node
    "do nothing by default"
!

compile: node
    "do nothing by default"
!

endMethodList
    "do nothing by default"
!

evaluate: node
    "This is not a do-nothing because its result affects the parsing
     process: true means 'start parsing methods', false means 'keep
     evaluating'. "
    self subclassResponsibility
! !

