"======================================================================
|
|   Smalltalk Tk-based GUI building blocks (abstract classes).
|
|   $Revision: 1.6.2$
|   $Date: 1999/08/31 11:23:18$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1990, 91, 92, 94, 95, 99 Free Software Foundation, Inc.
| Written by Steve Byrne.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"

!Delay class methodsFor: 'idle situations'!

idle
    Blox idle
!!

"----------------------------------- Gui class ---------------------------"

Object subclass: #Gui
    instanceVariableNames: 'blox'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Graphics-Windows'
!

Gui comment: '
I am a small class which serves as a base for complex objects which
expose an individual protocol but internally use a Blox widget for
creating their user interface.'!

!Gui methodsFor: 'accessing' !

blox
    "Return instance of blox subclass which implements window"
    ^blox.
!

blox: aBlox
    blox := aBlox. 
! !

"------------------------------ Event handling ------------------------"

Object subclass: #BEventTarget
       instanceVariableNames: 'eventReceivers'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BEventTarget comment: '
I track all the event handling procedures that you apply to an object.'!

!BEventTarget methodsFor: 'intercepting events'!

onAsciiKeyEventSend: aSelector to: anObject

    aSelector argumentCount = 1 ifFalse: [ ^self invalidArgsError: '1' ].
    self registerEventReceiver:  anObject.
    ^self
        bind: '<KeyPress>'
        to: #sendKeyEvent:oop:selector:
        of: self
        parameters: '*%A* ', anObject asOop printString, ' ', aSelector asTkString
!

onFocusEnterEventSend: aSelector to: anObject

    aSelector argumentCount = 0 ifFalse: [ ^self invalidArgsError: '0' ].
    ^self
        bind: '<FocusIn>'
        to: aSelector
        of: anObject
        parameters: ''
!

onFocusLeaveEventSend: aSelector to: anObject

    aSelector argumentCount = 0 ifFalse: [ ^self invalidArgsError: '0' ].
    ^self
        bind: '<FocusOut>'
        to: aSelector
        of: anObject
        parameters: ''
!

onKeyEvent: key send: aSelector to: anObject
    "Examples for key:   'Ctrl-1', 'Alt-X', 'Meta-plus'

     Special keys include: 'backslash', 'exclam', 'quotedbl', 'dollar',
     'asterisk', 'less', 'greater', 'asciicircum' (caret), 'question',
     'equal', 'parenleft', 'parenright', 'colon', 'semicolon', 'bar' (pipe
     sign), 'underscore', 'percent', 'minus', 'plus', 'BackSpace', 'Delete',
     'Insert', 'Return', 'End', 'Home', 'Prior' (Pgup), 'Next' (Pgdn),
     'F1'..'F24', 'Caps_Lock', 'Num_Lock', 'Tab', 'Left', 'Right', 'Up',
     'Down'.
     This same codes are received by the method invoked by onKeyEventSend:to:"

    aSelector argumentCount = 0 ifFalse: [ ^self invalidArgsError: '0' ].

    ^self
        bind: (self getKeyPressEventName: key)
        to: aSelector
        of: anObject
        parameters: ''
!

onKeyEventSend: aSelector to: anObject

    aSelector argumentCount = 1 ifFalse: [ ^self invalidArgsError: '1' ].
    ^self
        bind: '<KeyPress>'
        to: aSelector
        of: anObject
        parameters: '%K'
!

onKeyUpEventSend: aSelector to: anObject

    aSelector argumentCount = 1 ifFalse: [ ^self invalidArgsError: '1' ].
    ^self
        bind: '<KeyRelease>'
        to: aSelector
        of: anObject
        parameters: '%K'
!

onMouseDownEvent: button send: aSelector to: anObject

    aSelector argumentCount = 1 ifFalse: [ ^self invalidArgsError: '1' ].
    self registerEventReceiver:  anObject.
    ^self
        bind: '<ButtonPress-', button printString, '>'
        to: #sendMouseEvent:y:oop:selector:
        of: self
        parameters: '%x %y ', anObject asOop printString, ' ', aSelector asTkString
!

onMouseDoubleEvent: button send: aSelector to: anObject

    aSelector argumentCount = 1 ifFalse: [ ^self invalidArgsError: '1' ].
    self registerEventReceiver:  anObject.
    ^self
        bind: '<Double-', button printString, '>'
        to: #sendMouseEvent:y:oop:selector:
        of: self
        parameters: '%x %y ', anObject asOop printString, ' ', aSelector asTkString
!

onMouseEnterEventSend: aSelector to: anObject

    aSelector argumentCount = 0 ifFalse: [ ^self invalidArgsError: '0' ].
    ^self
        bind: '<Enter>'
        to: aSelector
        of: anObject
        parameters: ''
!

onMouseLeaveEventSend: aSelector to: anObject

    aSelector argumentCount = 0 ifFalse: [ ^self invalidArgsError: '0' ].
    ^self
        bind: '<Leave>'
        to: aSelector
        of: anObject
        parameters: ''
!

onMouseMoveEvent: button send: aSelector to: anObject

    aSelector argumentCount = 1 ifFalse: [ ^self invalidArgsError: '1' ].
    self registerEventReceiver:  anObject.
    ^self
        bind: '<B', button printString, '-Motion>'
        to: #sendMouseEvent:y:oop:selector:
        of: self
        parameters: '%x %y ', anObject asOop printString, ' ', aSelector asTkString
!

onMouseTripleEvent: button send: aSelector to: anObject

    aSelector argumentCount = 1 ifFalse: [ ^self invalidArgsError: '1' ].
    self registerEventReceiver:  anObject.
    ^self
        bind: '<Triple-', button printString, '>'
        to: #sendMouseEvent:y:oop:selector:
        of: self
        parameters: '%x %y ', anObject asOop printString, ' ', aSelector asTkString
!

onMouseUpEvent: button send: aSelector to: anObject

    aSelector argumentCount = 1 ifFalse: [ ^self invalidArgsError: '1' ].
    self registerEventReceiver:  anObject.
    ^self
        bind: '<ButtonRelease-', button printString, '>'
        to: #sendMouseEvent:y:oop:selector:
        of: self
        parameters: '%x %y ', anObject asOop printString, ' ', aSelector asTkString
!

onResizeSend: aSelector to: anObject

    aSelector argumentCount = 1 ifFalse: [ ^self invalidArgsError: '1' ].
    self registerEventReceiver:  anObject.
    ^self
        bind: '<Configure>'
        to: #sendMouseEvent:y:oop:selector:
        of: self
        parameters: '%w %h ', anObject asOop printString, ' ', aSelector asTkString
!!

!BEventTarget methodsFor: 'private'!

bind: event to: aSymbol of: anObject parameters: params
    self
        registerEventReceiver: anObject;
        primBind: event to: aSymbol of: anObject parameters: params
!

getKeyPressEventName: key
    | platform mod keySym |

    keySym := key isCharacter ifTrue: [ String with: key ] ifFalse: [ key ].
    (keySym includes: $-) ifFalse: [ ^'<KeyPress-', keySym, '>' ].

    mod := (ReadStream on: key) upTo: $-.
    keySym := key copyFrom: mod size + 2 to: key size.

    platform := Blox platform asSymbol.
    (mod = 'Meta') & (platform ~~ #unix)      ifTrue: [ mod := 'Alt' ].
    (mod = 'Alt')  & (platform == #unix)      ifTrue: [ mod := 'Meta' ].
    (mod = 'Alt')  & (platform == #macintosh) ifTrue: [ mod := 'Option' ].
    (mod = 'Ctrl') & (platform == #macintosh) ifTrue: [ mod := 'Cmd' ].
    ^'<', mod, '-KeyPress-', keySym, '>'
!

invalidArgsError: expected
    ^self error: 'invalid number of arguments, expected ', expected
!

primBind: event to: aSymbol of: anObject parameters: params
    self subclassResponsibility
!

registerEventReceiver: anObject
    eventReceivers isNil ifTrue: [ eventReceivers := IdentitySet new ].
    eventReceivers add: anObject
!

sendKeyEvent: key oop: oop selector: sel
    "Private - Filter ASCII events from Tcl to Smalltalk. We receive
     either *{}* for a non-ASCII char or *A* for an ASCII char, where
     A is the character. In the first case the event is eaten, in the
     second it is passed to a Smalltalk method"

    "key printNl.
    oop asInteger asObject printNl.
    '---' printNl."

    key size = 3 ifTrue: [
        oop asInteger asObject
            perform: sel asSymbol
            with: (key at: 2)
    ]
!

sendMouseEvent: x y: y oop: oop selector: sel
    "Private - Filter mouse events from Tcl to Smalltalk. We receive two
     strings, we convert them to a Point and then pass them to a Smalltalk
     method"

    "oop printNl.
    oop asInteger asObject printNl.
    '---' printNl."

    oop asInteger asObject
        perform: sel asSymbol
        with: (x asNumber @ y asNumber)
!!

" -------------------------------- Tcl interface ---------------------"

BEventTarget subclass: #Blox
       instanceVariableNames: 'primitive cache parent children'
       classVariableNames: 'Grab Interp Index ClipStatus Debug'
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

Blox comment: '
I am the superclass for every user interface object. I provide
common methods and a simple Tcl interface for internal use'!


Blox class
	defineCFunc: 'tclInit'
	withSelectorArgs: 'tclInit'
	returning: #cObject
	args: #()
!

Blox class
	defineCFunc: 'Tcl_Eval'
	withSelectorArgs: 'evalIn: interp tcl: cmd'
	returning: #int
	args: #(cObject string)
! 

Blox class
	defineCFunc: 'Tcl_GetStringResult'
	withSelectorArgs: 'resultIn: interp'
	returning: #string
	args: #(cObject)
! 

Blox class
	defineCFunc: 'bloxIdle'
	withSelectorArgs: 'idle'
	returning: #void
	args: #(self)
! 

!Blox class methodsFor: 'private - Tcl'!

debug
    ^Debug
!

debug: aBoolean
    Debug := aBoolean
!

tclEval: tclCode
    self debug ifTrue: [ stdout nextPutAll: tclCode; nl ].
    (self evalIn: Interp tcl: tclCode) = 1
        ifTrue: [ ^self error: self tclResult ].
!

tclResult
    ^self resultIn: Interp
!

uniqueName
    Index := Index + 1.
    ^'.b', Index printString
! !

!Blox methodsFor: 'private - Tcl'!

tclEval: tclCode
    self class debug ifTrue: [ stdout nextPutAll: tclCode; nl ].
    (self class evalIn: Interp tcl: tclCode) = 1
        ifTrue: [ ^self error: self tclResult ].
!

tclResult
    ^Blox resultIn: Interp
! !

" -------------------------------- 'meta' methods ---------------------"

!Blox class methodsFor: 'creating Tk interface methods'!

defineMutator: keyword option: tkOption toStringCode: extraCode
    "Produce code like...
   
    self tclEval: self connected, ' configure -label ' , (value  asTkString).
    self cache at: #label put: value"

    self
        defineMutator: keyword
        command: '%1 configure -', tkOption, ' %3'
        key: tkOption
        toStringCode: extraCode
!

defineAccessor: keyword option: tkOption fromStringCode: extraCode
    "Produce code like...

    self cache at: #label ifPresent: [ :value | ^value ].
    self tclEval: ('return [%1 cget -label]'
        bindWith: self connected
        with: self container.
    ^self cache at: #label put: (self tclResult )"

    self
        defineAccessor: keyword
        command: 'return [%1 cget -', tkOption, ']'
        key: tkOption
        fromStringCode: extraCode
!

defineMutator: keyword command: cmd key: key toStringCode: extraCode
    | s |
    s _
'%1: value
    self tclEval: (%2
        bindWith: self connected
        with: self container
        with: (value %3 asTkString)).
    self cache at: #%4 put: value'

        bindWith: keyword with: cmd storeString with: extraCode with: key.

    "stdout nextPutAll: s; nl; nl."
    self compile: s classified: 'accessing'
!

defineAccessor: keyword command: cmd key: key fromStringCode: extraCode
    | s |
    s _
'%1
    self cache at: #%4 ifPresent: [ :value | ^value ].
    self tclEval: (%2
        bindWith: self connected
        with: self container).
    ^self cache at: #%4 put: (self tclResult %3)'

        bindWith: keyword with: cmd storeString with: extraCode with: key.

    "stdout nextPutAll: s; nl; nl."
    self compile: s classified: 'accessing'
!

defineColorProtocolWithActivePrefix: active
    self
        defineAccessor: 'foregroundColor' option: 'foreground' fromStringCode: '';
        defineAccessor: 'backgroundColor' option: 'background' fromStringCode: '';
        defineAccessor: active, 'Foreground' option: active, 'foreground' fromStringCode: '';
        defineAccessor: active, 'Background' option: active, 'background' fromStringCode: '';

        defineMutator: 'foregroundColor' option: 'foreground' toStringCode: '';
        defineMutator: 'backgroundColor' option: 'background' toStringCode: '';
        defineMutator: active, 'Foreground' option: active, 'foreground' toStringCode: '';
        defineMutator: active, 'Background' option: active, 'background' toStringCode: ''
! !


" -------------------------------- generic methods ---------------------"

!Blox class methodsFor: 'instance creation'!

new
    ^self shouldNotImplement
!

new: parent
    | widget |
    ^super new
        init: parent
! !

!Blox class methodsFor: 'event dispatching'!

bloxInit
    | initResult |
    initResult := self tclInit.
    initResult isNil ifTrue: [ ^self ].
    initResult address = 0 ifTrue: [ ^self ].

    "If we reach this point, BLOX had not been initialized during this
     execution of the VM"
    Debug isNil ifTrue: [ Debug := false ].
    Interp := initResult.
    BPopupMenu initializeOnStartup.
    BCanvas initializeOnStartup.
    BWindow initializeOnStartup.
    Index := 0.
    ClipStatus := false.
    Grab := nil.
    ^self
!

dispatchEvents
    ClipStatus isString ifTrue: [
        self clipboard: ClipStatus
    ].

    self tclEval: '
        set terminateMainLoop 0
        tkwait variable terminateMainLoop'.
    
    "If we're outside the event loop, Tk for Windows is unable to
     render the clipboard and locks up the clipboard viewer app.
     Anyway save the contents for the next time we'll start a
     message loop"
    ClipStatus ifTrue: [
        ClipStatus := self clipboard.
        self clearClipboard
    ].
!

dispatchEvents: mainWindow
    ClipStatus isString ifTrue: [
        self clipboard: ClipStatus
    ].

    self tclEval: 'tkwait window ', mainWindow container.

    "If we're outside the event loop, Tk for Windows is unable to
     render the clipboard and locks up the clipboard viewer app.
     Anyway save the contents for the next time we'll start a
     message loop"
    ClipStatus ifTrue: [
        ClipStatus := self clipboard.
        self clearClipboard
    ].
!

terminateMainLoop
    self tclEval: 'set terminateMainLoop 1'.
    ^true
! !

!Blox class methodsFor: 'utility'!

beep
    self tclEval: 'bell'
!

clearClipboard
    self tclEval: 'clipboard clear'.
    ClipStatus := false
!

clipboard
    self tclEval: 'selection get -selection CLIPBOARD'.
    ^self tclResult
!

clipboard: aString
    aString isNil ifTrue: [ ^self ].
    self tclEval: 'clipboard clear'.
    self tclEval: 'clipboard append ', aString asTkString.
    ClipStatus := true
!

createColor: cyan magenta: magenta yellow: yellow
    ^self
        createColor: 65535 - cyan
        green: 65535 - magenta
        blue: 65535 - yellow
!

createColor: red green: green blue: blue
    "Answers a String, like '#FFFFC000C000' for pink"
    ^(String new: 13)
        at:  1 put: $#;
        at:  2 put: (Character digitValue: ((red   bitShift: -12) bitAnd: 15));
        at:  3 put: (Character digitValue: ((red   bitShift:  -8) bitAnd: 15));
        at:  4 put: (Character digitValue: ((red   bitShift:  -4) bitAnd: 15));
        at:  5 put: (Character digitValue: ( red                  bitAnd: 15));
        at:  6 put: (Character digitValue: ((green bitShift: -12) bitAnd: 15));
        at:  7 put: (Character digitValue: ((green bitShift:  -8) bitAnd: 15));
        at:  8 put: (Character digitValue: ((green bitShift:  -4) bitAnd: 15));
        at:  9 put: (Character digitValue: ( green                bitAnd: 15));
        at: 10 put: (Character digitValue: ((blue  bitShift: -12) bitAnd: 15));
        at: 11 put: (Character digitValue: ((blue  bitShift:  -8) bitAnd: 15));
        at: 12 put: (Character digitValue: ((blue  bitShift:  -4) bitAnd: 15));
        at: 13 put: (Character digitValue: ( blue                 bitAnd: 15));
        yourself
!

platform
    self tclEval: 'return $tcl_platform(platform)'.
    ^self tclResult.
! !

!Blox methodsFor: 'basic'!

deepCopy
    "It does not make sense to make a `deep' copy, because it
     would make data inconsistent across different objects"
    ^self shallowCopy
!

release
    primitive isNil ifFalse: [ self destroy ].
    super release.
! ! 

!Blox methodsFor: 'widget protocol'!

"NOTE: some of these may not be suitable for all Blox subclasses."

boundingBox
    ^(self x @ self y) extent: (self width @ self height)
!

boundingBox: rect
    self
        left: rect left
        top: rect top
        right: rect right
        bottom: rect bottom
!

childrenDo: aBlock
    children do: aBlock
!

childrenCount
    ^children size
!

destroy
    self tclEval: 'destroy ', self container
!

enabled
    ^self state ~= #disabled
!

enabled: enabled
    self state: (enabled ifTrue: [ #normal ] ifFalse: [ #disabled ])
!

exists
    ^primitive notNil
!

extent
    ^(self width @ self height)
!

extent: extent			"a point"
    self
        width: extent x
        height: extent y
!

isWindow
    ^false
!

left: left top: top right: right bottom: bottom
    self
        x: left
        y: top
        width: right - left
        height: bottom - top
!

parent
    ^parent
!

pos: position			"a point"
    self
        x: position x
	y: position y
!

posVert: aBlox
    self x: aBlox x; y: aBlox y + aBlox height
!

posHoriz: aBlox
    self x: aBlox x + aBlox width; y: aBlox y
!

toplevel
    self parent isNil ifTrue: [ ^self ].
    ^self parent toplevel
!

window
    ^self parent window
!

width: xSize height: ySize
    self 
        width: xSize;
        height: ySize
!

x: xPos y: yPos
    self
        x: xPos;
        y: yPos
!

x: xPos y: yPos width: xSize height: ySize
    self
        x: xPos y: yPos;
        width: xSize height: ySize
! !

!Blox methodsFor: 'creating children'!

make: array
    "Create children of the receiver. Answer a Dictionary of the children.
     Each element of array is an Array with the form:
         - first element = name, becomes the Dictionary's key
         - second element = class name symbol
         - third element = parameter setting array, for example
           #(width: 50 height: 30 backgroundColor: 'blue')
         - from the fourth element = children of the widget; each element has
           the same format described here"
    ^self make: array on: Dictionary new
!

make: array on: result
    "Private - Create children of the receiver, adding them to result;
     answer result. array has the format described in the comment to #make:"
    array do: [ :each | self makeChild: each on: result ].
    ^result
!

makeChild: each on: result 
    "Private - Create a child of the receiver, adding them to result;
     each is a single element of the array described in the comment to #make:"
    | current selector |
    current := result
        at: (each at: 1)
        put: ((Smalltalk classAt: (each at: 2)) new: self).

    each at: 3 do: [ :param |
        selector isNil
            ifTrue: [ selector := param ]
            ifFalse: [ current perform: selector with: param. selector := nil ]
    ].
    each size > 3 ifFalse: [ ^result ].
    each from: 4 to: each size do: [ :child |
        current makeChild: child on: result
    ].
! !

!Blox methodsFor: 'private'!

addChild: child
    ^children addLast: child
!

basicAddChild: child
    ^children addLast: child
!

primBind: event to: aSymbol of: anObject parameters: params

    ^self
        bind: event
        to: aSymbol
        of: anObject
        parameters: params
        prefix: 'bind ', self connected
!

bind: event to: aSymbol of: anObject parameters: params prefix: prefix
    "Low level event binding"
    | stream |
    stream := WriteStream with: prefix copy.
    stream
        space;
        nextPutAll: event;
        nextPutAll: ' {+callback ';
        print:      anObject asOop;
        space;
        nextPutAll: aSymbol asTkString;
        space;
        nextPutAll: params;
        nextPut:    $}.

    self tclEval: stream contents.
    ^event
!

cache
    "Answer the Tk option cache"
    ^cache
!

container
    self subclassResponsibility
!

connected
    self subclassResponsibility
!

init: parentWidget
    parent := parentWidget.
    cache := IdentityDictionary new.
    children := OrderedCollection new.
!

guiObject
    "Left for backward compatibility"
    ^primitive
!

primitive
    ^primitive
!

destroyed
    children := primitive := parent := nil
!

withChildrenDo: aBlock
    self value: aBlock.
    self childrenDo: aBlock
! !

Blox
    "#normal, #disabled, sometimes #active. Note this works on the container,
     not on the connected object!!"
    defineMutator: 'state'
    command: '%2 configure -state %3'
    key: 'state'
    toStringCode: '';

    defineAccessor: 'state'
    command: 'return [%2 cget -state]'
    key: 'state'
    fromStringCode: 'asSymbol'!

"------------------- Generic UI elements -------------------------"

Blox subclass: #BWidget
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BWidget comment: '
I am the superclass for every widget except those related to
menus. I provide more common methods and geometry management'!

!BWidget class methodsFor: 'creating Tk interface methods'!

defineGeometryMethods: keyword relativeTo: parentSelector
    | s |
    #(
'%1Absolute
    self tclEval: ''return [winfo %1 '', self connected, '']''.
    ^self tclResult asInteger'

'%1
    ^(self cache at: #%1) * self parent %2'

'%1: value
    | relative |
    self cache at: #%1 put: (relative := value / self parent %2).
    self tclEval:
        ''place '', self container, '' -rel%1 '', relative asFloat printString'

'%1Offset: value
    self tclEval:
        ''place '', self container, '' -%1 '', value asFloat printString'

'%1Pixels: value
    self %1: 0; %1Offset: value')

    do: [ :code |
        self
            compile: (code bindWith: keyword with: parentSelector)
            classified: 'geometry management'
    ]
! !

BWidget
    defineGeometryMethods: 'width' relativeTo: 'width';
    defineGeometryMethods: 'height' relativeTo: 'height';
    defineGeometryMethods: 'x' relativeTo: 'width';
    defineGeometryMethods: 'y' relativeTo: 'height';

    "#raised, #sunken, #flat, #ridge, #solid, #groove.
     #solid means 2-D.
     #groove means 3-D, with only the border sunken.
     #ridge means 3-D, with only the border raised.
     Note this works on the container, not on the connected object!!"
    defineMutator: 'effect'
    command: '%2 configure -relief %3'
    key: 'effect'
    toStringCode: '';

    defineAccessor: 'effect'
    command: 'return [%2 cget -relief]'
    key: 'effect'
    fromStringCode: 'asSymbol';

    "e.g. hand2 or left_ptr"
    defineAccessor: 'cursor' option: 'cursor' fromStringCode: 'asSymbol';
    defineMutator: 'cursor' option: 'cursor' toStringCode: '';

    "e.g. 'Helvetica 18 bold' "
    defineAccessor: 'font' option: 'font' fromStringCode: '';
    defineMutator: 'font' option: 'font' toStringCode: ''
!

!BWidget methodsFor: 'widget protocol'!

activate
    self tclEval: 'focus ', self connected
!

bringToTop
    self tclEval: 'raise ', self container
!

sendToBack
    self tclEval: 'lower ', self container
! !

!BWidget methodsFor: 'private'!

addChild: child
    child isWindow ifFalse: [
        self tclEval: 'place ', child container, ' -in ', self container
    ].
    ^self basicAddChild: child
!

create
    self subclassResponsibility
!

init: parentWidget
    super init: parentWidget.
    self create.
    self bind: '<Destroy>' to: #destroyed of: self parameters: ''.
    self setInitialSize.
    self parent notNil ifTrue: [ self parent addChild: self ].
!

setInitialSize
    "Make the Tk placer's status, the receiver's cache and the
     window status (as returned by winfo) consistent. Occupy the
     whole area by default"
    self
        x: 0
        y: 0
        width: self parent width
        height: self parent height
!

restoreWidgetSize
    "Make the Tk placer's status, the receiver's cache and the
     window status (as returned by winfo) consistent, based on
     winfo's information"

    self tclEval: 'return [winfo x ', self container, ']'.
    self x: self tclResult asNumber.
    
    self tclEval: 'return [winfo y ', self container, ']'.
    self y: self tclResult asNumber.

    self tclEval: 'return [winfo width ', self container, ']'.
    self width: self tclResult asNumber.

    self tclEval: 'return [winfo height ', self container, ']'.
    self height: self tclResult asNumber
! !

"------------------- Primitives --------------------------------------"

BWidget subclass: #BPrimitive
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BPrimitive comment: '
I am the superclass for every widget (except menus) directly
provided by the underlying GUI system.'!


!BPrimitive methodsFor: 'private'!

container
    ^primitive
!

connected
    ^primitive
!

create
    self create: ''
!

create: options
    self tclEval: self widgetType, self connected, options.
!

init: parentWidget
    primitive := self setWidgetName: parentWidget.
    super init: parentWidget.
!

setWidgetName: parentWidget
    ^parentWidget isNil
        ifTrue: [ self class uniqueName ]
        ifFalse: [ parentWidget container copy, self class uniqueName ].
!

widgetType
    self subclassResponsibility
! !

"------------------------------ vs. extended -------------------------"

BWidget subclass: #BExtended
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BExtended comment: '
Just like Gui, I serve as a base for complex objects which expose
an individual protocol but internally use a Blox widget for
creating their user interface. The instances of my subclasses,
however, have the standard widget protocol.
Just override my newPrimitive method to return another widget,
and you''ll get a class which interacts with the user like that
widget (a list box, a text box, or even a label) but exposes a
different protocol.'!


!BExtended methodsFor: 'accessing'!

primitive
    ^primitive
!

newPrimitive
    self subclassResponsibility
! !

!BExtended methodsFor: 'private'!

container
    ^self primitive container
!

connected
    ^self primitive connected
!

create
    primitive := self newPrimitive
!

widgetType
    self subclassResponsibility
! !

"------------------------------ Viewport controls --------------------------"

BPrimitive subclass: #BViewport
       instanceVariableNames: 'connected'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
!

BViewport comment: '
I represent an interface which is common to widgets that can be
scrolled, like list boxes or text widgets.'!


!BViewport methodsFor: 'accessing'!

connected
    ^connected
! !

!BViewport methodsFor: 'private'!

create: options
    "Create with both scrollbars"
    self create: options horizScrollbar: true vertical: true
!

create: options horizScrollbar: hscroll vertical: vscroll
    connected := self container, '.ctl'.
    self tclEval: 'frame ', self container, ' -relief sunken'.
    super create: options.
    self createScrollbar: hscroll vertical: vscroll.
    self tclEval: 'pack ', self connected, ' -expand 1 -fill both'.
!

createScrollbar: hscroll vertical: vscroll

    hscroll ifTrue: [
        self tclEval: ('
            scrollbar %1.hs -orient horiz -command {%1.ctl xview}
            %1.ctl configure -xscrollcommand {%1.hs set}
            pack %1.hs -fill x -side bottom' bindWith: self container)
    ].

    vscroll ifTrue: [
        self tclEval: ('
            scrollbar %1.vs -orient vert -command {%1.ctl yview}
            %1.ctl configure -yscrollcommand {%1.vs set}
            pack %1.vs -fill y -side right' bindWith: self container)
    ].
! !

"------------------------------ Menu superclass -----------------------"

Blox subclass: #BMenuObject
     instanceVariableNames: ''
     classVariableNames: ''
     poolDictionaries: ''
     category: 'Graphics-Windows'
! 

BMenuObject comment: '
I am an abstract superclass for widgets which make up a menu structure.'!


"Define position methods as dummy"
#('x' 'x: value' 'y' 'y: value' 'width' 'width: value' 'height' 'height: value')
    do: [ :each | BMenuObject compile: each classified: 'disabled methods' ]!

BMenuObject defineColorProtocolWithActivePrefix: 'active'!

"----------------------------------- ADDS TO THE STANDARD IMAGE ----------"


!String methodsFor: 'Tk interface'!

asTkString
    | i stream ch crFound |
    stream := WriteStream on: (self copyEmpty: self size + 10).
    stream nextPut: $".
    crFound := false.
    i := 1.
    [ i <= self size ] whileTrue: [
        ch := self at: i.
        ch = $" ifTrue: [ stream nextPut: $\ ].
        ch = $\ ifTrue: [ stream nextPut: $\ ].
        ch = $[ ifTrue: [ stream nextPut: $\ ].
        ch = $] ifTrue: [ stream nextPut: $\ ].
        ch = $$ ifTrue: [ stream nextPut: $\ ].

        ch = Character nl
            ifTrue: [
                "Under Windows, CR/LF-separated lines are common. Turn a
                 CR/LF pair into a single \n"
                crFound ifTrue: [ stream skip: -2 ].
                stream nextPut: $\.
                ch := $n
            ].

        "On Macs, CR-separated lines are common. Turn 'em into \n"
        (crFound := (ch == Character cr))
            ifTrue: [
                stream nextPut: $\.
                ch := $n
            ].

        stream nextPut: ch.
        i := i + 1
    ].
    stream nextPut: $".
    ^stream contents
! !
