/*
 * GNU POC (passwords on card) - manage passwords on smartcards
 * Copyright (C) 2001 Henning Koester <henning@crackinghacking.de>
 *
 * Please report bugs to bug-poc@gnu.org
 *
 * This file is part of POC.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* $Id: backup_card.c,v 1.7 2001/08/18 15:58:39 henning Exp $ */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "poc.h"
#include "poc_macros.h"
#include "poc_types.h"
#include "cardinfo.h"
#include "card.h"
#include "lang.h"
#include "misc.h"

/******************************************************************************
 *
 * Function    : backup_create
 *
 * Description : This function backups the content of a card (the whole
 *               memory to a file.)
 *
 * Input       : [1] filename (char)
 *                   The file, where the backup will be made to.
 *               [2] port (u16)
 *                   Com-Port
 *
 * Return      : POC_ERROR or POC_SUCCESS
 *
 *****************************************************************************/
bool
backup_create(const char * const filename, const u16 port) {
  FILE *f_ptr;                /* FILE pointer of the backup file. */
  u8 *buffer = NULL;          /* Buffer for storing card's data. */
  char ret;                   /* Hold returns of card calls (card.c) */
  u16 data_size;              /* Size of card's data. */
  u16 ctn = 0;                /* Card-terminal handle. */

#ifdef DEBUG
  printf("DEBUG: backup_create(filename [%s], port [%i]\n", filename, port);
#endif

  /* Try to open the backup file. */
  if ((f_ptr = fopen(filename, "wb")) == NULL) {
   
    /* Can't create backup file. Print an error and return to the caller. */
    print_err(STR_CREATE_FILE_ERR);
    printf("'%s': ", filename); fflush(stdout);
    perror("");
    return(POC_ERROR);
  }

  /* Initialize & reset card terminal. */
  if ( (card_init_terminal(ctn, port) != POC_SUCCESS) ||
       (card_reset_terminal(ctn) != POC_SUCCESS) ) {
    
    /* If initialization fails, or resetting, we will return to the caller. */
    card_close_terminal(ctn);
    return(POC_ERROR);
  }
  
  /* Request ICC and check whether a memory card is inserted. */
  if (card_request_icc(ctn) != MEMORY_CARD) {

    /* If no memory card is present we will return. */
    print_err(STR_NO_MEM_CARD);
    card_close_terminal(ctn);
    return(POC_ERROR);
  }

  /* Select card's memory so we can read it. */
  if (card_select_file(ctn) != POC_SUCCESS) {
    card_close_terminal(ctn);
    return(POC_ERROR);
  }

  /* Allocate 2 byte to store card's size. */
  if ( (buffer = realloc(buffer, 2)) == NULL) {
    print_err(ERR_PRFX_NM);
    perror("");
    return(POC_ERROR);
  }

  /* Read data size.
   * If 'ret' is not POC_SUCCESS, we will print an error message if a memory 
   * error occured and return. */
  if ( (ret = card_read_data(ctn, DATA_SIZE_OFFSET, 2, buffer)) != 
       POC_SUCCESS) {
    if (ret == POC_MEM_ERR) {
      print_err(ERR_PRFX_NM);
      perror("");
    }
    card_close_terminal(ctn);
    return(POC_ERROR);
  }

  /* Convert the byte order and store card's size in 'data_size'. */
  data_size = (buffer[0] << 8) | buffer[1];

  /* If the card is empty, we'll print an error and return. */
  if (data_size == 0) {
    print_err(STR_EMPTY_CARD);
    return(POC_ERROR);
  }

  /* 
   * Allocate more memory to store the data which will be read from the card
   * (data_size) plus 2 byte to store card-size-information (written in the
   * first 2 bytes of the card) plus 2 byte to store data-size-information
   * (stored in byte 3 and 4 in the card's memory.
   */
  if ( (buffer = realloc(buffer, data_size + 2 + 2)) == NULL) {

    /* If memory allocation failed, we'll return... */
    print_err(ERR_PRFX_NM);
    perror("");
    return(POC_ERROR);
  }

  /* Now read the everything what's stored on the card (offset '0'). */
  if ( (ret = card_read_data(ctn, 0, data_size + 2 + 2, buffer)) !=
       POC_SUCCESS) {

    /* Something went wrong. We return. */
    if (ret == POC_MEM_ERR) {
      print_err(ERR_PRFX_NM);
      perror("");
    }
    card_close_terminal(ctn);
    return(POC_ERROR);
  }

  /* No more communication is needed, so we'll close the terminal. */
  card_close_terminal(ctn);

  /* Write everything to the backup file. */
  fwrite(buffer, data_size + 2 + 2, 1, f_ptr);

  /* Close the file. */
  fclose(f_ptr);

  /* Change file's mode to 600 (-rw-------). */
  if ( (chmod(filename, S_IRUSR | S_IWUSR)) != 0) {
    print_err(STR_CHMOD_FAIL);
    perror("");
    return(POC_ERROR);
  }

  /* wipe sensetive data. */
  memset(buffer, 0, data_size + 2);
  overwrite_buffer(buffer);
  drop_mbuffer(buffer);

  return(POC_SUCCESS);
}

/******************************************************************************
 *
 * Function    : backup_restore
 *
 * Description : This function restores a backuped card from a backup file.
 *
 * Input       : [1] filename (char)
 *                   The file, where the backup is read from.
 *               [2] port (u16)
 *                   Com-Port
 *
 * Return      : POC_ERROR or POC_SUCCESS
 *
 *****************************************************************************/
bool
backup_restore(const char * const filename, const u16 port) {
  FILE *f_ptr;       /* FILE pointer of the file where the backup is stored. */
  u8 *buffer = NULL;       /* Buffer for storing card's data. */
  u16 ctn = 0;             /* Card-terminal. */
  char ret;                /* Hold returns of card calls (card.c) */
  u16 data_size;           /* Size of card's data. */

#ifdef DEBUG
  printf("DEBUG: backup_restore(filename [%s], port [%i]\n", filename, port);
#endif

  /* Open the backup file for later reading. */
  if ((f_ptr = fopen(filename, "rb")) == NULL) {

    /* Something went wrong while trying to open the file. Print an error
     * and return. */
    print_err(STR_OPEN_FILE_ERR);
    printf("'%s': ", filename); fflush(stdout);
    perror("");
    return(POC_ERROR);
  }

  /*
   * First we will read the data size and then allocate enough space for it
   * with additionally 2 bytes (2 for card-size information).
   */

  /* Allocate 2 byte of memory to store card's size. */
  if ( (buffer = realloc(buffer, 2)) == NULL) {

    /* Return on memory allocation error. */
    print_err(ERR_PRFX_NM);
    perror("");
    return(POC_ERROR);
  }

  /* Reposition the stream to the offset of the data size information
   * (data size information is stored in byte 3-4 of the file. Byte 1-2 
   * contain the card size information.)
   */
  fseek(f_ptr, 2, SEEK_SET);

  /* Read byte 3 and 4. */
  fread(buffer, 1, 2, f_ptr);
 
  /* Convert the data to and unsigned short and store it in 'data_size'. */
  data_size = buffer[0] << 8 | buffer[1];

  /* Now allocate more memory to read the whole file. */
  if ( (buffer = realloc(buffer, data_size + 2 + 2)) == NULL) {

    /* Return on memory allocation error. */
    print_err(ERR_PRFX_NM);
    perror("");
    return(POC_ERROR);
  }

  /* Reposition the stream to the beginning of the file. */
  fseek(f_ptr, 0, SEEK_SET);
  
  fread(buffer, 1, data_size + 2 + 2, f_ptr);  /* Read the whole file. */

  fclose(f_ptr);    /* Close the file. */


  /*
   * Now where we've read everything, we will initialize and reset the
   * card terminal. If everything worked fine we will write 'buffer'
   * to the card.
   */

  /* Initialize & reset card terminal. */
  if ( (card_init_terminal(ctn, port) != POC_SUCCESS) ||
       (card_reset_terminal(ctn) != POC_SUCCESS) ) {

    /* If something fails, we'll overwrite sensetive data and return. */
    overwrite_buffer(buffer);
    drop_mbuffer(buffer);
    card_close_terminal(ctn);
    return(POC_ERROR);
  }

  /* We will fetch the card's ICC and check whether a memory card was found
   * or not. */
  if (card_request_icc(ctn) != MEMORY_CARD) {

    /* No memory card found, so we'll return with an error. */
    print_err(STR_NO_MEM_CARD);
    overwrite_buffer(buffer);
    drop_mbuffer(buffer);
    card_close_terminal(ctn);
    return(POC_ERROR);
  }

  /* Select card's memory so we can write to it. */
  if (card_select_file(ctn) != POC_SUCCESS) {
    overwrite_buffer(buffer);
    drop_mbuffer(buffer);
    card_close_terminal(ctn);
    return(POC_ERROR);
  }

  if ( (ret = card_write_data(ctn, 0, data_size + 2 + 2, buffer)) != 
       POC_SUCCESS) {
    if (ret == POC_MEM_ERR) {
      print_err(ERR_PRFX_NM);
      perror("");
    }
    overwrite_buffer(buffer);
    drop_mbuffer(buffer);
    card_close_terminal(ctn);
    return(POC_ERROR);
  }

  /* Everything worked fine. */
  card_close_terminal(ctn);

  return(POC_SUCCESS);
}
