/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/*
 * Name          : pips/src/pipsVarInq.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : Thu Oct 13 21:36:52 WET DST 1994
 * Modified on   : 
 * Synopsis      : 
 *
 *   pipsVarInq (ncId, dataId, varName, dataType, images, height, width)
 *
 *   int ncId         IS   the identifier of the current netcdf file
 *   int dataId       IS   the identifier of the current variable
 *   char *varName    IS   name of variable to query
 *   nc_type dataType IS   the nc type of the variable
 *   long images      IS   returned number of images in 'dataId'
 *   long height      IS   returned height of images in 'dataId'
 *   long width       IS   returned width of images in 'dataId'
 * 
 * Description   : 
 *
 *   Returns information about the variables of a netcdf file.
 *   If 'varName' is NULL or its stringlength is 0, then subseqent calls with
 *   the same variables will return images from all variables starting with
 *   the first. If 'varName' is not NULL then it must have space for at least
 *   MAX_NC_NAME characters, and the variable name will be returned.
 *   'ncId' and 'dataId' should never have random values.
 *   Initialize them or set them to -1;
 *   Calls for different netcdf files or variables cannot be mixed. All
 *   internal variables are reset, if a new 'ncId' or 'dataId' is encountered.
 *   It returns true, as long as there are variables not returned yet.
 */

#include "pips.h"   /* PIPS system header file  */

#ifdef __STDC__

int pipsVarInq (int ncId, int *dataId, char *varName, nc_type *dataType,
                 long *images, long *height, long *width)

#else

pipsVarInq (ncId, dataId, varName, dataType, images, height, width)
    int ncId;
    int* dataId;
    char *varName;
    nc_type *dataType;
    long* images;
    long* height;
    long* width;

#endif
{
  static int ncIdCurr   = -2; /* The current processed netcdf file id        */
  static int dataIdCurr = -2; /* The current processed variable id           */
  static int extVar     = 0;  /* How many variables to extract: 1 or all     */
  static int nvars      = 0;  /* Current variable id and # of variables      */
  int ndims;                  /* Number of dimensions of 'dataId'            */
  int dimids [MAX_VAR_DIMS];  /* Array with dimension ids for 'dataId'       */

/*
 * Increase varId, decrease number of extracted variables
 */
  if (*dataId == dataIdCurr) {
    (*dataId)++;
    extVar--;             /* Counts number of extracted variables down*/
  }
  dataIdCurr = *dataId;

/*
 * If function is first invoked with 'ncId', set variables
 */
  if (ncId != ncIdCurr) {
    ncIdCurr = ncId;

/*
 * Get number of variables
 */
    ncinquire (ncId, 0, &nvars, 0, 0);

/*
 * If variable name was given, abort function, if it is invalid.
 * If no variable name was given, start with first variable.
 */
    (*dataId) = 0;
    extVar = nvars;
    if (varName && (strlen (varName) != 0)) {
      extVar = 1;
      if ((*dataId = ncvarid (ncId, varName)) == -1) {
        pipsPrintError (-1, PipsInvalidVar, varName, PipsDontExit);
        return (0);
      } /* end if */
    } /* end if */
  } /* end if ncId */

/*
 * Get information about current variable
 */
  ncvarinq (ncId, *dataId, varName, dataType, &ndims, dimids, 0);

/*
 * Take the last three dimensions of the variable.
 * 'images', 'height' and 'width' are set to 0, if the appropriate
 * dimension doesn't exist.
 */
  (*images) = 0;  (*height) = 0;  (*width) = 0;
  if (ndims > 2)
    ncdiminq (ncId, dimids[ndims-3], 0, images);
  if (ndims > 1)
    ncdiminq (ncId, dimids[ndims-2], 0, height);   
  if (ndims > 0)
    ncdiminq (ncId, dimids[ndims-1], 0, width);

  return (extVar > 0);
}  /* end pipsVarInq */


/* end pipsVarInq.c */
