/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/*
 * Name          : sumHistogram.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : Thu Oct 13 21:36:52 WET DST 1994
 * Modified on   : 
 *
 *  This file contains two functions:
 *
 *     pipsJobManager(), which is executed on the local host and
 *     pipsProcPar(),    which is executed on all hosts.
 *
 *  Both have their own description appended below.
 */

#include "pips.h"   /* PIPS system header file  */
#include "pvm3.h"   /* PVM system header file   */
#include <math.h>   /* Used: sqrt()             */

/*
 * +++ Set the minimum number of variables and parameters, 
 * the program expects on the command line
 */
int parameterNumber = 2;

/*
 * +++ Description of program for display on stderr by pipsPrintError()
 */
char *programName = "sumHistogram";           /* Used in program description */
char *description[] = {
  "  sumHistogram [-h] ncName var1 var2 [-r i1 i2 r1 r2 c1 c2] \\ ",
  "               [winSize [diffSigma]]",
  " ",
  "Where:",
  "  -h         Prints this help text.",
  "  ncName     Name of netCDF file to process, must be first parameter.",
  "  var1       Variable to operate on.",
  "  var2       Variable to store the result; must not be variable 1.",
  "  i1 i2      Specify the image numbers to process, first number is 0",
  "  r1 r2      Specify the row to process, first row is number 0",
  "  c1 c2      Specify the column to process, first column is 0",
  "  diffSigma  The sigma for the gaussian difference contribution",
  "  winSize    Window to examine for each pixel",
  " ",
  "Description:",
  "  sumHistogram opens 'ncName', spawns tasks on all processors in the",
  "  virtual parallel machine.",
  "  Expects long data as input.",
  "  If NUM[DIFF] denotes, who often difference DIFF occurs in the window",
  "  then SCORE = sum (NUM[DIFF] * exp (-DIFF^2 / diffSigma^2 / 2)) ",
  "  The numbers for the range (-r option) can be specified too large;",
  "  the maximum value is taken then.",
  "  Multiplies all values with 1000000 and stores result as long.",
  ""
}; /* end description */




/******************************************************************************
 *  P A R E N T   P R O C E S S  (on local host)
 ******************************************************************************
 *
 *   void pipsJobManager (ntask, tid, ncId, parac, parav)
 *
 *   int ntask        IS   Number of tasks available for parallel
 *                         data processing
 *   int tid []       IS   An array with task Id's of the tasks
 *   int ncId         IS   Id of opened netcdf file, returned by ncopen()
 *   int parac        IS   Number of valid arguments in parav[]
 *   char **parav     IS   Tail of argv containing command line options
 *
 * Description   : 
 *
 *   Here, the data SEGMENTATION is done.
 *   Segments the data into pieces that can be processed in parallel,
 *   sends them to the tasks (that run pipsProcPar) and recollects it.
 *   If the number of data fragments (left) is less than the number of
 *   tasks, a competition is started that also guaranties that all lost
 *   data (crashed hosts, etc) is reprocessed.   
 */

#ifdef __STDC__

void pipsJobManager (int ntask, int tid[], int ncId, int parac, char **parav)

#else

pipsJobManager (ntask, tid, ncId, parac, parav)
    int ntask;
    int tid[];
    int ncId;
    int parac;
    char **parav;

#endif
{
/*
 * Variables for the job distribution
 */
  int done = 0;         /* Total number of processed jobs                    */
  int *jobId;           /* Array with currently processed jobs (1 per task)  */
  int *perform;         /* how many jobs each task has allready accomplished */
  int index;            /* Index to tid[], jobId[] and perform[]             */
  long i;               /* Index to tid[], jobId[] and perform[]             */
  int job = 0;          /* Current job Id (0 .. njobs-1)                     */
  int finishedJob;      /* Id of job that has just been finished             */
  int newJobs = 1;      /* Indicates that new jobs are still available       */
  int njobs;            /* Number of data slabs to process parallel          */

/*
 * +++ Netcdf variables and variables for data segmentation
 * start, edge: recordDim, rowDim, columnDim
 */
  long start[3];        /* Start of each datapacket                          */
  long edge [3];        /* Shape (size) of each datapacket                   */
  int fileType;         /* Value to use for file type attribute              */
  nc_type dataType;     /* Data type used in the result variable             */
  int var0Id;           /* Id of the first variable                          */
  int var1Id;           /* Id of the second variable                         */
  long var0Height;      /* Height of image in variable var0Id                */
  long var0Width;       /* Width of image in variable varOId                 */
  long var1Height;      /* Height of image in variable var1Id                */
  long var1Width;       /* Width of image in variable var1Id                 */
  long rows;            /* Number of rows of data in one data fragment       */
  long images;          /* Number of images in current variable              */
  nc_type type;         /* Data type used in 'dataId'                        */
  nc_type resType;      /* Type of an existing variable for the results      */
  long winSize;         /* Size of window to examine arround center pixel    */
  float sigmaDiff;      /* Sigma for gaussian weighted difference contrib.   */
  long startImg;        /* First image to process                            */
  long endImg;          /* Last image to process                             */
  long startRow;        /* First row to process                              */
  long endRow;          /* Last row to process                               */
  long startCol;        /* First column to process                           */
  long endCol;          /* Last column to process                            */
  unsigned long *data;  /* holds data that has to be processed               */
  unsigned long *result;/* holds data that has been processed                */

/*
 * +++ Set the granularity to adjust the programs performance (see document.)
 */
  int granularity = 3;  /* The higher it is, the more jobs are generated     */


/******************************************************************************
 * +++  P R O C E S S   P A R A M E T E R S   A N D   P R E P A R A T I O N S 
 ******************************************************************************
 */

/*
 * +++ Check the input variables (existence and properties)
 * Get their dimensions and calculate minimum of each dimension
 */
  var0Id = ncvarid (ncId, parav[0]);
  if (! pipsVarInq (ncId, &var0Id, parav[0], &type, &images, &var0Height,
                     &var0Width)) {
    pipsPrintError (ncId, PipsInvalidVar, parav[0], PipsDontExit);
    return;
  } /* end if */

/*
 * +++ Check type of incoming data
 * Complain, if the variable doesn't have the expected data type
 */
  if (type != NC_LONG) {
    pipsPrintError (ncId, PipsDataIncompat, parav[0], PipsDontExit);
    return;
  } /* end if */

/*
 * +++ Evaluate the rest of command line parameters
 */
  index = 2;

/*
 * Set default values
 */
  startImg = 0;      
  endImg = images-1;
  startRow = 0;      
  endRow = var0Height-1;
  startCol = 0;      
  endCol = var0Width-1;

/*
 * Get ranges for images, rows and columns, if -r option given
 */
  if ((parac > index) && (strncmp ("-r", parav[index], 2) == 0)) {
    index++;

    if (parac > index)
      startImg = atol (parav[index]);
    if (startImg < 0)
      startImg = 0;
    index++;

    if (parac > index)
      endImg = atol (parav[index]);
    if (endImg > images-1)
      endImg = images-1;
    index++;

    if (parac > index)
      startRow = atol (parav[index]);
    if (startRow < 0)
      startRow = 0;
    index++;

    if (parac > index)
      endRow = atol (parav[index]);
    if (endRow > var0Height-1)
      endRow = var0Height-1;
    index++;

    if (parac > index)
      startCol = atol (parav[index]);
    if (startCol < 0)
      startCol = 0;
    index++;

    if (parac > index)
      endCol = atol (parav[index]);
    if (endCol > var0Width-1)
      endCol = var0Width-1;
    index++;

  } /* end if parac */

  winSize = 11;                    /* Default value */
  if (parac > index)
    winSize = atol (parav[index]);
  index++;

  sigmaDiff = 5;                  /* Default value */
  if (parac > index)
    sigmaDiff = atof (parav[index]);
  index++;

/*
 * +++ Make window dimension odd to allow for a central element.  
 */
  if (winSize % 2 == 0)
    winSize++;

/*
 * +++ Select type of outgoing data and fileType to create from it (later on)
 */
  dataType = NC_LONG;
  fileType = PipsPpmRaw;

/*
 * +++ Determine the total size of data per variable; number of rows to process
 */
  rows = ((endRow - startRow + 1) - (winSize - 1)) * (endImg - startImg + 1);

/*
 * Calculate the number of jobs (data fragments) and rows per job
 */
  rows = pipsDataPerJob (ntask, granularity, endImg - startImg + 1,
                          sqrt(rows), rows, &njobs);

/*
 * +++ Create the output variable (exit on failure)
 */
  resType = dataType;
  var1Width = (endCol - startCol + 1) - (winSize - 1);
  var1Height = rows * njobs;
  if ((var1Id = pipsVarDef (ncId, parav[1], &resType, &fileType,
                             &i, &var1Height, &var1Width)) < 0) {
    pipsPrintError (ncId, PipsInvalidVar, parav[1], PipsDontExit);
    return;
  } /* end if */

/*
 * Complain and exit, if the result variable exists and is unsuited for result
 */
  if (dataType != resType) {
    pipsPrintError (ncId, PipsVariableType, parav[1], PipsDontExit);
    return;
  } /* end if */

/*
 * Reset the processed job array - currently no task is processing a job
 */
  jobId = pipsNewFlagArray (ntask, PipsNoJob);

/*
 * Reset the number of jobs accomplished by each task
 */
  perform = pipsNewFlagArray (ntask, 0);

/*
 * +++ Allocate memory, one buffer for both, loading and saving of data
 * Make sure the buffer size is a multiple of the larger data type
 */
  data   = (unsigned long *) calloc (
           (var1Width + winSize - 1) * (rows + winSize - 1), sizeof(long));
  result = (unsigned long *) data;                /* uses same memory */

/******************************************************************************
 *  D A T A   S E G M E N T A T I O N
 ******************************************************************************
 * Distribute jobs until all have been processed
 */
  while (done < njobs) {
    pipsAwaitChildData (tid,jobId,perform,&done, ntask, &finishedJob, &index);
/******************************************************************************
 * +++ R E C E I V E   A N D   S T O R E   P R O C E S S E D   D A T A 
 ******************************************************************************
 */
/*
 * Unpack and save data, if task was busy with a job
 */
    if (finishedJob != PipsNoJob) {
/*
 * +++ Set the image number, the data fragment belongs to
 */
      start [0] = (finishedJob * rows) / var1Height;
/*
 * +++ Set the row number, the data fragment starts at
 */
      start [1] = (finishedJob * rows) % var1Height;
      start [2] = 0;
      edge  [0] = 1;
      edge  [1] = rows;
      edge  [2] = var1Width;
      pvm_upkulong (result, edge[0]*edge[1]*edge[2], 1);
      ncvarput (ncId, var1Id, start, edge, (void *) result);
    } /* end if finishedJob */

/*
 * Keep sending, while there is still an unfinished job left
 */
    if (pipsJobAvailable (jobId, &job, ntask, index, newJobs)) {
/******************************************************************************
 * +++  L O A D   A N D   S E N D   P R O C E S S   D A T A 
 ******************************************************************************
 */
/*
 * +++ Set the image number, the data fragment belongs to
 */
      start [0] = (job * rows) / var0Height;
/*
 * +++ Set the row number, the data fragment starts at
 */
      start [1] = (job * rows) % var0Height;
      start [2] = startCol;
      edge  [0] = 1;
      edge  [1] = rows + winSize - 1;
      edge  [2] = var1Width + winSize - 1;
      ncvarget (ncId, var0Id, start, edge, (void *) data);
/*
 * +++ Pack in the processing parameters
 */
      pvm_pklong  (&(edge[2]), 1, 1);
      pvm_pklong  (&rows, 1, 1);
      pvm_pklong  (&winSize,   1, 1);
      pvm_pkfloat (&sigmaDiff, 1, 1);
      pvm_pkulong (data, edge[0]*edge[1]*edge[2], 1);

/*
 * Send data packet to the idle child and tell it to commence work on it
 */
      pvm_send (tid[index], PipsCommence);
    } /* end if pipsJobAvailable */

/*
 * Select a job that hasn't been processed yet, if available
 * (for the next idle task)
 */
    pipsSelectNewJob (&newJobs, njobs, &job);
  } /* end while done < njobs */


/******************************************************************************
 *  E V A L U A T I O N   A N D   C L E A N U P
 ******************************************************************************
 */
/*
 * Report performance evaluation
 */
  pipsPrintPerformance (tid, perform, ntask, njobs);

/*
 * Free allocated memory
 */
  free (data);
  free (jobId);
  free (perform);
} /* pipsJobManager */







/******************************************************************************
 *  C H I L D   P R O C E S S  (on all hosts)
 ******************************************************************************
 *
 *   void pipsProcPar (void)
 *
 * Description   : 
 *
 *   Here the data PROCESSING is performed.
 *   Unpacks data received from the pipsJobManager (), processes it
 *   and packs the result data in again to be send back to the
 *   pipsJobManager. 
 */

#ifdef __STDC__

void pipsProcPar (void)

#else

pipsProcPar ()

#endif
{
/*
 * +++ Variables sent by the jobManager
 * NOTE: Declaring all variables 'static' increases speed dramatically
 */
  static long cols;
  static long rows;
  static long winSize;
  static float sigmaDiff;

/*
 * +++ 'Backup' of above variables to keep track of changes in variables
 */
  static long currCols = 0;
  static long currRows  = 0;
  static long  currWinSize = 0;
  static float currSigmaDiff = 0;

/*
 * +++ Variables for processing the data
 */
  static long i, x, y, wx, wy;
  static long middle;
  static long elements;
  static double weight;
  static double sum;
  static double quotient;
  static double hist[256];
  static double *dist = NULL;
  static unsigned long *data = NULL;
  static unsigned long *result = NULL;

/*
 * +++ Define macros for better readability
 */
#define RESULTCOLS  (cols - winSize + 1)
#define DATA(X,Y)   data   [(Y) * cols + (X)]
#define RESULT(X,Y) result [(Y) * RESULTCOLS + (X)]
#define DIST(X,Y)   dist   [((Y)+middle) * winSize + (X)+middle]

/******************************************************************************
 * R E C E I V E   P R O C E S S   D A T A 
 ******************************************************************************
 * +++ Unpack data that has arrived from the jobmanager
 */
  pvm_upklong (&cols, 1, 1);
  pvm_upklong (&rows, 1, 1);
  pvm_upklong (&winSize, 1, 1);
  pvm_upkfloat (&sigmaDiff, 1, 1);

/*
 * +++ Preparations after change of parameters (data from different image)
 * Memory has to be reallocated and variables re-initialized
 */
  if ((cols      != currCols)       ||
      (rows      != currRows)       ||
      (winSize   != currWinSize)    ||
      (sigmaDiff != currSigmaDiff)) {
    currCols = cols;
    currRows = rows;
    currWinSize = winSize;
    currSigmaDiff = sigmaDiff;

/*
 * +++ Code simplifying varibles
 */
    middle = winSize / 2;
    elements = winSize * winSize;

/*
 * +++ Allocate memory for variables
 */
    if (data)               /* Free allocated memory first */
      free (data);
    data = (unsigned long *) malloc (cols * (rows+winSize-1) * sizeof(long));

    if (result)             /* Free allocated memory first */
      free (result);
    result = (unsigned long *) malloc (RESULTCOLS * rows * sizeof (long));

    if (dist)             /* Free allocated memory first */
      free (dist);
    dist = (double *) malloc (elements * sizeof (double));

/*
 * +++ Calculate gaussian weighted distance mask
 */
    /* Set factor so that contribution of most distant point is 0.001 */
    quotient = log (0.001) / pow (sqrt (middle * middle * 2)-1, 2);
    DIST (0,0) = 1;
    for (x=1; x <= middle; x++)
      for (y=0; y <= x; y++) {
        weight = exp (pow (sqrt(x*x+y*y)-1,2) * quotient);
        DIST ( x, y) = weight;    DIST ( y, x) = weight;
        DIST ( x,-y) = weight;    DIST (-y, x) = weight;
        DIST (-x, y) = weight;    DIST ( y,-x) = weight;
        DIST (-x,-y) = weight;    DIST (-y,-x) = weight;
      } /* end for x */

/*
 * Calculate quotient for difference weighted values
 */
    quotient = -1 / (sigmaDiff * sigmaDiff * 2);
  } /* end if */

/*
 * +++ Receive the image data
 */
  pvm_upkulong (data, cols * (rows + winSize - 1), 1);
      
/******************************************************************************
 * +++ C A L C U L A T I O N S   O N   D A T A 
 ******************************************************************************
 */
/*   
 *  Loop over image   
 */
  for ( y = middle; y < rows + middle; y++ )
    for ( x = middle; x < cols - middle; x++ ) {
/*
 * Initialize the histogram
 */
      for (i=0; i<256; i++)
        hist [i] = 0;
/* 
 *  Loop over window; make a histogram of differences
 */
      for ( wy = y-middle; wy < ( y + middle + 1); wy++ )
        for ( wx = x-middle; wx < ( x + middle + 1); wx++ )
          hist [abs(DATA (wx, wy) - DATA (x, y))] += DIST (wx-x, wy-y);

/*
 * Multiply histogram values with their weighted difference
 */
      for (i=0; i<256; i++)
        hist [i] *= exp (i*i*quotient);

/*
 * Integrate the histogram
 */
      sum = 0;
      for (i=0; i<256; i++)
        sum += hist [i];

      RESULT (x-middle, y-middle) = sum * 1000000.0;
    } /* end for x */

  pvm_initsend (PvmDataDefault);
/******************************************************************************
 * S E N D   R E S U L T   D A T A 
 ******************************************************************************
 * +++ Pack result data and send to jobmanager (sending done by main() )
 */
  pvm_pkulong (result, RESULTCOLS * rows, 1);

} /* end pipsProcPar */


/* end sumHistogram.c */
