/*
   XGFontInfo

   NSFont helper for GNUstep GUI X/GPS Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Scott Christley
   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: February 1997
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May, October 1998
   Author:  Michael Hanni <mhanni@sprintmail.com>
   Date: August 1998

   This file is part of the GNUstep GUI X/GPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <gnustep/xgps/XGContext.h>
#include <gnustep/xgps/XGContextPrivate.h>
#include <gnustep/xgps/XGGState.h>
#include <Foundation/NSDebug.h>

/*
 * class global dictionary of existing fonts
 */
static NSMutableDictionary	*_globalFontDictionary = nil;

@implementation XGFontInfo

- (XFontStruct*) xFontStruct
{
  return font_info;
}

- initWithFontName: (NSString*)name matrix: (const float *)fmatrix
{
  Display	*xdpy = [XGContext currentXDisplay];
  XGFontInfo	*fontinfo;
  NSString	*xfontname;

  /*
   * Retrieve the XLFD matching the given fontName. DPS->X.
   */
  xfontname = XGXFontName(name, fmatrix[0]);

  /*
   * Do we have an already contructed XGFontInfo for this request? If yes we
   * return it.
   */
  if (_globalFontDictionary == nil)
    _globalFontDictionary = [NSMutableDictionary new];
  if ((fontinfo = [_globalFontDictionary objectForKey: xfontname]))
    {
      RELEASE(self);
      return RETAIN(fontinfo);
    }

  // Now we create an XGFontInfo to add to the original fontDict.
  [super init];
  ASSIGN(fontName, name);
  [_globalFontDictionary setObject: self forKey: xfontname];

  if (!xdpy)
    return self;

  // Load X font and get font info structure.
  if ((font_info = XLoadQueryFont(xdpy, [xfontname cString])) == NULL)
    {
      NSLog(@"Selected font: %@ is not available.\n"
	    @"Using system fixed font instead", xfontname);

      // Shouldn't get here, but still.
      // try default font
      if ((font_info = XLoadQueryFont(xdpy, "9x15")) == NULL)
        {
	  NSLog(@"Unable to open fixed font");
	  return self;
	}
    }
  else
    NSDebugLog(@"Loaded font: %@", xfontname);

  ASSIGN(familyName, XGFontFamily(xdpy, font_info));
  isFixedPitch = XGFontIsFixedPitch(xdpy, font_info);
  isBaseFont = NO;
  ascender = font_info->max_bounds.ascent;
  descender = -(font_info->max_bounds.descent);
  fontBBox = NSMakeRect(
    (float)(0 + font_info->min_bounds.lbearing),
    (float)(0 - font_info->max_bounds.ascent),
    (float)(font_info->max_bounds.rbearing - font_info->min_bounds.lbearing),
    (float)(font_info->max_bounds.ascent + font_info->max_bounds.descent));
  maximumAdvancement = NSMakeSize(font_info->max_bounds.width,
    (font_info->max_bounds.ascent +font_info->max_bounds.descent));
  minimumAdvancement = NSMakeSize(0,0);

  // FIXME: italicAngle, weight, underlinePosition, underlineThickness 
  // capHeight, xHeight encodingScheme are not set
  
  // Fill the afmDitionary
  [fontDictionary setObject: fontName forKey: NSAFMFontName];
  [fontDictionary setObject: familyName forKey: NSAFMFamilyName];

  return self;
}

- (void) dealloc
{
  if (font_info != NULL)
    XUnloadFont([XGContext currentXDisplay], font_info->fid);
  [super dealloc];
}

- (void) set
{
  XGContext *ctxt = GSCurrentContext();
  XGGState *current = [ctxt xrCurrentGState];
  [current setFontStruct: font_info];
}

- (float) widthOfString: (NSString*)string
{
  return XTextWidth(font_info, [string cString], [string length]);
}

- (NSMultibyteGlyphPacking)glyphPacking
{
  if (font_info->min_char_or_byte2 == 0 && 
      font_info->max_char_or_byte2 == 0)
    return NSOneByteGlyphPacking;
  else 
    return NSTwoByteGlyphPacking;
}

static
XCharStruct *
XGXCharStructForGlyph(XFontStruct *fontinfo, NSGlyph glyph)
{
  unsigned min1 = fontinfo->min_byte1;
  unsigned max1 = fontinfo->max_byte1;
  unsigned min2 = fontinfo->min_char_or_byte2;
  unsigned max2 = fontinfo->max_char_or_byte2;
  // if per_char is NULL assume max bounds
  XCharStruct *pc = NULL;

  if (fontinfo->per_char)
    {
      if (min1 == 0 && max1 == 0)
        {
	  if (glyph >= min2 && glyph <= max2)
	    pc = &(fontinfo->per_char[glyph - min2]);
        }
      else 
        {
	  unsigned b1 = glyph >> 8;
	  unsigned b2 = glyph & 255;
  
	  if (b1 >= min1 && b1 <= max1 && b2 >= min2 && b2 <= max2)
	    pc = &(fontinfo->per_char[(b1 - min1) * (max2 - min2 + 1) + 
				     b2 - min2]);
        }
    }
  
  return pc;
}

// glyph is an ascii char value
- (NSSize) advancementForGlyph: (NSGlyph)glyph
{
  XCharStruct *pc = XGXCharStructForGlyph(font_info, glyph);

  if (!pc)
    pc = &(font_info->max_bounds);

  return NSMakeSize((float)pc->width, 0);
}

- (NSRect) boundingRectForGlyph: (NSGlyph)glyph
{
  XCharStruct *pc = XGXCharStructForGlyph(font_info, glyph);

  if (!pc)
    return fontBBox;

  return NSMakeRect((float)pc->lbearing, (float)-pc->descent, 
		    (float)(pc->rbearing - pc->lbearing), 
		    (float)(pc->ascent + pc->descent));
}

- (BOOL) glyphIsEncoded: (NSGlyph)glyph
{
  XCharStruct *pc = XGXCharStructForGlyph(font_info, glyph);

  return (pc != NULL);
}

- (NSGlyph) glyphWithName: (NSString*)glyphName
{
  KeySym k = XStringToKeysym([glyphName cString]);

  if (k == NoSymbol)
    return 0;
  else
    return (NSGlyph)k;
}

- (NSPoint) positionOfGlyph: (NSGlyph)curGlyph
	    precededByGlyph: (NSGlyph)prevGlyph
		  isNominal: (BOOL*)nominal
{
  XCharStruct *pc1;
  XCharStruct *pc2;

  if (nominal)
    *nominal = YES;

  if (curGlyph == NSControlGlyph || prevGlyph == NSControlGlyph)
    return NSZeroPoint;

  if (curGlyph == NSNullGlyph)
    {
      NSSize advance = [self advancementForGlyph: prevGlyph];
      return NSMakePoint(advance.width, advance.height);
    }

  pc1 = XGXCharStructForGlyph(font_info, curGlyph);
  pc2 = XGXCharStructForGlyph(font_info, prevGlyph);

  if (!pc1 || !pc2)
    return NSZeroPoint;
    
  // FIXME: We use no kerning
  return NSMakePoint((float)pc2->width, 0);
}

- (float) pointSize
{
  return XGFontPointSize([XGContext currentXDisplay], font_info);
}

- (NSFontTraitMask) traits
{
  Display		*xdpy = [XGContext currentXDisplay];
  return XGTraitsOfFont(xdpy, font_info);
}

- (int) weight
{
  Display	*xdpy = [XGContext currentXDisplay];
  return XGWeightOfFont(xdpy,  font_info);
}

@end
