/* XGGState - Implements graphic state drawing for Xlib

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Adam Fedor <fedor@gnu.org>
   Date: Nov 1998
   
   This file is part of the GNU Objective C User Interface Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
   */

#include "gnustep/xgps/XGContext.h"
#include "gnustep/xgps/XGGState.h"
#include "gnustep/xgps/XGContextPrivate.h"
#include "gnustep/xgps/XGContextWindow.h"
#include "gnustep/xgps/XGDrawObject.h"
#include "XGDrawingEngine.h"
#include "SharedX/xrtools.h"
#include <base/fast.x>

#define XDPY (context->dpy)
#define XSCR (context->screen_number)


typedef struct _DrawObject_t {
  ctxt_object_t type;
  NSPoint point;
  NSZone *zone;
  unsigned retained;
} DrawObject_t;

static void FreeDraw(DrawObject_t *o)
{
  if (o->retained)
    o->retained--;
  else
    NSZoneFree(o->zone, o);
}

static void *RetainDraw(DrawObject_t *o)
{
  o->retained++;
  return o;
}

#define GSI_ARRAY_TYPES       0
#define GSI_ARRAY_EXTRA       DrawObject_t*

#define GSI_ARRAY_RELEASE(X)   FreeDraw(((X).ext))
#define GSI_ARRAY_RETAIN(X)    RetainDraw(((X).ext))

#ifdef GSIArray
#undef GSIArray
#endif
#include <base/GSIArray.h>

#define CHECK_GC \
  if (!xgcntxt) \
    [self createGraphicContext]

#define COPY_GC_ON_CHANGE \
  CHECK_GC; \
  if (sharedGC == YES) \
    [self copyGraphicContext]

#define CHECK_PATH \
  if (!path) \
    { \
      path = NSZoneMalloc(fastZone(self), sizeof(GSIArray_t)); \
      GSIArrayInitWithZoneAndCapacity((GSIArray)path, fastZone(self), 8); \
    }

#define PUSH_PATH_OBJECT(_type) \
  do { \
    DrawObject_t *o; \
    o = (DrawObject_t*)NSZoneMalloc(fastZone(self), sizeof(DrawObject_t)); \
    o->type = _type; \
    o->zone = fastZone(self); \
    o->retained = 0; \
    GSIArrayAddItem((GSIArray)path, (GSIArrayItem)o); \
  } while (0)

#define PUSH_PATH_OBJECT_POINT(_type, _point) \
  do { \
    DrawObject_t *o; \
    o = (DrawObject_t*)NSZoneMalloc(fastZone(self), sizeof(DrawObject_t)); \
    o->type = _type; \
    o->point = _point; \
    o->zone = fastZone(self); \
    o->retained = 0; \
    GSIArrayAddItem((GSIArray)path, (GSIArrayItem)o); \
  } while (0)

#define GST_CURRENTPOINT(_point) \
  if (_point.x < 0) \
    NSLog(@"DPS (xgps): No current point\n");

#define GS_NEW_ALPHA_BUFFER(context, depth, width, height) \
    RCreateXImage(context, depth, width, height)

#define AINDEX 5

@interface XGGState (Private)
- (void) _paintPath;
- (void) setClipMask;
- createGraphicContext;
- copyGraphicContext;
@end

#define	floor

/*
 *	Inline functions to convert from OpenStep view coordinates or
 *	OpenStep window coordinates to X window coordinates.
 */
static inline XPoint XGWindowPointToX(XGGState *s, NSPoint p)
{
  XPoint newPoint;

  newPoint.x = floor(p.x) + s->offset.x;

  newPoint.y = floor(s->offset.y - p.y);

  return newPoint;
}

static inline XRectangle XGWindowRectToX(XGGState *s, NSRect r)
{
  XRectangle newRect;

  newRect.x = floor(r.origin.x + s->offset.x);
  newRect.width = floor(r.origin.x + r.size.width + s->offset.x) - newRect.x;

  newRect.y = floor(s->offset.y - r.origin.y - r.size.height);
  newRect.height = floor(s->offset.y - r.origin.y) - newRect.y;

  return newRect;
}

static inline XPoint XGViewPointToX(XGGState *s, NSPoint p)
{
  p = [s->ctm pointInMatrixSpace: p];
  return XGWindowPointToX(s, p);
}

static inline XRectangle XGViewRectToX(XGGState *s, NSRect r)
{
  r = [s->ctm rectInMatrixSpace: r];
  return XGWindowRectToX(s, r);
}

@implementation XGGState

static	Region	emptyRegion;

+ (void) initialize
{
  static BOOL	beenHere = NO;

  if (beenHere == NO)
    {
      XPoint	pts[5];

      beenHere = YES;
      pts[0].x = 0; pts[0].y = 0;
      pts[1].x = 0; pts[1].y = 0;
      pts[2].x = 0; pts[2].y = 0;
      pts[3].x = 0; pts[3].y = 0;
      pts[4].x = 0; pts[4].y = 0;
      emptyRegion = XPolygonRegion(pts, 5, WindingRule);
      NSAssert(XEmptyRegion(emptyRegion), NSInternalInconsistencyException);
    }
}

/* Designated initializer. */
- initWithDrawContext: (XGContext *)drawContext
{
  [super init];

  xgcontext = drawContext;
  context = [drawContext xrContext];
  ctm = [[NSAffineTransform allocWithZone: fastZone(self)] init];
  path = NULL;
  point.x = -1;
  offset = NSMakePoint(0, 0);
  draw = 0;
  xfont = NULL;
  color.field[AINDEX] = 1.0;
  xgcntxt = None;
  return self;
}

- (void) dealloc
{
  if ( sharedGC == NO && xgcntxt ) 
    {
      XFreeGC(XDPY, xgcntxt);
    }
  if (clipregion)
    XDestroyRegion(clipregion);
  OBJC_FREE(xfont);
  if (path)
    {
      GSIArrayEmpty((GSIArray)path);
      NSZoneFree(fastZone(self), path);
    }
  [ctm release];
  [super dealloc];
}

- deepen
{
  Region new_region;

  /* Copy the GC */
  if (draw != 0)
    [self copyGraphicContext];

  if (path)
    path = GSIArrayCopyWithZone(path, fastZone(self));

  ctm = [ctm copyWithZone: fastZone(self)];

  if (xfont)
    {
      XFontStruct *new_font;
      OBJC_MALLOC(new_font, XFontStruct, 1); 
      memcpy(new_font, xfont, sizeof(XFontStruct));
      xfont = new_font;
    }

  /* Copy the clipregion */
  if (clipregion)
    {
      new_region = XCreateRegion();
      XIntersectRegion(clipregion, clipregion, new_region);
      clipregion = new_region;
    }
  return self;
}

- copyWithZone: (NSZone *)zone
{
  return [(XGGState *)NSCopyObject(self, 0, fastZone(self)) deepen];
}

- (void) setWindow: (int)number;
{
  window = number;
}

- (void) setDrawable: (Drawable)theDrawable;
{
  draw = theDrawable;
}

- (void) setGraphicContext: (GC)xGraphicContext
{
  GC source;
  unsigned long	mask;
  BOOL old_shared;

  source = xgcntxt;
  old_shared = sharedGC;
  if (xGraphicContext == None)
    return;
  if (xGraphicContext == xgcntxt)
    return;
  
  xgcntxt = xGraphicContext;
  sharedGC = YES;		/* Not sure if we really own the GC */
  /* Update the GC to reflect our settings */
  if (source == None)
    return;
  mask = GCForeground | GCFont | GCFunction | GCFillRule
    | GCCapStyle | GCJoinStyle | GCLineWidth;
  XCopyGC(XDPY, source, mask, xgcntxt); 

  if (source != None && old_shared == NO)
    XFreeGC(XDPY, source);
}

/* Set various characteristics of the graphic context */
- (void) setGCValues: (XGCValues)values withMask: (int)mask
{
  COPY_GC_ON_CHANGE;
  XChangeGC(XDPY, xgcntxt, mask, &values);
}

/* Set the GC clipmask.  The clipmask needs to be a combination of the
   current clipregion and the viewclipregion (which is part of the 
   DrawContext). */
- (void) setClipMask
{
  Region region;

  COPY_GC_ON_CHANGE;
  if (!clipregion && !context->viewclip)
    {
      XSetClipMask(XDPY, xgcntxt, None);
      return;
    }

  if (clipregion && context->viewclip)
    {
      region = XCreateRegion();
      XIntersectRegion(clipregion, context->viewclip, region);
    }
  else if (!context->viewclip)
    region = clipregion;
  else 
    region = context->viewclip;

  XSetRegion(XDPY, xgcntxt, region);
  if (clipregion && context->viewclip)
    XDestroyRegion(region);
}

- (void) setColor: (xr_device_color_t)acolor;
{
  float alpha = color.field[AINDEX];
  color = acolor;
  color.field[AINDEX] = alpha;
  gcv.foreground = xrColorToPixel(context, color);
  [self setGCValues: gcv withMask: GCForeground];
}

- (void) setFontStruct: (XFontStruct *)fontstruct
{
  if (xfont == NULL || xfont->fid != fontstruct->fid)
    {
      OBJC_FREE(xfont);
      OBJC_MALLOC(xfont, XFontStruct, 1); 
      memcpy(xfont, fontstruct, sizeof(XFontStruct));
      gcv.font = xfont->fid;
      [self setGCValues: gcv withMask: GCFont];
    }
}

- (void) setOffset: (NSPoint)theOffset
{
  offset = theOffset;
}

- (NSPoint) offset
{
  return offset;
}

- copyGraphicContext
{
  GC source;
  unsigned long	mask;
  
  if (draw == 0)
    DPS_ERROR(DPSinvalidid, @"Creating a GC with no Drawable defined");

  source = xgcntxt;
  mask = 0xffffffff; /* Copy everything (Hopefully) */
  xgcntxt = XCreateGC(XDPY, draw, 0, NULL);
  XCopyGC(XDPY, source, mask, xgcntxt); 
  sharedGC = NO;
  return self;
}

// Create a default graphics context.
- createGraphicContext
{
  if (draw == 0)
    DPS_ERROR(DPSinvalidid, @"Creating a GC with no Drawable defined");
  gcv.function = GXcopy;
  gcv.background = context->white;
  gcv.foreground = context->black;
  gcv.plane_mask = AllPlanes;
  gcv.fill_style = FillSolid;
  gcv.fill_rule  = WindingRule;
  xgcntxt = XCreateGC(XDPY, draw,
		      GCFunction | GCForeground | GCBackground | GCPlaneMask 
		      | GCFillStyle | GCFillRule,
		      &gcv);
  [self setClipMask];
  sharedGC = NO;
  return self;
}

- (NSRect)clipRect
{
  XRectangle r;
  r.width = 0; r.height = 0;
  if (clipregion)
    XClipBox(clipregion, &r);
  return NSMakeRect(r.x, r.y, r.width-1, r.height-1);
}

- (BOOL) hasGraphicContext
{
  return (xgcntxt) ? YES : NO;
}

- (BOOL) hasDrawable
{
  return (draw ? YES : NO);
}

- (int) window
{
  return window;
}

- (Drawable) drawable
{
  return draw;
}

- (GC) graphicContext
{
  return xgcntxt;
}

- (void) copyBits: (XGGState*)source fromRect: (NSRect)aRect 
				      toPoint: (NSPoint)aPoint
{
  XRectangle	dst;
  XRectangle    src;
  NSRect	flushRect;
  Drawable	from;

  flushRect.size = aRect.size;
  flushRect.origin = aPoint;

  CHECK_GC;
  if (draw == 0)
    DPS_ERROR(DPSinvalidid, @"No Drawable defined");
  from = source->draw;
  if (from == 0)
    DPS_ERROR(DPSinvalidid, @"No source Drawable defined");

  src = XGViewRectToX(source, aRect);
  dst = XGViewRectToX(self, flushRect);
  XCopyArea(XDPY, from, draw, xgcntxt,
                src.x, src.y, src.width, src.height, dst.x, dst.y);
}

- (void) _compositeGState: (XGGState *)source 
                fromRect: (NSRect)aRect
                 toPoint: (NSPoint)aPoint
                      op: (NSCompositingOperation)op
{
  XRectangle sr, dr;
  NSRect flushRect;
  NSRect drect;
  RXImage *source_im, *dest_im;
  gswindow_device_t *source_win, *dest_win;

  if (!source)
    source = self;

  /* Get the image from the source gstate */
  source_win = [XGContext _windowWithTag: source->window];
  if (!source_win)
    DPS_ERROR(DPSinvalidid, @"Invalid composite source gstate");
  if (source_win->buffer == 0 && source_win->map_state != IsViewable)
    {
      /* Can't get pixel information from a window that isn't mapped */
      DPS_ERROR(DPSinvalidaccess, @"Invalid gstate buffer");
    }

  sr = XGViewRectToX(source, aRect);
  source_im = RGetXImage(context, 
		 (source_win->buffer) ? source_win->buffer : source_win->ident,
		 sr.x, sr.y, sr.width, sr.height);

  /* Get our drawable info */
  dest_win = [XGContext _windowWithTag: window];
  if (!dest_win)
    DPS_ERROR(DPSinvalidid, @"Invalid composite gstate");

  /*
   * Find the current view clipregion and see if the image is inside it.
   */
  flushRect.size = aRect.size;
  flushRect.origin = aPoint;
  dr = XGViewRectToX(self, flushRect);

  /* Intersect the rects to find what part of the image we need to get */
  drect = NSMakeRect(dr.x, dr.y, dr.width, dr.height);
  if (NSIsEmptyRect(drect))
    return;
  
  if (dest_win->buffer == 0 && dest_win->map_state != IsViewable)
    {
      /* Why bother drawing? */
      return;
    }
  else
    dest_im = RGetXImage(context, draw, NSMinX(drect), NSMinY(drect), 
			 NSWidth(drect), NSHeight(drect));

  /* Force creation of our alpha buffer */
  if (dest_win->alpha_buffer == NULL)
    dest_win->alpha_buffer = GS_NEW_ALPHA_BUFFER(context, dest_win->depth,
					     NSWidth(dest_win->xframe),
					     NSHeight(dest_win->xframe));
  /* Composite it */
  _pixmap_combine_alpha(context, source_im, source_win->alpha_buffer, 
			dest_im, dest_win->alpha_buffer, sr.x, sr.y, drect,
			op, [xgcontext drawMechanism]);
  /* Draw into the window/buffer */
  RPutXImage(context, draw, xgcntxt, dest_im, 0, 0, NSMinX(drect), 
  	    NSMinY(drect), NSWidth(drect), NSHeight(drect));
  RDestroyXImage(context, dest_im);
  RDestroyXImage(context, source_im);
}

- (void) compositeGState: (XGGState *)source 
                fromRect: (NSRect)aRect
                 toPoint: (NSPoint)aPoint
                      op: (NSCompositingOperation)op
{
  BOOL do_copy, source_alpha, dest_alpha;
  XGCValues comp_gcv;

  if (!source)
    source = self;

  /* If we have no drawable, we can't proceed. */
  if (draw == 0)
    DPS_ERROR(DPSinvalidid, @"No Drawable defined");

  /* Check alpha */
#define CHECK_ALPHA							\
  do {									\
    gswindow_device_t *source_win, *dest_win;				\
    source_win = [XGContext _windowWithTag: [source window]];	\
    source_alpha = (source_win && source_win->alpha_buffer);		\
    dest_win = [XGContext _windowWithTag: [self window]];	\
    dest_alpha = (dest_win && dest_win->alpha_buffer);			\
  } while (0)

  do_copy = NO;
  switch (op)
    {
    case   NSCompositeClear:
      do_copy = YES;
      comp_gcv.function = GXclear;
      break;
    case   NSCompositeCopy:
      do_copy = YES;
      comp_gcv.function = GXcopy;
      break;
    case   NSCompositeSourceOver:
      CHECK_ALPHA;
      if (source_alpha == NO)
	do_copy = YES;
      else
	do_copy = NO;
      comp_gcv.function = GXcopy;
      break;
    case   NSCompositeSourceIn:
      CHECK_ALPHA;
      if (source_alpha == NO && dest_alpha == NO)
	do_copy = YES;
      else
	do_copy = NO;
      comp_gcv.function = GXcopy;
      break;
    case   NSCompositeSourceOut:
      do_copy = NO;
      comp_gcv.function = GXcopy;
      break;
    case   NSCompositeSourceAtop:
      CHECK_ALPHA;
      if (source_alpha == NO && dest_alpha == NO)
	do_copy = YES;
      else
	do_copy = NO;
      comp_gcv.function = GXcopy;
      break;
    case   NSCompositeDestinationOver:
      CHECK_ALPHA;
      if (dest_alpha == NO)
	return;
      else
	do_copy = NO;
      comp_gcv.function = GXcopy;
      break;
    case   NSCompositeDestinationIn:
      CHECK_ALPHA;
      if (source_alpha == NO && dest_alpha == NO)
	return;
      else
	do_copy = NO;
      comp_gcv.function = GXcopy;
      break;
    case   NSCompositeDestinationOut:
      do_copy = NO;
      comp_gcv.function = GXcopy;
      break;
    case   NSCompositeDestinationAtop:
      CHECK_ALPHA;
      if (source_alpha == NO && dest_alpha == NO)
	return;
      else
	do_copy = NO;
      comp_gcv.function = GXcopy;
      break;
    case   NSCompositeXOR:
      do_copy = NO;
      comp_gcv.function = GXxor;
      break;
    case   NSCompositePlusDarker:
      do_copy = NO;
      comp_gcv.function = GXcopy;
      break;
    case   NSCompositeHighlight:
      do_copy = NO;
      comp_gcv.function = GXxor;
      break;
    case   NSCompositePlusLighter:
      do_copy = NO;
      comp_gcv.function = GXcopy;
      break;
    }

  if (comp_gcv.function != GXcopy)
    [self setGCValues: comp_gcv withMask: GCFunction];

  if (do_copy)
    [self copyBits: source fromRect: aRect toPoint: aPoint];
  else
    [self _compositeGState: source 
                fromRect: aRect
                 toPoint: aPoint
                      op: op];

  if (comp_gcv.function != GXcopy)
    {
      comp_gcv.function = GXcopy;
      [self setGCValues: comp_gcv withMask: GCFunction];
    }
}

- (void)copyImage: (RXImage *)image ofSize: (NSSize)size
{
  NSRect	r;
  XRectangle	x;

  CHECK_GC;
  if (draw == 0)
    DPS_ERROR(DPSinvalidid, @"No Drawable defined");
  r.origin = point;
  r.size = size;
  x = XGViewRectToX(self, r);
  RPutXImage(context, draw, xgcntxt,
			image, 	0, 0,
			x.x, x.y,
			x.width, x.height);
}

- (NSPoint) pointInMatrixSpace: (NSPoint)aPoint
{
  return [ctm pointInMatrixSpace: aPoint];
}

- (NSRect) rectInMatrixSpace: (NSRect)rect
{
  rect = [ctm rectInMatrixSpace: rect];
  return rect;
}

/* Paint the current path using Xlib calls. All coordinates should already
   have been transformed to device coordinates (save for the final flip
   needed because the X origin is in the top-left). Filling and clipping
   may not work correctly because we don't really flatten curves (although
   I have code for this - maybe I should add it) - FIXME */
- (void) _doPath: (XPoint*)pts : (int)count draw: (ctxt_object_t)type
{
  int fill_rule;

  COPY_GC_ON_CHANGE;
  if (draw == 0)
    DPS_ERROR(DPSinvalidid, @"No Drawable defined");
  fill_rule = WindingRule;
  switch (type)
    {
    case path_stroke:
      XDrawLines(XDPY, draw, xgcntxt, pts, count, CoordModeOrigin);
      break;
    case path_eofill:
      gcv.fill_rule = EvenOddRule;
      [self setGCValues: gcv withMask: GCFillRule];
      /* NO BREAK */
    case path_fill:
      XFillPolygon(XDPY, draw, xgcntxt, pts, count, Complex, 
		   CoordModeOrigin);
      
      if (gcv.fill_rule == EvenOddRule)
	{
	  gcv.fill_rule = WindingRule;
	  [self setGCValues: gcv withMask: GCFillRule];
	}
      break;
    case path_eoclip:
      fill_rule = EvenOddRule;
      /* NO BREAK */
    case path_clip:
      {
	Region region, new_region;
	region = XPolygonRegion(pts, count, fill_rule);
	if (clipregion)
	  {
	    XIntersectRegion(clipregion, region, new_region);
	    XDestroyRegion(region);
	    XDestroyRegion(clipregion);
	  } else
	    new_region = region;
	clipregion = new_region;
	[self setClipMask];
      }
      break;
    case path_eoviewclip:
      fill_rule = EvenOddRule;
      /* NO BREAK */
    case path_viewclip:
      {
	Region region;
	region = XPolygonRegion(pts, count, fill_rule);
	if (context->viewclip)
	  XDestroyRegion(context->viewclip);
	context->viewclip = region;
	[self setClipMask];
      }
      break;
    default:
      break;
    }
}

- (void) _paintPath
{
  ctxt_object_t drawType;
  unsigned	count;
  DrawObject_t  *o;

  o = (GSIArrayLastItem((GSIArray)path)).ext;
  drawType = o->type;
  GSIArrayRemoveLastItem((GSIArray)path);

  count = GSIArrayCount((GSIArray)path);
  if (count)
    {
      XPoint	pts[count];
      unsigned	i = 0;
      unsigned	pos = 0;
      BOOL	start_path_set = NO;
      NSPoint	start_path = NSMakePoint(0,0);

      while (pos < count)
	{
	  NSPoint	p, new;
	  BOOL		doit;
	  
	  doit = NO;
	  o = (GSIArrayItemAtIndex((GSIArray)path, pos++)).ext;
	  p = o->point;
	  switch (o->type)
	    {
	      case path_moveto:
		if (i > 1)
		  {
		    [self _doPath: pts : i draw: drawType];
		  }
		i = 0;
		new = p;
		if (!start_path_set)
		  {
		    start_path = new;
		    start_path_set = YES;
		  }
		break;

	      case path_rmoveto:
		if (i > 1)
		  {
		    [self _doPath: pts : i draw: drawType];
		  }
		i = 0;
		GST_CURRENTPOINT(point);
		new.x += p.x; new.y += p.y;
		if (!start_path_set)
		  start_path_set = YES;
		break;

	      case path_lineto:
		GST_CURRENTPOINT(point);
		new = p;
		if (!start_path_set)
		  start_path_set = YES;
		break;

	      case path_rlineto:
		GST_CURRENTPOINT(point);
		new.x += p.x; new.y += p.y;
		if (!start_path_set)
		  start_path_set = YES;
		break;

	      case path_arc:
		break;

	      case path_curveto:
		break;

	      case path_charpath:
		break;

	      case path_closepath:
		new = start_path;
		doit = YES;
		break;

	      default:
		break;
	    }
	  point = new;
	  pts[i] = XGViewPointToX(self, new);
	  i++;
	  if (doit && i > 1) 
	    {
	      [self _doPath: pts : i draw: drawType];
	      i = 0;
	    }
	} /* while */

      if (i > 1) 
	{
	  [self _doPath: pts : i draw: drawType];
	}
    }

  /*
   * clip does not delete the current path, so we only clear the path if the
   * operation was not a clipping operation.
   */
  if ((drawType != path_clip) && (drawType != path_eoclip))
    GSIArrayRemoveAllItems((GSIArray)path);
}

- (XPoint) viewPointToX: (NSPoint)aPoint
{
  return XGViewPointToX(self, aPoint);
}

- (XRectangle) viewRectToX: (NSRect)aRect
{
  return XGViewRectToX(self, aRect);
}

- (XPoint) windowPointToX: (NSPoint)aPoint
{
  return XGWindowPointToX(self, aPoint);
}

- (XRectangle) windowRectToX: (NSRect)aRect
{
  return XGWindowRectToX(self, aRect);
}

@end

@implementation XGGState (ColorOps)

- (void)DPScolorimage 
{
}

- (void)DPScurrentblackgeneration 
{
}

- (void)DPScurrentcmykcolor: (float *)c : (float *)m : (float *)y : (float *)k 
{
  xr_device_color_t new = color;
  if (new.space != cmyk_colorspace)
    new = xrConvertToCMYK(new);
  *c = new.field[0];
  *m = new.field[1];
  *y = new.field[2];
  *k = new.field[3];
}

- (void)DPScurrentcolorscreen 
{
}

- (void)DPScurrentcolortransfer 
{
}

- (void)DPScurrentundercolorremoval 
{
}

- (void)DPSsetblackgeneration 
{
}

- (void)DPSsetcmykcolor: (float)c : (float)m : (float)y : (float)k 
{
  color.space = cmyk_colorspace;
  color.field[0] = c;
  color.field[1] = m;
  color.field[2] = y;
  color.field[3] = k;
  [self setColor:color];
}

- (void)DPSsetcolorscreen 
{
}

- (void)DPSsetcolortransfer 
{
}

- (void)DPSsetundercolorremoval 
{
}

@end

@implementation XGGState (GStateOps)

- (void)DPSconcat: (const float *)m
{
  [ctm concatenateWith: [NSAffineTransform matrixFrom: m]];
}

- (void)DPScurrentdash 
{
}

- (void)DPScurrentflat: (float *)flatness 
{
}

- (void)DPScurrentgray: (float *)gray 
{
  xr_device_color_t gcolor;
  gcolor = xrConvertToGray(color);
  *gray = gcolor.field[0];
}

- (void)DPScurrenthalftone 
{
}

- (void)DPScurrenthalftonephase: (float *)x : (float *)y 
{
}

- (void)DPScurrenthsbcolor: (float *)h : (float *)s : (float *)b 
{
  xr_device_color_t gcolor;
  gcolor = xrConvertToHSB(color);
  *h = gcolor.field[0]; *s = gcolor.field[1]; *b = gcolor.field[2];
}

- (void)DPScurrentlinecap: (int *)linecap 
{
  *linecap = gcv.cap_style - CapButt;
}

- (void)DPScurrentlinejoin: (int *)linejoin 
{
  *linejoin = gcv.join_style - JoinMiter;
}

- (void)DPScurrentlinewidth: (float *)width 
{
  *width = gcv.line_width;
}

- (void)DPScurrentmatrix: (float *)m
{
  [ctm getMatrix: m];
}

- (void)DPScurrentmiterlimit: (float *)limit 
{
}

- (void)DPScurrentpoint: (float *)x : (float *)y 
{
  NSAffineTransform *ictm;
  NSPoint user;

  ictm = [[ctm copyWithZone: fastZone(self)] autorelease];
  [ictm inverse];
  user = [ictm pointInMatrixSpace: point];
  *x = user.x;
  *y = user.y;
}

- (void)DPScurrentrgbcolor: (float *)r : (float *)g : (float *)b 
{
  xr_device_color_t gcolor;
  gcolor = xrConvertToRGB(color);
  *r = gcolor.field[0]; *g = gcolor.field[1]; *b = gcolor.field[2];
}

- (void)DPScurrentscreen 
{
}

- (void)DPScurrentstrokeadjust: (int *)b 
{
}

- (void)DPScurrenttransfer 
{
}

- (void)DPSinitgraphics 
{
  [ctm makeIdentityMatrix];
  point.x = -1;
  if (clipregion)
    XDestroyRegion(clipregion);
  clipregion = 0;
  /* FIXME: reset the GC */
  color.space = gray_colorspace; 
  color.field[0] = 0.0;
  [self setColor: color];
  color.field[AINDEX] = 1.0;
}

- (void)DPSinitmatrix 
{
  [ctm makeIdentityMatrix];
}

- (void)DPSrotate: (float)angle 
{
  [ctm rotateByAngle: angle];
}

- (void)DPSscale: (float)x : (float)y 
{
  [ctm scaleBy: x : y];
}

- (void)DPSsetdash: (const float *)pat : (int)size : (float)offset 
{
}

- (void)DPSsetflat: (float)flatness 
{
}

- (void)DPSsetgray: (float)gray 
{
  color.space = gray_colorspace;
  color.field[0] = gray;
  [self setColor: color];
}

- (void)DPSsethalftone 
{
}

- (void)DPSsethalftonephase: (float)x : (float)y 
{
}

- (void)DPSsethsbcolor: (float)h : (float)s : (float)b 
{
  color.space = hsb_colorspace;
  color.field[0] = h; color.field[1] = s; color.field[2] = b;
  [self setColor: color];
}

- (void)DPSsetlinecap: (int)linecap 
{
  gcv.cap_style = linecap + CapButt;
  [self setGCValues: gcv withMask: GCCapStyle];
}

- (void)DPSsetlinejoin: (int)linejoin 
{
  gcv.join_style = linejoin + JoinMiter;
  [self setGCValues: gcv withMask: GCJoinStyle];
}

- (void)DPSsetlinewidth: (float)width 
{
  int	w;

  /*
   * Evil hack to get drawing to work - with a line thickness of 1, the
   * rectangles we draw seem to lose their bottom right corners irrespective
   * of the join/cap settings - but with a thickness of zero things work.
   */
  if (width < 1.5)
    width = 0.0;

  w = (int)width;
  if (gcv.line_width != w)
    {
      gcv.line_width = w;
      [self setGCValues: gcv withMask: GCLineWidth];
    }
}

- (void)DPSsetmatrix: (const float *)m;
{
  [ctm setMatrix: m];
}

- (void)DPSsetmiterlimit: (float)limit 
{
}

- (void)DPSsetrgbcolor: (float)r : (float)g : (float)b 
{
  color.space = rgb_colorspace;
  color.field[0] = r; color.field[1] = g; color.field[2] = b;
  [self setColor: color];
}

- (void)DPSsetscreen 
{
}

- (void)DPSsetstrokeadjust: (int)b 
{
}

- (void)DPSsettransfer 
{
}

- (void)DPStranslate: (float)x : (float)y 
{
  NSPoint p;
  p.x = x;
  p.y = y; 
  [ctm translateToPoint: p];
}

@end

@implementation XGGState (PaintOps)

typedef enum {
  show_delta, show_array_x, show_array_y, show_array_xy
} show_array_t;

/* Omnibus show string routine that combines that characteristics of
   ashow, awidthshow, widthshow, xshow, xyshow, and yshow */
- (void) _showString: (const char *)s
	    xCharAdj: (float)cx
	    yCharAdj: (float)cy
		char: (char)c
	    adjArray: (const float *)arr
	     arrType: (show_array_t)type
	  isRelative: (BOOL)relative;
{
  int i;
  int len;
  int width;
  NSAffineTransform *trans;
  NSSize scale;

  /* Transformation should only be delta transformations (no offset) */
  trans = AUTORELEASE([ctm copyWithZone: fastZone(self)]);
  [trans setFrameOrigin: NSMakePoint(0,0)];

  len = strlen(s);
  scale = [trans sizeInMatrixSpace: NSMakeSize(1,1)];
  if (xfont == NULL)
    {
      NSLog(@"DPS (xgps): no font set\n");
      return;
    }
  COPY_GC_ON_CHANGE; 
  if (draw == 0)
    DPS_ERROR(DPSinvalidid, @"No Drawable defined");
  for (i = 0; i < len; i++)
    {
      NSPoint	delta;
      XPoint	xp;

      width = XTextWidth(xfont, s+i, 1);
      xp = XGViewPointToX(self, point);
      XDrawString(XDPY, draw, xgcntxt, xp.x, xp.y, s, 1);

      /* Note we update the current point according to the current 
	 transformation scaling, although the text isn't currently
	 scaled (FIXME). */
      if (type == show_array_xy)
	{
	  delta.x = arr[2*i]; delta.y = arr[2*i+1];
	}
      else if (type == show_array_x)
	{
	  delta.x = arr[i]; delta.y = 0;
	}
      else if (type == show_array_y)
	{
	  delta.x = 0; delta.y = arr[i];
	}
      else
	{
	  delta.x = arr[0]; delta.y = arr[1];
	}
      delta = [trans pointInMatrixSpace: delta];
      if (relative == YES)
	{
	  delta.x += width * scale.width;
	  delta.y += xfont->max_bounds.ascent * scale.height;
	}
      if (c && *(s+i) == c)
	{
	  NSPoint cdelta;
	  cdelta.x = cx; cdelta.y = cy;
	  cdelta = [trans pointInMatrixSpace: cdelta];
	  delta.x += cdelta.x; delta.y += cdelta.y;
	}
      point.x += delta.x;
      if (type != show_delta)
	point.y += delta.y;
    }
}

- (void)DPSashow: (float)x : (float)y : (const char *)s 
{
  float arr[2];

  arr[0] = x; arr[1] = y;
  [self _showString: s
    xCharAdj: 0 yCharAdj: 0 char: 0 adjArray: arr arrType: show_delta
    isRelative: YES];
}

- (void)DPSawidthshow: (float)cx : (float)cy : (int)c : (float)ax : (float)ay : (const char *)s 
{
  float arr[2];

  arr[0] = ax; arr[1] = ay;
  [self _showString: s
    xCharAdj: cx yCharAdj: cy char: c adjArray: arr arrType: show_delta
    isRelative: YES];
}

- (void)DPScopypage 
{
}

- (void)DPSeofill 
{
  CHECK_PATH;
  PUSH_PATH_OBJECT(path_eofill);
  [self _paintPath];
}

- (void)DPSerasepage 
{
}

- (void)DPSfill 
{
  CHECK_PATH;
  PUSH_PATH_OBJECT(path_fill);
  [self _paintPath];
}

- (void)DPSimage: dictionary
{
  id obj;
  BOOL uses_alpha, is_planar;
  NSRect rect;
  XRectangle dr, cr;
  NSRect drect, crect;
  int colors;
  id data;
  const unsigned char *raw_data;
  RXImage *dest_im;
  gswindow_device_t *dest_win;

  rect = NSZeroRect;
  uses_alpha = NO;
  colors = 1; /* Default is grey-scale */
  obj = [dictionary objectForKey: @"Width"];
  if (obj)
    rect.size.width = [obj doubleValue];
  obj = [dictionary objectForKey: @"Height"];
  if (obj)
    rect.size.height = [obj doubleValue];
  obj = [dictionary objectForKey: @"SamplesPerPixel"];
  if (obj)
    colors = [obj intValue];
  obj = [dictionary objectForKey: @"Alpha"];
  if (obj)
    uses_alpha = [obj boolValue];
  obj = [dictionary objectForKey: @"MultipleDataSources"];
  if (obj)
    is_planar = [obj boolValue];
  data = [dictionary objectForKey: @"DataSource"];
  if ([data respondsToSelector: @selector(cString)])
    raw_data = [data cString];
  else
    {
      unsigned char **rdata;
      rdata = (unsigned char **)[data pointerValue];
      raw_data = rdata[0];
    }

  if (is_planar)
    {
      NSLog(@"XGContext (DPSimage): Cannot handle planar images\n");
      return;
    }

  /* Get our drawable info */
  dest_win = [XGContext _windowWithTag: window];
  if (!dest_win)
    DPS_ERROR(DPSinvalidid, @"Invalid image gstate");

  /*
   * Find the current view clipregion and see if the image is inside it.
   */
  XClipBox(context->viewclip, &cr);
  rect.origin = point;
  if (viewIsFlipped)
    rect.origin.y -= rect.size.height;
  dr = [self viewRectToX: rect];

  /* Intersect the rects to find what part of the image we need to get */
  drect = NSMakeRect(dr.x, dr.y, dr.width, dr.height);
  crect = NSMakeRect(cr.x, cr.y, cr.width, cr.height);
  drect = NSIntersectionRect(drect, crect);
  if (NSIsEmptyRect(drect))
    return;

  if (dest_win->buffer == 0 && dest_win->map_state != IsViewable)
    {
      /* Why bother drawing? */
      return;
    }
  else
    dest_im = RGetXImage(context, draw, NSMinX(drect), NSMinY(drect), 
			 NSWidth(drect), NSHeight(drect));
  
  /* Force creation of our alpha buffer */
  if (uses_alpha && dest_win->alpha_buffer == NULL)
    dest_win->alpha_buffer = GS_NEW_ALPHA_BUFFER(context, dest_win->depth,
					     NSWidth(dest_win->xframe),
					     NSHeight(dest_win->xframe));

  /* Composite it */
  _bitmap_combine_alpha(context, raw_data,
			colors, uses_alpha,
			dest_im, dest_win->alpha_buffer, dr.x, dr.y, drect,
			0, [xgcontext drawMechanism]);
  /* Draw into the window/buffer */
  RPutXImage(context, draw, xgcntxt, dest_im, 0, 0, NSMinX(drect), 
  	    NSMinY(drect), NSWidth(drect), NSHeight(drect));
  RDestroyXImage(context, dest_im);
}

- (void)DPSimagemask: dictionary 
{
}

- (void)DPSkshow: (const char *)s 
{
}

- (void)DPSrectfill: (float)x : (float)y : (float)w : (float)h 
{
  XRectangle	bounds;
  
  CHECK_GC;
  if (draw == 0)
    DPS_ERROR(DPSinvalidid, @"No Drawable defined");

  bounds = XGViewRectToX(self, NSMakeRect(x, y, w, h));
  XFillRectangle(XDPY, draw, xgcntxt,
		 bounds.x, bounds.y, bounds.width, bounds.height);

#if 0
  /* FIXME: Need to do this eveverywhere */
  if (color.field[AINDEX] < 1.0)
    {
      /* Fill alpha also */
      int i, j;
      int alpha;
      gswindow_device_t *gs_win;
      gs_win = [XGContext _windowWithTag: window];
      if (!gs_win->alpha_buffer)
	return;

      alpha = (color.field[AINDEX] * 255);
      for (j=0; j < bounds.height; j++)
        for (i=0; i < bounds.width; i++)
   	  XPutPixel(((RXImage *)gs_win->alpha_buffer)->image, i, j, alpha);
    }
#endif
}

- (void)DPSrectstroke: (float)x : (float)y : (float)w : (float)h 
{
  XRectangle	bounds;
  
  CHECK_GC;
  if (draw == 0)
    DPS_ERROR(DPSinvalidid, @"No Drawable defined");

  bounds = XGViewRectToX(self, NSMakeRect(x, y, w, h));
  bounds.width--;
  bounds.height--;
  XDrawRectangle(XDPY, draw, xgcntxt,
		 bounds.x, bounds.y, bounds.width, bounds.height);
}

- (void)DPSshow: (const char *)s 
{
  int len;
  int width;
  NSSize scale;
  XPoint xp;

  len = strlen(s);
  if (xfont == NULL)
    {
      NSLog(@"DPS (xgps): no font set\n");
      return;
    }
  width = XTextWidth(xfont, s, len);
  COPY_GC_ON_CHANGE; 
  if (draw == 0)
    DPS_ERROR(DPSinvalidid, @"No Drawable defined");

  xp = XGViewPointToX(self, point);
  XDrawString(XDPY, draw, xgcntxt, xp.x, xp.y, s, len);

  /* Note we update the current point according to the current 
     transformation scaling, although the text isn't currently
     scaled (FIXME). */
  //scale = [ctm sizeInMatrixSpace: NSMakeSize(1, 1)];
  scale = NSMakeSize(1, 1);
  point.x += width * scale.width;
}

- (void)DPSshowpage 
{
}

- (void)DPSstroke 
{
  CHECK_PATH;
  PUSH_PATH_OBJECT(path_stroke);
  [self _paintPath];
}

- (void)DPSstrokepath 
{
}

- (void)DPSueofill: (const char *)nums : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSufill: (const char *)nums : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSustroke: (const char *)nums : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSustrokepath: (const char *)nums : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSwidthshow: (float)x : (float)y : (int)c : (const char *)s 
{
  float arr[2];

  arr[0] = 0; arr[1] = 0;
  [self _showString: s
    xCharAdj: x yCharAdj: y char: c adjArray: arr arrType: show_delta
    isRelative: YES];
}

- (void)DPSxshow: (const char *)s : (const float *)numarray : (int)size 
{
  [self _showString: s
    xCharAdj: 0 yCharAdj: 0 char: 0 adjArray: numarray arrType: show_array_x
    isRelative: NO];
}

- (void)DPSxyshow: (const char *)s : (const float *)numarray : (int)size 
{
  [self _showString: s
    xCharAdj: 0 yCharAdj: 0 char: 0 adjArray: numarray arrType: show_array_xy
    isRelative: NO];
}

- (void)DPSyshow: (const char *)s : (const float *)numarray : (int)size 
{
  [self _showString: s
    xCharAdj: 0 yCharAdj: 0 char: 0 adjArray: numarray arrType: show_array_y
    isRelative: NO];
}

@end

@implementation XGGState (PathOps)

- (void)DPSarc: (float)x : (float)y : (float)r : (float)angle1 : (float)angle2 
{
}

- (void)DPSarcn: (float)x : (float)y : (float)r : (float)angle1 : (float)angle2 
{
}

- (void)DPSarct: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)r 
{
}

- (void)DPSarcto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)r : (float *)xt1 : (float *)yt1 : (float *)xt2 : (float *)yt2 
{
}

- (void)DPScharpath: (const char *)s : (int)b 
{
}

- (void)DPSclip 
{
  CHECK_PATH;
  PUSH_PATH_OBJECT(path_clip);
  [self _paintPath];
}

- (void)DPSclippath 
{
}

- (void)DPSclosepath 
{
  CHECK_PATH;
  PUSH_PATH_OBJECT(path_closepath);
}

- (void)DPScurveto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)x3 : (float)y3 
{
  NSPoint p;
  double x, y, t, k = .025;

  CHECK_PATH;
  p.x = point.x;
  p.y = point.y;
  for(t = k; t <= 1+k; t += k)
    {
      x = (p.x+t*(-p.x*3+t*(3*p.x-p.x*t)))
        +t*(3*x1+t*(-6*x1+x1*3*t))+t*t*(x2*3-x2*3*t)+x3*t*t*t;
      y = (p.y+t*(-p.y*3+t*(3*p.y-p.y*t)))
        +t*(3*y1+t*(-6*y1+y1*3*t))+t*t*(y2*3-y2*3*t)+y3*t*t*t;
      [self DPSlineto: x : y];
    }
}

- (void)DPSeoclip 
{
  CHECK_PATH;
  PUSH_PATH_OBJECT(path_eoclip);
  [self _paintPath];
}

- (void)DPSeoviewclip 
{
  CHECK_PATH;
  PUSH_PATH_OBJECT(path_eoviewclip);
  [self _paintPath];
}

- (void)DPSflattenpath 
{
}

- (void)DPSinitclip 
{
  if (clipregion)
    XDestroyRegion(clipregion);
  clipregion = 0;
  [self setClipMask];
}

- (void)DPSinitviewclip 
{
  if (context->viewclip)
    XDestroyRegion(context->viewclip);
  context->viewclip = 0;
  [self setClipMask];
}

- (void)DPSlineto: (float)x : (float)y 
{
  NSPoint p;
  p.x = x; p.y = y;
  CHECK_PATH;
  PUSH_PATH_OBJECT_POINT(path_lineto, p);
  point = p;
}

- (void)DPSmoveto: (float)x : (float)y 
{
  NSPoint p;
  p.x = x; p.y = y;
  CHECK_PATH;
  PUSH_PATH_OBJECT_POINT(path_moveto, p);
  point = p;
}

- (void)DPSnewpath 
{
  CHECK_PATH;
  GSIArrayRemoveAllItems((GSIArray)path);
  point = NSMakePoint(0, 0);
}

- (void)DPSpathbbox: (float *)llx : (float *)lly : (float *)urx : (float *)ury 
{
}

- (void)DPSpathforall 
{
}

- (void)DPSrcurveto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)x3 : (float)y3 
{
}

- (void)DPSrectclip: (float)x : (float)y : (float)w : (float)h 
{
  XRectangle    xrect;

  CHECK_GC;

  xrect = XGViewRectToX(self, NSMakeRect(x, y, w, h));
  if (clipregion == 0)
    clipregion = XCreateRegion();
  XIntersectRegion(clipregion, emptyRegion, clipregion);
  XUnionRectWithRegion(&xrect, clipregion, clipregion);
  [self setClipMask];
}

- (void)DPSrectviewclip: (float)x : (float)y : (float)w : (float)h 
{
  XRectangle    xrect;

  CHECK_GC;

  xrect = XGViewRectToX(self, NSMakeRect(x, y, w, h));
  /* Check that we're clipping inside drawable (prevents unwanted
     effects in things like image drawing) */
  NSDebugLLog(@"XGGState", @"X rectviewclip %d %d %d %d",
  	      xrect.x, xrect.y, xrect.width,  xrect.height);
  if (draw)
    {
      NSRect rclip;
      Window root_rtn;
      int x, y;
      unsigned int width, height, b_rtn, d_rtn;
      XGetGeometry(XDPY, draw, &root_rtn, &x, &y, &width, &height,
      		   &b_rtn, &d_rtn);
      rclip = NSIntersectionRect(NSMakeRect(0, 0, width, height),
      			         NSMakeRect(xrect.x, xrect.y, xrect.width,
				 	    xrect.height));
      xrect.x = NSMinX(rclip);
      xrect.y = NSMinY(rclip);
      xrect.width = NSWidth(rclip);
      xrect.height = NSHeight(rclip);
      NSDebugLLog(@"XGGState", @"Clipped X rectviewclip %d %d %d %d",
  	      xrect.x, xrect.y, xrect.width,  xrect.height);
    }

  if (context->viewclip == 0)
    context->viewclip = XCreateRegion();
  XIntersectRegion(context->viewclip, emptyRegion, context->viewclip);
  XUnionRectWithRegion(&xrect, context->viewclip, context->viewclip);
  [self setClipMask];
}

- (void)DPSreversepath 
{
}

- (void)DPSrlineto: (float)x : (float)y 
{
  NSPoint p;
  p.x = x; p.y = y;
  CHECK_PATH;
  PUSH_PATH_OBJECT_POINT(path_rlineto, p);
  point.x += p.x;
  point.y += p.y;
}

- (void)DPSrmoveto: (float)x : (float)y 
{
  NSPoint p;
  p.x = x; p.y = y;
  CHECK_PATH;
  PUSH_PATH_OBJECT_POINT(path_rmoveto, p);
  point.x += p.x;
  point.y += p.y;
}

- (void)DPSsetbbox: (float)llx : (float)lly : (float)urx : (float)ury 
{
}

- (void)DPSsetucacheparams 
{
}

- (void)DPSuappend: (const char *)nums : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSucache 
{
}

- (void)DPSucachestatus 
{
}

- (void)DPSupath: (int)b 
{
}

- (void)DPSviewclip 
{
  CHECK_PATH;
  PUSH_PATH_OBJECT(path_viewclip);
  [self _paintPath];
}

- (void)DPSviewclippath 
{
}

- (void) DPSsetalpha: (float)a
{
  gswindow_device_t *gs_win;
  color.field[AINDEX] = a;
  gs_win = [XGContext _windowWithTag: window];
  if (!gs_win)
    return;
  if (!gs_win->alpha_buffer && a < 1.0)
    gs_win->alpha_buffer = GS_NEW_ALPHA_BUFFER(context, gs_win->depth,
					     NSWidth(gs_win->xframe),
					     NSHeight(gs_win->xframe));
}

- (void) DPScurrentalpha: (float *)alpha
{
  if (alpha)
    *alpha = color.field[AINDEX];
}


@end

