/*
   XGBitmapImageRep.m

   NSBitmapImageRep for GNUstep GUI X/GPS Backend

   Copyright (C) 1996-1999 Free Software Foundation, Inc.

   Author:  Adam Fedor <fedor@colorado.edu>
   Author:  Scott Christley <scottc@net-community.com>
   Date: Feb 1996
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May 1998
   Author:  Richard Frith-Macdonald <richard@brainstorm.co.uk>
   Date: Mar 1999

   This file is part of the GNUstep GUI X/GPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>
#include <stdlib.h>
#include <tiff.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include <gnustep/xgps/XGContext.h>
#include <gnustep/xgps/XGGState.h>
#include <Foundation/NSDebug.h>
#include <Foundation/NSUserDefaults.h>

/*
 * Backend structure for XGBitmapImageRep
 */
typedef struct _XGBitmapImageRep_struct
{
  BOOL		is_cached;
  RXImage	*xImage;
  Pixmap	xOpaquePixmap;

} XGBitmapImageRep_struct;

#define XCACHE (((XGBitmapImageRep_struct *)back_end_reserved)->is_cached)
#define XIMAGE (((XGBitmapImageRep_struct *)back_end_reserved)->xImage)
#define XPIXMAP (((XGBitmapImageRep_struct *)back_end_reserved)->xOpaquePixmap)

#define ALPHA_THRESHOLD 158


@implementation XGBitmapImageRep

static BOOL	alwaysCache = NO;

+ (void) initialize
{
  if (self == [XGBitmapImageRep class])
    {
      NSUserDefaults	*defaults = [NSUserDefaults standardUserDefaults];

      alwaysCache = [defaults boolForKey: @"GSCacheBitmaps"];
    
      [self registerImageRepClass: self];		// Register ourself
    }
}

- (id) initWithBitmapDataPlanes: (unsigned char **)planes
		     pixelsWide: (int)width
		     pixelsHigh: (int)height
		  bitsPerSample: (int)bps
		samplesPerPixel: (int)spp
		       hasAlpha: (BOOL)alpha
		       isPlanar: (BOOL)isPlanar
		 colorSpaceName: (NSString *)colorSpaceName
		    bytesPerRow: (int)rowBytes
		   bitsPerPixel: (int)pixelBits
{
  self = [super initWithBitmapDataPlanes: planes
			      pixelsWide: width
			      pixelsHigh: height
			   bitsPerSample: bps
			 samplesPerPixel: spp
				hasAlpha: alpha
				isPlanar: isPlanar
			  colorSpaceName: colorSpaceName
			     bytesPerRow: rowBytes
			    bitsPerPixel: pixelBits];

  // Allocate the  back-end structure
  back_end_reserved = malloc(sizeof(XGBitmapImageRep_struct));
  XCACHE = NO;
  XIMAGE = 0;
  XPIXMAP = 0;

  return self;
}

- (void) dealloc
{
  XGContext	*ctxt = (XGContext*)[XGContext currentContext];
  RContext	*xContext = [ctxt xrContext];

  if (XIMAGE != 0)			// free ximage cache
    {
      RDestroyXImage(xContext, XIMAGE);
    }
  if (XPIXMAP != 0)			// free pixmap cache
    {
      XFreePixmap(xContext->dpy, XPIXMAP);
    }
  free(back_end_reserved);		// Free the back-end structure

  [super dealloc];
}

/*
 * RGB color image with color values separate
 * Use colorimage postscript operator
 */
- (BOOL) _drawColorSeparate
{
  return YES;
}

/*
 * RGB color image with color values not separate
 * Use colorimage postscript operator
 */
- (BOOL) drawInRect: (NSRect)aRect
{
  if (XCACHE == NO || (hasAlpha == YES && alwaysCache == NO))
    {
      XGContext		*ctxt = (XGContext*)[XGContext currentContext];
      RContext		*xContext = [ctxt xrContext];
      XGGState		*current = [ctxt xrCurrentGState];
      Display		*xDisplay = xContext->dpy;
      int		xscreen = xContext->screen_number;
      Colormap		colorMap = XDefaultColormap(xDisplay, xscreen);
      unsigned char	*bData = (unsigned char*)[self bitmapData];
      Drawable		draw;
      GC		gc;
      int		x, y;
      XColor		c0;
      XColor		c1;
      XRectangle	xr;
      XRectangle	cr;
      XImage		*ximage;
      NSView		*v = [ctxt focusView];
      NSWindow		*w = [v window];
      NSRect            fr = [(XGWindow *)w xFrame];

      DPScurrentgcdrawable(ctxt, (void**)&gc, (void**)&draw, &x, &y);

      /*
       * If we have no drawable, we can't proceed.
       */
      if (draw == None)
	return NO;

      /*
       * Translate to X coordinates -
       */
      if (current->viewIsFlipped)
	aRect.origin.y -= aRect.size.height;
      xr = [current viewRectToX: aRect];

      /*
       * Find the current view clipregion and see if the image is inside it.
       * If it isn't, then we can't neccessarily get our image from the
       * screen, so, as a quick hack, we just guess at the background.
       */
      XClipBox(xContext->viewclip, &cr);

      /* Also, constrain the image to the window frame. 
       * This will prevent X errors due to trying to get an image 
       * which is outside of the window. 
       */
      if (xr.x + size.width > fr.size.width || xr.y + size.height > fr.size.height 
	  || xr.x < 0 || xr.y < 0
	  || cr.y > xr.y || cr.x > xr.x || cr.y + cr.height < xr.y + xr.height
	  || cr.x + cr.width < xr.x + xr.width || [w isVisible] == NO)
	{
	  XIMAGE = RGetXImage(xContext, RootWindow(xDisplay, xscreen),
				0, 0,
				(int)size.width, (int)size.height);
	  if (hasAlpha)
	    {
	      XColor	bg;
	      unsigned	row;

	      if ([v respondsToSelector: @selector(backgroundColor)])
		bg = [(id)[(id)v backgroundColor] xColor];
	      else
		bg = [(id)[w backgroundColor] xColor];

	      for (row = 0; row < size.height; row++)
		{
		  unsigned	col;

		  for (col = 0; col < size.width; col++)
		    {
		      XPutPixel(XIMAGE->image, col, row, bg.pixel);
		    }
		}
	    }
	}
      else
	{
	  /*
	   * Grab a copy of the background behind the image.
	   */
	  XIMAGE = RGetXImage(xContext, draw, xr.x, xr.y,
				(int)size.width, (int)size.height);
	}
      ximage = XIMAGE->image;

	// convert our internal bitmap data into an RXImage
      if (numColors >= 3)
	{
	  unsigned long		pixel;
	  XGDrawMechanism	drawMechanism = [ctxt drawMechanism];
	  unsigned short	oldAlpha = 0;
	  unsigned long		oldPixel = 0;
	  unsigned char		oldAr = 0;
	  unsigned char		oldAg = 0;
	  unsigned char		oldAb = 0;
	  unsigned char		oldBr = 0;
	  unsigned char		oldBg = 0;
	  unsigned char		oldBb = 0;

	  if (drawMechanism == XGDM_FAST16)
	    {
	      unsigned		row;

	      for (row = 0; row < size.height; row++)
		{
		  unsigned	col;

		  for (col = 0; col < size.width; col++)
		    {
		      unsigned char	r = *bData++;
		      unsigned char	g = *bData++;
		      unsigned char	b = *bData++;

		      /*
		       * Convert 8-bit components down to the 5-bit values
		       * that the display system can actually handle.
		       */
		      r >>= 3;
		      g >>= 3;
		      b >>= 3;
		      if (hasAlpha)
			{
			  unsigned short	alpha;

			  alpha = (unsigned short)*bData++;
			  if (alpha == 0)
			    continue;		// background unchanged.

			  alpha >>= 3;		// Convert to 5-bit.

			  if (alpha != 31)
			    {
			      unsigned short	ialpha = 31 - alpha;

			      /*
			       * Get the background pixel and convert to RGB.
			       */
			      pixel = XGetPixel(ximage, col, row);
			      if (pixel != oldPixel)
				{
				  oldPixel = pixel;
				  oldBb = pixel & 0x1f;
				  pixel >>= 6;
				  oldBg = pixel & 0x1f;
				  pixel >>= 5;
				  oldBr = pixel;
				  oldAlpha = 0;
				}
			      if (alpha != oldAlpha)
				{
				  oldAlpha = alpha;
				  oldAr = ialpha * oldBr;
				  oldAg = ialpha * oldBg;
				  oldAb = ialpha * oldBb;
				}

			      // mix in alpha to produce RGB out
			      r = (oldAr + (r*alpha))/31;
			      g = (oldAg + (g*alpha))/31;
			      b = (oldAb + (b*alpha))/31;
			    }
			}
		      pixel = (((r << 5) + g) << 6) + b;
		      XPutPixel(ximage, col, row, pixel);
		    }
		}
	    }
	  else if (drawMechanism == XGDM_FAST32)
	    {
	      unsigned		row;

	      for (row = 0; row < size.height; row++)
		{
		  unsigned	col;

		  for (col = 0; col < size.width; col++)
		    {
		      unsigned char	r = *bData++;
		      unsigned char	g = *bData++;
		      unsigned char	b = *bData++;

		      if (hasAlpha)
			{
			  unsigned short	alpha;

			  alpha = (unsigned short)*bData++;
			  if (alpha == 0)
			    continue;		// background unchanged.

			  if (alpha != 255)
			    {
			      unsigned short	ialpha = 255 - alpha;

			      pixel = XGetPixel(ximage, col, row);
			      if (pixel != oldPixel)
				{
				  oldPixel = pixel;
				  oldBr = (pixel & 0xff0000) >> 16;
				  oldBg = (pixel & 0xff00) >> 8;
				  oldBb = pixel & 0xff;
				  oldAlpha = 0;
				}
			      if (alpha != oldAlpha)
				{
				  oldAlpha = alpha;
				  oldAr = ialpha * oldBr;
				  oldAg = ialpha * oldBg;
				  oldAb = ialpha * oldBb;
				}

			      // mix in alpha to produce RGB out
			      r = (oldAr + (r*alpha))/255;
			      g = (oldAg + (g*alpha))/255;
			      b = (oldAb + (b*alpha))/255;
			    }
			}
		      pixel = (((r << 8) + g) << 8) + b;
		      XPutPixel(ximage, col, row, pixel);
		    }
		}
	    }
	  else
	    {
	      XColor		c2;
	      unsigned		row;

	      /*
	       * This block of code should be totally portable as it uses the
	       * 'official' X mechanism for converting from pixel values to
	       * RGB color values - on the downside, it's very slow.
	       */
	      pixel = (unsigned long)-1;	// Never valid?
	      c2.pixel = pixel;

	      for (row = 0; row < size.height; row++)
		{
		  unsigned	col;

		  for (col = 0; col < size.width; col++)
		    {
		      unsigned char	r = *bData++;
		      unsigned char	g = *bData++;
		      unsigned char	b = *bData++;

		      if (hasAlpha)
			{
			  unsigned short	alpha;

			  alpha = (unsigned short)*bData++;
			  if (alpha == 0)
			    continue;		// background unchanged.

			  if (alpha == 255)
			    {
			      c1.red = r << 8;
			      c1.green = g << 8;
			      c1.blue = b << 8;
			    }
			  else
			    {
			      unsigned short	ialpha = 255 - alpha;

			      /*
			       * RGB color components in X are 16-bit values -
			       * but our bitmap contains 8-bit values so we must
			       * adjust our values up to 16-bit, which we can do
			       * by increasing their alpha component by 256.
			       */
			      alpha <<= 8;

			      /*
			       * Get the background pixel and convert to RGB if
			       * we haven't already done the conversion earlier.
			       */
			      c0.pixel = XGetPixel(ximage, col, row);
			      if (c0.pixel != pixel)
				{
				  pixel = c0.pixel;
				  XQueryColor(xDisplay, colorMap, &c0);
				}

			      // mix in alpha to produce RGB out
			      c1.red = ((c0.red*ialpha) + (r*alpha))/255;
			      c1.green = ((c0.green*ialpha) + (g*alpha))/255;
			      c1.blue = ((c0.blue*ialpha) + (b*alpha))/255;
			    }
			}
		      else
			{
			  /*
			   * Simply convert our 8-bit RGB values to X-style
			   * 16-bit values, then determine the X colormap
			   * entry and set the pixel.
			   */
			  c1.red = r << 8;
			  c1.green = g << 8;
			  c1.blue = b << 8;
			}

		      /*
		       * Convert the X RGB value to a colormap entry if we
		       * don't already have one.  Then set the pixel.
		       * NB.  XAllocColor() will not necessarily return a
		       * color with exactly the rgb components we gave it, so
		       * we must record those components beforehand.
		       */
		      if (c2.pixel == (unsigned long)-1
			|| c2.red != c1.red
			|| c2.green != c1.green
			|| c2.blue != c1.blue)
			{
			  c2.red = c1.red;
			  c2.green = c1.green;
			  c2.blue = c1.blue;
			  XAllocColor(xDisplay, colorMap, &c1);
			  c2.pixel = c1.pixel;
			}
		      XPutPixel(ximage, col, row, c1.pixel);
		    }
		}
	    }
	}
      else
	{					// Grayscale ?
	  unsigned short	*cPtr;
	  unsigned		row;

	  cPtr = &c1.red;
	  for (row = 0; row < size.height; row++)
	    {
	      unsigned	col;

	      for (col = 0; col < size.width; col++)
		{
		  unsigned	i;

		  for (i = 0; i < numColors; i++)
		    {
		      // Create XColor from image data
		      cPtr[i] = (unsigned short)(257 * (char)*bData++);
		      XAllocColor(xDisplay, colorMap, &c1);
		      XPutPixel(ximage, col, row, c1.pixel);
		    }
		}
	    }
	}
      XCACHE = YES;		// This image might be re-used.
    }
  return YES;
}

- (BOOL) draw
{
  NSDebugLog(@"XGBitmapImageRep: draw\n");
  NSDebugLog(@"XGBitmapImageRep: color space %s\n", [_colorSpace cString]);
  NSDebugLog(@"XGBitmapImageRep: size %f %f\n", size.width, size.height);
  NSDebugLog(@"%d %d %d %d\n",bitsPerSample,_pixelsWide,_pixelsHigh,hasAlpha);
  NSDebugLog(@"%d %d %d %d\n",bytesPerRow,numColors,bitsPerPixel,compression);

  if ([_colorSpace compare: NSDeviceRGBColorSpace] == NSOrderedSame)
    {
      if (_isPlanar)
	return [self _drawColorSeparate];
//	else
//	return [self drawColorContig];
    }

  return NO;
}

- (BOOL) drawAtPoint: (NSPoint)aPoint
{
  NSRect	r;

  // FIX ME: do we need to return to orignal transformation?
  NSDebugLog(@"Drawing at point %f %f\n", aPoint.x, aPoint.y);
//  PStranslate(aPoint.x, aPoint.y);

  r.origin = aPoint;
  r.size = size;
  return [self drawInRect: r];
}

- (RXImage *) xImage
{
  return XIMAGE;
}

- (Pixmap) xPixmapMask
{
  if (XPIXMAP == None)
    {
      uint32 j, i;
      unsigned char	ialpha;
      unsigned char	*bData = [(NSBitmapImageRep *)self bitmapData];
      XGContext		*ctxt = (XGContext*)[XGContext currentContext];
      RContext		*xContext = [ctxt xrContext];
      Display		*xDisplay = xContext->dpy;
      XGWindow		*w = (XGWindow*)[(NSView*)[NSView focusView] window];
      Drawable		xDrawable = [w xDrawable];
      int		bitmapSize = (((int)size.width * (int)size.height)/8);
      char		*aData = calloc(1, bitmapSize);
      char		*cData = aData;

      if (numColors == 4)
	{
	  for (j = 0; j < bitmapSize; j++)
	    {
	      for (i = 0; i < 8; i++)
		{
		  bData += 3;
		  ialpha = (unsigned short)((char)*bData++);
		  if (ialpha > ALPHA_THRESHOLD)
		    *cData |= (0x01 << i);
		}
	      cData++;
	    }
	}
      else
	{
	  for (j = 0; j < bitmapSize; j++)
	    *cData++ = 0xff;
	}

      XPIXMAP = XCreatePixmapFromBitmapData(xDisplay,xDrawable,(char *)aData,
	      (int)size.width, (int)size.height, 1L, 0L, 1);
      free(aData);
    }

  return XPIXMAP;
}

- (Class) classForCoder: (NSCoder*)aCoder
{
  if ([self class] == [XGBitmapImageRep class])
    return [super class];
  return [self class];
}

@end
