/* 
   PXKDrawingEngine.m

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Date: March 1996
   
   This file is part of the GNUstep GUI X/DPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>
#include <gnustep/xdps/NSDPSContext.h>
#include <DPS/dpsXshare.h>
#if HAVE_DPS_DPSNXARGS_H
#include <DPS/dpsNXargs.h>
#endif

#include <AppKit/GSWraps.h>
#include "general.h"
#include "PXKDrawingEngine.h"

#define CUR_CONTEXT     ((NSDPSContext*)[NSDPSContext currentContext])

void PXKInitializeDrawingEngine(void);

//
// This functions performs all backend initialization
// that is required before main() is called
//
BOOL
initialize_gnustep_backend(void)
{
  // Have backend classes poseAs their super
  NSDebugLog(@"Using GNUstep GUI X/DPS Backend for backend implementation.\n");

  [NSGraphicsContext setDefaultContextClass: [NSDPSContext class]];
  [PXKScreen poseAsClass:[NSScreen class]];
  [PXKFontManager poseAsClass:[NSFontManager class]];
  [PXKFont poseAsClass:[NSFont class]];
  [PXKApplication poseAsClass:[NSApplication class]];
  [PXKMenuWindow poseAsClass:[NSMenuWindow class]];
  [PXKColor poseAsClass:[NSColor class]];

  // image classes
  [PXKImage poseAsClass: [NSImage class]];
  [PXKBitmapImageRep poseAsClass: [NSBitmapImageRep class]];
  [PXKCachedImageRep poseAsClass: [NSCachedImageRep class]];
  [PXKEPSImageRep poseAsClass: [NSEPSImageRep class]];

  [PXKCursor poseAsClass: [NSCursor class]];
  [PXKWindow poseAsClass:[NSWindow class]];

  // Initialize the XWindows drawing engine
  NSDebugLog(@"Initializing XWindows drawing engine\n");
  PXKInitializeDrawingEngine();

  /* Force font manager to initialize with the list of fonts available on
     system. */
  [NSFontManager sharedFontManager];

  if (GSDebugSet(@"NSDPSContext") == YES)
    {
      NSLog(@"NSDPSContext: Tracing Postscript \n");
      [(NSDPSContext *)[NSDPSContext currentContext] setOutputTraced: YES];
    }

  // success
  return YES;
}

/*
  cacheSystemImages() loads up any commonly used system images so that
  they are immediately available to the application.
  */
void
cacheSystemImages()
{
  [NSImage imageNamed: @"common_ArrowDown.tiff"];
  [NSImage imageNamed: @"common_ArrowLeft.tiff"];
  [NSImage imageNamed: @"common_ArrowRight.tiff"];
  [NSImage imageNamed: @"common_ArrowUp.tiff"];
  [NSImage imageNamed: @"common_Dimple.tiff"];
  [NSImage imageNamed: @"common_RadioOff.tiff"];
  [NSImage imageNamed: @"common_RadioOn.tiff"];
  [NSImage imageNamed: @"common_SwitchOff.tiff"];
  [NSImage imageNamed: @"common_SwitchOn.tiff"];
  [NSImage imageNamed: @"common_ret.tiff"];
}

/*
 * initializeDrawingSystem() is called once from the NSApplication object
 * when the application starts
 */

/* Initialize drawing engine */
void
PXKInitializeDrawingEngine(void)
{
  NSDPSContext *context;
  context = (NSDPSContext *)[NSDPSContext defaultContextWithInfo: nil];
  [NSDPSContext setCurrentContext: context];
}

/* Wraps */
/* Graphics Wraps */
void 
GSWSetMatrix(NSGraphicsContext *ctxt, float m[6])
{
  PSWSetMatrix(m);
}

/* Context helper wraps */
unsigned int 
GSWDefineAsUserObj(NSGraphicsContext *ctxt)
{
  return PSDefineAsUserObj();
}

void 
GSWViewIsFlipped(GSCTXT *ctxt, BOOL flipped)
{
}



/* Private functions, not part of the OpenStep specification */
NSPoint intersectionPoint(NSRect lineA, NSRect lineB)
{
    NSPoint aPoint;
    float r, s;
    float XA = lineA.origin.x;
    float XB = lineA.origin.x + lineA.size.width;
    float XC = lineB.origin.x;
    float XD = lineB.origin.x + lineB.size.width;
    float YA = lineA.origin.y;
    float YB = lineA.origin.y + lineA.size.height;
    float YC = lineB.origin.y;
    float YD = lineB.origin.y + lineB.size.height;
    float denominator = (XB-XA)*(YD-YC)-(YB-YA)*(XD-XC);
    
    /*
      The following code taken from comp.graphics.algorithms FAQ.
      
      This problem can be extremely easy or extremely difficult
      depends on your applications. If all you want is the
      intersection point, the following should work:
      
      Let A,B,C,D be 2-space position vectors. Then the directed
      line segments AB and CD are given by:
	  
	  AB=A+r(B-A), r in [0,1]
	  CD=C+s(D-C), s in [0,1]
		  
          If AB and CD intersect, then

	  A+r(B-A)=C+s(D-C)

          or

	  XA+r(XB-XA)=XC+s(XD-XC)
	  YA+r(YB-YA)=YC+s(YD-YC)  for some r,s in [0,1]
	  
          Solving the above for r and s yields
 
	       (YA-YC)(XD-XC)-(XA-XC)(YD-YC)
          r = -----------------------------  (eqn 1)
	       (XB-XA)(YD-YC)-(YB-YA)(XD-XC)
	  
	       (YA-YC)(XB-XA)-(XA-XC)(YB-YA)
	   s = -----------------------------  (eqn 2)
	       (XB-XA)(YD-YC)-(YB-YA)(XD-XC)
	       
	   */
    
    if ((denominator < 0.00000001) && (denominator > -0.00000001)) {
	fprintf(stderr, "in function intersect(), lines are parallel.\n");
	aPoint.x = -1;
	aPoint.y = -1; /* x and y are always positive in this toolkit */
    }
    
    r = ((YA-YC)*(XD-XC)-(XA-XC)*(YD-YC)) / denominator;
    s = ((YA-YC)*(XB-XA)-(XA-XC)*(YB-YA)) / denominator;
    
         /*
          Let I be the position vector of the intersection point, then

	  I=A+r(B-A)

          or

	  XI=XA+r(XB-XA)
	  YI=YA+r(YB-YA)    

	  */
	
    aPoint.x = XA+r*(XB-XA);
    aPoint.y = YA+r*(YB-YA);
    
    /*
	  By examining the values of r and s, you can also determine
	  some other limiting conditions:
	  
	  If 0 <= r <= 1 and 0 <= s <= 1, intersection exists
	  r < 0 or r>1 or s < 0 or s>1 line segments do not intersect
	  
	  If the denominator in eqn 1 is zero, AB and CD are parallel
          If the numerator in eqn 1 is also zero, AB and CD are
          coincident

          If the intersection point of the 2 lines are needed (lines in
          this context mean infinite lines) regardless whether the two
          line segments intersect, then

          If r > 1, I is located on extension of AB
          If r < 0, I is located on extension of BA
          If s > 1, I is located on extension of CD
          If s < 0, I is located on extension of DC

          Also note that the denominators of equations 1 and 2 are
          identical.

          References:
	  
          [O'Rourke] pp. 249-51
          [Gems3] pp. 199-202 "Faster Line Segment Intersection,"
	  */

    return aPoint;
}

float dist2D(NSPoint a, NSPoint b)
{
    return (a.x-b.x)*(a.x - b.x) / ((a.y-b.y)*(a.y-b.y));
}

NSRect clipLineToRect(NSRect aLine, NSRect aRect)
{
    NSRect resultLine = aLine;
    NSRect clipLine;
    NSPoint aPoint;
    float XA, XB, YA, YB;
    
    XA = aLine.origin.x;
    XB = aLine.origin.x + aLine.size.width;
    YA = aLine.origin.y;
    YB = aLine.origin.y + aLine.size.height;

    /*
     * Set height = width = -1  for lines that are totally
     * outside of the clipping rectangle.
     */

    if (((XA < aRect.origin.x) && (XB < aRect.origin.x)) ||
	((XA > aRect.origin.x + aRect.size.width) &&
	 (XB > aRect.origin.x + aRect.size.width)) ||
	((YA < aRect.origin.y) && (YB < aRect.origin.y)) ||
	((YA > aRect.origin.x + aRect.size.width) &&
	 (YB > aRect.origin.x + aRect.size.width)))
    {
	resultLine.size.width = resultLine.size.height = -1;
	return resultLine;
    }
    
    /* clip left */
    
    if ((XA <= aRect.origin.x) && (XB >= aRect.origin.x)) {
	
	NSDebugLog(@"clip left");
	
	clipLine = aRect;
	clipLine.size.width = 0.0;
	
	aPoint = intersectionPoint(aLine, clipLine);
    
	resultLine.origin.x = aPoint.x;
	resultLine.origin.y = aPoint.y;
	resultLine.size.width = XB - aPoint.x;
	resultLine.size.height = YB - aPoint.y;
    }

    /* clip right */
    
    if ((XA <= aRect.origin.x + aRect.size.width) &&
	(XB >= aRect.origin.x + aRect.size.width))
    {
	NSDebugLog(@"clip right");
	
	clipLine = aRect;
	clipLine.origin.x += clipLine.size.width;
	clipLine.size.width = 0.0;
	
	aPoint = intersectionPoint(aLine, clipLine);
    
	resultLine.origin.x = aPoint.x;
	resultLine.origin.y = aPoint.y;
	resultLine.size.width = XB - aPoint.x;
	resultLine.size.height = YB - aPoint.y;
    }

    /* clip bottom */
    
    if ((YA <= aRect.origin.y + aRect.size.height) &&
	(YB >= aRect.origin.y + aRect.size.height))
    {
	NSDebugLog(@"clip bottom");
	
	clipLine = aRect;
	clipLine.size.height = 0.0;
	
	aPoint = intersectionPoint(aLine, clipLine);
    
	resultLine.origin.x = aPoint.x;
	resultLine.origin.y = aPoint.y;
	resultLine.size.width = XB - aPoint.x;
	resultLine.size.height = YB - aPoint.y;
    }

    /* clip top */
    
    if ((YA <= aRect.origin.y) && (YB >= aRect.origin.y)) {

	NSDebugLog(@"clip top");	
	
	clipLine = aRect;
	clipLine.origin.x += clipLine.size.height;
	clipLine.size.height = 0.0;
	
	aPoint = intersectionPoint(aLine, clipLine);
    
	resultLine.origin.x = aPoint.x;
	resultLine.origin.y = aPoint.y;
	resultLine.size.width = XB - aPoint.x;
	resultLine.size.height = YB - aPoint.y;
    }

    return resultLine;
}

Window _findXWindow(Display* display,
		    Window topwindow,
		    Window window_to_check,
		    int x, int y)
{
  int		newx, newy;
  Window	window;

  // find the lowest X window in the heirarchy that the global x,y
  // coordinates are in.
  if (topwindow == (Window) None) 
    {
      fprintf(stderr,"_findXWindow() error: topwindow == (Window) None \n");

      return (Window) None;
    }

  if (window_to_check == (Window) None)
    {
      fprintf(stderr,
	"_findXWindow() error: window_to_check == (Window) None \n");

      return topwindow;
    }

  // find lowest X window by iterating thru the X window's heirarchy while
  // XTranslateCoordinates returns an X Window via the window ptr

  window = window_to_check;
  while ((XTranslateCoordinates(display, topwindow, window_to_check,
	    x, y, &newx, &newy, &window) != 0)
	    && (window != (Window) None))
    {
      if (window != (Window) None)
	{
	  topwindow = window_to_check;
	  window_to_check = window;
	  x = newx;
	  y = newy;
	}
    }

  if (window == (Window) None)
    window = window_to_check;

  return window;
}
