/* 
   PXKButtonCell.m

   NSButtonCell for GNUstep GUI X/DPS Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: February 1997
   A completely rewritten version of the original source by Scott Christley.
   
   This file is part of the GNUstep GUI X/DPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <stdlib.h>

#include <AppKit/NSColor.h>
#include <AppKit/NSGraphics.h>
#include <gnustep/xdps/PXKCell.h>
#include <gnustep/xdps/PXKButtonCell.h>
#include <gnustep/xdps/PXKView.h>
#include <gnustep/xdps/PXKWindow.h>
#include <gnustep/xdps/PXKDPSContext.h>
#include <gnustep/xdps/PXKImage.h>
#include <gnustep/xdps/PXKFont.h>
#include <gnustep/xdps/PXKDrawingFunctions.h>

@implementation PXKButtonCell

- (void)_currentBackgroundGray:(float*)backgroundGray
  titleGray:(float*)titleGray
{
#define SET_GRAY(pointer, value) \
  if (pointer) *pointer = value

  SET_GRAY (backgroundGray, NSLightGray);
  SET_GRAY (titleGray, NSBlack);

  if ([self state]) {
    unsigned int mask = [self showsStateBy];
    if (mask & NSChangeGrayCellMask) {
      SET_GRAY (backgroundGray, NSWhite);
      SET_GRAY (titleGray, NSLightGray);
    }
    else if (mask & NSChangeBackgroundCellMask) {
      SET_GRAY (backgroundGray, NSWhite);
      SET_GRAY (titleGray, NSBlack);
    }
  }

  if ([self isHighlighted]) {
    unsigned int mask = [self highlightsBy];
    if (mask & NSChangeGrayCellMask) {
      if (backgroundGray && *backgroundGray == NSLightGray) {
	SET_GRAY (backgroundGray, NSWhite);
	SET_GRAY (titleGray, NSLightGray);
      }
    }
    else if (mask & NSChangeBackgroundCellMask) {
      if (backgroundGray && *backgroundGray == NSLightGray)
	SET_GRAY (backgroundGray, NSWhite);
    }
  }

#undef SET_GRAY
}

- (void)drawWithFrame:(NSRect)cellFrame
	       inView:(NSView*)controlView
{
  float backgroundGray;

  /* Don't do anything if the cellFrame is the zero rectangle */
  if (!cellFrame.origin.x && !cellFrame.origin.y
      && !cellFrame.size.width && !cellFrame.size.height)
    return;

  /* Mark the cell's frame rectangle as needing flush */
  [[controlView window] _view: controlView needsFlushInRect: cellFrame];

  [self _currentBackgroundGray:&backgroundGray titleGray:NULL];

  PSgsave();

  /* Clear the cell frame */
  PSsetgray (backgroundGray);
  PSrectfill (cellFrame.origin.x, cellFrame.origin.y,
	      cellFrame.size.width, cellFrame.size.height);

  /* Now draw the border if needed. */
  if ([self isBordered]) {
    if ([self isHighlighted]) {
      unsigned int mask = [self highlightsBy];

      if (mask & NSPushInCellMask) {
	/* If the cell is highlighted use a border with the visual aspect
	    of a pressed button. This is different for toggle buttons.
	    Otherwise the button looks normal. Also compute the rect where
	    text and image get displayed. This takes into account the border's
	    size. */

	if ([self type] == NSToggleButton) {
	  GSDrawPressedToggleButton (cellFrame, cellFrame);
	  cellFrame.origin.x += 2;
	  cellFrame.origin.y += 2;
	  cellFrame.size.width -= 4;
	  cellFrame.size.height -= 4;
	}
	else {
	  GSDrawPressedPushButton (cellFrame, cellFrame);
	  cellFrame.origin.x += 2;
	  cellFrame.origin.y += 1;
	  cellFrame.size.width -= 3;
	  cellFrame.size.height -= 3;
	}
      }
      else {
	NSDrawButton (cellFrame, cellFrame);
	cellFrame.origin.x += 1;
	cellFrame.origin.y += 2;
	cellFrame.size.width -= 3;
	cellFrame.size.height -= 3;
      }
    }
    else {
      NSDrawButton (cellFrame, cellFrame);
      cellFrame.origin.x += 1;
      cellFrame.origin.y += 2;
      cellFrame.size.width -= 3;
      cellFrame.size.height -= 3;
    }
  }

  PSrectclip (cellFrame.origin.x, cellFrame.origin.y,
	      cellFrame.size.width, cellFrame.size.height);
  PStranslate (cellFrame.origin.x, cellFrame.origin.y);
  cellFrame.origin = NSZeroPoint;
  [self drawInteriorWithFrame:cellFrame inView:controlView];
  PSgrestore();
}

- (void)drawInteriorWithFrame:(NSRect)cellFrame inView:(NSView*)controlView
{
  BOOL showAlternate = NO;
  unsigned int mask;
  NSImage* imageToDisplay;
  NSString* titleToDisplay;
  NSCellImagePosition imagePosition;
  NSSize imageSize = {0, 0};
  NSRect rect;
  float titleGray;

  if ([self isHighlighted])
    mask = [self highlightsBy];
  else
    mask = [self showsStateBy];

  /* Determine the image and the title that will be displayed. If the
     NSContentsCellMask is set the image and title are swapped only
     if state is 1 or if highlighting is set (when a button is pushed its
     content is changed to the face of reversed state).
   */
  if (mask & NSContentsCellMask) {
    if ([self isHighlighted])
      showAlternate = ![self state];
    else
      showAlternate = [self state];
  }

  if (showAlternate) {
    imageToDisplay = [self alternateImage];
    titleToDisplay = [self alternateTitle];
    if (!titleToDisplay)
      titleToDisplay = [self title];
  }
  else {
    imageToDisplay = [self image];
    titleToDisplay = [self title];
  }

  if (imageToDisplay)
    imageSize = [imageToDisplay size];
  rect = NSMakeRect (cellFrame.origin.x, cellFrame.origin.y,
		     imageSize.width, imageSize.height);
  [self _currentBackgroundGray:NULL titleGray:&titleGray];

  switch ((imagePosition = [self imagePosition])) {
    case NSNoImage:
      /* Display the title */
      [self _displayTitle:titleToDisplay
	    inFrame:cellFrame
	    titleGray:titleGray];
      break;

    case NSImageOnly:
      /* Image is centered horizontally and vertically. */
      [self _displayImage:imageToDisplay inFrame:cellFrame];
      break;

    case NSImageLeft:
      /* Image is displayed to the left. */
      rect.origin = cellFrame.origin;
      rect.size.width = imageSize.width + 2 * xDist;
      rect.size.height = cellFrame.size.height;
      [self _displayImage:imageToDisplay inFrame:rect];

      /* Display the title */
      rect.origin.x += rect.size.width;
      rect.size.width = cellFrame.size.width - rect.size.width;
      [self _displayTitle:titleToDisplay inFrame:rect titleGray:titleGray];
      break;

    case NSImageRight:
      /* Image is displayed to the right. */
      rect.origin.x += cellFrame.size.width - (2 * xDist + imageSize.width);
      [self _displayImage:imageToDisplay inFrame:rect];

      /* Display the title */
      rect.size.width = rect.origin.x - cellFrame.origin.x;
      rect.origin.x = cellFrame.origin.x;
      [self _displayTitle:titleToDisplay inFrame:rect titleGray:titleGray];
      break;

    case NSImageBelow:
      /* Draw the image at the bottom of cell and the title upper. */
      cellFrame.size.height /= 2;
      [self _displayImage:imageToDisplay inFrame:cellFrame];
      cellFrame.origin.y += cellFrame.size.height;
      [self _displayTitle:titleToDisplay
	    inFrame:cellFrame
	    titleGray:titleGray];
      break;

    case NSImageAbove:
      /* Draw the image in the upper part of cell and the title below. */
      cellFrame.size.height /= 2;
      [self _displayTitle:titleToDisplay
	    inFrame:cellFrame
	    titleGray:titleGray];
      cellFrame.origin.y += cellFrame.size.height;
      [self _displayImage:imageToDisplay inFrame:cellFrame];
      break;

    case NSImageOverlaps:
      /* Simply draw the image followed by the title. */
      [self _displayImage:imageToDisplay inFrame:cellFrame];
      [self _displayTitle:titleToDisplay
	    inFrame:cellFrame
	    titleGray:titleGray];
      break;
  }
}

- (void)highlight:(BOOL)flag
	withFrame:(NSRect)cellFrame
	   inView:(NSView*)controlView
{
  if ([self isHighlighted] == flag)
    return;

  /* Set the highlighting to flag */
  [super highlight:flag withFrame:cellFrame inView:controlView];

  [self drawWithFrame:cellFrame inView:controlView];
}

@end
