// This is -*- C++ -*-
// $Id: Random.h,v 1.2 1999/03/16 18:29:29 alstrup Exp $

/* 
 * Random.h
 *
 * Copyright (C) 1998 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef _INC_RANDOM_H
#define _INC_RANDOM_H

#include <limits.h>
#include <vector>
#include <algorithm>

using namespace std;

const double ONE_OVER_UINT_MAX = 1.0/UINT_MAX;
const unsigned HALF_UINT_MAX = UINT_MAX/2;

class Random {
public:
  Random() : mt_(0), mti_(0) { }
  ~Random() { delete [] mt_; }

  void seed(unsigned);

  unsigned random();
  void random(unsigned*, int);

  // They always say it is bad to look at x % N, where x comes out of a RNG.
  // Is this a problem with the Mersenne Twister?  I dunno.
  // Of course, you introduce some bias (each value isn't equally probable
  // unless N is power of 2), but for N << 2^32 it shouldn't be that
  // much of a problem.  (Famous last words...)
  int random_integer(int a, int b) { return a + (random() % (b-a+1)); }
  void random_integer(int a, int b, int*, int);

  // Picks doubles in the interval [0,1) or [a,b)
  double random_uniform() { return random()*ONE_OVER_UINT_MAX; }
  double random_uniform(double a, double b) { return a+(b-a)*random_uniform();}
  void random_uniform(double*, int);
  void random_uniform(double a, double b, double*, int);

  double random_normal();
  double random_normal(double m, double sd) { return m+sd*random_normal(); }
  void random_normal(double*, int);
  void random_normal(double m, double sd, double*, int);

  bool random_bool() { return random() < HALF_UINT_MAX; }
  bool random_bool(double p_true) { return random_uniform() < p_true; }

  // Things will end badly if operator= doesn't work properly for class X.
  template<class X> void scramble(X* data, size_t N)
  {
    if (N < 2) return;
    for(size_t i=0; i<N-1; ++i) {
      size_t j = random_integer(i, N-1);
      swap(data[i], data[j]);
    }
  }

  template<class X> void scramble(vector<X>& data)
  {
    size_t N = data.size();
    if (N < 2) return;
    for(size_t i=0; i<N-1; ++i) {
      size_t j = random_integer(i, N-1);
      swap(data[i], data[j]);
    }
  }
    
private:
  void stir();

  unsigned* mt_;
  unsigned mti_;

  double extra_normal_;
  bool have_extra_normal_;
};

// Yikes!  Global data!
extern Random common_random;

#endif // _INC_RANDOM_H

// $Id: Random.h,v 1.2 1999/03/16 18:29:29 alstrup Exp $
